PRO ftr,ftrin,ftrout,RRANGE=rrange,NPOINT=npoint
;
;+
; NAME:
;	FTR
;
; PURPOSE:
;	This procedure calculates the Fourier Transform of a set
;
; CATEGORY:
;	XAID xafs data analysis package, 
;
; CALLING SEQUENCE:
;
;	FTR, ftrin, ftrout
;
; INPUTS:
;	ftrin:  a 2 or 3 col set with k,real(chi),imaginary(chi)
;	ftrout:	a 4-column set with the  congugated variable (col 0), 
;	the modulus (col 1), the real part (col 2) and imaginary 
;	part (col 3) of the Fourier Transform
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	RRANGE=[rmin,rmax] : range of the congugated variable for 
;		the transformation (default = [0.,6.])
;	NPOINT= number of points of the congugated variable for the output
;		(default = 512)
;
; OUTPUTS:
;	This function returns a 4-columns array (ftrout) with
;	the congugare variable (R) in column 0, the modulus of the
;	FT in col 1, the real part in col 2 and the imaginary part in
;	col 3.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Uses the standard formulae  F = Fr + i Fi   with
;	Fr = [1/(2pi)] sum[ Dk ( fr cos(2kr) - fi sin(2kr) ) ]
;	Fi = [1/(2pi)] sum[ Dk ( fr sin(2kr) + fi cos(2kr) ) ]
;
; EXAMPLE:
;		FTR,setin,fourier
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;-
on_error,2
;
if not(keyword_set(rrange)) then begin
  rmin=0.
  rmax=6.
endif else begin
  rmin=rrange(0)
  rmax=rrange(1)
endelse
;
if not(keyword_set(npoint)) then npoint=512
;
ncol = n_elements(ftrin(*,0))
tk = ftrin(0,*)
tchi = ftrin(1,*)
if (ncol eq 2) then ichi=tchi*0.0 else ichi=ftrin(2,*)
four=fltarr(npoint)
r=fltarr(npoint)
foui=fltarr(npoint)
j=n_elements(tk)
;
dr = (rmax-rmin)/(npoint-1)                                                 
;
for i=0,npoint-1 do begin
  r_val=dr*i+rmin                                                      
  r(i) =r_val
  ab=shift(tk,-1)-tk
  a=(ab/2.+tk)*2.*r_val
  xr= (shift(tchi,-1) - tchi)/2.+tchi
  yr= (shift(ichi,-1) - ichi)/2.+ichi
  ca=cos(a)                                                               
  sa=sin(a)                                                               
  c=xr*ca-yr*sa 
  s=xr*sa+yr*ca 
  ww1 = c*ab
  ww2 = s*ab
  c1 = total(ww1) -  ww1(j-1)
  s1 = total(ww2) -  ww2(j-1)
  four(i)=c1/sqrt(2.*!pi)
  foui(i)=s1/sqrt(2.*!pi)
endfor
;                                                                               
; prepare the output                                                            
;                                                                               
        rf=four*2. 
        iff=foui*2. 
        mf=sqrt(rf^2+iff^2)                                           
ftrout=fltarr(4,npoint)
ftrout(0,*)=r              ;abscisas axis
ftrout(1,*)=mf             ;modulus
ftrout(2,*)=rf             ;real part
ftrout(3,*)=iff            ;imaginary part
;
;
return
end
