;+
;
;   ===========================  XSh_srcomp   =========================
;
;   Xsrcomp is an widget based graphical interface to calculate
;   the spectrum of a Bending Magnet Synchrotron Radiation source.
;   It interfaces the program SRCOMP, a part of the ray-tracing package SHADOW.
;   It creates an intermediate file SRCOMP which can be used as a source
;   for TRANSMIT. The SRCOMP file is deleted when QUITing Xsrcomp.
;
;   The main window contains four buttons:
;
;	FILE:
;		LOAD INPUT FILE: Reads a srcomp input file.
;		SAVE INPUT DATA AS: Writes a srcomp input data into a file.
;		WRITE FILES FOR XOP/OPTICS: This option permits to
;		  write the files XSH_SRCOMPE and XSH_SRCOMPW to disk. These 
;		  files contain the spectra in #photons/sec/0.1%bw and in 
;		  Watts/eV respectively. They may be used by other applications
;		  as for example, transmit, or for any other purpose.
;		QUIT: to exit from the program
;
;	SET PARAMETERS: to define the parameters for the calculation
;		MACHINE NAME:	just fill the name.
;		B FROM: For selecting B field from either Magnetic
;			radius of the bending magnet or magnetic field.
;		MAGNETIC FIELD: The magnetic field of the bending 
;			magnet in Tesla.
;		MACHINE RADIUS:	The radius of the Bending 
;			Magnet in meters. 
;		BEAM ENERGY:	The electron energy in GeV.
;		(The three precedent parameters are related by:
;			Radius [m] = 3.336 * Energy[GeV] / B[Tesla])
;		BEAM CURRENT:	The electron beam current in Ampers.
;		HORIZ DIV:	The horizontal divergence in mrad.
;		MIN PHOTON ENERGY: 	Minimum photon energy for the 
;					calculated spectrum, in eV.
;		MAX PHOTON ENERGY: 	Maximum photon energy for the 
;					calculated spectrum, in eV.
;		PHOTON ENERGY STEP: 	Step energy for the calculated	
;					spectrum, in eV.
;		VERT DIV:  Vertical divergence of the photon beam
;		accepted (write a big number for considering all the beam).
;		
;	SHOW: Produces a graphic of the Bending Magnet spectrum. The
;		user can select the units. 
;
;	HELP:   Shows this text
;
;
; ----------------------------------------------------------------------
;       Author: M. Sanchez del Rio (srio@esrf.fr) , ESRF, Nov 1993
;       Modification history:
;	20/9/94	F.L add "load input file" option
;	29/9/94	F.L add "save input file as..." option
;	25/10/94 MSR remove common blocks and makes some cosmetics.
;		Reads default input structure from XOP_DEFAULTS.
;       96/01/19 MSR changes labels Xtransmit -> XOP/Optics
;       96/06/26 MSR includes the possibility of setting the magnetic
;		field instead the magnetic radius. Xplot log default graphs.
;       97/01/23 MSR ports to Windows95. Cosmetics. Version 1.2.
;       97/10/01 MSR renames read_ascii by rascii. Get the default
;		structure from shadow_defaults() instead of xop_defaults()
;		Makes use of SDEP(). Considered obsolete XOP application 
;		(substituted by BM).
;       97/10/02 MSR makes use of xop_wd,/default
;       97/12/05 MSR renames to xsh_srcomp and adapts it for being used in 
;		the SHADOW-GUI. Version 1.3.
;       00/29/06 MSR allows passing input parameters as an argument
;		(to be called from xsh_source)
;
;-
;
;========================================================================
;
FUNCTION XSH_SRCOMP_VERSION
return,'1.3'
end
;
;========================================================================
;
PRO XSH_SRCOMP_INP_WRITE,inp,NAME=name
;
; write the contents of the structure with parameters for running srcomp
;
IF NOT(KEYWORD_SET(name)) THEN name='xsh_srcomp_tmp.inp'
;
size_inp = SIZE(inp)
IF (size_inp(0) ne 1) THEN BEGIN
  tmp = xsh_defaults_utils('xsh_srcomp')
  inp = tmp.parameters
ENDIF
;
OPENW,unit,name,/GET_LUN
PRINTF,unit,STRCOMPRESS(inp.Machine_name,/REMOVE_ALL)
PRINTF,unit,'Bending Magnet'

case STRCOMPRESS(inp.RB_choice(0),/REMOVE_ALL)  of
 '0': tmp = inp.Machine_R_m
 '1': tmp = 3.336 * inp.Beam_energy_GeV / inp.BField_T
 else:
endcase
PRINTF,unit,STRCOMPRESS(tmp,/REMOVE_ALL)

PRINTF,unit,STRCOMPRESS(inp.Beam_energy_GeV,/REMOVE_ALL)
PRINTF,unit,STRCOMPRESS(inp.Current_A,/REMOVE_ALL)
PRINTF,unit,STRCOMPRESS(inp.Hor_div_mrad,/REMOVE_ALL)
PRINTF,unit,STRCOMPRESS(inp.Phot_energy_min,/REMOVE_ALL)
PRINTF,unit,STRCOMPRESS(inp.Phot_energy_max,/REMOVE_ALL)
PRINTF,unit,STRCOMPRESS(inp.Phot_energy_step,/REMOVE_ALL)
PRINTF,unit,'3'
PRINTF,unit,'0.1'
PRINTF,unit,'1'
PRINTF,unit,STRCOMPRESS(inp.Vert_div_mrad,/REMOVE_ALL)
FREE_LUN,unit
;
END
;
;======================================================================
;
FUNCTION  XSH_SRCOMP_INP_READ,file,struct
;
; read an Srcomp input file, ans stores the result in struct
;
; the return value is the error flag: MUST BE DEFINED TO -1 BEFORE
;
on_error,2
i_out=-1
openr,Unit,file,/get_lun
;
;	waste values
line0 = string(20)
line1 = 1.0
line2 = 1.0
line3 = 1.0
line4 = 1.0
line5 = 1.0
line6 = 1.0
line7 = 1.0
line8 = 1.0
waste_string = string(20)
;
;	read file and write into the structure
readf,Unit, line0
readf,Unit, waste_string
readf,Unit, line1
readf,Unit, line2
readf,Unit, line3
readf,Unit, line4
readf,Unit, line5
readf,Unit, line6
readf,Unit, line7
readf,Unit, waste_string
readf,Unit, waste_string
readf,Unit, waste_string
readf,Unit, line8
free_lun,Unit
;
struct.Machine_name = line0
struct.Machine_R_m = line1
struct.Beam_energy_Gev = line2
struct.Current_A = line3
struct.Hor_div_mrad = line4
struct.Phot_energy_min = line5
struct.Phot_energy_max = line6
struct.Phot_energy_step = line7
struct.Vert_div_mrad = line8

struct.RB_choice(0) = '0'
struct.BField_T = 3.336 * struct.Beam_energy_Gev / struct.Machine_R_m
;
i_out = 0     ; successful completation of work
return,i_out
end 
;
;==========================================================================
;
PRO xsh_srcomp_event,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('XSH_SRCOMP_EVENT: Error caught: '+$
	!err_string,/Error)
   GOTO,out
ENDIF
;
; register the events
;
Widget_control, event.id, Get_uvalue=eventuval

stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uvalue=state,/No_Copy

CASE eventuval OF
	'QUIT': BEGIN
		case sdep() of
		  'WINDOWS': command = ['del SRCOMP','del MACHINE',$
			'del xsh_srcomp_tmp.inp']
		  'UNIX': command = '/bin/rm -f srcomp SRCOMP MACHINE xsh_srcomp_tmp.inp'
		  else: command = ''
		endcase
		xsh_run,command
		WIDGET_CONTROL,event.top,/DESTROY
		RETURN
		END
	'INPUT_LOAD' : BEGIN
		infile=Dialog_PickFile(GROUP=event.top,/NOCONF)
		if strcompress(infile,/rem) EQ '' then goto,out
		k_err = -1
		Message,/Info,'Loading XSh_Srcomp input file.'
		tmp = state.str.parameters
		k_err = XSH_SRCOMP_INP_READ(infile,tmp)
		IF k_err LT 0 THEN BEGIN
		  itmp = Dialog_Message(DIALOG_PARENT=event.top,/ERROR, $
		  [' Error reading file...'+infile,$
                         ' This seems not an XSh_Srcomp ', $
                         ' input file  '])
		  return
		  GoTo,out
		ENDIF
		state.str.parameters = tmp
		END
	'SETDEF': BEGIN
		itmp = Dialog_Message(DIALOG_PARENT=event.top,$
		/Question,['This option initializes the',$
		'XSh_SrComp parameters to their default values.',$
		'Then you must click Set_parameters to run the program.',$
		'Please confirm.'],title='xsh_srcomp')
		if itmp eq 'No' then goto,out
		tmp = xsh_defaults_utils('xsh_srcomp')
		state.str.parameters = tmp.parameters
		END
	'SETPAR': begin
		inp = state.str.parameters

                if sdep() EQ 'WINDOWS' then helpcmd = "xdisplayfile1,'"+$
                  "\doc\xsh_srcomp.par',group=event.top" else $
                  helpcmd = "xdisplayfile1,'"+$
                  "/doc/xsh_srcomp.par',group=event.top"
		XSCRMENU,inp,GROUP=base,NCOLUMN=2,/NOTYPE,$
		  TITLES=state.str.titles, FLAGS=state.str.flags,  $
 		  ACTION=action, WTITLE='input values for SRComp',/INTERPRET, $
		  HELP=helpcmd
		if action eq 'DONT' then GoTO,out
		state.str.parameters = inp
		PRINT,'############# start running srcomp ####################'
		XSH_SRCOMP_INP_WRITE,inp
		command ='srcomp < xsh_srcomp_tmp.inp' 
		Message,/Info,$
			'--> SRCOMP calculation in process.. please wait ...   '
		xsh_run,command
		if sdep() EQ 'UNIX' then begin
		  command = '/bin/mv  SRCOMP srcomp'
		  message,/info,'Executing: '+command
		  SPAWN,command
		endif
		PRINT,'############# end running srcomp ######################'
		end
	'INPUT_SAVE': BEGIN
		ffile = Dialog_PickFile(/Write,file='xsh_srcomp.inp')
		if strcompress(ffile,/rem) EQ '' then goto,out
		inp = state.str.parameters
		XSH_SRCOMP_INP_WRITE,inp,name=ffile
		itmp = Dialog_Message(DIALOG_PARENT=event.top,$
		/info,'File '+ffile+' written to disk.')
		END
        'HELP': Xhelp,'xsh_srcomp',GROUP=event.top
        'WRITE': BEGIN
		;
		; Prepare file with Power, needed for XOP/Optics
		;
                ffile = findfile('srcomp')
		inp = state.str.parameters
                size_inp = SIZE(inp)
                IF (ffile(0) EQ '' OR  size_inp(1) EQ 0) THEN BEGIN
                  message,/info,'File SRCOMP not found. '
                  itmp = Dialog_Message(DIALOG_PARENT=event.top,$
			/ERROR,['data SRCOMP not found...',$
                        ' Set Parameters before...  '])
		  GoTo,out
                ENDIF
		tmp = rascii('srcomp')
		bpass = 0.1/100.
		radi = inp.Machine_R_m
		gamma = 1957.*inp.Beam_energy_Gev
		lambda = 4.*!pi*radi/3./(gamma^3)
		lambda = lambda * 1.0e10
		yval= tmp(0,*)/(12398.52/lambda)
		coeff_3 = 1.256e10*bpass*gamma
		coeff = 1.641e-13*gamma*lambda/coeff_3
		;
		tmp2 = tmp
		tmp2(1,*) = tmp(1,*)* coeff* tmp2(0,*)/yval
		openw,Unit,'XSH_SRCOMPW',/GET_LUN
		printf,Unit,tmp2
		FREE_LUN,Unit
		if sdep() EQ 'WINDOWS' then command='copy srcomp XSH_SRCOMPE' $
		  else command = '/bin/cp srcomp XSH_SRCOMPE'
		message,/info,'Executing: '+command
		spawn,command
		itmp = Dialog_Message(DIALOG_PARENT=event.top,$
		 /INFO,'Files XSH_SRCOMPE and XSH_SRCOMPW'+$
		 ' written to disk.')
		END
	'SHOW':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval

		IF (checkfile('srcomp') NE 1) THEN BEGIN
		  message,/info,'File SRCOMP not found. '
		  itmp = Dialog_Message(DIALOG_PARENT=event.top,$
			/ERROR,['data SRCOMP not found...',$ 
			' Set Parameters before...  '])
		  goto,out
		ENDIF
		srcomp_data=rascii('srcomp')
		;PLOTFILE,'srcomp',srcomp_data,/NODATA
		xtitle_ev='Photon Energy [eV]'
		xtitle_A ='Photon wavelength [Angs]' 
		srcomp_plot=srcomp_data
  
		bpass = 0.1/100.
		radi = state.str.parameters.Machine_R_m
		gamma = 1957.*state.str.parameters.Beam_energy_Gev
		lambda = 4.*!pi*radi/3./(gamma^3) 
		lambda = lambda * 1.0e10
		yval= srcomp_data(0,*)/(12398.52/lambda)
		coeff_3 = 1.256e10*bpass*gamma
	  	
  
		CASE eventval OF
		  "eV,Phot/sec/eV":    BEGIN
		       ytitle = 'Phot/sec/eV'
		       coeff = 1.013e6*lambda*gamma/coeff_3
		       srcomp_plot(1,*) = srcomp_data(1,*)* $
			  coeff/yval
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_ev,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END
		  "Angs,Phot/sec/Angs":BEGIN
		       ytitle = 'Phot/sec/Angs'
		       coeff = 2.998e-1*gamma^4/radi/coeff_3
		       srcomp_plot(0,*) = $
			  12398.52/srcomp_data(0,*)
		       srcomp_plot(1,*) = srcomp_data(1,*) $
			  *coeff*yval
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_A,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END
		  "Angs,Phot/sec/0.1%bw":BEGIN
		       ytitle = 'Phot/sec/0.1%bw'
		       srcomp_plot(0,*) = $
			  12398.52/srcomp_data(0,*)
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_A,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END
		  "eV,Phot/sec/0.1%bw":BEGIN
		       ytitle = 'Phot/sec/0.1%bw'
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_ev,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END		
		  "Angs,Watts/Angs":	     BEGIN
		       ytitle = 'Watts/Angs'
		       coeff = 1.421e-26*gamma^7/radi^2/coeff_3
		       srcomp_plot(1,*) = srcomp_data(1,*)* $
			  coeff*yval^2
		       srcomp_plot(0,*) = $
			  12398.52/srcomp_data(0,*)
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_A,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END
		  "eV,Watts/eV":	     BEGIN
		       ytitle = 'Watts/eV'
		       coeff = 1.641e-13*gamma*lambda/coeff_3
		       srcomp_plot(1,*) = srcomp_data(1,*)* $
			  coeff* srcomp_plot(0,*)/yval
		       XPLOT,srcomp_plot,GROUP=event.top, $
		       xtitle=xtitle_ev,ytitle=ytitle, title=title, $
		       /xlog,/ylog
		       END
		  "Parameters": XDISPLAYFILE1,'MACHINE',GROUP=event.top
		    ENDCASE
		  END
  ENDCASE

out:
IF N_Elements(state) GT 0 THEN $
  IF Widget_Info(stateid,/Valid_ID) THEN $
    Widget_Control, stateid, Set_Uvalue=state, /No_Copy

END
;
;==============================================================================
;
PRO XSH_SRCOMP, str, GROUP=group
on_error,2
;
; create the base widget
;
if xregistered('xsh_srcomp') GE 1 then return

;
; widget definition
;

base=WIDGET_BASE(/COLUMN,TITLE='Xsh_srcomp '+xsh_srcomp_version(),$
	MBAR=wMenuBar)

wButtons = widget_base(base,/Column) ; also to store state

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp = widget_button(wFile,VALUE='Load XSh_SRComp input file...', $
		UVALUE='INPUT_LOAD')
  wtmp = widget_button(wFile,VALUE='Save input data as...', $
		UVALUE='INPUT_SAVE')
  wtmp = widget_button(wFile,VALUE='Write Files for Transmit', $
		UVALUE='WRITE')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SETPAR')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  showtitles = ["eV,Phot/sec/eV","Angs,Phot/sec/Angs","Angs,Phot/sec/0.1%bw",$
    "eV,Phot/sec/0.1%bw","Angs,Watts/Angs","eV,Watts/eV","Parameters"]
  for i=0,n_elements(showtitles)-1 do $
    wtmp = widget_button(wResults,VALUE=showtitles(i), UVALUE='SHOW')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='xsh_srcomp', UVALUE='HELP')
;
wtmp = widget_button(BASE,VALUE='Set Parameters', UVALUE='SETPAR')

if sdep() EQ 'WINDOWS' then font = 'VERDANA*BOLD*ITALIC*24' else $
  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

wtmp = WIDGET_LABEL( BASE, FONT=font, VALUE='SRCOMP')
wtmp = WIDGET_LABEL( BASE, FONT=font, VALUE='  Bending Magnet Spectrum  ')
wtmp = WIDGET_LABEL( BASE, FONT=font, VALUE='(SHADOW utilities)')


;
;state
;
IF N_Elements(str) EQ 0 THEN str = xsh_defaults_utils('xsh_srcomp')
wids = {dummy:0L}
state = { str:str, wids:wids}
;
;actions
;
Widget_control,Widget_Info(base,/Child),Set_uvalue=state,/No_Copy
Widget_control,base,/Realize
Xmanager,'xsh_srcomp',base,Group_leader=group,/no_block
end
