pro PLOTXY,shadow_in,col1,col2,shadow_out, GROUP=group,_EXTRA=extra,$
XRANGE=xran,YRANGE=yran, XYRANGE=xyran, $
NFILES=NFILES,MULFAC=mulfac, WEIGHT=weight,xplot=xplot, $
NOLOST=nolost, RETRACE=retrace1,POSITION=position, $
CHARSIZE=CHARSIZE, CART = cart, PUBLISH = publish, EXTRACT=extract, $
INFOSH=INFOSH, NOERASE = NOERASE , $
TITLE=title,SLIT=slit, NOCOMM=nocomm, $
HISTO=histo, NBINS=nbins, $
CONTOUR = contour,NLEVELS=nlevels, SMOOTH=SMOOTH, $
CARRAY=CARRAY, $
CCOL = ccol, CLIMITS=climits, CBINS=cbins, CREF=cref, CFILE=cfile,   $
C_DATA=c_data,NC_BINS=nc_bins, $
GAUSSFIT=gaussfit, FWHM_FIT=fwhm_fit, $
NSPLIT=nsplit,int_val=int_val,int_err=int_err, $
CALFWHM=calfwhm,FWHM_VAL=fwhm_val, FWHM_ERROR=FWHM_ERROR,FWHM_MEAN=FWHM_MEAN,$
toph=toph,topv=topv
;+
; NAME:
;	PLOTXY
; PURPOSE:
;	an IDL implementation of the PLOTXY SHADOW's utility
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	plotxy,shadow_in,col1,col2,shadow_out,keyword_parameters
; INPUTS:
;	Number of input variables can be from one to four:
;	When using one input, it should be an array of points with two
;	  columns (i.e. dblarr(2,npoints). The first column contains X
;	  coordinates and the second column the Y values.
;	When using two inputs, the first one is the array for X and the
;	  second one for the Y.
;	With three inputs [usual case]:
;	shadow_in   may be:
;           a shadow file (between quotes) (e.g. 'begin.dat') or
;           an IDL-SHADOW structure
;	col1    x column to plot
;	col2    y column to plot
;           the colum meaning is the following:
;           1   X spatial coordinate [user's unit]
;           2   Y spatial coordinate [user's unit]
;           3   Z spatial coordinate [user's unit]
;           4   X' direction or divergence [rads]
;           5   Y' direction or divergence [rads]
;           6   Z' direction or divergence [rads]
;           7   X component of the electromagnetic vector (s-polariz)
;           8   Y component of the electromagnetic vector (s-polariz)
;           9   Z component of the electromagnetic vector (s-polariz)
;          10   Lost ray flag
;          11   Energy [eV]
;          12   Ray index
;          13   Optical path length
;          14   Phase (s-polarization)
;          15   Phase (p-polarization)
;          16   X component of the electromagnetic vector (p-polariz)
;          17   Y component of the electromagnetic vector (p-polariz)
;          18   Z component of the electromagnetic vector (p-polariz)
;          19   Wavelength [A]
;          20   R= SQRT(X^2+Y^2+Z^2)
;          21   angle from Y axis
;          22   the magnituse of the Electromagnetic vector
;          23   |E|^2 (total intensity)
;          24   total intensity for s-polarization
;          25   total intensity for p-polarization
;          26   K = 2 pi / lambda [A^-1]
;          27   K = 2 pi / lambda * col4 [A^-1]
;          28   K = 2 pi / lambda * col5 [A^-1]
;          29   K = 2 pi / lambda * col6 [A^-1]
;          30   S0-stokes = |Es|^2 + |Ep|^2
;          31   S1-stokes = |Es|^2 - |Ep|^2
;          32   S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p)
;          33   S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p)
;	When four inputs are given, the fourth is optional:
; OPTIONAL INPUT PARAMETERS:
;        shadow_out = name of IDL-SHADOW structure where store the file
; KEYWORD PARAMETERS (INPUT):
;        XRANGE = [ min_x,max_x ]
;        YRANGE = [ min_y,max_y ]
;        XYRANGE = [ min,max ] (sets XRANGE=YRANGE)
;        NOLOST = consider losses
;            0 all rays (default option)
;            1 exclude losses;
;            2 only losses;
;        NBINS = number of bins (def=25);
;        CART = boolean when if present and non-zero
;               forces the data space to have cartesian coordinates
;               (i.e. square has aspect ratio one)
;        NOCOMM =  0 with comment section (default)
;                  1 without comment section 
;        HISTO  = -1 skip histograms
;                  0 with histograms (default)
;                  1 overplot histogram weigthed with intensity
;                  2 plot only histograms weigthed with intensity
;		   3 (same as -1)
;        TITLE  = chain for title	
;        SLIT   = [ lb_x,lb_y,ru_x,ru_y ] where 
;                     lb_x and lb_y  are the coordinates of the
;                     left bottom vertex of the slit and
;                     ru_x and ru_y are the coordinates of the 
;                     right upper vertex of the slit
;		IF lb_x=ru_x and lb_y=lb_y and GETFWHM keyword is set,
;               then the overploted slit is lb_x times the horizontal 
;               fwhm and lb_y times the vertical one. 
;	GAUSSFIT = makes a gaussian fit of the histo
;		0  No (default)
;		1  Gauss+2nd degree polynomial (using gauss_fit routine)
;		2  Gaussian: using (fit_voigt(...,/Gaussian) routine)
;		3  PseudoVoigt: using (fit_voigt() routine)
;		4  Lorentzian: using (fit_voigt(...,/Lorentzian) routine)
;                       publication, i.e. username, date, time, filepath, etc.
;                       When flag is present and non-zero, output is supressed.
;        PUBLISH = flag to supress graphic output not suitable for
;        CONTOUR =  0 (default) no contour (scattered plot)
;		    1  makes contour curves (no reflectivity considered)
;		    2  makes contour curvex including reflectivity
;		    -1 or -2 same as 1 and 2 (respectively) but 
;			using colored polycontours.
;		    3 (same as -1)
;		    4 (same as -2)
;		    5 Pixelized contour (no reflectivity considered)
;		    6 Pixelized contour including reflectivity
;		    Nota Bene: when using contours, any keyword accepted
;		    by the IDL "contour" routine is accepted by plotxy and 
;		    passed to contour.
;
;	 CALFWHM  = 1 : calculates the fwhm values of the histograms
;	            2 : calculates the fwhm values of the histograms and 
;			overplots a slit with these values.
;	 INFOSH = when set to 1, calls also the INFOSH procedure.
;	 NOERASE = when set to 1, noerase the previous plot. Many plotxy
;		plots can be combined in one page by using the 
;		!p.position value and the /noerase keyword.
;	 CHARSIZE = starting character dimension
;	 MULFAC = A multiplicative factor to change the X and Y scales.
;		Set to a scalar for applying both H and V scales, or
;		to [Mx,My] to apply independently H and V scales.
;        GROUP The parent id for the caller widget (used to positioning
;               error and information windows).
;	RETRACE Set this keyword to a distance for retracing (in vacuum) 
;		the shadow file before to be plotted. Only applied if
;		the input is a shadow file or structure.
;	POSITION A four element arrays [x0,y0,x1,y1] defining the 
;		position of the plot (as in !p.position)
;	WEIGTH An array of weight (if inputs are arrays)
;	XPLOT Set this keyword to -1 for launching an xplot window with
;             the plotted points. Set this keyword to the xplot window
;             identifier to send data to an existing xplot window. 
;       NSPLIT, set this keyword to an integer > 1 to calculate errors
;             in intensity value and fwhm by splitting the ray block in
;              NSPLIT groups. 
;
; CONTOUR KEYWORD PARAMETERS:
;        SMOOTH =  to smooth the contour plot graphic. It smooths the
;		3D histogram with the data to be display by using the
;		IDL's smooth founction with width equal to the smooth value.
;	 NC_BINS the number of bins used for calculating the 2D histogram
;		in the case of CONTOUR (Default: NC_BINS=NBINS). 
;		It may be a scalar (used for both X and Y) or a 2-elements 
;		array [Nx,Ny].
;	 NLEVELS = When contour is selected, nlevels indicates the
;		  number of contour levels (Default = 6)
;
; COLOR KEYWORD PARAMETERS:
;        CCOL = Column by which to color code scatter plot points.
;               If this parameter is present and non-zero, each point
;               will be plotted with a color which depends on the
;               value in the column specified and the current color
;               table.  If the parameter is an integer, it is interpreted
;               as the column number of the data to use for color information.
;               If the parameter is an array, it is interpreted as the actual
;               data by which to color the rays.
;        CARRAY = An array with the numerical values of the for the
;		color column (used insted the CCOL xolumn in the file). When 
;		this keyword is set, the CCOL keyword is irrelevant.
;        CLIMITS = [min,max], the limits applied to the values in the
;               color information column, the default is to use the
;               range of the data in that column.
;        CBINS   =   number of bins for the color histogram (defaut=100)
;        CREF    =   when set to 1 takes into account reflectivity
;                    in the color histogram (Default=0, no reflectivity)
;        CFILE   =   name of the shadow file or IDL-SHADOW structure
;                    from where the color column is read. To be set when
;                    the file is different from the shadow_in.
;
; KEYWORD PARAMETERS (OUTPUT):
; 	 FWHM_VAL = IF CALFWHM is set, fwhm_val stores the FWHM value of 
;		  the histogram in an IDL variable dblarr(2) with the 
;                 values for the horizontal and vertical histograms 
;                 respectively.
; 	 FWHM_MEAN = IF CALFWHM and NSPLIT are set, fwhm_mean stores 
;                 the FWHM mean value (one standard deviation) of 
;		  the histogram in an IDL variable dblarr(2).
; 	 FWHM_ERROR = IF CALFWHM and NSPLIT are set, fwhm_error stores 
;                 the FWHM error value (one standard deviation) of 
;		  the histogram in an IDL variable dblarr(2).
;	 FWHM_FIT = IF CALFWHM is set, fwhm_fit stores the FWHM value of 
;                 the gauss fit in an IDL varianle
;        INT_VAL  = The value of the intensity
;        INT_ERR  = IF CALFWHM is set,  int_err stores the value of 
;                   the error in intensity (1-sigma)
;	 EXTRACT = extract stores the data plotted (scattered points) in
;	           an IDL variable ftarr(2,n_points_good)
;	 C_DATA = If CONTOUR is set, set this keyword to a named variable 
;               to retrieve the 2D histogram used for contours.
;		It returns a structure {x:x, y:y, z:z} containing
;		the surface (z tag) and two arrays with the x and y arrays.
;               This option permits you the to reuse of the contours
;		generated for further manipulation, e.g. 
;		IDL> contour,c_data.z,c_data.x,c_data.y
;	TOPH: extract the peak of the horizontal histogram fit
;	TOPV: extract the peak of the vertical histogram fit
;
; OUTPUTS:
;	a graphic. It may also create an IDL-SHADOW structure (optional)
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	It may be conflict between CART axis and CONTOUR. To check!!
; PROCEDURE:
;	Similar to SHADOW's original.
; KNOWN BUGS:
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, May 1991
;       91-12-11 MSR includes SLIT, GAUSSFIT and XYRANGE options.
;       92-07-10 MSR includes INPUT_TYPE option
;       92-07-14 S. Turner apply WHERE routine to eliminate loop
;                for plotting only good rays.  Also change position parameters
;                to put axes labels back on the plot.
;       7-15-92  SWT adds color to plot, ccol selects colorization,
;                and indicates which column to use.
;       7-16-92  SWT automates the selection of the input type,
;                parameter INPUT_TYPE is no longer needed.
;       7-16-92  SWT includes an option to have cartesian coords for plot.
;       7-20-92  SWT provides option of color-coding points from
;                a different ray file, also makes font size for annotation
;                scale with plot size for all devices, notably 'ps',
;                which previously did not work.
;       92-09-02 MSR includes columns > 18
;       92-09-03 MSR includes CREF, CFILE, and few modifications
;       92-10-02 MSR adds GETPHASE to correct phases for stokes parameters
;       92-10-02 MSR adds option 2 to NOLOST
;       92-12-04 MSR adds option 2 to HISTO
;       93-04-05 MSR adds NLEVELS, and color contours plots
;       93-04-19 MSR changes GAUSSFIT by GAUSS_FIT
;       93-05-14 MSR allows either GAUSS_FIT or GAUSSFIT
;       93-07-08 MSR includes printing the intensity and command
;	93-10-21 MSR adds CALFWHM,FWHM_FIT and FWHM_VAL keywords to 
;		 extract values from histograms.
;	93-10-26 MSR adds  EXTRACT keyword
;	93-11-03 MSR fix a bug in colored contours
;	94-09-27 MSR makes to print the FWHM in the screen when /CALFWHM is set
;	95-02-10 MSR adds the infosh keyword
;	95-06-09 MSR clean-up by using getshcol function. Adds /noerase
;		 and allow multiple plots in one page by using the
;		 !p.position system variable. Cosmetics.
;	95-06-14 MSR for contours: removes the /spline and add smooth kw.
;		Short code by using histosh... Other small changes.
;	95-09-25 MSR  fix bug in /cart option. Adds CHARSIZE kw.
;	96-01-30 MSR  fix bug in number of lost rays when /nolost is not used.
;		      Fix bug in limits when contour is set.
;	96-05-03 MSR  Add keyword NFILES to be passed to READSH
;	96-08-20 MSR  Add keyword MULFAC
;	97-12-05 srio@esrf.fr duplicates the negative keyword values (to
;			be used with the interface xsh_plotxy) and uses catch.
;	98-01-07 srio@esrf.fr adds the possibility to input two arrays
;			and adds CARRAY keyword.
;	98-03-10 srio@esrf.fr adds Dialog_Message in catch section.
;	98-04-07 srio@esrf.fr uses column_name do display it.  Cosmetics.
;	98-04-07 srio@esrf.fr adds _EXTRA and C_DATA keywords
;	98-07-10 srio@esrf.fr adds NC_BINS kw.
;	00-03-27 srio@esrf.fr changed !d.n_colors -> (!d.n_colors<255)
;	06-02-16 srio@esrf.fr added the RETRACE keyword
;	06-02-21 srio@esrf.fr added pseudovoigt fit to histograms. 
;		Rearranges mulfac
;	16-10-06 srio@esrf.fr changes fitting routines. Added lorentzian
;			Added weight kw. 
;	11-06-07 srio@esrf.eu Added pixelized contours. 
;	25-06-07 srio@esrf.eu Makes 2D histograms using hist_nd2
;	15-04-08 srio@esrf.eu Adjust comments box... Added xplot kw.
;	2008-09-09 srio@esrf.eu sort the code for making the main plot
;		at the end, for allowing cursor display and zoom.
;	2009-01-12 srio@esrf.eu added NSPLIT keyword to make statistics.
;	2009-03-03 srio@esrf.eu added CALFWHM=2 and calculate intensity
;		in slits.
;	2010-12-03 srio@esrf.eu added TOPV,TOPH kw
; 
;-
on_error,2
catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error, $
	'PLOTXY: Error caught: '+!err_string)
   if n_elements(oldposition) NE 0 then goto,out else return
endif

if n_params() LT 3 then begin
  message,/info,'Using arrays...'
  case n_params() of
  2: begin
       arr1 =  shadow_in
       arr2 =  col1
     end
  1: begin
       arr1 =  shadow_in(0,*)
       arr2 =  shadow_in(1,*)
     end
  else: begin
     message,/info,'Error with inputs.'
     return
     end
  endcase
  input_type = 1
  publish = 1
  nocomm = 1
endif else input_type = 0

intSlit=-1 ; to store intensity in slit (-1=no slit)
xxl=0 & xxr=0 ; fwhm extrema for horizontal histogram
yyl=0 & yyr=0 ; fwhm extrema for vertical histogram
IF N_Elements(NSPLIT) EQ 0 THEN NSPLIT=0
IF NSPLIT EQ 1 THEN NSPLIT=0 ; nothing to split
;
; save system variables that will be modified
;
oldposition = !p.position
oldcharsize = !p.charsize
oldxstyle = !x.style
oldystyle = !y.style
;
if not(keyword_set(noerase)) then erase
;
; sets coordinate origins...
;
IF N_Elements(position) EQ 4 THEN BEGIN
  xx0 = position[0]
  yy0 = position[1]
  xx1 = position[2]
  yy1 = position[3]

ENDIF ELSE BEGIN
  xx0 = 0 ; !p.position(0)
  yy0 = 0 ; !p.position(1)
  xx1 = 0 ; !p.position(2)
  yy1 = 0 ; !p.position(3)
ENDELSE
if xx0 EQ xx1 then begin
  xx0 = 0.0  &  xx1 = 1.0
endif
if yy0 EQ yy1 then begin
  xx0 = 0.0  &  yy1 = 1.0
endif
xx = abs(xx1-xx0)
yy = abs(yy1-yy0)
;
; load shadow-idl structure and define  arrays and constants
;
if input_type EQ 0 then begin
  str = readsh(shadow_in,nfiles=nfiles,GROUP=group)
  ;if not(keyword_set(str)) then return
  if type(str) NE 8 then message,'Bad input.'
  IF Keyword_set(retrace1) THEN str=Retrace(str,DIST=retrace1)
  shadow_out = str
endif
;
; set character size to fraction of plot width, independent of output dev.
;
if not(keyword_set(charsize)) then begin
  bfy = double(!d.y_ch_size)/!d.y_size
  bfx = double(!d.x_ch_size)/!d.x_size
  xsize = 0.0065/bfx
  ysize = 0.018/bfy
  !p.charsize = min([xsize,ysize])  * xx
endif else !p.charsize = charsize
;
publish = keyword_set(publish)
;
!x.style = 1
!y.style = 1
;
;
;
if input_type EQ 0 then begin
  if (keyword_set(ccol)) then begin
    if keyword_set(cfile) then cstr = readsh(cfile,GROUP=group) else cstr = str
    carr = getshcol(cstr,ccol,nolost=nolost)
  endif
endif else begin
  if (keyword_set(carray)) then begin
    carr = carray
    ccol=1
  endif
endelse

if keyword_set(contour) then $
  if not(keyword_set(nlevels)) then nlevels=6
;
; setup the arrays
;
if input_type EQ 0 then begin
  arr1 = getshcol(str,col1,nolost=nolost)
  arr2 = getshcol(str,col2,nolost=nolost)
endif
if Keyword_Set(mulfac) then begin
  IF N_Elements(mulfac) EQ 1 THEN BEGIN
	mulfac1=[mulfac,mulfac] 
	mulfactxt = 'mulfac='+StrCompress(mulfac,/Rem)
  ENDIF ELSE BEGIN
	mulfac1=mulfac
	mulfactxt = 'mulfac='+Vect2String(mulfac)
  ENDELSE
  arr1 = arr1*double(mulfac1[0])
  arr2 = arr2*double(mulfac1[1])
endif else begin
  mulfac1=[1D0,1D0]
endelse

if input_type EQ 0 then wi = getshcol(str,23,nolost=nolost) else $
  wi = arr1*0 + 1
IF Keyword_Set(weight) THEN wi = weight
w = arr1*0 + 1
;
;   defaults
;
nbins_default = 25
if not(keyword_set(nbins)) then  nbins = nbins_default
if keyword_set(nocomm) then  nocomm = 1 else nocomm = 0
if not(keyword_set(histo)) then  histo = 0
if not(keyword_set(title)) then  title = ''
;
; variable to extract
;
extract = dblarr(2,n_elements(arr1))
extract(0,*)=arr1  &  extract(1,*)=arr2

;
; main graphic, prepare
;
!p.position = [xx0,yy0,xx0,yy0] + [0.025*xx,0.030*yy,0.5247*xx,0.6855*yy]
if histo eq 3 then histo=-1
if keyword_set(contour) then begin
  if contour EQ 3 then contour=-1
  if contour EQ 4 then contour=-2
endif
if histo eq -1 then $
  !p.position = [xx0,yy0,xx0,yy0] + [0.025*xx,0.030*yy,0.6957*yy,0.904*yy]
;
off1 = 0.1D0*abs( max(arr1)-min(arr1) )
off2 = 0.1D0*abs( max(arr2)-min(arr2) )

if off1 lt 1D-10 then off1=0.5*max(arr1)
if off2 lt 1D-10 then off2=0.5*max(arr2)

;
; plot limits
;
if keyword_set(xyran) then begin
  xran = xyran
  yran = xyran
endif

if keyword_set(xran) then begin
  IF xran[0] NE xran[1] THEN xrange = xran
endif
  
IF N_Elements(xrange) NE 2 THEN BEGIN
  xrange =[min(arr1)-off1,max(arr1)+off1]
  IF xrange[0] NE xrange[1] THEN xrange = round1( xrange,fix(alog(xrange(1)-xrange(0))/alog(10))-2)
ENDIF
IF abs(xrange[0] -xrange[1]) LT 1D-10 THEN xrange=xrange[0]+[-.1,.1]

if keyword_set(yran) then begin
  IF yran[0] NE yran[1] THEN yrange = yran
endif 

IF N_Elements(yrange) NE 2 THEN BEGIN
  yrange =[min(arr2)-off2,max(arr2)+off2]
  IF yrange[0] NE yrange[1] THEN yrange = round1( yrange,fix(alog(yrange(1)-yrange(0))/alog(10))-2)
ENDIF
IF abs(yrange[0]-yrange[1]) LT 1D-10 THEN yrange=[-.1,.1]

if keyword_set(cart) then begin
    cart_axes,xrange,yrange,position = !p.position, noerase=noerase
endif

;
;   color histogram
;
cbins_default = 100
if keyword_set(ccol) then begin
    if not(keyword_set(climits)) then begin
	cmax = max(carr)
	cmin = min(carr)
        off1 = 0.1*abs( cmax-cmin )
        if off1 lt 1e-10 then off1=0.5*max(arr1)
        cmin = cmin - off1
        cmax = cmax + off1
    endif else begin
	cmax = climits(1)
	cmin = climits(0)
    end

    position = [xx0,yy0,xx0,yy0] + [.5522*xx,.714*yy,0.975*xx,.904*yy] 
    myclimits = [cmin,cmax]
    if not(keyword_set(cbins)) then cbins=cbins_default 
    if keyword_set(cref) then begin
      color_histo,carr,wi,position = position,climits = myclimits, nbins=cbins
    endif else begin
      color_histo,carr,w,position = position,climits = myclimits, nbins=cbins
    endelse
end
if keyword_set(contour) then begin
  if n_elements(contour) EQ 1 then $
  if ((contour lt 0) OR (contour GE 3)) then begin
    toplevel = nlevels
    IF contour GE 3 THEN toplevel=255
    position =  [xx0,yy0,xx0,yy0] + [.5522*xx,.714*yy,0.975*xx,.904*yy] 
    ccont = findgen(topLevel*10)
    cwei = ccont*0.+1.0
    color_histo,ccont,cwei,position = position,climits = [0.,topLevel-1], $
     nbins=topLevel
  endif
endif

;;
;; slit, if any
;;
;if keyword_set(slit) then oplot,$
;  [slit(0),slit(2),slit(2),slit(0),slit(0)], $
;  [slit(1),slit(1),slit(3),slit(3),slit(1)]
;;

if histo ne -1 then begin
;
;	upper histogram
;
if keyword_set(ccol) then fill=1
if keyword_set(contour) then $
if n_elements(contour) EQ 1 then if contour lt 0 then fill=1
!p.position = [xx0,yy0,xx0,yy0] + [0.025*xx,.714*yy,.5247*xx,.904*yy]

if histo EQ 0 then $
   hh = histosh(arr1,nbins=nbins,xrange=xrange,/plot,fill=fill)
if histo EQ 1 then begin
  hh = histosh(arr1,nbins=nbins,xrange=xrange,/plot)
  hh = histosh(arr1,wi,nbins=nbins,xrange=xrange,/plot,fill=fill)
endif
if histo EQ 2 then $
   hh = histosh(arr1,wi,nbins=nbins,xrange=xrange,/plot,fill=fill)


hx = reform(hh(0,*))  &  hy = reform(hh(1,*))

; gaussian fit (if selected)
if keyword_set (gaussfit) then begin
 ahfit = 0
 CASE gaussfit OF
 1: begin
   hfit = gauss_fit (hx,hy,ahfit)
   oplot,hx,hfit
   end
 2: begin
   hfit = fit_voigt (hx,hy,ahfit,/Gaussian)
   hfit2 = voigt1(hx,ahfit)
   oplot,hx,hfit2
   ahfit[2]=ahfit[2]/2.35 ; for compatibility with other formulas
   end
 3: begin
   hfit = fit_voigt (hx,hy,ahfit)
   hfit2 = voigt1(hx,ahfit)
   oplot,hx,hfit2
   ahfit[2]=ahfit[2]/2.35 ; for compatibility with other formulas
   end
 4: begin
   hfit = fit_voigt (hx,hy,ahfit,/Lorentzian)
   hfit2 = voigt1(hx,ahfit)
   oplot,hx,hfit2
   ahfit[2]=ahfit[2]/2.35 ; for compatibility with other formulas
   end
 else:
 ENDCASE
 print,'PLOTXY: Fit on horizontal histogram: '
 print,'PLOTXY:                                       sigma=',ahfit(2)
 print,'PLOTXY:                                       fwhm =',2.35*ahfit(2)
 toph = ahfit(1)
endif
; if selected, calculate fwhm
if keyword_set(calfwhm) then begin
   IF calfwhm EQ 2 THEN BEGIN
      IF N_Elements(slit) EQ 0 THEN slit=[1,1,1,1]
   ENDIF
   fwhm_val = dblarr(2)
   fwhm_mean = dblarr(2)
   fwhm_error = dblarr(2)
   fwhm_val(0) = getfwhm(hh,/zerobase,/plot,xxl=xxl,xxr=xxr)
   print,' '
   print,'PLOTXY: FWHM of the horizontal histogram is :',fwhm_val(0)
   print,'PLOTXY:        FWHM (extrema): ',vect2string([xxl,xxr])
 IF keyword_set(NSPLIT)  THEN BEGIN
   tmpAll = DblArr(nSplit)
   FOR jj=0L,NSplit-1 DO BEGIN
     tmpy = GetShCol(shadow_out,col1,noLost=noLost,Block=[jj,nSplit],verbose=0)
     IF histo EQ 0 THEN tmpw = tmpy*0+1 ELSE $
       tmpw=GetShCol(shadow_out,23,noLost=noLost,Block=[jj,nSplit],verbose=0)
     tmph = histosh(tmpy*mulFac1[0],tmpw,nbins=nbins,xrange=xrange)
     tmpFwhm = getfwhm(tmph,/zerobase)
     tmpAll[jj]=tmpFwhm
   ENDFOR
     tmpM = moment(tmpAll)
     fwhm_mean[0] = tmpM[0]
     fwhm_error[0] = Sqrt(tmpM[1])
     print,'PLOTXY:        nSplit: ',nSplit
     print,'PLOTXY:        FWHM (mean): ',fwhm_mean[0]
     print,'PLOTXY:        FWHM (stDev): ',fwhm_error[0]
 ENDIF 
endif

;
;	right histogram
;
!p.position = [xx0,yy0,xx0,yy0] + [.5522*xx,0.030*yy,.696*xx,.686*yy] 
if histo EQ 0 then $
  hh = histosh(arr2,nbins=nbins,xrange=yrange,/plot,/swap,fill=fill)
if histo EQ 1 then begin
  hh = histosh(arr2,nbins=nbins,xrange=yrange,/plot,/swap)
  hh = histosh(arr2,wi,nbins=nbins,xrange=yrange,/plot,/swap,fill=fill)
endif
if histo EQ 2 then $
  hh = histosh(arr2,wi,nbins=nbins,xrange=yrange,/plot,/swap,fill=fill)
hx = reform(hh(0,*))  &  hy = reform(hh(1,*))
if keyword_set (gaussfit) then begin
 avfit = 0
 CASE gaussfit OF
 1: begin
   vfit = gauss_fit (hx,hy,avfit)
   oplot,vfit,hx  
   end
 2: begin
   vfit = fit_voigt (hx,hy,avfit,/Gaussian)
   vfit2 = voigt1(hx,avfit)
   oplot,vfit2,hx  
   avfit[2]=avfit[2]/2.35 ; for compatibility with other formulas
   end
 3: begin
   vfit = fit_voigt (hx,hy,avfit)
   vfit2 = voigt1(hx,avfit)
   oplot,vfit2,hx  
   avfit[2]=avfit[2]/2.35 ; for compatibility with other formulas
   end
 4: begin
   vfit = fit_voigt (hx,hy,avfit,/Lorentzian)
   vfit2 = voigt1(hx,avfit)
   oplot,vfit2,hx  
   avfit[2]=avfit[2]/2.35 ; for compatibility with other formulas
   end
 else:
 ENDCASE
 print,'PLOTXY: Fit on vertical histogram: '
 print,'PLOTXY:                                       sigma=',avfit(2)
 print,'PLOTXY:                                       fwhm =',2.35*avfit(2)
 fwhm_fit = [2.35*ahfit(2),2.35*avfit(2)]
 topv = avfit(1)
endif
; if selected, calculate fwhm
if keyword_set(calfwhm) then begin
 fwhm_val(1) = getfwhm(hh,/reverse_plot,/zerobase,xxl=yyl,xxr=yyr)
 print,' '
 print,'PLOTXY: FWHM of the vertical histogram is :',fwhm_val(1)
 print,'PLOTXY:        FWHM (extrema): ',vect2string([yyl,yyr])
 IF keyword_set(NSPLIT)  THEN BEGIN
   FOR jj=0L,NSplit-1 DO BEGIN
     tmpy = GetShCol(shadow_out,col2,noLost=noLost,Block=[jj,nSplit],verbose=0)
     tmph = histosh(tmpy*mulFac1[1],tmpw,nbins=nbins,xrange=yrange)
     tmpFwhm = getfwhm(tmph,/zerobase)
     tmpAll[jj]=tmpFwhm
   ENDFOR
     tmpM = moment(tmpAll)
     fwhm_mean[1] = tmpM[0]
     fwhm_error[1] = Sqrt(tmpM[1])
     print,'PLOTXY:        nSplit: ',nSplit
     print,'PLOTXY:        FWHM (mean): ',fwhm_mean[1]
     print,'PLOTXY:        FWHM (stDev): ',fwhm_error[1]
 ENDIF 
endif
;
endif


csize = !p.charsize * 1.4 
tsize = !p.charsize * 2.0 
user = getenv('USER') ;spawn,'echo $USER',user,/SH
host = getenv('HOST') ;spawn,'echo $HOST',host,/SH
if nocomm eq 0 then begin
  !p.position = [xx0,yy0,xx0,yy0] + [0.7033*xx,0.030*yy,0.975*xx,0.6855*yy]
  plot,[0,1],/noData,/noErase,xrange=[0,1],yrange=[0,1],xticks=0,yticks=0,$
    ticklen=0,xstyle=4,ystyle=4
  plots,0,0
  plots,0,0,/cont
  plots,1,0,/cont
  plots,1,1,/cont
  plots,0,1,/cont
  plots,0,0,/cont

  if not(publish) and sdep() eq 'UNIX' then $
    xyouts,xx0+xx*.72,yy0+yy*.66,/norm,siz=csize,user+'@'+host
  if keyword_set(xran) or keyword_set(yran) or keyword_set(xyran) $
    then begin 
    xyouts,xx0+xx*.72,yy0+yy*.41,/norm,siz=csize,'EXTERNAL LIMITS'
  endif else begin
    xyouts,xx0+xx*.72,yy0+yy*.41,/norm,siz=csize,'INTERNAL LIMITS'
  endelse
  if input_type EQ 0 then begin
    if keyword_set(nolost) then begin
      if nolost eq 1 then xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--GOOD ONLY'
      if nolost eq 2 then xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--ONLY LOSSES'
    endif else begin
      xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--ALL RAYS'
    endelse

    
    ; intensity
    intens,str,int_val,nolost=nolost,nsplit=nsplit,splitError=int_err,$
           verbose=0
    xyouts,xx0+xx*.72,yy0+yy*.30,/norm,siz=.8*csize,'INTENS  = '
    IF Keyword_Set(nsplit) THEN $
       tmp = ' '+STRING("261B)+' '+String(int_err,format='(F10.2)') ELSE $
       tmp=''
    xyouts,xx0+xx*.83,yy0+yy*.30,/norm,siz=.8*csize,$
	StrCompress( string(int_val,format='(F10.2)')+tmp )

    xyouts,xx0+xx*.72,yy0+yy*.28,/norm,siz=.8*csize,'TOT RAYS = '
    xyouts,xx0+xx*.83,yy0+yy*.28,/norm,siz=.8*csize,$
	string(str.npoint,format='(I10)')

    ; lost rays
    xyouts,xx0+xx*.72,yy0+yy*.26,/norm,siz=.8*csize,'LOST = '
    xyouts,xx0+xx*.83,yy0+yy*.26,/norm,siz=.8*csize, $
	string(str.npoint-n_elements(where(getshcol(str,10) > 0)),format=$
	'(I10)')

    ; good rays
    xyouts,xx0+xx*.72,yy0+yy*.24,/norm,siz=.8*csize,'GOOD = '
    xyouts,xx0+xx*.83,yy0+yy*.24,/norm,siz=.8*csize, $
	string(n_elements(where(getshcol(str,10) > 0)),format=$
	'(I10)')

    xyouts,xx0+xx*.72,yy0+yy*.18,/norm,siz=.8*csize,'Horizontal: '
    ;xyouts,xx0+xx*.83,yy0+yy*.18,/norm,siz=.8*csize,col1
    xyouts,xx0+xx*.80,yy0+yy*.18,/norm,siz=.8*csize,column_name(col1)
    xyouts,xx0+xx*.72,yy0+yy*.16,/norm,siz=.8*csize,'Vertical:   '
    ;xyouts,xx0+xx*.83,yy0+yy*.16,/norm,siz=.8*csize,col2
    xyouts,xx0+xx*.80,yy0+yy*.16,/norm,siz=.8*csize,column_name(col2)
  endif
  if keyword_set(ccol) then begin
    xyouts,xx0+xx*.72,yy0+yy*.20,/norm,siz=.8*csize,'Color: '
    ;xyouts,xx0+xx*.83,yy0+yy*.20,/norm,siz=.8*csize,ccol
    xyouts,xx0+xx*.80,yy0+yy*.20,/norm,siz=.8*csize,column_name(ccol)
  endif

  ; fit
  if keyword_set(gaussfit) then begin
    xyouts,xx0+xx*.72,yy0+yy*.10,/norm,siz=.8*csize,'FitHorizFWHM: '
    xyouts,xx0+xx*.83,yy0+yy*.10,/norm,siz=.8*csize,2.35*ahfit(2)
    xyouts,xx0+xx*.72,yy0+yy*.05,/norm,siz=.8*csize,'FitVertFWHM:   '
    xyouts,xx0+xx*.83,yy0+yy*.05,/norm,siz=.8*csize,2.35*avfit(2)
  endif

  ; fwhm
  if keyword_set(calfwhm) then begin
    IF Keyword_Set(nsplit) THEN BEGIN
       tmp=StrCompress('HistoHorizFWHM:'+$
           String(fwhm_val[0],format='(G8.2)')+STRING("261B)+$
           String(fwhm_error[0],format='(G8.2)'),/Rem )
       xyouts,xx0+xx*.72,yy0+yy*.12,/norm,siz=.8*csize,tmp

       tmp=StrCompress('HistoVertFWHM:'+$
           String(fwhm_val[1],format='(G8.2)')+STRING("261B)+$
           String(fwhm_error[1],format='(G8.2)'),/Rem )
       xyouts,xx0+xx*.72,yy0+yy*.07,/norm,siz=.8*csize,tmp

    ENDIF ELSE BEGIN
       xyouts,xx0+xx*.72,yy0+yy*.12,/norm,siz=.8*csize,'HistoHorizFWHM: '
       xyouts,xx0+xx*.83,yy0+yy*.12,/norm,siz=.8*csize,fwhm_val[0]

       xyouts,xx0+xx*.72,yy0+yy*.07,/norm,siz=.8*csize,'HistoVertFWHM:   '
       xyouts,xx0+xx*.83,yy0+yy*.07,/norm,siz=.8*csize,fwhm_val[1]
    ENDELSE
  endif

endif

;
; main graphic, prepare
;
!p.position = [xx0,yy0,xx0,yy0] + [0.025*xx,0.030*yy,0.5247*xx,0.6855*yy]
;if histo eq 3 then histo=-1
;if keyword_set(contour) then begin
;  if contour EQ 3 then contour=-1
;  if contour EQ 4 then contour=-2
;endif
if histo eq -1 then $
  !p.position = [xx0,yy0,xx0,yy0] + [0.025*xx,0.030*yy,0.6957*yy,0.904*yy]


; prepare contours
if (keyword_set(contour)) then begin
  if not(keyword_set(nc_bins)) then nc_bins = nbins
  if N_Elements(nc_bins) EQ 1 THEN nc_bins=Replicate(nc_bins,2)
  ;if keyword_set(p_nbins) then nbins=p_nbins
  ;binsizex =(xrange(1)-xrange(0))/double(nc_bins[0])
  ;binsizey =(yrange(1)-yrange(0))/double(nc_bins[1])
  ;bs = [binsizex,binsizey]
  if n_elements(contour) EQ 1 then begin
    data = Make_Set(Reform(arr1),Reform(arr2))
    if ((contour eq 1) or (contour eq -1) or (contour eq 5)) then $
      hz = hist_nd2(data,bs,wei=w,nbins=nc_bins,$
         min=[xrange[0],yrange[0]],max=[xrange[1],yrange[1]],loc=loc)
    if ((contour eq 2) or (contour eq -2) or (contour eq 6)) then $
      hz = hist_nd2(data,bs,wei=wi,nbins=nc_bins,$
         min=[xrange[0],yrange[0]],max=[xrange[1],yrange[1]],loc=loc)
  endif else begin
      flg = getshcol(shadow_out,10)
      if keyword_set(nolost) then tmp = contour( where(flg > 0)) $
        else tmp = contour
      hz = hist_nd2(data,bs,wei=tmp,nbins=nc_bins,$
         min=[xrange[0],yrange[0]],max=[xrange[1],yrange[1]],loc=loc)
  endelse
  hx = loc[0:nc_bins[0]-1]  
  hy = loc[nc_bins[0]:Total(nc_bins)-1]  
endif

if keyword_set(cart) then $
    ;cart_axes,xrange,yrange,position = !p.position, noerase=noerase
    cart_axes,xrange,yrange,position = !p.position, noerase=1
if N_Elements(smooth) EQ 0 then smooth=0
;
;
; main graphic, plot
;

if keyword_set(contour) then begin
   message,/info,'2D histo minimum: '+strcompress(min(hz))
   message,/info,'2D histo maximum: '+strcompress(max(hz))
   if n_elements(contour) EQ 1 then begin
       fill = 0
       if (contour lt 0) then fill = 1 
       if ((contour eq 3) or (contour EQ 4)) then fill = 1 
   endif
   if keyword_set(smooth) then begin 
      if (smooth GE min(nc_bins) ) then begin
        smooth = fix(min(nc_bins)/2.)
        text = 'smooth > nbin: set smooth to '+strcompress(smooth)
        message,/info,text
        if sdep(/w) then itmp = Dialog_Message( DIALOG_PARENT=group,text)
      endif
      hz=smooth(hz,smooth)
   endif 
   c_data = {x:hx, y:hy, z:hz}
   IF ((contour eq 5) or (contour EQ 6)) THEN BEGIN
     plotimage,bytscl(hz), $
      imgxrange=hx[[0,N_Elements(hx)-1]],imgyrange=hy[[0,N_Elements(hy)-1]],$
      xrange=xrange,yrange=yrange,/NOERASE,interp=smooth,_extra=extra
   ENDIF ELSE BEGIN
     contour,hz,hx,hy,nlevels=nlevels, /NOERASE, fill=fill, $
	xrange=xrange,yrange=yrange, _EXTRA=extra
   ENDELSE
endif else begin
    plot,arr1,arr2,psym=3,xrange=xrange,yrange=yrange, /NOERASE
    if keyword_set(ccol) then begin
      x = dblarr(1)
      y = dblarr(1)
      for i = 0L,n_elements(carr) -1 do begin
	  ; clr = 250*((carr(i)-cmin)/(cmax-cmin))+3
	  ; clr = (clr < 253) > 0
	  ; changed 98/03/24 srio@esrf.fr
	  clr = (!d.n_colors<255)*((carr(i)-cmin)/(cmax-cmin))-1
	  clr = (clr < ( (!d.n_colors<255) -1) ) > 0
	  x(0) = arr1(i) 
	  y(0) = arr2(i)
	  oplot,x,y,color = clr,psym = 2,symsize = 0.25
      endfor
    endif
endelse

;
; slit, if any
;
if keyword_set(slit) then if N_Elements(slit) EQ 4 then begin
  slit1=double(slit)
  IF N_Elements(slit) EQ 2 THEN $
      slit1=Double([slit[0],slit[0],slit[1],slit[1]])
  

  IF slit1[0] EQ slit1[2] THEN BEGIN ; use fwhm
    xxm = 0.5D0*(xxl+xxr)
    xxw = abs(xxl-xxr)*slit1[0]
    slit1[0]=xxm-0.5D0*xxw
    slit1[2]=xxm+0.5D0*xxw
  ENDIF
  IF slit1[1] EQ slit1[3] THEN BEGIN ; use fwhm
    yym = 0.5D0*(yyl+yyr)
    yyw = abs(yyl-yyr)*slit1[1]
    slit1[1]=yym-0.5D0*yyw
    slit1[3]=yym+0.5D0*yyw
  ENDIF

  IF ((slit1[0] NE slit1[2]) AND (slit1[1] NE slit1[3])) then begin
    intSlit=0
    oplot,[slit1(0),slit1(2),slit1(2),slit1(0),slit1(0)], $
    [slit1(1),slit1(1),slit1(3),slit1(3),slit1(1)]
    ;
    ; compute the intensity
    ; 
    sx1=min(slit1[[0,2]],max=sx2)
    sy1=min(slit1[[1,3]],max=sy2)
    ig1 = where(arr1 GE sx1 AND arr1 LE sx2)
    IF ig1[0] NE -1 THEN BEGIN
      tmp1 = arr2[ig1]
      ig2 = where(tmp1 GE sy1 AND tmp1 LE sy2)
      IF ig2[0] NE -1 THEN BEGIN
        tmpI = wi[ig1]
        tmpII=tmpI[ig2]
        intSlit=Total(tmpII)
      ENDIF
    ENDIF
  ENDIF
endif
;

;
;	comments line
;
if sdep() EQ 'UNIX' then spawn,'date',/noshell,date else date=''
cd,c=pwd
;spawn,'pwd',/noshell,pwd
if not(publish) then $
  xyouts,xx0+xx*.025,yy0+yy*.92,/norm,siz=csize,pwd+sdep(/ds)+str.name+'  '+date
xyouts,xx0+xx*.025,yy0+yy*.98,/norm,siz=tsize,title
; intensity in slit
IF (intSlit GE 0) THEN BEGIN
  xyouts,xx0+xx*.72,yy0+yy*.32,/norm,siz=.8*csize,'INTENS IN SLIT = '
  xyouts,xx0+xx*.83,yy0+yy*.32,/norm,siz=.8*csize,$
    string(intSlit,format='(F10.2)')
ENDIF

;if nocomm eq 0 then begin
;  !p.position = [xx0,yy0,xx0,yy0] + [0.7033*xx,0.030*yy,0.975*xx,0.6855*yy]
;  plot,[0,1],/noData,/noErase,xrange=[0,1],yrange=[0,1],xticks=0,yticks=0,$
;    ticklen=0,xstyle=4,ystyle=4
;  plots,0,0
;  plots,0,0,/cont
;  plots,1,0,/cont
;  plots,1,1,/cont
;  plots,0,1,/cont
;  plots,0,0,/cont
;
;  if not(publish) and sdep() eq 'UNIX' then $
;    xyouts,xx0+xx*.72,yy0+yy*.66,/norm,siz=csize,user+'@'+host
;  if keyword_set(xran) or keyword_set(yran) or keyword_set(xyran) $
;    then begin 
;    xyouts,xx0+xx*.72,yy0+yy*.41,/norm,siz=csize,'EXTERNAL LIMITS'
;  endif else begin
;    xyouts,xx0+xx*.72,yy0+yy*.41,/norm,siz=csize,'INTERNAL LIMITS'
;  endelse
;  if input_type EQ 0 then begin
;    if keyword_set(nolost) then begin
;      if nolost eq 1 then xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--GOOD ONLY'
;      if nolost eq 2 then xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--ONLY LOSSES'
;    endif else begin
;      xyouts,xx0+xx*.72,yy0+yy*.35,/norm,siz=csize,'--ALL RAYS'
;    endelse
;    intens,str,int_val,nolost=nolost
;    xyouts,xx0+xx*.72,yy0+yy*.30,/norm,siz=.8*csize,'INTENS  = '
;    xyouts,xx0+xx*.83,yy0+yy*.30,/norm,siz=.8*csize,$
;	string(int_val,format='(F10.2)')
;    xyouts,xx0+xx*.72,yy0+yy*.28,/norm,siz=.8*csize,'TOT  = '
;    xyouts,xx0+xx*.83,yy0+yy*.28,/norm,siz=.8*csize,$
;	string(str.npoint,format='(I10)')
;    xyouts,xx0+xx*.72,yy0+yy*.26,/norm,siz=.8*csize,'LOST = '
;    ;xyouts,xx0+xx*.83,yy0+yy*.26,/norm,siz=.8*csize,str.npoint-n_elements(arr1)
;    xyouts,xx0+xx*.83,yy0+yy*.26,/norm,siz=.8*csize, $
;	string(str.npoint-n_elements(where(getshcol(str,10) > 0)),format=$
;	'(I10)')
;    xyouts,xx0+xx*.72,yy0+yy*.18,/norm,siz=csize,'Horizontal: '
;    ;xyouts,xx0+xx*.83,yy0+yy*.18,/norm,siz=csize,col1
;    xyouts,xx0+xx*.80,yy0+yy*.18,/norm,siz=csize,column_name(col1)
;    xyouts,xx0+xx*.72,yy0+yy*.16,/norm,siz=csize,'Vertical:   '
;    ;xyouts,xx0+xx*.83,yy0+yy*.16,/norm,siz=csize,col2
;    xyouts,xx0+xx*.80,yy0+yy*.16,/norm,siz=csize,column_name(col2)
;  endif
;  if keyword_set(ccol) then begin
;    xyouts,xx0+xx*.72,yy0+yy*.20,/norm,siz=csize,'Color: '
;    ;xyouts,xx0+xx*.83,yy0+yy*.20,/norm,siz=csize,ccol
;    xyouts,xx0+xx*.80,yy0+yy*.20,/norm,siz=csize,column_name(ccol)
;  endif
;
;  if keyword_set(gaussfit) then begin
;    xyouts,xx0+xx*.72,yy0+yy*.10,/norm,siz=csize,'FitHorizFWHM: '
;    xyouts,xx0+xx*.83,yy0+yy*.10,/norm,siz=csize,2.35*ahfit(2)
;    xyouts,xx0+xx*.72,yy0+yy*.05,/norm,siz=csize,'FitVertFWHM:   '
;    xyouts,xx0+xx*.83,yy0+yy*.05,/norm,siz=csize,2.35*avfit(2)
;  endif
;
;  if keyword_set(calfwhm) then begin
;    xyouts,xx0+xx*.72,yy0+yy*.12,/norm,siz=csize,'HistoHorizFWHM: '
;    xyouts,xx0+xx*.83,yy0+yy*.12,/norm,siz=csize,fwhm_val(0)
;    xyouts,xx0+xx*.72,yy0+yy*.07,/norm,siz=csize,'HistoVertFWHM:   '
;    xyouts,xx0+xx*.83,yy0+yy*.07,/norm,siz=csize,fwhm_val(1)
;  endif
;
;endif
;
;
; prints the command
;
if not(keyword_set(publish)) then begin

command='plotxy,'+str.name+','+strcompress(col1,/rem)+','+  $
  strcompress(col2,/rem)
if keyword_set(xran) then command=command+',XRANGE=['+  $
   strcompress(xran(0),/rem)+','+strcompress(xran(1),/rem)+']'
if keyword_set(yran) then command=command+',YRANGE=['+  $
   strcompress(yran(0),/rem)+','+strcompress(yran(1),/rem)+']' 
if keyword_set(nolost) then command=command+',NOLOST='+ $
   strcompress(nolost,/rem)
if nbins NE nbins_default then command=command+',NBINS='+ $
   strcompress(nbins,/rem) 
if keyword_set(nocomm) then command=command+',NOCOMM='+ $
   strcompress(nocomm,/rem)
if keyword_set(histo) then command=command+',HISTO='+ $
   strcompress(histo,/rem)
if keyword_set(slit) then command=command+',SLIT=['+ $
   strcompress(slit(0),/rem)+','+strcompress(slit(1),/rem)+','+ $
   strcompress(slit(2),/rem)+','+strcompress(slit(3),/rem)+']' 
if keyword_set(xyran) then command=command+',XYRANGE=['+ $
   strcompress(xyran(0),/rem)+','+strcompress(xyran(1),/rem)+']'
if keyword_set(calfwhm) then command=command+',CALFWHM='+ $
   strcompress(calfwhm,/rem) 

if keyword_set(gaussfit) then command=command+',GAUSSFIT='+ $
   strcompress(gaussfit,/rem) 
if keyword_set(ccol) then command=command+',CCOL='+ $
   strcompress(ccol,/rem) 
if keyword_set(climits) then command=command+',CLIMITS=['+ $
   strcompress(climits(0),/rem)+','+strcompress(climits(1),/rem)+']' 
if keyword_set(cbins) then $
   if cbins NE cbins_default then command=command+',CBINS='+ $
   strcompress(cbins,/rem) 
if keyword_set(cref) then command=command+',CREF='+ $
   strcompress(cref,/rem) 
if keyword_set(cfile) then command=command+',CFILE='+cstr.name
if keyword_set(contour) then if n_elements(contour) EQ 1 then $
   command=command+',CONTOUR='+ strcompress(contour,/rem) 
if keyword_set(nlevels) then command=command+',NLEVELS='+ $
   strcompress(nlevels,/rem) 
if keyword_set(smooth) then command=command+',SMOOTH='+ $
   strcompress(smooth,/rem) 
if keyword_set(retrace1) then command=command+',RETRACE='+ $
   strcompress(retrace1,/rem) 
if keyword_set(mulfac) then command=command+','+mulfactxt
if keyword_set(nsplit) then if nsplit GE 2 then command=command+','+$
   'NSPLIT='+StrCompress(nsplit,/Rem)

   xyouts,xx0+xx*.025,yy0+yy*.95,/norm,siz=csize,'command: '+command
endif

; call infosh if selected
if keyword_set(INFOSH) then $
  if keyword_set(nolost) then infosh,str,/nolost else infosh,str
;
;	reset values
;
out:
;; !p.position = oldposition
;; !p.charsize = oldcharsize
;; !x.style = oldxstyle
;; !y.style = oldystyle

IF Keyword_Set(xplot) THEN BEGIN
  IF Widget_Info(xplot,/Valid_ID) THEN BEGIN
    print,'plotxy: using existing xplot window'
  ENDIF ELSE BEGIN
    xplot,p=xplot,wtitle='Points from plotxy'
  ENDELSE
  xplot_loadfile,xplot,make_set(reform(arr1),reform(arr2))
  xplot_controls_action,xplot,PSYMSIGN='1',PSYMBOL=3
ENDIF

end
