PRO conicmesh,c,ww=ww,ll=ll,nx=nx,ny=ny,x=x,y=y, $
  modify=flatSag, pp=p,qq=q,theta=theta, $
  presurface=presurface, surf=surf, dialog_parent=dialog_parent, $
  use_second_solution=use_second_solution
;+
;
;       NAME:
;               CONICMESH
;       PURPOSE:
;               This function calculates a mesh surface
;               using the conic coefficients from conicset()
;       CATEGORY:
;               SHADOW tools
;       CALLING SEQUENCE:
;               conicmesh,c
;
; 	INPUTS:
;		c: 10 conic coefficients, like the  output of 
;			conicset()
;	
; 	KEYWORD PARAMETERS
;		ww: the surface width in cm (along X)
;		ll: the surface length in cm (along Y)
;        	nx: number of points in X
;        	ny: number of points in Y
;        	x: The array with x (in this case, nx and ww are not used)
;        	y: The array with y (in this case, ny and ll are not used)
;		modify: a flag for allowing to modify the
;			conic surface: 
;			0: no modification
;			1: REPLICATE central meridional prof
;			2: REPLICATE central sag prof
;			3: replicate central mer + add central sag
;			4: fixed in sag + central meridional
;			5: cone in sag (averaged at edges)+ central meridional
;			6: cone in sag (tg at pole)+ central meridional
;		pp: the p distance in cm (needed if modify=4,5)
;		qq: the q distance in cm (needed if modify=4,5)
;		theta: grazing incident angle in rad (needed if 
;			modify=4,5)
;		presurface: set to a file name to write the 
;			surface in SHADOW predurface format.
;		surf: set to a named variable to get the surface 
;			in a structuto of type { x:x, y:y, z:z}
;		use_second_solution: set this flag to use the second solution
;                       from the two possible surfaces
;		dialog_parent: parent widget id to pass to i/o windows.
;
;	AUTHOR: 
;		M. Sanchez del Rio srio@esrf.eu
;	
;	MODIFICATION HISTORY:
;		2007-11-08 written
;		2009-07-24 added option 6, fixed bug with 5
;		2010-06-22 added dialogs for modifying sagittal
;                          focal distances
;		2015-02-04 srio@esrf.eu added negative sagittal radius in case 4
;
;-
on_error,2
;
; mesh
; 
IF N_Elements(flatSag) EQ 0 THEN flatSag=0

IF N_Elements(x) GT 1 THEN BEGIn
  xx=x
ENDIF ELSE BEGIN
  IF N_Elements(nx) EQ 0 THEN nx = 21
  IF N_Elements(ww) EQ 0 THEN ww = 10
  xx = makearray1(nx,-double(ww)/2,double(ww)/2)
ENDELSE

IF N_Elements(y) GT 1 THEN BEGIn
  yy=y
ENDIF ELSE BEGIN
  IF N_Elements(ny) EQ 0 THEN ny = 11
  IF N_Elements(ll) EQ 0 THEN ll = 10
  yy = makearray1(ny,-double(ll)/2,double(ll)/2)
ENDELSE


zz = DblArr(nx,ny)

For i=0,nx-1 do begin
  For j=0,ny-1 do begin
   
  x = xx[i] 
  y = yy[j]

  aa = c[2]
  bb = c[4]*y+c[5]*x+c[8]
  cc = c[0]*x^2 + c[1]*y^2 + c[3]*x*y + c[6]*x + c[7]*y + c[9]

  IF aa NE 0 THEN BEGIN
    discr = bb^2-4*aa*cc
    s1 = (-bb+Sqrt(discr))/2/aa
    s2 = (-bb-Sqrt(discr))/2/aa
    ss = min([s1,s2],max=ss2)
    IF keyword_Set(use_second_solution) THEN ss=ss2
  ENDIF ELSE BEGIN
    ss = -cc/bb
  ENDELSE

  zz[i,j] = ss


  EndFor
EndFor

;
; change curvatures
;

CASE flatSag of
  0:
  1: BEGIN ; REPLICATE central meridional prof
    zCenter = Reform(zz[nx/2,*])
    FOR i=0,nx-1 DO zz[i,*]=zCenter
    END
  2: BEGIN ; REPLICATE central sag prof
    zCenter = Reform(zz[*,ny/2])
    FOR j=0,ny-1 DO zz[*,j]=zCenter
    END
  3: BEGIN ; replicate central mer + add central sag
    zCenter = Reform(zz[nx/2,*])
    sagCenter = Reform(zz[*,ny/2])
    FOR i=0,nx-1 DO zz[i,*]=zCenter ; replicate central meridional
    FOR j=0,ny-1 DO zz[*,j]=zz[*,j]+sagCenter ; add central sag
    END

  4: BEGIN ; fixed in sag + central meridional
    zCenter = Reform(zz[nx/2,*])

    IF Keyword_set(p) THEN BEGIN
    tmp={p:p[0],q:q[0],theta:theta[0]}
    XScrMenu,tmp,/interpret,/noType,wtitle='Sagittal distances',$
     titles=['p for calculating Rs [cm]:','q for calculating Rs [cm]:',$
             'theta [grazing, rad]:'],action=action, $
     dialog_parent=dialog_parent
    IF action EQ 'DONT' THEN RETURN
    p[0]=tmp.p
    q[0]=tmp.q
    theta[0]=tmp.theta
    pp = sqrt(p[0]^2+yy^2-2*p[0]*yy*cos(!dpi-theta[0]))
    qq = sqrt(q[0]^2+yy^2-2*q[0]*yy*cos(theta[0]))
    rsag = 2D0*pp*qq/(pp+qq)*sin(theta[0])
    ENDIF

    zz=zz*0
    rSagJ = rsag[ny/2]
    xedit, rSagJ,Text='Using Rsag value [cm]: ',Title='Sagittal Radius',$
      Dialog_parent=dialog_parent,action=action
    IF action EQ 'CANCEL' THEN RETURN
    
    message,/info,'Using Rsag: '+StrCompress(rSagJ)
    FOR j=0,ny-1 DO BEGIN
      sagI = -xx^2
      sagI = -sqrt(sagI+rSagJ^2) + abs(rSagJ)
      if (rSagJ gt 0) then begin
          zz[*,j]=zz[*,j]+sagI ; add sag
      endif else begin
          zz[*,j]=zz[*,j]-sagI ; add sag
      endelse
    ENDFOR
    FOR i=0,nx-1 DO zz[i,*]=zz[i,*]+zCenter
    END


  5: BEGIN ; conic in sag (averaged at the edges) + central meridional
    zCenter = Reform(zz[nx/2,*])

    IF Keyword_set(p) THEN BEGIN
    tmp={p:p[0],q:q[0],theta:theta[0]}
    XScrMenu,tmp,/interpret,/noType,wtitle='Sagittal distances',$
     titles=['p for calculating Rs [cm]:','q for calculating Rs [cm]:',$
             'theta [grazing, rad]:'],action=action, $
     dialog_parent=dialog_parent
    IF action EQ 'DONT' THEN RETURN
    p[0]=tmp.p
    q[0]=tmp.q
    theta[0]=tmp.theta
    
    pp = sqrt(p[0]^2+yy^2-2*p[0]*yy*cos(!dpi-theta[0]))
    qq = sqrt(q[0]^2+yy^2-2*q[0]*yy*cos(theta[0]))
    rsag = 2D0*pp*qq/(pp+qq)*sin(theta[0])
    ENDIF

    zz=zz*0
    rSagJ = rsag[ny/2]
    rsag_linear = rsag*0
    FOR j=0,ny-1 DO BEGIN
      sagI = -xx^2
      rSagJ = rsag[0]+(Float(j)/Float(ny-1))*(rsag[ny-1]-rsag[0])
      rSag_linear[j]= rSagJ
      sagI = -sqrt(sagI+rSagJ^2) + rSagJ
      zz[*,j]=zz[*,j]+sagI ; add sag
    ENDFOR
    FOR i=0,nx-1 DO zz[i,*]=zz[i,*]+zCenter
    xplot,make_set(yy,rsag,rsag_linear),/no,coltitles=['y[cm]','Rsag (exact) [cm]','Rsag (linear-averaged at edges) [cm]'],xtitle='-1',ytitle='-1'
    END

  6: BEGIN ; conic in sag (tangent at pole) + central meridional
    zCenter = Reform(zz[nx/2,*])

    IF Keyword_set(p) THEN BEGIN
    tmp={p:p[0],q:q[0],theta:theta[0]}
    XScrMenu,tmp,/interpret,/noType,wtitle='Sagittal distances',$
     titles=['p for calculating Rs [cm]:','q for calculating Rs [cm]:',$
             'theta [grazing, rad]:'],action=action, $
     dialog_parent=dialog_parent
    IF action EQ 'DONT' THEN RETURN
    p[0]=tmp.p
    q[0]=tmp.q
    theta[0]=tmp.theta
    
    pp = sqrt(p[0]^2+yy^2-2*p[0]*yy*cos(!dpi-theta[0]))
    qq = sqrt(q[0]^2+yy^2-2*q[0]*yy*cos(theta[0]))
    rsag = 2D0*pp*qq/(pp+qq)*sin(theta[0])
    ENDIF

    zz=zz*0
    rSagJ = rsag[ny/2]
    rsag_linear = rsag*0
    FOR j=0,ny-1 DO BEGIN
      sagI = -xx^2
      rSagJ = rsag[ny/2]+ (rsag[1+ny/2]-rsag[ny/2])/(yy[1+ny/2]-yy[ny/2])*yy[j]
      rSag_linear[j]= rSagJ
      sagI = -sqrt(sagI+rSagJ^2) + rSagJ
      zz[*,j]=zz[*,j]+sagI ; add sag
    ENDFOR
    FOR i=0,nx-1 DO zz[i,*]=zz[i,*]+zCenter
    xplot,make_set(yy,rsag,rsag_linear),/no,coltitles=['y[cm]','Rsag (exact) [cm]','Rsag (linear-tangent at pole) [cm]'],xtitle='-1',ytitle='-1'
    END


else:
ENDCASE
;
IF N_elements(presurface) THEN BEGIN
  matr2presur,zz,xx,yy,File=presurface
  itmp = Dialog_Message(['File written to disk: '+presurface[0],'',$
          'For using it in Shadow, do not forget to run PRESURFACE (Outputs tab)'],$
         dialog_parent=dialog_parent,/Info)
ENDIF

surf = {z:zz,x:xx,y:yy}

END




