function getshcol,shadow_in,col, NOLOST=nolost, block=block, verbose=verbose
;+
; NAME:
;	GETSHCOL
; PURPOSE:
;	extract a column array(s) from a SHADOW file or structure.
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	output=getshcol(shadow_in,col,keyword_parameters)
; INPUTS:
;	shadow_in   may be:
;           a shadow file (between quotes) (e.g. 'begin.dat') or
;           an IDL-SHADOW structure
;	col    number of SHADOW column to extract or array with
;		the SHADOW columns to extract.
;           the colum meaning is the following:
;           1   X spatial coordinate [user's unit]
;           2   Y spatial coordinate [user's unit]
;           3   Z spatial coordinate [user's unit]
;           4   X' direction or divergence [rads]
;           5   Y' direction or divergence [rads]
;           6   Z' direction or divergence [rads]
;           7   X component of the electromagnetic vector (s-polariz)
;           8   Y component of the electromagnetic vector (s-polariz)
;           9   Z component of the electromagnetic vector (s-polariz)
;          10   Lost ray flag
;          11   Energy [eV]
;          12   Ray index
;          13   Optical path length
;          14   Phase (s-polarization)
;          15   Phase (p-polarization)
;          16   X component of the electromagnetic vector (p-polariz)
;          17   Y component of the electromagnetic vector (p-polariz)
;          18   Z component of the electromagnetic vector (p-polariz)
;          19   Wavelength [A]
;          20   R= SQRT(X^2+Y^2+Z^2)
;          21   angle from Y axis
;          22   the magnitude of the Electromagnetic vector
;          23   |E|^2 (total intensity)
;          24   total intensity for s-polarization
;          25   total intensity for p-polarization
;          26   K=2 pi/lambda [A^-1]
;          27   Kx = K * col4 [A^-1]
;          28   Ky = K * col5 [A^-1]
;          29   Kx = K * col6 [A^-1]
;          30   S0-stokes = |Es|^2 + |Ep|^2
;          31   S1-stokes = |Es|^2 - |Ep|^2
;          32   S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p)
;          33   S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p)
;          34   CohS      = |Es| sin(phase_s)
;          35   CohP      = |Ep| sin(phase_p)
; KEYWORD PARAMETERS:
;        NOLOST = consider losses
;            0 all rays (default option)
;            1 exclude losses;
;            2 only losses;
;        VERBOSE = flag passed to readsh
; OUTPUTS:
;	an array with the extracted column(s)
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	None.
; PROCEDURE:
;	Straightforward.
; KNOWN BUGS:
;	None.
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, June 1995
;	98-06-15 srio@esrf.fr fixes a bug when extracting an array of
;		columns being Energy one of those. Cosmetics.
;	06-02-21 srio@esrf.fr adds cols 19,26-29. Works in Double.
;	2009-01-21 srio@esrf.eu adds verbose keyword.
;	2010-11-26 srio@esrf.eu patches to return zero if no good rays found
;	2013-05-30 srio@esrf.eu updated physical constant hc following shadow3
;-
on_error,2
;
; load shadow-idl structure 
;
str = readsh(shadow_in,nolost=nolost,verbose=verbose)
if not(keyword_set(str)) then return,0
if str.npoint EQ 0 then return,0
shadow_out = str

npoint = str.npoint
ncol = str.ncol

; define which points (rays) are going to be extracted

IF N_Elements(block) EQ 2 THEN BEGIN ; defined block (for statistics)
  col12 = str.ray(11,*)
  imax = max(col12)
  iblock = block[0]
  nblock = block[1]
  rays_per_block = imax/nblock
  first_ray = iblock*rays_per_block
  last_ray = first_ray + rays_per_block -1
  first_ray = first_ray + 1 ; shadow index start from one
  last_ray = last_ray + 1 ; shadow index start from one
  wp = where( (col12 GE first_ray) AND (col12 LE last_ray) )
  IF wp[0] EQ -1 THEN Message,'No good rays found in block '+ $
          StrCompress(iblock)+' out of '+StrCompress(nblock) 
ENDIF ELSE BEGIN
  wp = LIndGen(str.npoint)
ENDELSE

;
; extends rays with extra columns and defines output
;
extracols = where(col GT 18,nextracols)

if (nextracols EQ 0) then begin
   tmp = str.ray
   tmp=tmp[col-1,*]
   tmp=tmp[*,wp]
   output=temporary(tmp)
endif else begin
   raysmore = dblarr(37,npoint)
   raysmore(0:ncol-1,wp) = str.ray(*,wp)

   r_pol = sqrt(str.ray(0,wp)^2+str.ray(1,wp)^2+str.ray(2,wp)^2)
   angle_pol = acos(str.ray(4,wp))
   as_pol = sqrt(str.ray(6,wp)^2+str.ray(7,wp)^2+str.ray(8,wp)^2)
   as2_pol = str.ray(6,wp)^2+str.ray(7,wp)^2+str.ray(8,wp)^2
   if str.ncol eq 18 then begin
     ap_pol = sqrt(str.ray(15,wp)^2+str.ray(16,wp)^2+str.ray(17,wp)^2)
     ap2_pol = str.ray(15,wp)^2+str.ray(16,wp)^2+str.ray(17,wp)^2
     phase = getphase(str)
   endif else begin
     ap_pol = as_pol*0.0
     ap2_pol = as2_pol*0.0
     phase=dblarr(4,npoint)
   endelse
  cos_pol = cos(phase(0,wp)-phase(1,wp))
  sin_pol = sin(phase(0,wp)-phase(1,wp))
  sin_s = sin(phase(0,wp))
  sin_p = sin(phase(1,wp))
  cos_s = cos(phase(0,wp))
  cos_p = cos(phase(1,wp))
  
  raysmore(18,wp) =  2.0D0*!dpi*1D8/str.ray(10,wp)
  raysmore(19,wp) =  r_pol
  raysmore(20,wp) =  angle_pol
  raysmore(21,wp) =  sqrt(as2_pol+ap2_pol)  ;sqrt(int)
  raysmore(22,wp) =  as2_pol+ap2_pol
  raysmore(23,wp) =  as2_pol
  raysmore(24,wp) =  ap2_pol

  raysmore(25,wp) =  str.ray(10,wp)*1D-8
  raysmore(26,wp) =  str.ray(10,wp)*1D-8*str.ray(3,wp)
  raysmore(27,wp) =  str.ray(10,wp)*1D-8*str.ray(4,wp)
  raysmore(28,wp) =  str.ray(10,wp)*1D-8*str.ray(5,wp)

  raysmore(29,wp) =  as2_pol+ap2_pol
  raysmore(30,wp) =  as2_pol-ap2_pol
  raysmore(31,wp) =  2*as_pol*ap_pol*cos_pol
  raysmore(32,wp) =  2*as_pol*ap_pol*sin_pol
  raysmore(33,wp) =  as_pol*sin_s
  raysmore(34,wp) =  ap_pol*sin_p
  raysmore(35,wp) =  as_pol*cos_s
  raysmore(36,wp) =  ap_pol*cos_p

  tmp = raysmore
  tmp=tmp[col-1,*]
  tmp=tmp[*,wp]
  output=temporary(tmp)

  ;output = raysmore(col-1,wp) 
endelse
;
; if a column is energy, then change units to eV
;
ctmp = where(col eq 11,count)
if count GT 0 then begin
  for i=0L,n_elements(ctmp)-1 do begin
    message ,/info ,'GETSHCOL: Energy column selected: set units to eV'
    output(ctmp[i],wp) = output(ctmp[i],wp) $
	;/6.283185307179586476925287D0*1.239852D-4
	/6.283185307179586476925287D0*12398.4192920042D-8
  endfor
endif
;
return,output
end
