;
;========================================================================
;
FUNCTION xanes_normalize,y,x, $
   method=method, jump=jump, degree=degree, $
   e0=e0, estart=eMin1, eend=eMax1, $
   xplot=xplot, verbose=verbose, $
   preRange=preRange, postRange=xrange,  $  ; outputs
   set=set2, coltitles=coltitles,txt=txt    ; outputs

;
;+
; NAME:
;	XANES_NORMALIZE
;
; PURPOSE:
;	This function normalizes a XANES spectrum.
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	i) setOut = XANES_NORMALIZE(setIn)
;	ii) yOut = XANES_NORMALIZE(y,x)
;
; INPUTS:
;       Input either: 
;	  setIn:  fltarr(2,npoints) set of data (usually (E,Mu))
;	  y:  Array with ordinates (usually Mu)
;
; OPTIONAL INPUTS:
;	  x:  Array with abscissas (usually Energy)
;	
; KEYWORD PARAMETERS (INPUT):
;	METHOD:	Normalization method: 
;               0 Use the jump (constant) value
;		1 use a polynomial fit of the postEdge (default)
;
;       JUMP: (when Method=0) defines the Jump. If undefined, the 
;             jump is calculated using getJump() function.
;
;       DEGREE: (when Method=1) defines the polynomial degree for the
;             post-edge fit. Default=3
;
;       E0:   The edge value in eV or keV. If not set, it is calculated
;             using getEo()
;
;       ESTART: The number of eV or keV (same units as the abscissas)
;             for defining the preedge fit interval. 
;             Xmin for the fit is the minimum of the abscissas plus EMIN
;	      Default=5. (Set ESTART='0' for forcing zero because ESTART=0 takes
;                         the default)
;             
;       EEND: The number of eV or keV (same units as the abscissas)
;             for defining the preedge and postedge fit intervals. 
;             Xmax for the preedge fit is e0 minus EEND
;             Xmin for the postedge fit is e0 plus EEND
;	      Default=20. (Set EEND='0' for forcing zero because EEND=0 takes
;                         the default)
;             
;	XPLOT:	Set this keyword to inject results into an Xplot window
;
;	VERBOSE: Set this keyword for a verbose output
;
; KEYWORD PARAMETERS (OUTPUT):
;
;      preRANGE: a two elements array with the limits for the preedge fit
;
;      postRANGE: a two elements array with the limits for the postedge fit
;
;      SET: a multicolumn array containing the output and intermediate 
;           calculations.
;    
;      COLTITLES: a string array with the labels of set
;
;      TXT: a string array with the called command and used keyword values
;
; OUTPUTS:
;	This function returns the normalized xanes spectrum in either
;       a 1-dim array or a fltarr(2,npoints) set of data, depending 
;       how the input is passed.
;
; PROCEDURE:
;	A linear fit of the preedge is substracted. The resulting 
;       spectrum divided by either a constant (jump, when method=0) 
;       or by a polynomial fit of the postedge.
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;       2009-05-13 srio@esrf.eu written.
;
;-
;
;on_error,2

Catch, error_status
IF error_status NE 0 THEN BEGIN
  itmp = Dialog_Message(/Error, $
    !error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF

IF N_Elements(method) EQ 0 THEN method=1
IF N_Elements(degree) EQ 0 THEN degree=3
ss = size(y)

flag_set=0
CASE ss[0] OF
  1: BEGIN
     IF N_Elements(x) GT 0 THEN BEGIN 
       set1=Make_Set(Reform(x),Reform(y))
     ENDIF ELSE BEGIN
       set1=Make_Set(FindGen(N_Elements(y)),Reform(y))
     ENDELSE
     END
  else: BEGIN
     IF ss[1] EQ 1 THEN BEGIN
       IF N_Elements(x) GT 0 THEN BEGIN 
         set1=Make_Set(Reform(x),Reform(y)) 
       ENDIF ELSE BEGIN
         set1=Make_Set(FindGen(N_Elements(y)),Reform(y)) 
       ENDELSE
     ENDIF ELSE BEGIN
       set1=y[0:1,*]
       flag_set=1
     ENDELSE
     END
ENDCASE
  
;
; for storing outputs
;
set2=FltArr(7,N_Elements(set1[0,*]))
set2[0,*] = FIndgen(N_Elements(set1[0,*]))
set2[1:2,*]=set1

;
; Default Eo, Xmin, imax 
;

IF Not(Keyword_Set(e0)) THEN e0 = gete0(set1)


maxSet1 = Max(set1[0,*],min=minSet1)
IF (maxSet1-minSet1) GT 10 THEN kev_flag=0 ELSE kev_flag=1

IF Not(Keyword_Set(eMin1)) THEN BEGIN
  eMin = 5 
  IF kev_flag THEN emin=emin*1e-3
ENDIF ELSE BEGIN
  eMin=Float(eMin1)
ENDELSE

IF Not(Keyword_Set(eMax1)) THEN BEGIN
  eMax = 20 
  IF kev_flag THEN eMax=eMax*1e-3
ENDIF ELSE BEGIN
  eMax = Float(eMax1)
ENDELSE

xMin = Min(set1[0,*])+eMin
xMax = e0-emax
xrange=[e0+eMax,Max(set1[0,*])]

IF keyword_Set(verbose) THEN BEGIN
  print,'XANES_NORMALIZATION:'
  print,'           e0:       '
  print,'              at:    '+StrCompress(e0,/Remove_All)
  print,'           preedge: '
  print,'              from:  '+StrCompress(xMin,/Remove_All)
  print,'              to:    '+StrCompress(xMax,/Remove_All)
  print,'           method:   '+StrCompress(method,/Remove_All)
  IF method EQ 1 THEN $
  print,'           degree:   '+StrCompress(degree,/Remove_All)
ENDIF


;
; remove pre-edge linear fit
;
fitset,set1,fit1,method=0,xrange=[xMin,xMax]
set1[1,*] = set1[1,*]-Reform(fit1[1,*])

set2[3,*] = Reform(fit1[1,*])
set2[4,*] = set1[1,*]

CASE method OF 
  0: BEGIN
     ; Default Jump
     IF Not(keyword_set(jump)) THEN jump = getjump(set1,/NoPlot)
     ; normalize to jump 
     print,'           jump: '+StrCompress(jump,/Remove_All)
     yOut=Reform(set1[1,*])/jump
     set2[5,*]=jump
     END
  1: BEGIN
     ; polynomial for of post-edge

     IF keyword_Set(verbose) THEN BEGIN
       print,'           postedge: '
       print,'              from:  '+StrCompress(xrange[0],/Remove_All)
       ;print,'              index: '+StrCompress(index+iMax,/Remove_All)
       print,'              to:    '+StrCompress(xrange[1],/Remove_All)
       ;print,'              index: '+StrCompress(nn-1,/Remove_All)
     ENDIF
     fitset,set1,fit1,method=1,xrange=xrange,degree=degree
     yOut=Reform(set1[1,*])/Reform(fit1[1,*])
     set2[5,*]=Reform(fit1[1,*])
     END
  else:
ENDCASE

preRange=[xmin,xmax]
set2[6,*]=yOut
coltitles = ['Index','X','Mu','PreEdge','Mu-PreEdge', $
              'PostEdge','(Mu-PreEdge)/PostEdge)']

;
; build txt (for labeling)
;
txt = 'xanes_normalize(y,x'
CASE method OF
  0: txt=txt+',method=0,jump='+StrCompress(jump,/Remove_All)
  1: txt=txt+',degree='+StrCompress(degree,/Remove_All)
  else:
ENDCASE
txt=txt+',eStart='+StrCompress(eMin,/Remove_All)
txt=txt+',eEnd='+StrCompress(eMax,/Remove_All)
txt=txt+')'

IF N_Elements(xplot) GT 0 THEN xplot,set2,xcol=2,coltitles=coltitles,/no_block

IF flag_set THEN out=set2[[1,6],*] ELSE out=yout
RETURN,out
END


