;
;+
;
;   ===========================  Xshundul   =========================
;
;   Xshundul is an widget based graphical interface to generate
;   the SHADOW undulator source.
;
;   The main window contains four buttons:
;
;	File: 
;		Load XShUndul input file...
;		Write XShUndul input file...
;		QUIT: to exit from the program
;
;	SET PARAMETERS: to define the parameters for the calculation
;		
;	SHOW: 
;		Some views of the result file 'begin.dat'
;
;	HELP:   Shows this text
;
;
; ----------------------------------------------------------------------
;       Author: M. Sanchez del Rio (srio@esrf.fr) , ESRF, Dec 1995
;       Modification history:
;		96/01/08 MSR allows input file from command line.
;		96/11/05 MSR uses xsh_run to spawn shadow commands.
;		96/03/02 MSR adds no_block kw to xmanager
;		98/03/17 MSR modifies the input parameter (ptr) to plug into
;			the SHADOW-VUI.
;		00/05/30 MSR adds Only_Run keyword
;		00/07/17 MSR allows modifying number of rays.
;		2008/09/11 MSR fixes a bug (error when closing and
;			reopening the parameter window). (Noticed by 
;			Dave Gore). v1.1. 
;               2012/05/29 srio@esrf.eu added "reject rays" option.
;               20131212 srio@esrf.eu implemented SRW. Block emittances
;               20130109 srio@esrf.eu implemented Gaussian approximation
;
;-
;
;========================================================================
;
;-
;		
;========================================================================
;
FUNCTION XSHUNDUL_NAMEVERSION
return,'XShUndul 1.2'
end
;		
;========================================================================
;
pro xshundul_set_getstr,state
on_error,2
inp = state.str
wids = state.wids
widget_control,wids.lambdau,get_value=lambdau
widget_control,wids.k,get_value=k
widget_control,wids.e_energy,get_value=e_energy
widget_control,wids.nperiods,get_value=nperiods
widget_control,wids.emin,get_value=emin
widget_control,wids.emax,get_value=emax
widget_control,wids.intensity,get_value=intensity
widget_control,wids.maxangle,get_value=maxangle
widget_control,wids.seed,get_value=seed
widget_control,wids.nrays,get_value=nrays
widget_control,wids.sx,get_value=sx
widget_control,wids.sz,get_value=sz
widget_control,wids.ex,get_value=ex
widget_control,wids.ez,get_value=ez
widget_control,wids.f_bound_sour,get_value=f_bound_sour
;new
widget_control,wids.ng_e,get_value=ng_e
widget_control,wids.ng_p,get_value=ng_p
widget_control,wids.ng_t,get_value=ng_t
widget_control,wids.ng_plot,get_value=ng_plot
widget_control,wids.undul_phot_flag,get_value=undul_phot_flag
widget_control,wids.flag_emittance,get_value=flag_emittance

inp.lambdau = float(lambdau(0))
inp.k = float(k(0))
inp.e_energy = float(e_energy(0))
inp.nperiods = fix(nperiods(0))
inp.emin = float(emin(0))
inp.emax = float(emax(0))
inp.intensity = float(intensity(0))
inp.maxangle = float(maxangle(0))
inp.seed = long(seed(0))
inp.nrays = long(nrays[0])
inp.sx = float(sx(0))
inp.sz = float(sz(0))
inp.ex = float(ex(0))
inp.ez = float(ez(0))
inp.f_bound_sour = f_bound_sour[0]
inp.ng_e = long(ng_e[0])
inp.ng_p = long(ng_p[0])
inp.ng_t = long(ng_t[0])

inp.undul_phot_flag[0] = undul_phot_flag[0]
inp.flag_emittance[0] = flag_emittance[0]
inp.ng_plot[0] = ng_plot[0]

state.str = inp
end

;		
;========================================================================
;
PRO xshundul_set_mappanels,wids
CATCH, error_status
IF error_status NE 0 THEN BEGIN
   Message,/info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,$
        'XSHUNDUL_SET_MAPPANELS: error caught: '+!err_string)
   Catch, /Cancel
   RETURN
ENDIF

widget_control,wids.flag_emittance,get_value=tmp
; remember that it maps the parent, so sx and sz are mapped together...
widget_control,wids.sx,map=tmp
widget_control,wids.ex,map=tmp

widget_control,wids.undul_phot_flag,get_value=tmp
imap=1
if (fix(tmp) EQ 3) then imap=0
widget_control,wids.k,map=imap
widget_control,wids.e_energy,map=imap
widget_control,wids.preE0,map=imap
widget_control,wids.precc,map=imap
widget_control,wids.prefr,map=imap
widget_control,wids.ng_e,map=imap
widget_control,wids.maxangle,map=imap

END
;		
;========================================================================
;
PRO xshundul_set_event,event

CATCH, error_status
IF error_status NE 0 THEN BEGIN
   Message,/info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,$
        'XSHUNDUL_SET_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   RETURN
ENDIF

widget_control,event.id,get_uval=uval

if not(keyword_set(uval)) then return
widget_control,event.top,get_uvalue=state ; ,/no_copy
case uval of
   'UPDATE_PANELS': BEGIN
        xshundul_set_mappanels,state.wids
        END
   'F_BOUND_SOUR': BEGIN
	Widget_Control,state.wids.F_BOUND_SOUR,Get_Value=index
	state.str.F_BOUND_SOUR = index
	CASE index OF
	  '0': goto,out  ; No
	  '1': BEGIN ; histo3 file with phase-space volume
		titles = ['File with phase-space volume:',$
                          'Max number of rejected rays (set 0 for infinity)']
		wTitle='Optimize source using phase-space volume'
		flags = ['1','1']
		;tmp=state.str.FILE_BOUND
		data={ file_bound:state.str.FILE_BOUND, $
                       ntotalpoint:state.str.NTOTALPOINT }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret, $
		  NCol=1, Flags=flags
		IF action EQ 'DONT' THEN Return
		state.str.F_BOUND_SOUR = index
		state.str.FILE_BOUND = data.FILE_BOUND
		state.str.NTOTALPOINT = data.NTOTALPOINT
                ;Widget_Control, event.top, set_UValue=state
                goto,out
		END
	  '2': BEGIN ; slit/acceptance file
		titles = ["Slit distance [cm] (set 0 for angular acceptance)",$
                          "min x [cm] ( or min x' if distance=0)", $
                          "max x [cm] ( or max x' if distance=0)", $
                          "min z [cm] ( or min z' if distance=0)", $
                          "max z [cm] ( or max z' if distance=0)", $
                          "Max number of rejected rays (set 0 for infinity)"]

		wTitle='Optimize source using slit/acceptance'
		flags = replicate('1',6)
		;tmp=state.str.FILE_BOUND
		data={ slit_distance:state.str.slit_distance, $
		       slit_xmin:state.str.slit_xmin, $
		       slit_xmax:state.str.slit_xmax, $
		       slit_zmin:state.str.slit_zmin, $
		       slit_zmax:state.str.slit_zmax, $
                       ntotalpoint:state.str.NTOTALPOINT  }
		XScrMenu,data,Titles=titles,/NoType,WTitle=wtitle, $
		  Action=action, Dialog_Parent=event.top, /Interpret, $
		  NCol=1, Flags=flags
		IF action EQ 'DONT' THEN Return
		state.str.slit_distance=data.slit_distance
		state.str.slit_xmin=data.slit_xmin
		state.str.slit_xmax=data.slit_xmax
		state.str.slit_zmin=data.slit_zmin
		state.str.slit_zmax=data.slit_zmax
		state.str.ntotalpoint=data.ntotalpoint
		state.str.FILE_BOUND='xsh_slit_tmp.dat'
                ;Widget_Control, event.top, set_UValue=state 
                goto,out
		END
	ENDCASE 

        END
  'run':begin
	widget_control,/hourglass
	xshundul_set_getstr,state
	xshundul_run,state.str,group=event.top
	end
  'done':begin
	xshundul_set_getstr,state
	if ptr_valid(state.input) then *(state.input) = state.str
	widget_control,event.top,/destroy
	return
	end
  'cancel':begin
	widget_control,event.top,/destroy
	return
	end
  'update':begin
	xshundul_set_getstr,state
	inp = state.str
	;e0 = 950. * inp.e_energy^2 /(1+ inp.k^2/2)/(100*inp.lambdau)
        g1 = inp.e_energy*1d3/Physical_Constants('MEE') ; gamma
        lambda0 = inp.lambdau*(1+ inp.k^2/2)/2/g1^2*1e10  ; Angstroms
        e0 = physical_constants('hc')/lambda0
	;cc = 1./(1.957*inp.e_energy)*sqrt( (1+inp.k^2/2)/2/inp.nperiods)
	;fr = 1./(1.957*inp.e_energy)*sqrt( (1+inp.k^2/2))
	cc = 1./(g1*1d-3)*sqrt( (1+inp.k^2/2)/2/inp.nperiods)
	fr = 1./(g1*1d-3)*sqrt( (1+inp.k^2/2))
	widget_control,state.wids.pree0,set_value=$
	'1st, 3rd, 5th harmonic energy [eV]: '+vect2string([e0,e0*3,e0*5])
	widget_control,state.wids.precc,set_value=$
	'1st, 3rd, 5th harmonic central cone aperture (sigma) [mrad]: '+$
	vect2string([cc,cc/sqrt(3),cc/sqrt(5)])
	widget_control,state.wids.prefr,set_value=$
	'1st, 3rd, 5th harmonic first ring aperture [mrad]: '+$
	vect2string([fr,fr/sqrt(3),fr/sqrt(5)])
	end
  'undul_phot_flag':begin
	state.str.undul_phot_flag(0) = strcompress(event.index,/rem)
	end
   else:
endcase
out:
widget_control,event.top,set_uvalue=state ; ,/no_copy
end
;		
;========================================================================
;
PRO xshundul_set,inp,GROUP=group,input=input
;
on_error,2


if xregistered('xshundul_set') then begin
  itmp = Dialog_Message(Dialog_Parent=group,/Question,/Default_No,$
    ['SHADOW VUI has detected an open window for Undulator Source',$
     'Do you want another one?'])
  if itmp EQ 'No' then return
endif
if not(keyword_set(inp)) then inp = xsh_defaults('xshundul')
if not(keyword_set(group)) then wgroup=0L else wgroup=group
if not(keyword_set(input)) then input=0


wWindow = widget_base(title='Undulator Parameters ('+$
   xshundul_nameversion()+')',/COLUMN)


wtmp = widget_base(wWindow,/ROW)
;wtmp1 = widget_button(wtmp,value='Accept',uvalue='done')
wtmp1 = widget_button(wtmp,value='Close',uvalue='done')
;wtmp1 = widget_button(wtmp,value='Cancel',uvalue='cancel')
wtmp1 = widget_button(wtmp,value='Discard changes',uvalue='cancel')
wtmp1 = widget_button(wtmp,value='Run Shadow/source',uvalue='run')


;
; undulator/machine pars
;
wbase = widget_base(wWindow,/COL,/FRAME)
case sdep() of
  'WINDOWS': font = 'VERDANA*BOLD*ITALIC*24' 
  'UNIX': font='-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
  else: font = ''
endcase

wtmp = widget_label(wbase,value='Undulator/Machine parameters', FONT=font)

wbase0 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase0,value='Period [m]: ',/align_left)
wlambdau = widget_text(wbase0,value=strcompress(inp.lambdau,/rem),/EDIT,$
	XSIZE=6,uvalue='update',/All_Events)
wtmp = widget_label(wbase0,value='N periods: ', /align_left)
wnperiods = widget_text(wbase0,value=strcompress(inp.nperiods,/rem),/EDIT,$
	xsize=6, uvalue='update',/All_Events)
wbase01 = widget_base(wbase0,/ROW) ; new base to allow mapping
  wtmp = widget_label(wbase01,value='K Parameter: ', /align_left)
  wK = widget_text(wbase01,value=strcompress(inp.K,/rem),/EDIT,xsize=6, $
	uvalue='update',/All_Events)

wbase0 = widget_base(wbase,/ROW)
wbase01 = widget_base(wbase0,/ROW) ; new base to allow mapping
  wtmp = widget_label(wbase01,value='Energy [GeV]: ', /align_left)
  we_energy = widget_text(wbase01,value=strcompress(inp.e_energy,/rem),/EDIT,$
	xsize=6,uvalue='update',/All_Events)
  wtmp = widget_label(wbase01,value='Intensity [A]: ', /align_left)
  wintensity = widget_text(wbase01,value=strcompress(inp.intensity,/rem),$
	XSIZE=6,/EDIT)


wbase00 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase00,value='Use emittances : ',/align_left)
wflag_emittance = cw_droplist(wbase00,value=inp.flag_emittance,uvalue='UPDATE_PANELS')
wbase0 = widget_base(wbase00,/ROW)
  wtmp = widget_label(wbase0,value='Sigma X [cm]: ', /align_left)
  wsx = widget_text(wbase0,value=strcompress(inp.sx,/rem),/EDIT,XSIZE=8)
  wtmp = widget_label(wbase0,value='Sigma V [cm]: ', /align_left)
  wsz = widget_text(wbase0,value=strcompress(inp.sz,/rem),/EDIT,XSIZE=8)

wbase0 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase0,value='Emittance X [rad.cm]: ', /align_left)
wex = widget_text(wbase0,value=strcompress(inp.ex,/rem),/EDIT,XSIZE=12)
wtmp = widget_label(wbase0,value='Emittance Z [rad.cm]: ', /align_left)
wez = widget_text(wbase0,value=strcompress(inp.ez,/rem),/EDIT,XSIZE=12)

;
; pre-calculation pars
;

wbase = widget_base(wWindow,/COL,/FRAME)
wtmp = widget_label(wbase,value='Pre-calculation parameters', FONT=font)

;e0 = 950. * inp.e_energy^2 /(1+ inp.k^2/2)/(100*inp.lambdau)
;cc = 1./(1.957*inp.e_energy)*sqrt( (1+inp.k^2/2)/2/inp.nperiods)
;fr = 1./(1.957*inp.e_energy)*sqrt( (1+inp.k^2/2))

g1 = inp.e_energy*1d3/Physical_Constants('MEE') ; gamma
lambda0 = inp.lambdau*(1+ inp.k^2/2)/2/g1^2*1e10  ; Angstroms
e0 = physical_constants('hc')/lambda0
cc = 1./(g1*1d-3)*sqrt( (1+inp.k^2/2)/2/inp.nperiods)
fr = 1./(g1*1d-3)*sqrt( (1+inp.k^2/2))


wpreE0 = widget_label(wbase, /align_left, value= $
	'1st, 3rd, 5th harmonic energy [eV]: '+vect2string([e0,e0*3,e0*5]) )

wprecc = widget_label(wbase, /align_left, value= $
	'1st, 3rd, 5th harmonic central cone aperture (sigma) [mrad]: '+$
	vect2string([cc,cc/sqrt(3),cc/sqrt(5)]))

wprefr = widget_label(wbase, /align_left, value= $
	'1st, 3rd, 5th harmonic first ring aperture [mrad]: '+$
	vect2string([fr,fr/sqrt(3),fr/sqrt(5)]))

;
; calculation pars
;

wbase = widget_base(wWindow,/COL,/FRAME)
wtmp = widget_label(wbase,value='Calculation parameters', FONT=font)

wbase0 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase0,value='Emin [eV]: ',/align_left)
wemin = widget_text(wbase0,value=strcompress(inp.emin,/rem),/edit,xsize=9)
wtmp = widget_label(wbase0,value='Emax [eV]: ',/align_left)
wemax = widget_text(wbase0,value=strcompress(inp.emax,/rem),/edit,xsize=9)
wbase01 = widget_base(wbase0,/ROW) ; new base to allow mapping
  wtmp = widget_label(wbase01,value='Maximum semiaperture angle [mrad]: ',/align_left)
  wmaxangle = widget_text(wbase01,value=strcompress(inp.maxangle,/rem),/edit,$
	xsize=6)

wbase0 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase0,value='Number of rays : ',/align_left)
wnrays = widget_text(wbase0,value=strcompress(inp.nrays,/rem),/edit,$
	xsize=8)
wtmp = widget_label(wbase0,value='Random seed : ',/align_left)
wseed = widget_text(wbase0,value=strcompress(inp.seed,/rem),/edit,$
	xsize=8)
wtmp = widget_label(wbase0,value='Calculation code : ',/align_left)
wundul_phot_flag = cw_droplist(wbase0,value=inp.undul_phot_flag,UValue='UPDATE_PANELS')

wbase0 = widget_base(wbase,/ROW)
wtmp = widget_label(wbase0,value='Gridding points in energy: ',/align_left)
wng_e = widget_text(wbase0,value=strcompress(inp.ng_e,/rem),/edit,xsize=6)
wtmp = widget_label(wbase0,value='in theta: ',/align_left)
wng_t = widget_text(wbase0,value=strcompress(inp.ng_t,/rem),/edit,xsize=6)
wtmp = widget_label(wbase0,value='in phi: ',/align_left)
wng_p = widget_text(wbase0,value=strcompress(inp.ng_p,/rem),/edit,xsize=6)
wng_plot = cw_droplist(wbase0,title='Plot grid: ',value=inp.ng_plot)


wbase0 = widget_base(wbase,/ROW)
list = [strcompress(inp.f_bound_sour[0],/Remove_all), $
        'No','Using file with phase-space volume...', $
        'Using slit/acceptance...']
wF_BOUND_SOUR = CW_DropList(wbase0,Value=list,  $
      Title='Optimize source (reject rays):',UValue='F_BOUND_SOUR')


wids = { lambdau:wlambdau, k:wk, e_energy:we_energy, nperiods:wnperiods, $
  emin:wemin, emax:wemax, intensity:wintensity, maxangle:wmaxangle, $
  seed:wseed, sx:wsx, sz:wsz, ex:wex, ez:wez, $
  pree0:wpree0, precc:wprecc, prefr:wprefr, nrays:wnrays, $
  f_bound_sour:wf_bound_sour, $
  undul_phot_flag:wundul_phot_flag, $
  flag_emittance:wflag_emittance, $
  ng_e:wng_e, ng_t:wng_t, ng_p:wng_p, ng_plot:wng_plot }

state = { caller:wgroup, str:inp , wids:wids, input:input }
widget_control,wWindow,set_uvalue=state,/REALIZE
widget_control,wundul_phot_flag,set_droplist_select=fix(inp.undul_phot_flag(0))
widget_control,wflag_emittance,set_droplist_select=fix(inp.flag_emittance(0))
widget_control,wf_bound_sour,set_droplist_select=fix(inp.f_bound_sour(0))
xshundul_set_mappanels,wids

if keyword_set(input) then no_block=1 else no_block=0
xmanager,'xshundul_set', wWindow, group=group,no_block=no_block
END
;		
;========================================================================
;
PRO xshundul_run,str, remove=remove, group=group


common shadow3,shadow3_mode, shadow3_binary
;
on_error,2

; if using Gaussian approx, run Geometrical source
if  strcompress(str.undul_phot_flag(0),/rem) EQ '3' then begin
  print,'Using Geometrical approximation'

  openw,Unit,'shadow3.inp',/get_lun
  printf,Unit,'input_source'
  ;
  ; run input_source
  ;
  printf,Unit,'1'
  printf,Unit,'0'
  printf,Unit,strcompress(str.nrays,/rem)
  printf,Unit,strcompress(str.seed,/rem)
  
  ;printf,Unit,'0' ; optimization
  Printf,Unit,strcompress(str.f_bound_sour,/remove_all) ; Do you want to optimize the source ?
  case str.f_bound_sour of
    0:
    1: BEGIN
       Printf,Unit,strcompress(str.file_bound,/remove_all) 
       Printf,Unit,strcompress(str.ntotalpoint,/remove_all) 
       END
    2: BEGIN
       file = strcompress(str.file_bound,/remove_all) 
       openw,unit2,file,/get_lun
       printf,unit2,str.slit_distance, str.slit_xmin, str.slit_xmax, $
                                        str.slit_zmin, str.slit_zmax
       free_lun,unit2
       Printf,unit,strcompress(str.file_bound,/remove_all) 
       Printf,Unit,strcompress(str.ntotalpoint,/remove_all) 
       END
    else:
  endcase
  
  printf,Unit,'0' ; Regular(Geom/BM) source
  printf,Unit,'3' ; gauss
  ; calculate sizes of the photon undulator beam
  ; see formulas 25 & 30 in Ellaume' (Onaki & Ellaeume)
  l1 = str.lambdau*str.nperiods*1d2
  lambda1 = physical_constants('hc')/(str.emin+str.emax)*2*1d-8
  s_phot = 2.740/(4D0*!dpi)*sqrt(l1*lambda1)
  sp_phot = 0.69*sqrt(lambda1/l1)
  print,'XSHUNDUL_RUN:  Undul Length [cm]: ',l1
  print,'XSHUNDUL_RUN:  Phot wavelength [cm]: ',lambda1
  print,'XSHUNDUL_RUN:  s_phot = 2.740/(4 pi)*sqrt(lambda L) [cm]: ',s_phot
  print,'XSHUNDUL_RUN:  sp_phot = 0.69*sqrt(lambda/L) [rad]: ',sp_phot
  print,'XSHUNDUL_RUN:  s_elec (z) [cm]: ',str.sz
  print,'XSHUNDUL_RUN:  sp_elec (z) [rad] : ',str.ez/str.sz
  if fix(str.flag_emittance[0]) EQ 0 then begin
      printf,Unit,s_phot
      printf,Unit,s_phot
  endif else begin
      printf,Unit,strcompress(sqrt(s_phot^2+str.sx^2),/rem)
      printf,Unit,strcompress(sqrt(s_phot^2+str.sz^2),/rem)
  endelse
  printf,Unit,'1' ; No depth
  printf,Unit,'3' ; gauss (divergences)
  printf,Unit,'1.0' ; Limits (1=Infinity)
  printf,Unit,'1.0' ; Limits (1=Infinity)
  printf,Unit,'1.0' ; Limits (1=Infinity)
  printf,Unit,'1.0' ; Limits (1=Infinity)
  if fix(str.flag_emittance[0]) EQ 0 then begin
      printf,Unit,sp_phot
      printf,Unit,sp_phot
  endif else begin
      printf,Unit,strcompress(sqrt(sp_phot^2+(str.ez/str.sz)^2),/rem) ; first vertical!!!!
      printf,Unit,strcompress(sqrt(sp_phot^2+(str.ex/str.sx)^2),/rem)
  endelse
  printf,Unit,'1' ; phot energy
  printf,Unit,'3' ; box (1=single)
  printf,Unit,'0' ; eV
  printf,Unit,strcompress(str.emin,/rem)
  printf,Unit,strcompress(str.emax,/rem)
  printf,Unit,'1' ; OPD
  printf,Unit,'1' ; 18 cols
  printf,Unit,'0.0' ; pol phase
  printf,Unit,'1.0' ; pol degree
  printf,Unit,'1' ; coh
  
  printf,Unit,'source'
  printf,Unit,'systemfile'
  printf,Unit,'exit'
  free_lun,Unit
  command = shadow3_binary+' < shadow3.inp'
  widget_control,/hourglass
  xsh_run,command
  return
endif

; if using SRW, do it first and return
if  strcompress(str.undul_phot_flag(0),/rem) EQ '2' then begin
   write_gfile,str,'xshundul.json',/json
   ;command = ['. ./shadow3/setpythonpath.sh init','python srw_xshundul.py']
   ;outfile = 'shadowvui_srw.sh'
   ;openw,unit,outfile,/get_lun
   ;for i=0L,n_elements(command)-1 do printf,unit,command[i]
   ;free_lun,unit
   ;command = "chmod +x "+outfile+" ; ./"+outfile
   ;print,"Executing : "+command
   ;spawn,command
   xsh_python,'srw_xshundul.py'

   ; visualize 
   
   if  fix(str.ng_plot[0]) EQ 1 then begin
       xplot,spec='srw_xshundul.spec',xcol=2,ycol=3,parent=p
       xplot_mesh,p,flag=1,col=-1,interactive=0          
       xplot_controls_action,p,charsize=2.0,/norefresh
       xplot_refresh,p  
   endif
   return
endif


;
; run epath
;
CASE shadow3_mode OF
  0: openw,Unit,'xsh_epath_tmp.inp',/get_lun
  else: BEGIN
        openw,Unit,'shadow3.inp',/get_lun
        printf,Unit,'epath'
        END
ENDCASE
printf,Unit,'2'
printf,Unit,strcompress(str.lambdau,/rem)
printf,Unit,strcompress(str.k,/rem)
printf,Unit,strcompress(str.e_energy,/rem)
printf,Unit,'101'
printf,Unit,'1.'
printf,Unit,'xshundul.par'
printf,Unit,'xshundul.traj'
printf,Unit,'0'
IF shadow3_mode EQ 0 THEN free_lun,Unit

;xsh_run,'epath < xsh_epath_tmp.inp'
command = ['epath < xsh_epath_tmp.inp']

;
; run undul_set
;
IF shadow3_mode EQ 0 THEN BEGIN
  openw,Unit,'xsh_undul_set_tmp.inp',/get_lun
ENDIF ELSE BEGIN
  printf,Unit,'undul_set'
ENDELSE
printf,Unit,'0'
printf,Unit,'0'
;printf,Unit,'11'
;printf,Unit,'31'
;printf,Unit,'31'
printf,Unit,strcompress(str.ng_e,/remove_all)  ; points in energy
printf,Unit,strcompress(str.ng_t,/remove_all)  ; points in theta
printf,Unit,strcompress(str.ng_p,/remove_all)  ; points in phi
printf,Unit,'xshundul.traj'
printf,Unit,strcompress(str.nperiods,/rem)
printf,Unit,strcompress(str.emin,/rem)
printf,Unit,strcompress(str.emax,/rem)
printf,Unit,strcompress(str.intensity,/rem)
printf,Unit,strcompress(str.maxangle,/rem)
printf,Unit,'0'
printf,Unit,'1000'
IF shadow3_mode EQ 0 THEN BEGIN
   free_lun,Unit
   ;xsh_run,'undul_set < xsh_undul_set_tmp.inp'
   command = [command,'undul_set < xsh_undul_set_tmp.inp']
ENDIF 

; 
;run undul_phot(shadow) or cdf_z(urgent)
;
if  strcompress(str.undul_phot_flag(0),/rem) EQ '0' then begin
  ;xsh_run,'undul_phot'
  IF shadow3_mode EQ 0 THEN BEGIN
    command = [command,'undul_phot']
  ENDIF ELSE BEGIN
    printf,Unit,'undul_phot'
  ENDELSE
endif else begin ; using URGENT code
  IF shadow3_mode EQ 0 THEN BEGIN
    openw,Unit,'UNDUL_Z.PAR',/get_lun
    printf,Unit,str.lambdau, 0.0, str.k, str.nperiods, str.e_energy, $
	  str.intensity, 0, 0
    free_lun, Unit
    ;xsh_run,'cdf_z'
    command = [command,'cdf_z']
    ; 
    ; check if cdf_z is installed
    ;
    ffile = Getenv('SHADOW_ROOT')+SDep(/ds)+'bin'+SDep(/ds)+'cdf_z'
    IF SDEP() EQ 'WINDOWS' THEN  ffile=ffile+'.exe'
    IF checkfile(ffile) NE 1 THEN BEGIN
       itmp = Dialog_Message(/Error,Dialog_Parent=group,[$
          ' File Not Found: '+ffile,'','',$
          ' To run Undulator with URGENT code you must download the '+$
                  'cdf_z code from: ','',$
          ' http://ftp.esrf.eu/pub/scisoft/shadow/user_contributions','',$
          ' and install it in your shadow bin directory (listed before)'])
       RETURN
    ENDIF
  ENDIF ELSE BEGIN ; shadow3
    ;printf,Unit,'cdf_znew'
    printf,Unit,'undul_phot_urgent'
  ENDELSE
endelse
;
;run undul_phot_dump in shadow3
;
if  strcompress(str.ng_plot(0),/rem) EQ '1' then begin
    delete_files,['uphot.spec']
    IF shadow3_mode EQ 0 THEN BEGIN ; not implemented
    ENDIF ELSE BEGIN
      printf,Unit,'undul_phot_dump'
    ENDELSE
endif
;
;run undul_cdf
;
IF shadow3_mode EQ 0 THEN BEGIN
  openw,Unit,'xsh_undul_cdf_tmp.inp',/get_lun
ENDIF ELSE BEGIN
  printf,Unit,'undul_cdf'
ENDELSE
printf,Unit,'0'
printf,Unit,'1'
printf,Unit,'xshundul.sha'
printf,Unit,'xshundul.info'


IF shadow3_mode EQ 0 THEN BEGIN
  free_lun,Unit
  
  ;xsh_run,'undul_cdf < xsh_undul_cdf_tmp.inp'
  command = [command,'undul_cdf < xsh_undul_cdf_tmp.inp']
ENDIF 

;
; run input_source
;
IF shadow3_mode EQ 0 THEN BEGIN
  openw,Unit,'xsh_input_source_tmp.inp',/get_lun
ENDIF ELSE BEGIN
  printf,Unit,'input_source'
ENDELSE
printf,Unit,'1'
printf,Unit,'0'
printf,Unit,strcompress(str.nrays,/rem)
printf,Unit,strcompress(str.seed,/rem)

;printf,Unit,'0' ; optimization
Printf,Unit,strcompress(str.f_bound_sour,/remove_all) ; Do you want to optimize the source ?
case str.f_bound_sour of
  0:
  1: BEGIN
     Printf,Unit,strcompress(str.file_bound,/remove_all) 
     Printf,Unit,strcompress(str.ntotalpoint,/remove_all) 
     END
  2: BEGIN
     file = strcompress(str.file_bound,/remove_all) 
     openw,unit2,file,/get_lun
     printf,unit2,str.slit_distance, str.slit_xmin, str.slit_xmax, $
                                      str.slit_zmin, str.slit_zmax
     free_lun,unit2
     Printf,unit,strcompress(str.file_bound,/remove_all) 
     Printf,Unit,strcompress(str.ntotalpoint,/remove_all) 
     END
  else:
endcase

printf,Unit,'2' ; undulator
printf,Unit,'xshundul.sha'
if fix(str.flag_emittance[0]) EQ 0 then begin
    printf,Unit,'0.0'
    printf,Unit,'0.0'
    printf,Unit,'0.0'
    printf,Unit,'0'
    printf,Unit,'0.0'
    printf,Unit,'0'
endif else begin
    printf,Unit,strcompress(str.sx,/rem)
    printf,Unit,strcompress(str.sz,/rem)
    printf,Unit,strcompress(str.ex,/rem)
    printf,Unit,'0'
    printf,Unit,strcompress(str.ez,/rem)
    printf,Unit,'0'
endelse
printf,Unit,'3'
printf,Unit,'1'
printf,Unit,'1'

IF shadow3_mode EQ 0 THEN BEGIN
  free_lun,Unit
  ;xsh_run,'input_source < xsh_input_source_tmp.inp'
  command = [command,'input_source < xsh_input_source_tmp.inp']
ENDIF

; 
; run gen_source
;
;xsh_run,'gen_source start.00'
IF shadow3_mode EQ 0 THEN BEGIN
  command = [command,'gen_source start.00']
ENDIF ELSE BEGIN
  printf,Unit,'source'
  printf,Unit,'systemfile'
  printf,Unit,'exit'
  free_lun,Unit
  command = shadow3_binary+' < shadow3.inp'
ENDELSE

;
; run everything
;
widget_control,/hourglass
xsh_run,command

if (keyword_set(remove) AND (shadow3_mode GT 0 ) ) then begin
  command='/bin/rm xsh_*_tmp.inp'
  message,/info,'Executing: '+command
  spawn,command
endif
print,'*************** SHADOW UNDULATOR SOURCE COMPLETED ***************'

; visualize 
if  fix(str.ng_plot[0]) EQ 1 then begin
    xplot,parent=p,spec='uphot.spec',xcol=1,ycol=4,wtitle="Photon angle distributions"
    xplot_mesh,p,flag=1,col=-2,interactive=0          
    xplot_settitles,p,xtitle="THETA [rad]",YTITLE="PHI [rad]"
    ;xplot_settitles,p,xtitle="-1",YTITLE="-1"
    xplot_controls_action,p,charsize=2.0,/norefresh
    xplot_refresh,p  
endif

end
;		
;========================================================================
;
PRO xshundul_event,event
;
; register the events
;
WIDGET_CONTROL, event.id, GET_UVALUE=eventuval
widget_control, event.top, GET_UVALUE=inp ; , /no_copy

CASE eventuval OF
	'QUIT': BEGIN
		WIDGET_CONTROL,event.top,/DESTROY
		return
		END
	'FILELOAD': BEGIN
		ffile = dialog_pickfile(title='Select Xwiggler input file...',/NOCONF)
		if strcompress(ffile,/rem) EQ '' then return
		version = ''
		restore,ffile,/VERBOSE
		if version NE xshundul_nameversion() then itmp = $
			widget_message( $
		  'Input file and current XShUndul have different versions.')
		END
	'FILEWRITE': BEGIN
		ffile = dialog_pickfile(/WRITE,file='xshundul.inp')
		if strcompress(ffile,/rem) EQ '' then goto,out
		version = xshundul_nameversion()
		command = 'save,version,inp,'+$
		'FILENAME='+"'"+strcompress(ffile,/rem)+"'"+',/VERBOSE'
		tmp = execute(command)
		itmp = widget_message(/INFO,'File '+ffile+ $
			' written to disk.')
		END
	'SETPAR': begin
		; help,'calling _set; event.top = ',event.top
		xshundul_set,inp,GROUP=event.top
		end
        'HELP': BEGIN
                 DOC_LIBRARY,'xshundul',PRINT='cat > xshundul.hlp'
                 XDISPLAYFILE1,'xshundul.hlp',GROUP=event.top,HEI=48,/REMOVE
                END
	'SH13':	begin
		if not(checkfile('begin.dat')) then begin
		  tmp = widget_message(/error,["File 'begin.dat' not found.",$
			'Please run SHADOW.'])
		  goto,out
		endif
		command = " plotxy,'begin.dat',1,3,/gauss"
		xwindow,group=event.top,wtitle=command,buffer=command,/edit
		end
	'SH46':	begin
		if not(checkfile('begin.dat')) then begin
		  tmp = widget_message(/error,["File 'begin.dat' not found.",$
			'Please run SHADOW.'])
		  goto,out
		endif
		command = "plotxy,'begin.dat',4,6,/gauss"
		xwindow,group=event.top,wtitle=command,buffer=command,/edit
		end
	'SH11':	begin
		if not(checkfile('begin.dat')) then begin
		  tmp = widget_message(/error,["File 'begin.dat' not found.",$
			'Please run SHADOW.'])
		  goto,out
		endif
		command = "histo1,'begin.dat',11"
		xwindow,group=event.top,wtitle=command,buffer=command,/edit
		end
	'SHINFOSH': begin
		if not(checkfile('begin.dat')) then begin
		  tmp = widget_message(/error,["File 'begin.dat' not found.",$
			'Please run SHADOW.'])
		  goto,out
		endif
		infosh,'begin.dat'
		end
	'SHSOURCINFO':	begin
		SOURCINFO
                XDISPLAYFILE1,'und1.par',GROUP=event.top,HEI=24
                XDISPLAYFILE1,'und1.info',GROUP=event.top,HEI=24
		end
ENDCASE
out:
widget_control,event.top,set_uvalue=inp ; ,/NO_COPY
END
;
;==============================================================================
;
PRO XSHUNDUL, input, GROUP=group, Only_Run=only_run
;
on_error,2

;
;if xregistered('xshundul') then return

if n_params() EQ 1 then begin
  ;if not(checkfile(file)) then begin
  ;  itmp = widget_message('Error opening file '+file,/ERROR)
  ;  return
  ;endif
  ;restore,file,/VERBOSE
  ;if version NE xshundul_nameversion() then begin
  ;  itmp = widget_message($
  ;  'Input file and current XShUndul have different versions.')
  ;  return
  ;endif
; help,input
  inp = *input
endif else begin
  inp = xsh_defaults('xshundul')
  input = 0
endelse

IF Keyword_Set(only_Run) THEN BEGIN
  XShUndul_Run,inp,group=group
  RETURN
ENDIF

if not(keyword_set(input)) then begin
  base=WIDGET_BASE(/COLUMN,TITLE=xshundul_nameversion())
  XPdMenu, [	'"File" {', $

			'"Load XShUndul input file..."   FILELOAD',$
			'"Write XShUndul input file..."  FILEWRITE',$
			'"Quit"				QUIT',$
                        '}',$
		'"Set parameters"			SETPAR',$
		'"Show" {',$
                        '"Real Space X-Z (Cols 1,3)"    	SH13',$
                        '"Divergence Space X-Z (Cols 4,6)"    	SH46',$
                        '"Energy dependence (Col 11)"    	SH11',$
                        '"Numerical values"    			SHINFOSH',$
                        '"Sourcinfo"    			SHSOURCINFO',$
                        '}',$
                '"Help"					HELP'$
          ], base


  case sdep() of
    'UNIX': font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
    'WINDOWS': font = 'VERDANA*BOLD*ITALIC*36'  
    else: font=''
  endcase
  junk = WIDGET_LABEL( BASE, $
      FONT=font, VALUE=' SHADOWVUI ')
  junk = WIDGET_LABEL( BASE, $
      FONT=font, VALUE='   XShUndul')
  junk = WIDGET_LABEL( BASE, $
      FONT=font, VALUE='Shadow Undulator Source')
  ;help,base
  widget_control,base,set_uvalue=inp ; ,/NO_COPY
  WIDGET_CONTROL,/REALIZE,base
  XMANAGER,'xshundul',base,GROUP_LEADER=group,/no_block
endif else begin
  xshundul_set,inp,GROUP=group,input=input
endelse

end
;
