function READ_MOSFLM, filename, raster=raster, par,  detector=detector
;+
; NAME:
;	READ_MOSFLM
;
; PURPOSE:
;
; 	Read in an (ASCII) ".x" file from the MOSFLM software. Returns a 
;	structure array containing the information for each reflection.
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	ref = READ_MOSFLM(filename,par,[raster=raster],[detector=detector]) ;
;
; INPUTS:
;	filename : the path name of the file to process (with extension)
;	[raster] : the raster [microns] if different from 150.
;	[detector]: set this to the type of detector which is used [mar18,mar30,mar345,marccd]. Default = mar18
;
; OUTPUTS:
;	ref : the structure array containing the reflections.
;	[par] : optionnal output containing general info. 
;		It is also a structure. [see procedure]
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;   CRYSTAL	FLTARR(6) : alpha,beta,gamma,a*,b*,c* in reciprocal space
;   PHIS        FLTARR(3) : the crystal setting angles [degrees]
;   CDD         FLOAT     : crystal to detector distance [mm]
;   RASTER      FLOAT     : pixel size [um]
;   RASTX       INT       : number of pixel in X dimension
;   RASTY       INT       : number of pixel in Y dimension
;   CENX        FLOAT     : X position of beam center [pixels] from lower left of image
;   CENY        FLOAT     : Y position of beam center [pixels] from lower left of image
;   RADIUS      FLOAT     : maximum radius from beam center [mm] used for prdicted spots
;   OSC_START   FLOAT	  : start value of the oscillation
;   OSC_END     FLOAT	  : end value of the oscillation
;   WAVE	FLOAT	  : wavelength
;   LMIN        FLOAT     : minimum wavelength [A] (useless)
;   LMAX        FLOAT     : maximum wavelength [A] (useless)
;   DMIN        FLOAT     : resolution limit [A] used for predicted spots
;   YSCALE      FLOAT     : Y scale for pixel non squareness
;   TWIST       FLOAT     : correction for twist [0.01 degrees]
;   TILT        FLOAT     : correction for tilt [0.01 degrees]
;   BULGE       FLOAT     : correction for bulge [0.01 degrees]
;   R_REJ       FLOAT     : minimum radius from beam center [mm] used for predicted spots
;   GAIN        FLOAT     : gain of the detector at 1.5418 A
;   BG_GAIN     FLOAT     : gain of the detector at the mean background wavelength
;   BEAMSIZE    FLOAT     : beamsize [mm]
;   DARK_CURRENT    FLOAT : dark current of the detector [counts]
;   BOXSIZE     STRUCTURE : size of box for data integration [pixel]
;   NOISE_PERCENTAG INT   : maximum percentage of the pixels taken into account for 
;			    background calculation in integration box
;      Remark --> The last 6 parameters are entered later with the progrm WRITEPAR_LAUE.PRO
;		  They are used for IDL data integration. 
;
;      Data -->
;	M       INTARR(3) :The Miller indices h,k,l
;  	X       FLOAT     : X position from beam center [mm]
;   	Y       FLOAT     : Y position from beam center [mm]
;  	XPIX    FLOAT     : X position in image [raster]
;   	YPIX    FLOAT     : Y position in image [raster]
;   	L       FLOAT     : Wavelength [A]
;   	MUL     INT       : Multiplicity index
;   	NODAL   INT       : 0 if not nodal ; 1 if yes (ie |h|< 12 && |k|<12 && |l|<12)
;   	OVLP    INT       : 0 if not overlap ; 1 if yes. This field will usually be 
;				overwritten if the routine FIND_OVLP.PRO is used
;   	OVLP2   INT       : Also for overlaps
;  	BC_COEFF INTARR(3): Coefficients for background.
;   	ID      LONG       : Identification # of spot (used for IDL integration)
;   	CONNECT  INTARR(6) : Contains the identification # of the overlapping spots
;   	CLOSE   INT        : 0 if not close, 1 if yes
;  	M2      INTARR(3)  : Nodal Miller indices for second harmonic (ie: 2* .m)
;   	NIDX    INT        : Nodal spot selection index
;   	L2      FLOAT      : Wavelength for second harmonic (for doubles only, otherwise 0)
;   	IPOINT  INT        : Pointer to .ge2 file for a multiple (>2) 
;   	DMINTH  FLOAT      : Dmin threshold squared for next harmonic
;   	INTB    FLTARR(6)  : Box integrated intensities for 6 films
;  	ISIGB   FLTARR(6)  : Sigmas for box integrated intensities for 6 films
;  	INTP    FLTARR(6)  : Profile fitted integrated intensities for 6 films
;  	ISIGP   FLTARR(6)  : Sigmas for profile fitted integrated intensities for 6 films
;  	TTH     FLOAT      : Two theta angle [degrees]
;   	RES     FLOAT      : Resolution [A] in direct space
;   	STRENGTH FLOAT     : Strength of average profile [arb units]
;   	CHISQUARE FLOAT    : chi_square of profile fitting in DENZO
;   	COSINE  FLOAT      : cosine of incidence angle at detector
;   	LP      FLOAT      : Lorentz polarization factor
;       PARTIALITY FLOAT   : Partiality [0=full ; 1=partial]
;   	PAR1    FLOAT      : Additional parameter (unknown) 1
;   	PAR2    FLOAT      : Additional parameter (unknown) 2
;
; MODIFICATION HISTORY:
;	D.Bourgeois, November 1994.
; 	0.Svensson, January 1995
;-

;on_error,2                              ;Return to caller if an error occurs

;open the 'x' file 
dum = FINDFILE(filename,count=count)
IF count EQ 1 THEN PRINT,'Reading ',filename ELSE BEGIN 
PRINT,'File not found : ',filename
GOTO,escape
ENDELSE

;give a default detector
IF N_ELEMENTS(detector) EQ 0 THEN detector='mar18'

par = {denzo_par, $
	crystal: FLTARR(6),$
	phis: FLTARR(3), $
	mosa : 0.0, $
	low_res : 0.0, $
	hi_res : 0.0, $
	cdd: 0.0 , $
	wave: 0.0 , $ 
	raster: 0.0 , $
  	rastx: 0 , $
	rasty: 0 , $
	cenx: 0.0 , $
	ceny: 0.0 , $
	radius: 0.0 , $
	osc_start: 0.0, $
	osc_end: 0.0, $
	lmin: 0.0, $
	lmax: 0.0, $
	dmin: 0.0, $
  	yscale: 1.0, $
	twist: 0.0, $
	tilt: 0.0, $
	bulge: 0.0, $
  	r_rej: 0.0, $
	gain: 1.0, $
	bg_gain: 1.0, $
	beamsize: 0.3, $
	sample_depth: 0.0, $
	dark_current: 0.0, $
  	boxsize: {box, x:0,y:0} , $
	satvalue : 0L, $
	ovlp_dist : 0.0, $
	bc_extend : 2, $
	region_level : 0.5, $
	ovlp_min_dist : 0.0, $
	noise_percentage : 30.}


 
predictions = {mosflm, $
	m: [0,0,0], $
	x: 0.0, $
	y: 0.0, $
	l: 0.0, $
	mul: 1, $
        strength: 0.0, $
        xpix : 0.0, $
        ypix : 0.0, $
        chisquare : 0.0, $
	cosine: 0.0, $
	nodal: 0.0, $
	ovlp: 0, $
	ovlp2: 0, $
	id: 0L, $
	sector:0, $
	connect: LONARR(24), $
	close: 0.0, $
	m2: [0,0,0], $
	nidx: 0, $
	l2: 0.0, $
	ipoint: 0, $
	dminth: 0.0, $
	intb:	[0.0,0.0,0.0,0.0,0.0,0.0], $
	isigb:	[0.0,0.0,0.0,0.0,0.0,0.0], $
	intp:	[0.0,0.0,0.0,0.0,0.0,0.0], $
	isigp:	[0.0,0.0,0.0,0.0,0.0,0.0], $
  	tth: 0.0, $
	res: 0.0, $
	bc_coeff: FLTARR(3), $
        flag: 0, $
	gfit : 0.0, $
        lp: 0.0, $
        partiality: 0.0, $
	phi:0.,$
	par1 : 0., $
	par2 : 0. }

line = STRING(90)

;get the number of reflexions (approximate) in filename
SPAWN,'cat '+filename+' | wc',n_ref
ref = REPLICATE({mosflm}, n_ref(0))

OPENR,lun,filename,/GET_LUN
;find the parameters :

;IF KEYWORD_SET(noyork) THEN skip_header=5 ELSE skip_header=7
;FOR i=1,skip_header DO READF, lun, line

cr = STRING("15b)

index = 0L
;format = '(3I4,I2,2F8.1,F7.2,F6.1,F6.3,2F7.1,F6.3,F8.1,F6.3,F6.1)'
format = '(3I4,2I4,4F12.4,F5.2,2F9.2,F7.2,3F6.2)'

REPEAT BEGIN
 IF index/100 EQ index/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",index,100.*index/FLOAT((SIZE(ref))(1)),cr

 READF,lun,line
 IF (STRLEN(line) GE 40) THEN BEGIN

  IF (STRPOS(line,'**') NE -1) THEN BEGIN
    line=STR_REPLACE(line,'*',' ',/NO_COMPRESS)
   REPEAT BEGIN
    line=STR_REPLACE(line,'*','9',/NO_COMPRESS)
   ENDREP UNTIL (STRPOS(line,'*') EQ -1) 
  ENDIF 
  
  READS,line,format=format,h,k,l,m_isym,batch,boxint,sigi,ipr,sigipr,fractioncalc,xdet,ydet,rot,width,lp,mpart

  ref(index).m(0) = h
  ref(index).m(1) = k
  ref(index).m(2) = l
  ref(index).partiality = fractioncalc
  ref(index).intp(0) = ipr
  IF lp GT 0 THEN ref(index).intp(5) = ipr/lp ELSE ref(index).intp(5) = 0
  ref(index).intb(1) = boxint
  IF lp GT 0 THEN ref(index).intb(5) = boxint/lp ELSE ref(index).intb(5) = 0
  ref(index).chisquare = 0
  ref(index).isigp(0) = sigipr 
  ref(index).cosine = width
  ref(index).xpix = xdet
  ref(index).ypix = ydet
  ref(index).lp = lp
  ref(index).strength = 0
  ref(index).id = index
  ref(index).mul = 1
      index = index+1
 ENDIF 
ENDREP UNTIL ((STRLEN(line) LE 40) OR EOF(lun))

CASE detector OF
 'mar18' : BEGIN
		par.rastx = 1200 
		par.rasty = 1200 
	   END

 'mar30' : BEGIN
		par.rastx = 2000 
		par.rasty = 2000 
	   END

 'mar345' : BEGIN
		par.rastx = 2300 
		par.rasty = 2300 
	   END

 'marccd' : BEGIN
		par.rastx = 2048 
		par.rasty = 2048 
	   END
ENDCASE
PRINT,'Assumed raster [mm] : ',0.001*par.raster
PRINT,'Assumed size [X,Y] [raster] : ', par.rastx,par.rasty
;par.cenx = x_beam/(0.001*par.raster)
;par.ceny = y_beam/(0.001*par.raster)
PRINT,'Position of beam center [x,y] [raster] : ',par.cenx,par.ceny

FREE_LUN,lun

ref = ref(0:index-1)
;ref.x = (ref.xpix-par.cenx+1)*par.raster/1000.
;ref.y = (ref.ypix-par.ceny+1)*par.raster/1000.
;ref.tth = ATAN(SQRT(ref.x^2+ref.y^2)/par.cdd)*!RADEG
;ref.res = par.wave/(2.0*SIN(ref.tth/!RADEG/2.0))
;ref.l = par.wave
;ref.phi = 0.5*(par.osc_start+par.osc_end)




PRINT,'**** NUMBER OF READ REFLEXIONS : ',(SIZE(ref))(1)
PRINT,''
RETURN,(ref)

escape :
END







