function READ_LDM, filename
;+
; NAME:
;	READ_LDM
;
; PURPOSE:
;
; 	Read in an (ASCII) ".ldm" from the Laue suite CCP4. Returns a 
;	structure array containing the information for indexing
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	par = READ_LDM(filename) ;
;
; INPUTS:
;	filename : the path name of the file to process with extension
;
; OUTPUTS:
;	par : the structure array containing the data
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;   PHIS        FLTARR(3) : the crystal setting angles [degrees]
;   CELL        FLTARR(6) : the crystal cell
;   SPINDLE	FLOAT  	  : the spindle value
;   CTOF        FLOAT     : crystal to detector distance [mm]
;   RASTER      FLOAT     : pixel size [um]
;   RASTX       INT       : number of pixel in X dimension
;   RASTY       INT       : number of pixel in Y dimension
;   CENX        FLOAT     : X position of beam center [pixels] from lower left of image
;   CENY        FLOAT     : Y position of beam center [pixels] from lower left of image
;   RADIUS      FLOAT     : maximum radius from beam center [mm] used for prdicted spots
;   LMIN        FLOAT     : minimum wavelength [A]
;   LMAX        FLOAT     : maximum wavelength [A]
;   DMIN        FLOAT     : resolution limit [A] used for predicted spots
;   YSCALE      FLOAT     : Y scale for pixel non squareness
;   TWIST       FLOAT     : correction for twist [0.01 degrees]
;   TILT        FLOAT     : correction for tilt [0.01 degrees]
;   BULGE       FLOAT     : correction for bulge [0.01 degrees]
;
;
; MODIFICATION HISTORY:
;	D.Bourgeois, June 2004.
;-

;on_error,2                              ;Return to caller if an error occurs

par = { $
		cell : FLTARR(6), $
	  	phis : FLTARR(3), $
		spindle : 0.0, $
		low_res : 0.0, $
		hi_res : 0.0, $
		ctof : 0.0, $
		raster : 0.0, $
		rastx : 0, $
		rasty : 0, $
		cenx : 0.0, $
		ceny : 0.0, $
		radius : 0.0, $
		osc_start: 0.0, $
		osc_end: 0.0, $
		lmin : 0.0, $
		lmax : 0.0, $
		dmin : 0.0, $
		yscale : 0.0, $
		twist : 0.0, $
		tilt : 0.0, $
		bulge : 0.0, $
		r_rej : 0.0 }



;initialize par
dummy=0.0
cell = FLTARR(6)
axis = FLTARR(3)
angle = FLTARR(3)
phis = FLTARR(3)
lmin=0.0 & lmax=0.0 
xc_shift=0.0 & yc_shift=0.0 & centerx=0.0 & centery=0.0
radius=0.0 & dmin=0.0
twist=0.0 & tilt=0.0 & bulge=0.0
yscale=0.0
r_rej=0.0
rastx=0 & rasty=0
raster=0.0
ctof=0.0
spindle=0.0

 ;open the 'ldm' file 
  ldmfile=FINDFILE(filename,COUNT=count)
  IF count EQ 1 THEN BEGIN
   line=''
   OPENR,lun,filename,/GET_LUN
   PRINT,'Reading .ldm file ...'
   WHILE (NOT EOF(lun)) DO BEGIN
    READF, lun, line
    IF (STRMID(line, 0, 6) EQ 'NXRAST') THEN BEGIN
     READS,line,FORMAT='(6X,I5,7X,I5)',rastx,rasty
    ENDIF
    IF (STRMID(line, 0, 11) EQ 'RASTER_SIZE') THEN BEGIN
     READS,line,FORMAT='(11X,F7.3)',raster
    ENDIF
    IF (STRMID(line, 0, 2) EQ 'A ') THEN BEGIN
     READS,line,FORMAT='(2X,F6.3)',a
     np = STRPOS(line, 'B ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(2X,F6.3)',b
     np = STRPOS(line, 'C ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(2X,F6.3)',c
     np = STRPOS(line, 'ALPHA ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(6X,F6.3)',alpha
     np = STRPOS(line, 'BETA ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(5X,F6.3)',beta
     np = STRPOS(line, 'GAMMA ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(6X,F6.3)',gamma
     cell = [a,b,c,alpha,beta,gamma]
    ENDIF

    IF (STRMID(line, 0, 4) EQ 'PHIX') THEN BEGIN
     READS,line,FORMAT='(5X,F6.3)',phix
     np = STRPOS(line, 'PHIY')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(5X,F6.3)',phiy
     np = STRPOS(line, 'PHIZ')
     subline = STRMID(line, np, 10)
     READS,subline,FORMAT='(5X,F6.3)',phiz
     phis = [phix,phiy,phiz]
    ENDIF


    IF (STRMID(line, 0, 5) EQ 'TWIST') THEN BEGIN
     READS,line,FORMAT='(6X,F7.3)',twist
     np = STRPOS(line, 'TILT')
     subline = STRMID(line, np, 20)
     READS,subline,FORMAT='(5X,F7.3)',tilt
     np = STRPOS(line, 'BULGE')
     subline = STRMID(line, np, 20)
     READS,subline,FORMAT='(6X,F7.3)',bulge
    ENDIF


    IF (STRMID(line, 0, 4) EQ 'CTOF') THEN BEGIN
     READS,line,FORMAT='(5X,F7.3)',ctof
    ENDIF

    IF (STRMID(line, 0, 7) EQ 'SPINDLE') THEN BEGIN
     READS,line,FORMAT='(8X,F7.3)',spindle
    ENDIF


   ENDWHILE
   FREE_LUN,lun
  ENDIF ELSE MESSAGE,'Could not find .ldm file !'

par.cell=cell
par.phis=phis
par.spindle=spindle
par.osc_start=spindle
par.osc_end=spindle
par.lmin=lmin
par.lmax=lmax
par.dmin=dmin
par.ctof=ctof
par.radius=radius
par.cenx = 1000*(centerx(0) + xc_shift(0))/raster
par.ceny = 1000*(centery(0) + yc_shift(0))/raster*yscale(0)
par.twist=twist(0)
par.tilt=tilt(0)
par.bulge=bulge(0)
par.yscale=yscale(0)
par.r_rej=r_rej ;this parameter not present in the geasc file
par.rastx=rastx ;this parameter not present in the geasc file
par.rasty=rasty ;this parameter not present in the geasc file
par.raster=raster ;this parameter wrong in the geasc file

RETURN,par
END

