pro REFINE_COORD, sel_refl, refl, par, info=info, $
	tol=tol, dampfactor=dampfactor, stepfactor=stepfactor, $
	max_iter=max_iter, sigma=sigma, $
	phix=phix, phiy=phiy, phiz=phiz, cdd=cdd, cenx=cenx, ceny=ceny, $
	yscale=yscale, twist=twist, tilt=tilt, bulge=bulge, $
	alphastar=alphastar, betastar=betastar, gammastar=gammastar, $
	astar=astar, bstar=bstar, cstar=cstar
;+
; NAME:
;	REFINE_COORD
;
; PURPOSE:
;	Refine some parameters that determine the positions of Laue spots
;	on the detector given the observed positions.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	REFINE_COORD, sel_refl, refl, par, info=info, $
;	phix=phix, phiy=phiy, phiz=phiz, cdd=cdd, cenx=cenx, ceny=ceny, $
;	yscale=yscale, twist=twist, tilt=tilt, bulge=bulge, $
;	alphastar=alphastar, betastar=betastar, gammastar=gammastar, $
;	astar=astar, bstar=bstar, cstar=cstar
;
; INPUTS:
;	The positions of the spots are given in mm.
;	SEL_REFL: Array of reflections in the same form as output from 
;		the routine READ_GEASC2.PRO. The tags X and Y should 
;		contain the observed positions. These are used for the 
;		refinement. The positions are updated after refinement
;		(see below).
;	REFL: Same type as SEL_REFL but containing all reflections. The tags 
;		X and Y should contain the observed positions. In this array
;		is the new predicted positions put after refinement. Thus 
;		in the case of repeated calls of REFINE_COORD use 
;	PAR: The parameters (from READ_GEASC2.PRO + WRITEPAR_LAUE2.PRO). 
;		Structure containing the parameters that we are refining.
;
; KEYWORDED PARAMETERS:
;	INFO: If set then some information about the refinement is printed.
;	TOL: The "tolerance". If the relative shift in the CHISQ is less
;		than this (but it IS decreasing) then the refinement is
;		considered to be converged. Default=0.001.
;	DAMPFACTOR: The shift that MRQCOF determines for a parameter is
;		multiplied by this number before the shift is applied.
;		Default=1.0.
;	STEPFACTOR: Used when calculating the partial derivatives. The
;		partial derivatives are calculated as central approximations
;		with default stepsizes (different for each parameter) and
;		these default stepsizes are multiplied with STEPFACTOR.
;		Default 0.1.
;	MAX_ITER: The maximum number of iterations. Default 50.
;	SIGMA: If input then this is used as the sigma in the calculation
;		of CHISQ. The calculated sigma is output. Default is the
;		raster size. [mm]
;
;	The keywords corresponding to the parameters to refine should be set.
;	Possible refinement parameters:
;	PHIX, PHIY, PHIZ, CDD, CENX, CENY, YSCALE, TWIST, TILT, BULGE,
;	ALPHASTAR, BETASTAR, GAMMASTAR, ASTAR, BSTAR, CSTAR
;	Note: If YSCALE is refined then the whole scale should also
;	be refined by for example refining CDD at the same time.
;
; OUTPUTS:
;	After completion the coordinates in REFL and the parameters in 
;	PAR are put to the refined values. SEL_REFL are updated also updated.
; 	The observed positions haven't changed but since they are given in 
;	respect to the center the values must change. Further the new YSCALE 
;	would change their value in mm (they are determined from the image 
;	in pixels):
;	SIGMA: The calculated sigma (of distance from predicted to observed
;		position). [mm]
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	If INFO is set then some information is printed on the terminal.
;
; RESTRICTIONS:
;	In CALC_COORD_REFINE:
;	At present the missetting angles are supposed to be relative
;	a* being along the rotation axis (along the coordinate y-axis)
;	and b* being in the plane of the rotation axis and the beam
;	axis as close to the beam axis as possible (direction of the beam).
;	In other words a*,b* being given in LAUEGEN.
;
; PROCEDURE:
;	Calls MRQMIN for the refinement.
;	The refinement is considered converged if CHISQ decreases relatively
;	less than TOL (but is really decresing, not increasing) or if
;	CHISQ is less than 0.1*number_of_reflections. The refinement is 
;	also terminated if MAX_ITER iterations have been done. 
;	CHISQ is calculated with the raster size as assumed sigma if
;	SIGMA is not given.
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, June 1995.	
;
;-

IF NOT KEYWORD_SET(tol) THEN tol= 0.001
IF NOT KEYWORD_SET(dampfactor) THEN dampfactor= 1.0
IF NOT KEYWORD_SET(stepfactor) THEN stepfactor= 0.1
IF NOT KEYWORD_SET(max_iter) THEN max_iter= 50
IF NOT KEYWORD_SET(sigma) THEN sigma= par.raster/1000.0

x= REPLICATE({ m: INTARR(3), x: 0., y: 0. }, (size(sel_refl))(1))
x.m= sel_refl.m & x.x= sel_refl.x & x.y= sel_refl.y
y= REPLICATE({ x: 0., y: 0. }, (size(sel_refl))(1))
y.x= sel_refl.x & y.y= sel_refl.y
; Can effect when the refinement is terminated (see below): 
nopar=16
a= DBLARR(nopar)
a(0)= par.phis.x
a(1)= par.phis.y
a(2)= par.phis.z + par.spindle
a(3)= par.cdd
a(4)= 0.0	; Center (x,y), in [mm]. Offset relative to the value used  
a(5)= 0.0	; when determining the positions in Y.
a(6)= 1.0	; yscale
a(7)= par.twist
a(8)= par.tilt
a(9)= par.bulge
a(10)= par.crystal.angle(0)
a(11)= par.crystal.angle(1)
a(12)= par.crystal.angle(2)
a(13)= par.crystal.axis(0)
a(14)= par.crystal.axis(1)
a(15)= par.crystal.axis(2)
a0=a & list= INTARR(16) & i=0
a0(4)=		par.cenx
a0(5)=		par.ceny
a0(6)=		par.yscale
parname= ['PhiX', 'PhiY', 'PhiZ', 'CtoD', 'CenX', 'CenY', 'Yscale', $
	'Twist', 'Tilt', 'Bulge', 'Alpha*', 'Beta*', 'Gamma*', $
	'A*', 'B*', 'C*' ]
IF KEYWORD_SET(phix) THEN BEGIN & list(i)=0 & i=i+1 & ENDIF
IF KEYWORD_SET(phiy) THEN BEGIN & list(i)=1 & i=i+1 & ENDIF
IF KEYWORD_SET(phiz) THEN BEGIN & list(i)=2 & i=i+1 & ENDIF
IF KEYWORD_SET(cdd) THEN BEGIN & list(i)=3 & i=i+1 & ENDIF
IF KEYWORD_SET(cenx) THEN BEGIN & list(i)=4 & i=i+1 & ENDIF
IF KEYWORD_SET(ceny) THEN BEGIN & list(i)=5 & i=i+1 & ENDIF
IF KEYWORD_SET(yscale) THEN BEGIN & list(i)=6 & i=i+1 & ENDIF
IF KEYWORD_SET(twist) THEN BEGIN & list(i)=7 & i=i+1 & ENDIF
IF KEYWORD_SET(tilt) THEN BEGIN & list(i)=8 & i=i+1 & ENDIF
IF KEYWORD_SET(bulge) THEN BEGIN & list(i)=9 & i=i+1 & ENDIF
IF KEYWORD_SET(alphastar) THEN BEGIN & list(i)=10 & i=i+1 & ENDIF
IF KEYWORD_SET(betastar) THEN BEGIN & list(i)=11 & i=i+1 & ENDIF
IF KEYWORD_SET(gammastar) THEN BEGIN & list(i)=12 & i=i+1 & ENDIF
IF KEYWORD_SET(astar) THEN BEGIN & list(i)=13 & i=i+1 & ENDIF
IF KEYWORD_SET(bstar) THEN BEGIN & list(i)=14 & i=i+1 & ENDIF
IF KEYWORD_SET(cstar) THEN BEGIN & list(i)=15 & i=i+1 & ENDIF
mfit= i
IF (mfit eq 0) THEN MESSAGE, 'REFINE_COORD called without refinement '+ $
	'parameters specified.'
lista= INTARR(i)
lista= list(0:i-1)
funcs='calc_coord_refine'
alambda=-1.0
; Calculate the predicted coordinates for later comparison:
x2= REPLICATE({ m: INTARR(3), x: 0., y: 0. }, (size(refl))(1))
x2.m= refl.m 
y2= REPLICATE({ x: 0., y: 0. }, (size(refl))(1))
y2.x= refl.x & y2.y= refl.y
dist= CALC_COORD_REFINE(x2,y2,a)
refl.x= x2.x & refl.y= x2.y

MRQMIN, x, y, sigma, a, lista, covar, alpha, mbeta, chisq, funcs, alambda, $
	dampfactor, stepfactor=stepfactor
chisq0=chisq & iter=0
; Calculate average distance and rms (of selected refl, before refinement):
d0= 1000.0*TOTAL(sqrt((x.x-y.x)^2+(x.y-y.y)^2))/(size(x))(1)
rms0= 1000.0*sqrt(TOTAL((x.x-y.x)^2+(x.y-y.y)^2)/(size(x))(1))
; Iterate until chisq is less than 0.1 or 
;               chisq drops less than 0.001*chisq (but continue if it
;			increases!)
REPEAT BEGIN
  ochisq= chisq
  MRQMIN, x, y, sigma, a, lista, covar, alpha, mbeta, chisq, funcs, alambda, $
	dampfactor, stepfactor=stepfactor
  iter= iter+1
ENDREP UNTIL ((chisq lt 0.1*(size(x))(1)) or $
	(((ochisq-chisq)/ochisq lt tol) and ((ochisq-chisq) gt 0)) or $
	(iter ge max_iter))
alambda=0
MRQMIN, x, y, sigma, a, lista, covar, alpha, mbeta, chisq, funcs, alambda, $
	dampfactor, stepfactor=stepfactor

par.phis.x= 	a(0)
par.phis.y= 	a(1)
par.phis.z= 	a(2) - par.spindle
par.cdd= 	a(3)
par.cenx= 	a(4)/(par.raster/1000.0) + par.cenx
par.ceny= 	a(5)*a(6)*par.yscale/(par.raster/1000.0) + par.ceny
par.yscale= 	a(6)*par.yscale
par.twist= 	a(7)
par.tilt= 	a(8)
par.bulge= 	a(9)
par.crystal.angle(0)= 	a(10)
par.crystal.angle(1)= 	a(11)
par.crystal.angle(2)= 	a(12)
par.crystal.axis(0)= 	a(13)
par.crystal.axis(1)= 	a(14)
par.crystal.axis(2)= 	a(15)
; Update the positions in sel_refl. The observed positions haven't changed
; but since they are given in respect to the center the values must change.
; Further the new yscale would change their value in mm (they are determined
; from the image in pixels):
sel_refl.x= sel_refl.x - a(4)
; The new center off-set in y (a(5)) is determined in "observed position
; space" so it should be subtracted before the new yscale is applied:
sel_refl.y= (sel_refl.y - a(5))/a(6)
; Calculate average distance and rms (of selected refl, after refinement):
d= 1000.0*TOTAL(sqrt((x.x-y.x)^2+(x.y-y.y)^2))/(size(x))(1)
rms= 1000.0*sqrt(TOTAL((x.x-y.x)^2+(x.y-y.y)^2)/(size(x))(1))

; Calculate the new predicted coordinates and put in REFL:
x2= REPLICATE({ m: INTARR(3), x: 0., y: 0. }, (size(refl))(1))
x2.m= refl.m 
y2= REPLICATE({ x: 0., y: 0. }, (size(refl))(1))
y2.x= refl.x & y2.y= refl.y
dist= CALC_COORD_REFINE(x2,y2,a)
refl.x= x2.x & refl.y= x2.y
; Calculate average distance and rms (between old and new predicted pos.
; of all predicted reflections):
d2= 1000.0*TOTAL(dist)/(size(dist))(1)
rms2= 1000.0*sqrt(TOTAL(dist^2)/(size(dist))(1))

IF KEYWORD_SET(info) THEN BEGIN
  a(4)=		par.cenx
  a(5)=		par.ceny
  a(6)=		par.yscale
  PRINT, 'REFINE_COORD:'
  PRINT, 'Number of reflections used in refinement: ', (size(sel_refl))(1)
  PRINT, 'Number of iterations: ', iter
  PRINT, 'ChiSq: ', chisq0, ' -> ', chisq
  PRINT, 'Shift: '
  FOR i=0,mfit-1 DO PRINT, parname(list(i)), a(list(i))-a0(list(i)), $
	'  (', a(list(i)), ', Sigma:', sqrt(covar(i,i)), ')' 
  PRINT, '(All reflections):'
  PRINT, '  Average distance from old to new predicted [um] :', d2
  PRINT, '  RMS distance from old to new predicted [um] : ', rms2
  PRINT, 'Before Refinement (Selected reflections):'
  PRINT, '  Average distance from predicted to observed [um] :', d0
  PRINT, '  RMS distance from predicted to observed [um] : ', rms0
  PRINT, 'After Refinement (Selected reflections):'
  PRINT, '  Average distance from predicted to observed [um] :', d
  PRINT, '  RMS distance from predicted to observed [um] : ', rms
ENDIF

sigma= rms/1000.0

END