function PREDICT, par, rot_axis, spindle, lambdapol=lpol, bfactor=bfact, $
		pred_cutoff=pred_cutoff
;+
; NAME:
;	PREDICT
;
; PURPOSE:
;	Predict a Laue pattern based on the parameters as given
;	by READ_GEASC2.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	spot_list= PREDICT(par [, rot_axis, spindle, lambdapol=lambdapol, $
;			bfactor=bfactor, pred_cutoff=pred_cutoff])
;
; INPUTS:
;	PAR:    As from READ_GEASC.
;		NOTE: PAR.SPINDLE is always added to PhiZ before a possible 
;		rotation around another axis as defined by ROT_AXIS and
;		SPINDLE. On output PAR.SPINDLE is set to zero and PhiX, PhiY
;		and PhiZ are changed to the new position if a rotation
;		is made, i.e. if ROT_AXIS and SPINDLE are set.
;		If ROT_AXIS and SPINDLE are not set PAR is not changed but
;		PAR.SPINDLE is internally added to PhiZ.
;		(The user might want to use PAR.SPINDLE to store the value 
;		of the spindle (as defined around ROT_AXIS) 
;		but since the definition is different from the LAUEGEN 
;		definition if the rotation axis is not around the z-axis
;		I find it safest to set it to zero and then you can do
;		whatever you want!) 
;	ROT_AXIS: If given then it should contain a FLTARR(3) array with
;		the cosines of the rotation axis in the (x,y,z)-system.
;	SPINDLE: Positive rotation around the above axis in degrees. 
;
; KEYWORDED PARAMETERS:
;	Some parameters for a resolution dependent bandwidth:
;	LPOL: Polynomial that corresponds to an expected normalisation
;		curve. (As output from Lauenorm, i.e. the curve with
;		which the intensities are multiplied.) The curve is 
;		normalised so that the inverse peaks at 1.0 (in the
;		given lambdamin and lambdamax).
;	BFACTOR: B-factor as expected output in TRUNCATE.
;	PRED_CUTOFF: The programs calculates the following value:
;		  POLSCALE / POLY(lambda,lambdapol) * exp(-0.5*b*dstar2) *
;		    0.0025 / (dstar2*lambda^2)
;		(See RDBW.PRO, POLSCALE is set to maximum of 1/POLY). 
;		If this value is below PRED_CUTOFF then the reflection 
;		is expected not to diffract considerably and is not 
;		predicted.
;	LPOL defaults to [1], BFACTOR to 0.0 and PRED_CUTOFF to 0.001
;	(so default values gives hard limits).
;
; OUTPUTS:
;	Returns a list of spots.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	The spindle tag of PAR is set to zero and the missetting angles
;	are changed if ROT_AXIS and SPINDLE are given.
;
; RESTRICTIONS:
;	In CALC_COORD_REFINE:
;	At present the missetting angles are supposed to be relative
;	a* being along the rotation axis (along the coordinate y-axis)
;	and b* being in the plane of the rotation axis and the beam
;	axis as close to the beam axis as possible (direction of the beam).
;	In other words a*,b* being given in LAUEGEN.
;
; PROCEDURE:
;	Hopefully the same as is used in LAUEGEN.
; Loop to get a list of activated rlp:s.
; From LCHK:
;> Purpose: Find all reflections within dstar-max in the 
;> hemisphere +h, all k,l.
; (HERE: all h,k,l)
;>       In essence the routine flow is to
;>       (1) increase h from 0 to max
;            decrease h from -1 to min (NOT IN LCHK)
;>       (2) increase k from 0 to max, then
;>           decrease k from -1 to min
;>       (3) increase l from 0 to max, then
;>           decrease l from -1 to min
;>  with h varying least, l most rapidly. The max and min values are
;>  determined by a dstar which exceeds the limit. Complications are
;>  caused by non-orthogonal cells: for given h,k l=0 may exceed dstar-
;>  max, but some other l may not. The logic is that if at least one l
;>  for given h,k is found within dstar-max, then the l loops end when
;>  dstar exceeds dstar-max. If such an l is not found then the loops end
;>  in both directions when the calculated dstar is increasing. Similarly
;>  for the control of k loops, where the test is for at least one k,l
;>  inside for given h.
; (Algorithm in code rewritten from LRL.FOR)
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

IF NOT KEYWORD_SET(lpol) THEN lpol=[1]
IF NOT KEYWORD_SET(bfact) THEN bfact=0.0
IF NOT KEYWORD_SET(pred_cutoff) THEN pred_cutoff=0.001

;define one reflexion (See READ_GEASC)
one_refl={ geasc_type, $
		m: INTARR(3), $
		x: 0.0, $
		y: 0.0, $
		l: 0.0, $
		mul: 0, $
		nodal: 0, $
		ovlp: 0,$
		ovlp2: 0,$
		id:0, $
		connect: INTARR(24), $
		close:0, $
		m2: INTARR(3), $
		nidx: 0, $
		l2: 0.0,$
		ipoint: 0, $
		dminth: 0.0, $
		intb: FLTARR(6), $
		isigb: FLTARR(6), $
		intp: FLTARR(6), $
		isigp: FLTARR(6),$
		tth: 0.0, $
		res: 0.0, $
		flag: 0, $
		gfit: 0.0, $
		bc_coeff: FLTARR(3)	}

; Get matrix relating hkl to the rlp coordinates (X=M#H) with possibly
; another rotation:
; If ROT_AXIS and SPINDLE given then rotate: 
IF (N_PARAMS() eq 3) THEN BEGIN
  ; Rotate. Note that in CALC_ROTATION par.spindle is added to
  ; PhiZ before rotation and then par.spindle is set to zero:
  par= CALC_ROTATION(par, rot_axis, spindle)
  mat= CALC_ROTMAT(par)
ENDIF ELSE mat= CALC_ROTMAT(par)

; Create the vector A that is used in the call of CALC_COORD_REFINE.
nopar=16
a= DBLARR(nopar)
a(0)= par.phis.x
a(1)= par.phis.y
a(2)= par.phis.z+par.spindle
a(3)= par.cdd
a(4)= 0.0
a(5)= 0.0
a(6)= 1.0
a(7)= par.twist
a(8)= par.tilt
a(9)= par.bulge
a(10)= par.crystal.angle(0)
a(11)= par.crystal.angle(1)
a(12)= par.crystal.angle(2)
a(13)= par.crystal.axis(0)
a(14)= par.crystal.axis(1)
a(15)= par.crystal.axis(2)

; Get symmetry operators. If SG_NO==0 then only test for systematic absences
; due to centering will be done.
sym= READ_SYMMETRY(par.sg_no)
sym.centering_no= par.centering_no
SET_LATT, sym.centering_no, isys, ksys

; Get limits for array (from SLIP):
imat=INVERT(mat)
l= sqrt(2.0)/par.dmin
x= INTARR(3,5)
x(*,0)= imat # [-l,0,0]
x(*,1)= imat # [0,-l,0]
x(*,2)= imat # [0,l,0]
x(*,3)= imat # [0,0,-l]
x(*,4)= imat # [0,0,l]
hmin= MIN(x(0,*), max=hmax) & hmin=hmin-1 & hmax=hmax+1
kmin= MIN(x(1,*), max=kmax) & kmin=kmin-1 & kmax=kmax+1
lmin= MIN(x(2,*), max=lmax) & lmin=lmin-1 & lmax=lmax+1

; Inner point of ray activated (set to infinity=255 from start):
ip= BYTARR(hmax-hmin+1,kmax-kmin+1,lmax-lmin+1)+255
; Multiplicity of ray:
mul= BYTARR(hmax-hmin+1,kmax-kmin+1,lmax-lmin+1)
ntries=0l & num_reflex=0l
dstar_max2=1/par.dmin^2 
lammin=par.lmin & lammax=par.lmax 
; Get scale factor for lambda polynomial:
x= (lammax-lammin)/299.0*FINDGEN(300)+lammin
y= POLY(x,lpol)
polsc= MIN(y)
h=0 & next_h=1
PRINT, 'Searching for reflections in diffracting position.'
; Comments:
; 1)found_any_l=1 etc. means that an rlp has been found inside the
;  resolution sphere in the present loop. Does not mean that there
;  is an rlp actually diffracting.
REPEAT BEGIN
  h0=h
  k=0 & next_k=1
  found_any_k=0
  dstar2_k_max = 1000.0
  REPEAT BEGIN
    k0=k
    WHILE (SYS_AHK([h,k,0],sym=sym,isys=isys,ksys=ksys))(0) DO k=k+next_k
    l=0 & next_l=1
    IF ((h eq 0)and(k eq 0)) THEN l=1
    found_any_l= 0
    dstar2_l_max= 1000.0
    REPEAT BEGIN
      l0=l
      ntries= ntries+1
      WHILE (SYS_ABS([h,k,l],sym=sym,isys=isys,ksys=ksys))(0) DO l=l+next_l
      x = mat # [h,k,l]
      dstar2= total(x*x)
      ; Within resolution sphere?
      IF (dstar2 le dstar_max2) THEN BEGIN
	; Between the wavelenght cut-offs?
	lambda= -2.0*x(0)/(x(0)^2+x(1)^2+x(2)^2)
;        IF (((x(0)+rmin)^2+x(1)*x(1)+x(2)*x(2)) ge rmin2) THEN $
;        IF (((x(0)+rmax)^2+x(1)*x(1)+x(2)*x(2)) le rmax2) THEN $
	IF (lambda ge lammin) THEN $
	IF (lambda le lammax) THEN $
	IF (RDBW(lambda,dstar2,lpol,bfact,polsc=polsc) $
		gt pred_cutoff) THEN BEGIN
	  ; We have found an activated rlp:
	  ; Find nodal indices:
	  found=0 & div=max(abs([h,k,l]))
	  WHILE (not found) DO BEGIN
	    IF (((h mod div) eq 0) and ((k mod div) eq 0) $
		and ((l mod div) eq 0)) THEN BEGIN
	      found=1
	      ip(h/div-hmin,k/div-kmin,l/div-lmin)= $
			ip(h/div-hmin,k/div-kmin,l/div-lmin) < div
	      mul(h/div-hmin,k/div-kmin,l/div-lmin)= $
			mul(h/div-hmin,k/div-kmin,l/div-lmin) + 1
	    ENDIF ELSE div=div-1
	  ENDWHILE
	  num_reflex= num_reflex + 1
	ENDIF
	; The rest of the REPEAT-statement is only concerned with
	; optimisation of the loop limits:
        found_any_k=1
        found_any_l=1
	l=l+next_l
      ENDIF ELSE BEGIN
	IF (found_any_l) THEN BEGIN
	  IF (next_l eq 1) THEN BEGIN
	    next_l=-1
	    l=-1
	  ENDIF ELSE k=k+next_k
	ENDIF ELSE BEGIN
	  IF (dstar2 lt dstar2_l_max) THEN BEGIN
	    dstar2_l_max= dstar2
	    l=l+next_l
	  ENDIF ELSE BEGIN
	    IF (next_l eq 1) THEN BEGIN
	      next_l=-1
	      l=-1
	    ENDIF ELSE BEGIN
	      IF (found_any_k) THEN BEGIN
	        IF (next_k eq 1) THEN BEGIN
	          next_k=-1
	          k=-1
	        ENDIF ELSE h=h+next_h
	      ENDIF ELSE BEGIN
		IF (dstar2 lt dstar2_k_max) THEN BEGIN
		  dstar2_k_max= dstar2
		  k=k+next_k
		ENDIF ELSE BEGIN
		  IF (next_k eq 1) THEN BEGIN
		    next_k=-1
		    k=-1
		  ENDIF ELSE BEGIN
		    IF (next_h eq 1) THEN BEGIN
		      next_h=-1
		      h=-1
		    ENDIF 
		  ENDELSE
		ENDELSE
	      ENDELSE
	    ENDELSE
	  ENDELSE
	ENDELSE
      ENDELSE
    ENDREP UNTIL (l eq l0)
  ENDREP UNTIL (k eq k0)
ENDREP UNTIL (h eq h0)
PRINT, '  Number of rlp:s tried:', ntries

; Get spot list from reflection list and remove spots outside detector 
; limits:
IF (num_reflex eq 0) THEN MESSAGE, 'No predicted reflections!'
index= WHERE(mul ne 0, num_spots)
l= index/((hmax-hmin+1)*(kmax-kmin+1))+lmin
i= index mod ((hmax-hmin+1)*(kmax-kmin+1))
k= i/(hmax-hmin+1)+kmin
h= i mod (hmax-hmin+1)+hmin
spot_list= REPLICATE(one_refl, num_spots)
spot_list.m= TRANSPOSE([[ip(index)*h],[ip(index)*k],[ip(index)*l]])
; Get indices of the second harmonic if a multiple:
m2= (mul(index) gt 1) * (ip(index)+1)
spot_list.m2= TRANSPOSE([[m2*h],[m2*k], [m2*l]])
; If the next rlp is a systematic absence then add 2 instead of 1:
m2= (mul(index) gt 1) * (ip(index)+1+SYS_ABS(spot_list.m2,sym=sym))
spot_list.m2= TRANSPOSE([[m2*h],[m2*k], [m2*l]])
spot_list.mul= mul(index)
spot_list.l2= (mul(index) gt 1) * (ip(index)/(ip(index)+1))
spot_list.dminth= ((ip(index)+mul(index))/ip(index))^2
PRINT, '  Number of reflections:', num_reflex
PRINT, '  Number of spots      :', num_spots
PRINT, 'Removing spots not reaching the detector.'
dummy= CALC_COORD_REFINE(spot_list, {x:0.0,y:0.0}, a)
r2= spot_list.x^2+spot_list.y^2
spot_list=spot_list(WHERE((r2 le par.radius^2) and (r2 ge par.r_rej^2), $
		num_spots))
PRINT, '  Number of spots      :', num_spots

; Calculate data (lambda etc.) for the spots:
PRINT, 'Calculating data for the spots.'
x = mat # spot_list.m
dstar2= total(x*x,1)
spot_list.l= -2.0*(x(0,*))(*)/dstar2
spot_list.id= INDGEN(num_spots)
spot_list.connect(*)= -1
spot_list.l2= spot_list.l * spot_list.l2
spot_list.dminth= dstar2 * spot_list.dminth
spot_list.intb(*)= -9999.0
spot_list.isigb(*)= -9999.0
spot_list.intp(*)= -9999.0
spot_list.isigp(*)= -9999.0
spot_list.res= 1/sqrt(dstar2)
spot_list.tth= 2.0*180/!PI*asin(spot_list.l/(2.0*spot_list.res))

RETURN, spot_list

END







