function PRED_II, ref, lambdapol, bfactor, scale=scale,mult=mult,res_mult=res_mult,threshold=threshold,w_res=w_res,w_mult=w_mult,weak=weak,$
	use_rdbw=use_rdbw,use_mono=use_mono,mono_file=mono_file,lscale=lscale,lmin=lmin,lmax=lmax,ii_weak=ii_weak,ii_strong=ii_strong,$
	get_norm_coeff=get_norm_coeff,set_norm_coeff=set_norm_coeff,norm_coeff=norm_coeff,get_correlation=get_correlation
;+
; NAME:
;	PRED_II
;
; PURPOSE:
;	Calculate a factor that gives an estimation of the intensity of
;	a reflection in [ADU] from a monochromatic structure factor, and a 
;	predicted normalization curve (if the option USE_RDBW is not used)
;	ii = scale*spectrum(lambdapol)*res^2*Fmono^2*exp(-0.5*b*dstar2)
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	ref2= PRED_II(ref,lambdapol, bfactor,scale=scale,[mult=mult],[res_mult=res_mult],[threshold=threshold],$
;		[w_res=w_res],[w_mult=w_mult],[weak=weak],[/USE_RDBW],[/USE_MONO],[MONO_FILE=MONO_FILE],[/LSCALE],[ii_weak=ii_weak],[ii_strong=ii_strong])
;
; INPUTS:
;	ref : the reflexion list : should contain flags : .res (resolution
;		in A), .l (lambda), .intp(1) (monochromatic F, from get_monof)
;	lambdapol : the coefficient of the polynom representing the 
;		normalization curve.
;		How to obtain lambdapol :
;		1/ Lauenorm : in IDL do :
;			read_ln,'my_lauenorm.log',bins,lambda,lambdapol,ranges,polorder,lmin,lmax
;			The curve can be checked with :
;			l=lmin(0)+indgen(100)/100.*(lmax(0)-lmin(0)) & plot,l,1/poly(l,pol)
;			Then it is saved with : save,lambdapol,file='my_lambdacurve.idl'
;		2/ Lscale : in IDL do :
;			read_lscale,'my_lauenorm.log',bins,lambda,lambdapol,ranges,polorder,lmin,lmax
;			The curve can be checked with :
;			l=FLOAT(lmin)+indgen(100)/100.*(FLOAT(lmax)-FLOAT(lmin)) & plot,l,1/chebychev(lambdapol,l)
;			Then it is saved with : save,lambdapol,file='my_lambdacurve.idl'
;	BFACTOR: B-factor as expected output in TRUNCATE.
;
; KEYWORDED PARAMETERS:
;	scale: The factor is multiplied with this (default 1.0) if RDBW is NOT used.
;	mult : if set, the calculation is done also for the doubles
;		which have to be input after the routine get_monof has been
;		run with this keyword also set.
;	res_mult : the equivalent of result, for the multiples
;	threshold : a threshold below which the reflecions are disregarded [in % of the 
;		maximum factor]. Defaults = 1%
;	w_res : the indices of ref satisfying factor > threshold
;	w_mult : the indices of mult satisfying factor > threshold
;	weak : the reflexions predicted to be weak.
;	use_rdbw : if this keyword is set, then a Resolution Dependant Band Width filter is applied. This is
;		an alternative to the case where information from a THEORETICAL normalization curve (eg : 
;		calculted with Mathematica) is used, with also information from monochromatic data. If the
;		USE_RDBW keyword is used, then lambdapol should be taken from the experimentally determined
;		normalization curve, and no info is taken from monochromatic data.
;	use_mono : if this keyword is set, then information from a monochromatic data set is taken into account. 
;		The monochromatic structure factor amplitudes are then also
;		taken into account when assessing the predicted integrated intensities. Be careful that there is no mis-indexing.
;		If the monochromatic data set is incomplete relative to the Laue data set, a dummy monochromatic structure factor 
;		amplitude is taken, equal to the median value of the monochromatic Fmono's. 
;		A difficulty in using monochromatic reference data set is that proper symmetry equivalent must be found for each measurement.
;		The monochromatic F's for singles have  to be put in the field .INTP(1) of 'ref'. For doubles see 'mult'. Multiples higher than doubles are ignored.
;	lscale : if this keyword is set, then the calculation involves chebychev polynomia as in the program LSCALE. 
;		Use in conbination with /USE_RDBW. Default is to use Lauenorm.
;	LMIN : the minimum wavelength to consider (for LSCALE)
;	LMAX : the maximum wavelength to consider (for LSCALE)
;	ii_weak : the predicted intensities for the weak reflections [singles only]
;	ii_strong : the predicted intensities for the strong reflections [singles only]
;	SET_NORM_COEFF : If the calculation has already been done for the contribution of the lambda curve, it can be passed through SET_NORM_COEFF
;		This allows to save a lot of time if the computation has to be done many times
;	GET_NORM_COEFF : if the computation has to be done many times, the result of the contribution of the lambda curve can be saved in NORM_COEFF
;		if /GET_NORM_COEFF is given.
;
; OUTPUTS:
;	ref2 : the modified ref (weak reflexions removed, doubles modified accordingly)
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	The integrated intensity for Laue is given by :
;		II[ADU] = scale*exp(-0.5*bfactor*dstar2)*I0[lambda]*lambda^2*F^2*polarization*lorentz*detector_response
;		where dstar = 1/d^2 for the reflection in question.
;	 If /USE_RDBW is NOT used : then :
;					1/ lambdapol represents : I0[lambda] (function of ID, mirror, slits, detector ...)
;					2/ lorentz(=1/(2*sin(theta)^2)) --> theta given for each reflection
;					3/ lambda^2 --> lambda given for each reflection
;					4/ polarization is neglected for synchrotron work. 
;					5/ detector response is assumed to be wavelenght independant and therefore is neglected 
;					6/ scale is a global scale factor [default=1]
;					7/ exp(-0.5*bfactor*dstar2) is a temperature factor -->  dstar2 given for each reflection
;					8/ F^2 is from a monochromatic data set if /USE_MONO is specified
;
;
;	 If /USE_RDBW is used : then :
;					1/ lambdapol represents : I0[lambda]*lambda^2*detector_response. Comes from LAUENORM or LSCALE
;					2/ lorentz(=1/(2*sin(theta)^2)) --> theta given for each reflection
;					3/ polarization is neglected for synchrotron work. 
;					4/ scale is taken to be 1.
;					5/ exp(-0.5*bfactor*dstar2) is a temperature factor -->  dstar2 given for each reflection
;					6/ F^2 is from a monochromatic data set if /USE_MONO is specified
;
;
;
;	
; MODIFICATION HISTORY:
;	Written by Dominique Bourgeois, December 1996, March 1999.
;
;-

IF NOT KEYWORD_SET(scale) THEN scale=1.0
IF N_ELEMENTS(threshold) EQ 0 THEN threshold=1.

IF N_ELEMENTS(lmin) EQ 0 THEN lmin=MIN(ref.l)
IF N_ELEMENTS(lmax) EQ 0 THEN lmax=MAX(ref.l)

IF N_ELEMENTS(norm_coeff) EQ 0 THEN norm_coeff=0 
IF KEYWORD_SET(get_norm_coeff) THEN get_norm_coeff=1 ELSE get_norm_coeff=0
IF N_ELEMENTS(set_norm_coeff) EQ 0 THEN set_norm_coeff=0

PRINT,'Sorting all reflexions ...'
ref2=ref(SORT(ref.id))

IF KEYWORD_SET(mult) THEN BEGIN
 IF (SIZE(mult))(0) EQ 0 THEN BEGIN
  PRINT,'There are no multiples to treat.'
 ENDIF ELSE BEGIN
  ;do it also for the multiples
  PRINT,'Sorting doubles ...'
  mult2=mult(SORT(mult.id))
 ENDELSE
ENDIF ELSE BEGIN ; keyword mult not set : CAN NOT WORK WITH /USE_MONO
 IF KEYWORD_SET(use_mono) THEN MESSAGE,'Monochromatic SF will not be defined for multiples : can not treat this case !'
 PRINT,'Now dealing with multiples ...'
 PRINT,'Due to pb in Lauegen, only dealing with doubles ...'
 w_mul=WHERE((ref2.mul GT 1) AND (ref2.mul LE 2),ct_mul)
 dum=WHERE((ref2.mul GT 1),ct_mul_all)
 IF ct_mul GT 0 THEN BEGIN
  PRINT,'Found ',ct_mul,' doubles'
  PRINT,'Will ignore ',ct_mul_all-ct_mul,' triples, quadruples, etc ...'
  mult=ref2(w_mul)
  max_m=MAX(mult.mul)
  one_mult={id:0,l_m:FLTARR(max_m),m_m:INTARR(3,max_m),ii_m:FLTARR(max_m),res_m:FLTARR(max_m)}
  mult2=REPLICATE(one_mult,ct_mul) 
  mult2.id=mult.id
  ;in mult2 we put all the relevant miller indices and corresponding
  ;wavelengths
  PRINT,'Expanding wavelengths and Miller indices ...'
  cr = STRING("15b)
  FOR i=0,ct_mul-1 DO BEGIN
   IF i/100 EQ i/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",i,100.*i/FLOAT(ct_mul),cr
   a1=1/mult(i).l & a2=1/mult(i).l2
   n1 = a1/(a2-a1)
   l0 = 1/(a2-a1)
   res0 = n1*mult(i).res
   m0 = ROUND(mult(i).m/n1)
   FOR j=0,mult(i).mul-1 DO BEGIN
    mult2(i).l_m(j)=l0/(n1+j)
    mult2(i).res_m(j)=res0/(n1+j)
    mult2(i).m_m(*,j)=ROUND(m0*(n1+j))
   ENDFOR ; for each j
  ENDFOR ; for each i
 ENDIF
ENDELSE


;this is the expected integrated intensity
PRINT,'Calculating expected intensities ...'
dmax=8.
PRINT,'Normalization will be taken at a resolution of : '+STRING(dmax)+' A'


IF KEYWORD_SET(use_mono) THEN BEGIN
 PRINT,'Using monochromatic data set ! ' 
ENDIF ELSE $
 PRINT,'No monochromatic data set will be used !'

;&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&
;&&&&&&&&&&&&&& RDBW OPTION &&&&&&&&&&&&&&&&&
;&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&

IF KEYWORD_SET(use_rdbw) THEN BEGIN ; use experimental lambda curve
 PRINT,'Using RDBW option ...'
 IF KEYWORD_SET(lscale) THEN $
  PRINT,'Using normalization curve from LSCALE' $
 ELSE $
  PRINT,'Using normalization curve from LAUENORM'

 ;note that we dont care about multiples at this stage.
 res = RDBW(ref2.l, (1/ref2.res)^2d,lambdapol,bfactor,polscale=1,dmax=dmax,lmin=lmin,lmax=lmax,lscale=lscale,max_res=max_res,use_mono=use_mono,f_mono=ref2.intp(1),get_norm_coeff=get_norm_coeff,set_norm_coeff=set_norm_coeff,norm_coeff=norm_coeff)

 IF KEYWORD_SET(get_correlation) THEN BEGIN
  correl=CORRELATE(res,ref2.intp(0))
  PRINT,'************************************************************************'
  PRINT,'*** CORRELATION BETWEEN OBSERVATIONS AND PREDICTIONS IS :',correl
  PRINT,'************************************************************************'
  IF use_mono AND correl LE 0.6 THEN MESSAGE,'Correlation between calculated and experimental data is poor ! Check for misindexing of the monochromatic data ! '
 ENDIF

 IF KEYWORD_SET(mult) AND ((SIZE(mult))(0) NE 0) THEN BEGIN
  PRINT,'Calculating expected intensities for doubles (first and then second harmonic) ...'
  res_mult0 = RDBW(mult2.l_m(0),(1/mult2.res_m(0))^2d,lambdapol,bfactor,polscale=1,dmax=dmax,lmin=lmin,lmax=lmax,lscale=lscale,use_mono=use_mono,f_mono=mult2.ii_m(0))
  res_mult1 = RDBW(mult2.l_m(1),(1/mult2.res_m(1))^2d,lambdapol,bfactor,polscale=1,dmax=dmax,lmin=lmin,lmax=lmax,lscale=lscale,use_mono=use_mono,f_mono=mult2.ii_m(1))
 ENDIF 
ENDIF $ 
ELSE BEGIN ; use calculated lambda curve + mono data 

;&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&
;&&&&&&&&&&&&&& CALCULATED NORMALIZATION CURVE OPTION &&&&&&&&&&&&&&&&&
;&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&

 dstar2min=1/FLOAT(dmax^2)
 n_samples=500
 l=FLOAT(lmin)+indgen(n_samples)/FLOAT(n_samples)*(FLOAT(lmax)-FLOAT(lmin))
 w_max=WHERE(POLY(ref2.l,lambdapol) EQ MAX(POLY(ref2.l,lambdapol)),ct_max)
 max_l = l(w_max(0))

 IF KEYWORD_SET(use_mono) THEN BEGIN

  max_res=scale * POLY(max_l,lambdapol)* exp(-0.5*bfactor*dstar2min) * d_max^2 * MAX(ref2.intp(1))^2
  max_res2=scale * POLY(max_l,lambdapol) * exp(-0.5*bfactor*MIN((1/FLOAT(ref2.res))^2d)) * MAX(ref2.intp(1))^2
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'

  res =  scale*POLY(ref2.l,lambdapol)*exp(-0.5*bfactor*(1/ref2.res)^2d) * ref2.res^2*ref2.intp(1)^2
  IF KEYWORD_SET(mult) AND ((SIZE(mult))(0) NE 0) THEN BEGIN
   PRINT,'Calculating expected intensities for doubles...'
   res_mult0 = scale*POLY(mult2.l_m(0),lambdapol)*exp(-0.5*bfactor*(1/mult2.res_m(0))^2d) * mult2.res_m(0)^2*mult2.ii_m(0)^2
   res_mult1 = scale*POLY(mult2.l_m(1),lambdapol)*exp(-0.5*bfactor*(1/mult2.res_m(1))^2d) * mult2.res_m(1)^2*mult2.ii_m(1)^2
  ENDIF 

 ENDIF ELSE BEGIN ;not using Fmono

  max_res=scale * POLY(max_l,lambdapol)* exp(-0.5*bfactor*dstar2min) * d_max^2 
  max_res2=scale * POLY(max_l,lambdapol) * exp(-0.5*bfactor*MIN((1/FLOAT(ref2.res))^2d)) 
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'

  res =  scale*POLY(ref2.l,lambdapol)*exp(-0.5*bfactor*(1/ref2.res)^2d) * ref2.res^2
  IF KEYWORD_SET(mult) AND ((SIZE(mult))(0) NE 0) THEN BEGIN
   PRINT,'Calculating expected intensities for doubles...'
   res_mult0 = scale*POLY(mult2.l_m(0),lambdapol)*exp(-0.5*bfactor*(1/mult2.res_m(0))^2d) * mult2.res_m(0)^2
   res_mult1 = scale*POLY(mult2.l_m(1),lambdapol)*exp(-0.5*bfactor*(1/mult2.res_m(1))^2d) * mult2.res_m(1)^2
  ENDIF
 ENDELSE

ENDELSE

;End of the two options.

w_singles = WHERE(ref2.mul EQ 1,ct_singles)
w_res = WHERE(res(w_singles) GT threshold/100.*max_res,ct_res)
w_res_bad = WHERE(res(w_singles) LE threshold/100.*max_res,ct_res_bad)


IF ct_res GT 0 THEN ii_strong=res(w_singles(w_res)) ELSE ii_strong=0
IF ct_res_bad GT 0 THEN ii_weak=res(w_singles(w_res_bad)) ELSE ii_weak=0

 PRINT,'Removing weak singles ...'
 ;remove the bad singles
IF ct_res_bad GT 0 THEN ref2(w_singles(w_res_bad)).id = -1

 IF KEYWORD_SET(mult) AND ((SIZE(mult))(0) NE 0) THEN BEGIN
  w_mult0 = WHERE(res_mult0 GT threshold/100.*max_res,ct_res_mult0)
  w_mult1 = WHERE(res_mult1 GT threshold/100.*max_res,ct_res_mult1)

 ; IF (ct_res_mult0 EQ 0) OR (ct_res_mult1 EQ 0)  THEN MESSAGE,'Reduce cuttoff ! No multiples selected !'
  table_mult0=INTARR((SIZE(mult2))(1))
  table_mult1=INTARR((SIZE(mult2))(1))
  IF (ct_res_mult0 GT 0) THEN table_mult0(w_mult0) = 1
  IF (ct_res_mult1 GT 0) THEN table_mult1(w_mult1) = 2
  table_diff=table_mult1-table_mult0
  ;table_diff = 2 --> only the second harmonic is present
  ;table_diff = -1 --> only the first harmonic is present
  ;table_diff = 1 --> both are present
  ;table_diff = 0 --> none is present
  w_second=WHERE(table_diff EQ 2,ct_second)
  w_first=WHERE(table_diff EQ -1,ct_first)
  w_both=WHERE(table_diff EQ 1,ct_both)
  w_none=WHERE(table_diff EQ 0,ct_none)
  PRINT,'Number of doubles with only second harmonic present : ',ct_second
  PRINT,'Number of doubles with only first harmonic present : ',ct_first
  PRINT,'Number of doubles with both harmonics present : ',ct_both
  PRINT,'Number of doubles with none harmonic present : ',ct_none

  ;now put the correct doubles into ref2, with correct multiplicity
  w_doubles = WHERE(ref2.mul EQ 2,ct_doubles)

  IF ct_second GT 0 THEN BEGIN
   PRINT,'Removing weak multiples ...'
   ref2(w_doubles(w_second)).mul = 1
   ref2(w_doubles(w_second)).m = mult2(w_second).m_m(*,1)
   ref2(w_doubles(w_second)).l = mult2(w_second).l_m(1)
   ref2(w_doubles(w_second)).res = mult2(w_second).res_m(1)
   ref2(w_doubles(w_second)).intp(1) = mult2(w_second).ii_m(1)
  ENDIF

  IF ct_first GT 0 THEN BEGIN
   ref2(w_doubles(w_first)).mul = 1
   ref2(w_doubles(w_first)).m = mult2(w_first).m_m(*,0)
   ref2(w_doubles(w_first)).l = mult2(w_first).l_m(0)
   ref2(w_doubles(w_first)).res = mult2(w_first).res_m(0)
   ref2(w_doubles(w_first)).intp(1) = mult2(w_first).ii_m(0)
  ENDIF

  ;don't change (ref2(w_doubles))(w_both)

  IF ct_none GT 0 THEN ref2(w_doubles(w_none)).id = -1
 
 ENDIF ;keyword_set(mult)

 IF N_ELEMENTS(ct_mul_all) EQ 0 THEN w_mul_all = WHERE(ref2.mul GT 2,ct_mul_all)
 w_weak=WHERE(ref2.id EQ -1,ct_weak)
 IF ct_weak GT 0 THEN weak=ref2(w_weak) ELSE weak=0
 w_ok=WHERE(ref2.id GE 0,ct_ok)
 IF ct_ok GT 0 THEN ref2=ref2(w_ok) ELSE ref2=0

PRINT,'Number of selected reflections [singles]: ', ct_res,' ie : ',100*ct_res/FLOAT((SIZE(ref(WHERE(ref.mul EQ 1))))(1)),' %'

IF KEYWORD_SET(mult) AND ((SIZE(mult))(0) NE 0) THEN BEGIN
 PRINT,'Number of selected doubles: ', ct_both,' ie : ',100*(ct_both)/FLOAT((SIZE(mult2))(1)),' %'
 PRINT,'Number of recovered singles [from doubles]: ', ct_first+ct_second,' ie : ',100*(ct_first+ct_second)/FLOAT((SIZE(mult2))(1)),' % [from doubles]'
 PRINT,'Number of untouched higher order multiples: ', ct_mul_all
ENDIF

RETURN,ref2

END


