pro MRQCOF, x, y, sig, a, lista, alpha, mbeta, chisq, funcs, $
	stepfactor=stepfactor
;+
; NAME:
;	MRQCOF
;
; PURPOSE:
;	See MRQMIN. Used in MRQMIN to calculate ALPHA, MBETA and CHISQ.
;
; CATEGORY:
;	Numerical model refinement.
;
; CALLING SEQUENCE:
; 	MRQCOF, x, y, sig, a, lista, alpha, mbeta, chisq, funcs
;
; INPUTS:
;	X: Input data, independent variable. Only passed to the user supplied
;		function FUNCS.
;	Y: Input data, dependent variable. Only passed to the user supplied
;		function FUNCS.
;	SIG: The standard deviation of the measurements Y. If not known set
;		to 1.0. (Vector of same size as X or scalar.)
;	A: A vector of the parameters to the function FUNCS (and some of these
;		are refined).
;	LISTA: A vector containing the subscript in A of the parameters to
;		refine (and thus of the same length as the number of parameters
;		to refine).
;	FUNCS: String containing the name of the function to call to determine
;		the agreement between Y and model Y and to calculate the
;		partial derivatives.
;		The function is called with "d= FUNCS(x,y,a,dyda)" where
;		X,Y and A is as input to MRQCOF 
;		and DYDA should be an array of #elementsX x 
;		#elementsA containing the partial derivatives on output. 
;		The function should return the difference between model and 
;		measurements as a vector of same size as Y. When FUNCS is 
;		called DYDA is set to be a vector of same size as A containing
;		zeros at the positions of the parameters not being refined. 
;		This can be used in FUNCS to save time (only calculate the
;		partial derivatives for the parameters who has a non-zero
;		value in the given dyda).
;	
;
; KEYWORDED PARAMETERS:
;	STEPFACTOR: The default steps used to calculate the derivatives
;		are multiplied by this factor. (Only passed on to FUNCS.)
;
; OUTPUTS:
;	ALPHA: The curvature matrix.
;	MBETA: Vector containing -0.5 times the gradient of ChiSq in respect
;		to the elements in LISTA.
;	CHISQ: The value of the minimisation function. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	MRQCOF is based on the routine mrqcof described in section 14.4 
;	of Numerical Recipes in C: The Art of Scientific Computing, 
;	W.H. Press, B.P. Flanery, S.A. Teukolsky, W.T. Vetterling,
;	published by Cambridge University Press.
;
; MODIFICATION HISTORY:
;	Adopted by Thomas Ursby, June 1995
;
;-

ndata= (size(x))(1)
ma= (size(a))(1)
mfit= (size(lista))(1)
dyda= INTARR(ma)
dyda(lista)=1

; Initialize symmetric alpha, mbeta
alpha= DBLARR(mfit,mfit)
mbeta= DBLARR(mfit)
chisq=0.0

; Calculate the shifts:
dy= CALL_FUNCTION(funcs, x, y, a, dyda, stepfactor=stepfactor)
sig2i=1.0/(sig^2)
FOR j=0,mfit-1 DO BEGIN
  wt= dyda(*,lista(j))*sig2i
  FOR k=0,j DO alpha(j,k)= TOTAL(wt*dyda(*,lista(k)))
  mbeta(j)= -TOTAL(wt*dy)
ENDFOR
chisq= TOTAL(dy^2*sig2i)
FOR j=1,mfit-1 DO FOR k=0,j-1 DO alpha(k,j)=alpha(j,k)

END