FUNCTION INTEGRATE_BOX,spot_list,reb,image,image_b,par,peak_variance,success_flag,bc_coeff=bc_coeff,show_spots=show_spots,show_m=show_m,miller=miller,study_win=study_win,main_win=main_win
;+
; NAME:
;	INTEGRATE_BOX
;
; PURPOSE:
;	Box integration routine. Optimizes the area of integration via the routines
;	GET_BOXREG.PRO and GET_BOXLEVEL.PRO.
;
; CATEGORY:
;	Laue processing.
;
; CALLING SEQUENCE:
;	integrated = INTEGRATE_BOX(spot_list,reb,image,image_b,par,peak_variance,success_flag,bc_coeff=bc_coeff,/show_spots,/show_m,miller=miller,$
;		study_win=study_win,main_win=main_win)
;
; INPUTS:
;	spot_list : a structure containing the spot information
;	image : the original image
;	image_b : the byte image with the spot region marked.
;	reb : the rebinning factor
;	par : the parameters of the Laue experiment
;	show_spots : keyword set if the whole procedure is to be checked.
;	show_m : keyword set if one reflection specified by miller is to be
;		 checked
;	miller : INTARR[3] containing the Miller indices of the reflection 
;		to be studied if the keyword /show_m is set
;	study_win : window id of the window displaying surface plots, contour ... in debug modes
;	main_win : window id of the main window displaying the image
;
; OUTPUTS:
;	integrated : a FLTARR containing the integrated intensities
;	peak_variance : [optionnal] gives the expected variance of peaks
;	bc_coeff : [optionnal] gives the background parameters [0,n_back,bc_mean]
;	success_flag : [optionnal] returns 0 if integration successful
;		1, if problem with background, 2 if problem at integration
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	For each spot the following procedure is followed :
;	1/ The optimum cutoff level is determined by the routine GET_BOXLEVEL.PRO
;	2/ A region of integration which is double of the one defined by the contour
;		at the previously determined fit_level is chosen.
;	3/ The integrated intensity is taken as the total of the background subtracted
;		pixels in that region. The variance is also calculated via the routine
;		GET_BOXVAR.PRO
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, June 97.
;-


;*************
;GET GENERAL PARAMETERS
;*************

IF NOT N_ELEMENTS(main_win) THEN main_win=!D.WINDOW
IF NOT N_ELEMENTS(study_win) THEN BEGIN
 IF KEYWORD_SET(show_spots) OR KEYWORD_SET(show_m) THEN WIN,512,512
 study_win=!D.WINDOW
ENDIF

;get the dimension and size of the spot_list
number_spots=(SIZE(spot_list))(1)

;create the array for the integrated intensities
integrated = FLTARR(number_spots)
peak_variance = FLTARR(number_spots)
success_flag = FLTARR(number_spots)
bc_coeff = FLTARR(3,number_spots)

;arrays for statistics
stat = {ok:0L,bad:0L}
cr = STRING("15b)
stat_format = "($,'ok: ',I4,' bad: ',I4,' n_spots: ',I5,' ie : ',F4.0,'%',A)"

;*******************
;START INTEGRATING
;*******************

;run the procedure for each spot
FOR i=0,number_spots-1 DO BEGIN ;main loop going on for each spot.

IF KEYWORD_SET(show_m) THEN $
IF ((spot_list(i).m(0) EQ miller(0)) AND (spot_list(i).m(1) EQ miller(1)) AND (spot_list(i).m(2) EQ miller(2))) THEN show_spots=1

;***
;PRINT INFO
;***

IF KEYWORD_SET(show_spots) THEN BEGIN
WSET,main_win
dummy=PIXLAUE2(spot_list(i),reb,image,par,draw=8,color=2)
WSET,study_win

PRINT,''
PRINT,'Peak number :',i
PRINT,'Miller indices :',spot_list(i).m
ENDIF

;recalculate the background with the help of the byte image
xpos = FIX(spot_list(i).xpix)
ypos = FIX(spot_list(i).ypix)
;select the right area on the image
area = image(xpos - FIX(par.boxsize.x/2.):xpos + FIX(par.boxsize.x/2.), ypos - FIX(par.boxsize.y/2.):ypos + FIX(par.boxsize.y/2.))
area_b = area*(image_b(xpos - FIX(par.boxsize.x/2.):xpos + FIX(par.boxsize.x/2.), ypos - FIX(par.boxsize.y/2.):ypos + FIX(par.boxsize.y/2.)) EQ 0)

bc = BACK2D_POLY4(area_b,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,max_slope=par.bg_max_slope)
IF flag EQ 0 THEN BEGIN ; background is ok !
 spot_list(i).bc=bc
 spot_list(i).flag = flag
 spot_list(i).bc_coeff = [0,n_back,bc_mean]
ENDIF

area = area - spot_list(i).bc

;only use the correct restrained area
IF MAX(par.overall_region) NE 0 THEN area(par.overall_region)=0
area(*par.regions(spot_list(i).sector))=0

;area(par.overall_region2)=0

;the value of success flag is the one found in the .flag field of spot_list
success_flag(i) = spot_list(i).flag

;save the background parameters
bc_coeff(*,i)=[0,spot_list(i).bc_coeff(1),spot_list(i).bc_coeff(2)]

;get the optimum cutoff level [% of MAX(area)]
cutoff = GET_BOXLEVEL(area,spot_list(i).bc_coeff(1),spot_list(i).bc_coeff(2),par.dark_current)

;get the best contour for integration
IF par.mode EQ 'mono' THEN $ 
 region = GET_BOXREG(area,cutoff,par.boxsize) $
ELSE $
 region = GET_BOXREG(area,cutoff,par.boxsize,/FINE_SEARCH)

;the integrated intensity is then
integrated(i) = TOTAL(area(region))

;the variance is :
n_peak = (SIZE(region))(1)
gain = par.gain*1.5418/spot_list(i).l
peak_variance(i) = SQRT(GET_BOXVAR(integrated(i),n_peak,spot_list(i).bc_coeff(2),spot_list(i).bc_coeff(1),gain,par.bg_gain,par.dark_current))
IF KEYWORD_SET(SHOW_SPOTS) THEN BEGIN ;show_spots
PRINT,'Spot position [X,Y], [mm]:',spot_list(i).x,spot_list(i).y
PRINT,'Percentage of used profile [%] :',100*(FLOAT(n_peak)/(par.boxsize.x*par.boxsize.y))
PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(area EQ MAX(area)),area)
PRINT,'Peak energy [keV], wavelength [A] :', 12.4/spot_list(i).l, spot_list(i).l
PRINT,'Gain :',gain
PRINT,'Flag [0 : OK ; 1 : bad backgroung ; 2 : bad ii] :',success_flag(i)
PRINT,'Integrated intensity :',integrated(i)
PRINT,'Peak variance :',peak_variance(i)



PRINT,'Original profile'
SURFACE,area
OK=''
READ,'OK ? [Y/N]',OK
CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],C_COLORS=!D.N_COLORS - 60
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Kept region from Original profile'
temp = area
temp(*) = 0
temp(region) = area(region)
CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 190
OK=''
READ,'OK ? [Y/N]',OK

ENDIF ;show_spots

;IF success_flag(i) EQ 0 THEN PRINT,FORMAT = '($, "*")' ELSE PRINT,FORMAT = '($, "b")'
IF success_flag(i) EQ 0 THEN stat.ok=stat.ok+1 ELSE stat.bad=stat.bad+1

IF (i MOD 100) EQ 0 THEN PRINT,FORMAT=stat_format,stat,i+1,100.*(i+1)/FLOAT(number_spots),cr

IF KEYWORD_SET(show_m) THEN show_spots=0
ENDFOR  ;end of the main loop going on for each spot.

PRINT,''
PRINT,'  --> Number of spots integrated :',number_spots
PRINT,''
RETURN,integrated

END



