function ESTIMATE_BOXSIZE,image,ref,parameters,ovlp_dist=ovlp_dist,max_number_ref=max_number_ref,r1=r1,r2=r2,$
	alpha1=alpha1,alpha2=alpha2,satvalue=satvalue,show=show,study_win=study_win,main_win=main_win,bin_fac=bin_fac,mode=mode

;+
; NAME:
;	ESTIMATE_BOXSIZE
;
; PURPOSE:
;	Estimate boxsize and overlap distance before starting a Prow integration session.
;
; CATEGORY:
;	Image processing.
;
; CALLING SEQUENCE:
;	boxsize = ESTIMATE_BOXSIZE(image,ref,parameters,[ovlp_dist=ovlp_dist],[max_number_ref=max_number_ref],$
;		[r1=r1],[r2=r2],[alpha1=alpha1],[alpha2=alpha2],[satvalue=satvalue],[show=show],$
;		[study_win=study_win],[main_win=main_win],[bin_fac=bin_fac]
;
; INPUTS:
;	image : the original image
;	parameters : the parameters of the experiment
;	show : keyword set if the reference profile is to be visualized
;	study_win : window id of the window displaying surface plots, contour ... in debug modes
;
; OUTPUTS:
;	boxsize: the suggested boxsize.
;	ovlp_dist: [optional] the suggested overlap distance.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	1/ Select the spots in between r1,r2,alpha1,alpha2, in descending order of intensity.
;	2/ Remove saturated spots, spots which have a bad FWHM, spots which are shifted.
;	3/ Compute the average spot for a large boxsize (to be sure the whole spot will be included), and 
;		remove its background.
;	4/ Display this to the user.
;	5/ Estimate the number of pixels in the spot which are above noise, get a spot diameter and a cutoff
;		level.
;	6/ Choose a box size of 2 the spot diameter
;	7/ Calculate the long axis and short axis lengths with the chosen cutoff, and multiply the boxsize 
;		with the anisotropy factor.
;	8/ Repeat 7 with the final boxsize suggested.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, August 2000.
;-

IF N_ELEMENTS(mode) EQ 0 THEN mode='mono'

centerX = parameters.cenx - 1
centerY = parameters.ceny - 1
scaleY = parameters.yscale
pix_size = parameters.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY

; Pick the right area:
at = ATAN(ref.y,ref.x)*180/!Pi
sq = SQRT(ref.x^2 + ref.y^2)

x_size=(SIZE(image))(1) & y_size=(SIZE(image))(2)

IF KEYWORD_SET(show) THEN BEGIN ;show
 WSET,main_win
 LAUE_DRAW_SEC,LONARR(x_size,y_size,/NOZERO),bin_fac,parameters,alpha1,alpha2,r1,r2
ENDIF

IF alpha1 GT 180 THEN alpha1_b = alpha1 - 360 ELSE alpha1_b = alpha1 
IF alpha2 GT 180 THEN alpha2_b = alpha2 - 360 ELSE alpha2_b = alpha2

  IF alpha1_b LT alpha2_b THEN $
   w = WHERE( at LT alpha2_b AND $
		at GE alpha1_b AND $
		sq GE r1 AND $
		sq LE r2,n_ref) ELSE $
   w = WHERE( at LT alpha2_b OR $
		at GE alpha1_b AND $
		sq GE r1 AND $
		sq LE r2,n_ref)

 IF n_ref GT 0 THEN BEGIN
  sel_ref=ref(w)
 ENDIF ELSE BEGIN
  ok=DIALOG_MESSAGE('No predictions found !',/ERROR)
  GOTO,end_of_routine
 ENDELSE


;Remove saturated spots
peak_val=image(sel_ref.xpix,sel_ref.ypix)
w_ok=WHERE(peak_val LT satvalue,ct_ok)
IF ct_ok GE 1 THEN sel_ref=sel_ref(w_ok)


; Pick the right spots.
IF MAX(ref.intp(5) EQ 0) OR mode NE 'mono' THEN BEGIN
 ;then this was not loaded from DENZO, and we must assess peak values.
 peak_val=image(sel_ref.xpix,sel_ref.ypix)
 s=REVERSE(SORT(peak_val))
 sel_ref=sel_ref(s(0:(max_number_ref-1)<((SIZE(sel_ref))(1)-1)))
ENDIF ELSE BEGIN 
 ;we can use the denzo integration in this case.
 ;sort in descending order
 s=REVERSE(SORT(sel_ref.intp(5)))
 sel_ref=sel_ref(s(0:(max_number_ref-1)<((SIZE(sel_ref))(1)-1)))
ENDELSE
n_ref=(SIZE(sel_ref))(1)
PRINT,'Number of selected spots: ',n_ref

;Compute the average spot for a large boxsize of 35 pixels
 large_boxsize=35
 large_box:
 one_spot={x:0L,y:0L,sp:LONARR(large_boxsize,large_boxsize,/NOZERO),bc:LONARR(large_boxsize,large_boxsize,/NOZERO),fwhm:0.0,max_val:0L}
 all_spots=REPLICATE(one_spot,n_ref)
 all_spots.x = FIX(sel_ref.x/pix_sizeX+centerX)
 all_spots.y = FIX(sel_ref.y/pix_sizeY+centerY)
 all_spots.max_val=image(all_spots.x,all_spots.y)
 FOR k=0,(n_ref-1) DO BEGIN
  all_spots(k).sp = image(all_spots(k).x - FIX(large_boxsize/2.) : $
    all_spots(k).x + FIX(large_boxsize/2.) , all_spots(k).y - $
    FIX(large_boxsize/2.):all_spots(k).y + FIX(large_boxsize/2.))
  all_spots(k).bc =  BACK2D_POLY4(all_spots(k).sp,25,1.2,0)
  dum = WHERE((all_spots(k).sp - all_spots(k).bc) GE MAX(all_spots(k).sp - all_spots(k).bc)/2.0,count)
  all_spots(k).fwhm = 0.001*pix_size*SQRT(count)
 ENDFOR

;Remove spots which have a bad FWHM, spots which are shifted.
 sigma = STDEV(all_spots.fwhm,mean) 
 w_fwhm = WHERE((all_spots.fwhm - mean)^2 LE 5.0*sigma^2,count)
 IF (count GT 0) THEN BEGIN
  all_spots=all_spots(w_fwhm) 
  sel_ref=sel_ref(w_fwhm) 
  PRINT,'Number of spots rejected for bad FWHM: ',n_ref-count
  n_ref=count
 ENDIF

  bs=large_boxsize
  bsx2 = FIX(bs/2.)
  bsy2 = FIX(bs/2.)

  FOR k=0,(n_ref-1) DO all_spots(k).max_val=MAX(all_spots(k).sp(bsx2-bsx2/2:bsx2+bsx2/2,bsy2-bsy2/2:bsy2+bsy2/2))

  w_shift = WHERE( $
  all_spots.sp(FIX(bs*bsy2+bsx2)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*bsy2+bsx2-1)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*bsy2+bsx2+1)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2-1)+bsx2)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2-1)+bsx2-1)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2-1)+bsx2+1)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2+1)+bsx2)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2+1)+bsx2-1)) EQ LONG(all_spots.max_val) OR $
  all_spots.sp(FIX(bs*(bsy2+1)+bsx2+1)) EQ LONG(all_spots.max_val) $
  ,count)
 IF (count GT 0) THEN BEGIN
  all_spots=all_spots(w_shift) 
  sel_ref=sel_ref(w_shift) 
  PRINT,'Number of spots rejected for bad shift: ',n_ref-count
  n_ref=count
 ENDIF

;Compute the average spot and remove its background.
average_spot=all_spots(0).sp
FOR k=1,(n_ref-1) DO average_spot=average_spot+all_spots(k).sp

average_spot=LONG(average_spot/FLOAT(n_ref))
average_bc = BACK2D_POLY4(average_spot,25,1.2,0)

;Display this to the user.
 PRINT,'Examine the spot and click Done to continue'
 XSURFACE1,average_spot-average_bc

;Estimate the number of pixels in the spot which are above noise, get a spot diameter and a cutoff level.
;The area for 5.0% of a gaussian corresponds to twice the one for 47 % [check ?]
;The area for 2.0% of a gaussian corresponds to twice the one for 38 % [check ?]
;The area for 1.0% of a gaussian corresponds to twice the one for 32 % [check ?]
;The area for 0.5% of a gaussian corresponds to twice the one for 27 % [check ?]
;The area for 0.25% of a gaussian corresponds to twice the one for 22 % [check ?]
;The area for 0.1% of a gaussian corresponds to twice the one for 18 % [check ?]

sp = average_spot-average_bc 
max_sp=MAX(sp(bs/2-bs/4:bs/2+bs/4,bs/2-bs/4:bs/2+bs/4),w_max_sp)
xy_max_sp=WHERESUB(w_max_sp(0),sp(bs/2-bs/4:bs/2+bs/4,bs/2-bs/4:bs/2+bs/4))+bs/2-bs/4

;First estimate the surface at FWHM
cutoff=50.0
arr_spot_size=FLTARR(2,35)
;Remove the passes
pass=SEARCH_PASS(sp,tolerance=2)
i=0
sp2=sp & sp2(pass) = 0
REPEAT BEGIN
 fwhm_surface=SEARCH2D(sp2,xy_max_sp(0),xy_max_sp(1),max_sp/100.0*cutoff,max_sp,/DIAGONAL)
 spot_size = (SIZE(fwhm_surface))(1)
 arr_spot_size(0,i)=cutoff
 arr_spot_size(1,i)=spot_size
 cutoff=cutoff/1.2
 i=i+1
ENDREP UNTIL cutoff LE 0.1
;Take the maximum of second derivative
 der1=REFORM(DERIV(arr_spot_size(0,*),arr_spot_size(1,*)))
 der1=SMOOTH(der1,5)
 der2=DERIV(arr_spot_size(0,*),der1)
 shift_der2=SHIFT(der2,1) & shift_der2(0)=der2(0)
 w_max_der2=WHERE((der2-shift_der2) GT 10,ct_max_der2)
 WSET,study_win
 PLOT_OI,arr_spot_size(0,*),arr_spot_size(1,*),XTITLE='Cutoff [%]',YTITLE='Number of pixels in spot',CHARSIZE=1.5
 IF ct_max_der2 EQ 0 THEN BEGIN
  ok=DIALOG_MESSAGE('Error with peak search (Is the integration mode correct ?) !',/ERROR)
  GOTO,end_of_routine
 ENDIF

 spot_cutoff=(arr_spot_size(0,*))(w_max_der2(0))
 spot_diameter=2*SQRT((arr_spot_size(1,*))(w_max_der2(0))/!PI)
; PRINT,'Values of second derivative around peaked value: ',der2((0>w_max_der2(0)-5):(w_max_der2(0)+5)<((SIZE(der2))(1)-1))
 PRINT,'First suggested spot diameter [pixels]: ',spot_diameter
 PRINT,'First suggested spot cutoff [%]: ',spot_cutoff

;Choose a box size of 2 the spot diameter and calculate the long axis and short 
;axis lengths with the chosen cutoff
new_bs=FIX(spot_diameter*2)
IF new_bs/2 EQ new_bs/2. THEN new_bs=new_bs+1

IF new_bs GT large_boxsize THEN BEGIN
 large_boxsize = LONG(1.4*large_boxsize)
 PRINT,'Increasing starting box ...'
 GOTO, large_box
ENDIF

PRINT,'New boxsize chosen = ',new_bs

sp = sp(bs/2-new_bs/2:bs/2+new_bs/2,bs/2-new_bs/2:bs/2+new_bs/2)
pass=SEARCH_PASS(sp,tolerance=2)
sp2=sp & sp2(pass) = 0
PRINT,'Examine the spot and click Done to continue'
;XSURFACE1,sp2<(MAX(sp))*spot_cutoff/100.

WSET,study_win
bs=new_bs
xc = FIX(bs/2.)
yc = FIX(bs/2.)
longaxis = FIND_LONGAXIS(sp2,xc,yc,w,w1,w2,reb_box,LEVEL=spot_cutoff,/CONTINUOUS)

	   IF longaxis GT 0 THEN BEGIN
            reb_box2=reb_box
	    reb_box2(*)=0
	    reb_box2(w1)=1
	    reb_box2(w2)=1
 	    CONTOUR,reb_box2,LEVELS=[0.99999],XSTYLE=4,YSTYLE=4
	    dum=TVRD()
	    w_dum=WHERE(dum NE 0,ct_dum)
 	    IF ct_dum NE 0 THEN BEGIN
		xy_dum=WHERESUB(w_dum,dum)
		x_start=FLOAT(xy_dum(0))
		y_start=IMAGINARY(xy_dum(0))
		x_end=FLOAT(xy_dum(ct_dum-1))
		y_end=IMAGINARY(xy_dum(ct_dum-1))
		CONTOUR,reb_box,LEVELS=[sp(xc,yc)*spot_cutoff/100.,sp(xc,yc)],XRANGE=[0,(SIZE(reb_box))(1)-1],YRANGE=[0,(SIZE(reb_box))(2)-1]
		PLOTS,[[x_start,y_start],[x_end,y_end]],/DEV
	    ENDIF
            PRINT,'Long axis length [raster] : ',longaxis
	    ENDIF

bs = 2*ROUND(longaxis)
IF bs/2 EQ bs/2. THEN bs=bs+1
PRINT,'**************************************************'
PRINT,'Suggested boxsize [raster]: ',bs,' x',bs
PRINT,'Suggested overlap distance: [mm]',longaxis*pix_size*0.001
PRINT,'**************************************************'
;erase the displayed region if necessary
IF KEYWORD_SET(show) THEN BEGIN ;show
 WSET,main_win
 LAUE_DRAW_SEC,LONARR(x_size,y_size,/NOZERO),bin_fac,parameters,alpha1,alpha2,r1,r2
ENDIF
ovlp_dist=longaxis*pix_size*0.001
RETURN,bs
end_of_routine:
END




