pro WEIGHT_EXT, filename, sthol, sigmaa, sigmap, sthollim, $
		centric=centric, epsilon=epsilon, $
		resolution=resolution, par=par, sdfac=sdfac, $
		deconvolute=deconvolute, calc_sigmad=calc_sigmad, $
		no_bins=no_bins, $
		labin=labin, labout=labout, output=output, format=format, $
		sigmad=sigmad,proba_cutoff=proba_cutoff,plot=plot	
;+
; NAME:
;	WEIGHT_EXT
;
; PURPOSE:
; 	(ext=extended). New way of inputing centric and epsilon.
;
;	Calculates a probability weighted average of DeltaF (output
;	DELFOPT). This can be used either for difference maps or for 
;	refinement. For maps phase probability weights should also be used
;	(output DELFMAP).
;	The relative change : RELOPT = WCMB*DELFOPT/((F1+F2)/2.) can also be calculated.
;
;	The  probability weighted average of DeltaF is:
;
;		DELFOPT = q * (F2 - F1)
;
;		                  SigmaD^2/2
;		q= ---------------------------------------- (nc)
;		            SigmaD^2/2 + SIGobs^2
;			
;		                   SigmaD^2
;		q= ----------------------------------------  (c)
;		              SigmaD^2 + SIGobs^2
;	where
;		SigmaD^2 = epsilon SigmaP^2 (1-SigmaA^2)
;		SigmaP^2 = Sum(f^2) = <F^2>/epsilon
;		SIGobs = the standard deviation of deltaF (distribution assumed
;			to be Gaussian). Thus if deltaF is the difference 
;			between two measured F:s then SIGobs^2=SIG1^2+SIG2^2
;			whereas if one is calculated then SIGobs=SIG1
;		nc, c = non-centro symmetric reflections and centro symmetric
;			reflections respectively.
;		
;	For difference maps:
;
;		DELFMAP = w2 * DELFOPT - (w1-w2) * F1
;	where
;		W1 = SigmaA-weight for (F1,Fc) 
;		W2 = SigmaA-weight for (F2,Fc) 
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
;	WEIGHT_EXT, filename, sthol, sigmaa, sigmap, sthollim [, $
;		centric=centric, epsilon=epsilon, $
;		resolution=resolution, par=par, sdfac=sdfac, $
;		deconvolute=deconvolute, calc_sigmad=calc_sigmad, $
;		labin=labin, labout=labout, output=output, format=format, $
;		proba_cutoff=proba_cutoff,plot=plot]
;
; INPUTS:
;	FILENAME: Name of the input MTZ file (or MTZ dump file; see
;		READ_MTZ.PRO).
;		See LABIN for required labels.
;	STHOL, SIGMAA, SIGMAP and STHOLLIM are as output from READ_SIGMAA
;	so normally you don't need to worry about the details below. If
;	the option CALC_SIGMAD is used then these do not need to be given.
;	STHOL: = STHOLLIM(*)-STHOLLIM(0)/2.
;		( or maybe it is actually the average SinTheta/Lamba in
;		  the bin when it comes from SIGMAA?)
;	SIGMAA: Values of SigmaA in resolution bins (as output from SIGMAA).
;	SIGMAP: Values of SigmaP in resolution bins (as output from SIGMAA).
;	STHOLLIM: A vector with elements giving the maximum value of 
;		Sin(Theta)/Lambda in the corresponding bin for SIGMA_A etc. 
;		The first bin should be between 0 and sqrt(binsize) and all 
;		the bins of the same size in (Sin(Theta)/Lambda)^2 
;		(i.e. "binsize").
;
; KEYWORDED PARAMETERS:
;	CENTRIC: An array BYTARR(7) containing information of whether the 
;		the different classes of reflections are centro-symmetric.
;		The seven different set of reflections are in order:
;		hkl,hk0,h0l,0kl,h00,0k0,00l
;		If the set is centric then the corresponding element should
;		be set to 1. 
;		Default: No centric sets.
;	EPSILON: BYTARR(7) in the same way as for CENTRIC but this 
;		array contains the value of epsilon.
;		Default: epsilon=1 for all sets.
;		Epsilon is the ratio <F^2>/Sum(f^2) (see Int 
;		Tables (1985), vol II, p355-6). Epsilon is thus not the same
;		as alpha in the above reference since alpha assumes that also
;		systematic absences are included in the average. 
;		The systematic absent reflections should not be included in 
;		the input (or alternatively alpha should be input instead
;		and the absences included (all of them, but what if data not
;		complete?); in this case systematic absences should off course
;		still be set to zero which is not done by the program).
;	PAR: The cell parameters. Structure containing the tags a,b,c,alpha, $
;		beta,gamma. Angles in degrees. Used for caclulating the d-
;		spacing to assign right bin for SIGMA_WILSON (see D_CALC). 
;		Defaults to cubic with a=50.0.
;	SDFAC: The e.s.d.:s of the measurements are multiplied with this
;		factor prior to calculating the weighting factor.
;		The output is also adjusted (the labels SIG1 and SIG2).
;	RESOLUTION: FLTARR(2) with the limits in Angstroms (either order). 
;		Reflections outside these limits are removed. In any case
;		reflections of higher resolution than max((2*sthol)^2) 
;		are removed.
;		Default [1000,0.1]
;	DECONVOLUTE: If set then the measurement errors are deconvoluted
;		from SigmaD: 
;			SigmaD'^2= (SigmaD^2-2*SigmaObs^2)>(SigmaD^2/2)
;		where SigmaObs is the rms of the SIGobs in resolution
;		bins.
;	CALC_SIGMAD: If set then SIGMAA and SIGMAP are not 
;		used but instead the required SigmaD is determined with 
;		WEIGHT_DISTR_EXT.PRO (assumes DeltaF<<F).
;		NO_BINS should be set in this case.
;	NO_BINS: Number of bins (of SIGMAA,SIGMAP,STHOL and STHOLLIM).
;		Only used if CALC_SIGMAD is set. Default 10.
;	LABIN: To assign the columns required by the program to the right
;		columns in the input file.
;		STRARR(2,n) where LABIN(0,i) contains the label to be used
;		in the program and LABIN(1,i) contains the label found in 
;		the input MTZ file. LABIN is converted to upper case before 
;		comparing with the MTZ file. 'n' can be any number so that 
;		not all labels have to be edited. They can be in any order.
;		The required labels are:
;		-F1 and F2
;		-At least SIG1 or SIG2. If both are given then 
;			SIGobs^2=SDFAC^2*(SIG1^2+SIG2^2)
;		 otherwise SIGobs=SDFAC*SIG1 (or SIG2). Note that no check is 
;		 made whether it reasonable to have two sigmas.
;		Also recognised is:
;		-W1 and W2. Default is 1.0
;	LABOUT: To change name of the columns for the output. 
;		STRARR(2,n) where LABIN(0,i) contains the label from the input
;		and LABOUT(1,i) contains the label that will be output. Since
;		the labels are not output at the moment this only useful
;		for changing column names for OUTPUT. NOTE that LABIN only
;		change internal names so a label called 'sigobs' on input 
;		and that is renamed to 'sig1' with LABIN will still be called
;		'sigobs' when LABOUT is applied. 
;	OUTPUT: If given then only those columns whose names (after LABOUT has 
;		been applied) matches OUTPUT will be output and in the order
;		as they appear in OUTPUT. (STRARR(n)) 
;		The calculated "q" (see above) is called Q so if 'Q' is an 
;		element of in OUTPUT then the column Q with the factor will 
;		be contained in the output ASCII file. DELFOPT, RELOPT and 
;		DELFMAP can be specified (see above). If 'QF' is given then
;		a column called QF is output containing Q multiplied with
;		F (where F is a column called F after LABOUT has been 
;		applied). The column F will still be output seperatly if 
;		'F' is in OUTPUT. 
;	FORMAT: Output format (given as argument to PRINTF). See OUTPUTS
;		below.
;	PROBA_CUTOFF : Those reflection showing MAX(p(DeltaF)) < PROBA_CUTOFF
;		will be rejected. This is probably wise to reject strong
;		differences having abusively small sigmas. Defaults to zero
;		(nothing rejected). A good value is 0.001 ??
;	PLOT : If set, plot various things
;
; OUTPUTS:
;	Writes an ASCII file with the input name (FILENAME) after having 
;	removed a possible extension and added '.txt'. This file contains 
;	the indices and the output columns as specified by OUTPUT in the 
;	format '(3I4,nF9.3)' where n is the number of columns (not counting
;	hkl). If the keyword FORMAT is given then this will be used as output
;	format (no check is done about the number of columns, or anything
;	actually). See keyword OUTPUT for more details about the output. 
;	SIGMAD: Sigma_D in the same resolution bins as SigmaA and SigmaP. 
;		Note that the value used for each reflection is a linear
;		interpolation of the this and modified with epsilon.
;		(See PURPOSE and DECONVOLUTE.)
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, November 1995
;	Major changes, February-April 1996
;	04-09-1996 Fixed bugs (If no reflections in the first half of first
;		SIGMAD-bin or last half of the last bin the program would
;		crash, If trying to output q*F the program would crash).
;		Also added NO_BINS input so that STHOL and STHOLLIM do
;		not need to be input if CALC_SIGMAD option is used.
;		More reflection classes should be added (hhh,hh0,h0h,
;		0hh,hhl,hlh,lhh ???)
;		Some diagnostics should maybe also be added.
;		Should the bins be between smin and smax and 
;		not 0 and smax?
;-

; =======================================
; Read in the MTZ-file:
ref= READ_MTZ(filename,labels,bad_ref=bad_ref)

; Edit labels:
labels= STRUPCASE(labels)
labels_int= labels
IF (N_ELEMENTS(labin) gt 0) THEN BEGIN
  labin= STRUPCASE(labin)
  IF ((size(labin))(0) eq 1) THEN no_labin=1 ELSE no_labin=(size(labin))(2)
  FOR i=0,no_labin-1 DO BEGIN
    index= WHERE(labels eq labin(1,i), ct)
    IF (ct eq 0) THEN PRINT, 'Input label '+labin(1,i)+' not found.' $
    ELSE labels_int(index)=labin(0,i)
  ENDFOR
ENDIF

; =======================================
; Find the right columns and calculate sigobs:
if1= 	WHERE(labels_int eq 'F1' 	,ct1)-3
if2= 	WHERE(labels_int eq 'F2' 	,ct2)-3
isig1= 	WHERE(labels_int eq 'SIG1'	,ct3)-3
isig2= 	WHERE(labels_int eq 'SIG2'	,ct4)-3
iw1= 	WHERE(labels_int eq 'W1'	,ct5)-3
iw2= 	WHERE(labels_int eq 'W2'	,ct6)-3
IF ((ct1 eq 0) or (ct2 eq 0) or ((ct3 eq 0)and(ct4 eq 0))) THEN $
	MESSAGE,'All required labels not found. (F1, F2, SIG1 and/or SIG2)'


;Find out which bad_ref can be kept.
w_bad_ok = WHERE((bad_ref.entry(if1) NE -9999) AND (bad_ref.entry(if2) NE -9999),ct_bad_ok)
IF ct_bad_ok GE 1 THEN bad_ref=bad_ref(w_bad_ok)

IF (ct3 GT 0) THEN BEGIN
 w_bad_ok = WHERE(bad_ref.entry(isig1) NE -9999,ct_bad_ok)
 IF ct_bad_ok GE 1 THEN bad_ref=bad_ref(w_bad_ok)
ENDIF

IF (ct4 GT 0) THEN BEGIN
 w_bad_ok = WHERE(bad_ref.entry(isig2) NE -9999,ct_bad_ok)
 IF ct_bad_ok GE 1 THEN bad_ref=bad_ref(w_bad_ok)
ENDIF

IF (ct5 GT 0) THEN BEGIN
 w_bad_ok = WHERE(bad_ref.entry(iw1) NE -9999,ct_bad_ok)
 IF ct_bad_ok GE 1 THEN bad_ref=bad_ref(w_bad_ok)
ENDIF

IF (ct6 GT 0) THEN BEGIN
 w_bad_ok = WHERE(bad_ref.entry(iw2) NE -9999,ct_bad_ok)
 IF ct_bad_ok GE 1 THEN bad_ref=bad_ref(w_bad_ok)
ENDIF

;Add these bad_ref to the correct ref
IF (SIZE(bad_ref))(0) NE 0 THEN BEGIN
 s_ref=(SIZE(ref))(1) &  s_bad_ref=(SIZE(bad_ref))(1) 
 PRINT,'Number of recovered reflections (Probably Fmono not measured ?):',s_bad_ref
 PRINT,'Total number of treated reflections : ',s_ref+s_bad_ref

 tmp_ref=ref
 ref=REPLICATE(tmp_ref(0),s_ref+s_bad_ref)
 ref(0:s_ref-1).m=tmp_ref.m
 ref(0:s_ref-1).entry=tmp_ref.entry
 ref(s_ref:s_ref+s_bad_ref-1).m=bad_ref.m
 ref(s_ref:s_ref+s_bad_ref-1).entry=bad_ref.entry
ENDIF

; Remove reflections with sigma<=0:
IF (ct3 gt 0) THEN BEGIN
  index= WHERE(ref.entry(isig1) gt 0, ct)
  IF (ct gt 0) THEN ref=ref(index)
ENDIF
IF (ct4 gt 0) THEN BEGIN
  index= WHERE(ref.entry(isig2) gt 0, ct)
  IF (ct gt 0) THEN ref=ref(index)
ENDIF
IF (ct5 eq 0) THEN w1=1.0 ELSE w1=ref.entry(iw1)
IF (ct6 eq 0) THEN w2=1.0 ELSE w2=ref.entry(iw2)

; =======================================
; Definitions:
classes= [[1,1,1],[1,1,0],[1,0,1],[0,1,1],[1,0,0],[0,1,0],[0,0,1]]
IF (N_ELEMENTS(centric) eq 0) THEN centric=BYTARR(7) 
IF (N_ELEMENTS(epsilon) eq 0) THEN epsilon=BYTARR(7)+1
IF (N_ELEMENTS(resolution) eq 0) THEN resolution=[1000.0,0.1] $
ELSE resolution= FLOAT(resolution(REVERSE(SORT(resolution))))
IF (N_ELEMENTS(sdfac) eq 0) THEN sdfac=1.0

; =======================================
; Get sthol limits:
slim= 1/(2*resolution)
; If CALC_SIGMAD is not used then we must make sure that SIGMAA-program
; wasn't used with lower high resolution limit (in STHOLLIM):
IF (N_ELEMENTS(calc_sigmad) eq 0) THEN BEGIN
  no_bins= (size(sthollim))(1)
  slim(1)= slim(1) < sthollim(no_bins-1) 
ENDIF ELSE slim(1)= slim(1)
; s is Sin(Theta)/Lambda for each reflection:
s= 1/(2*d_calc(ref.m,par=par))

; Remove reflections outside the resolution limits:
ind= WHERE((s ge slim(0)) and (s le slim(1)),ct)
IF (ct gt 0) THEN ref= ref(ind)

; =======================================
; sdfac * sqrt( sig1^2 + sig2^2) :
sno= ct3+ct4*2
CASE (sno) OF
 1: sigobs= sdfac * ref.entry(isig1)
 2: sigobs= sdfac * ref.entry(isig2)
 3: sigobs= sdfac * sqrt(ref.entry(isig1)^2 + ref.entry(isig2)^2)
 ELSE: Message, 'Bug!'
ENDCASE

; =======================================
; Calculate the rms of SIGobs in resolution bins if DECONVOLUTION
; has been requested: (We use later sigobsbin with the value of
; sthol of sigmaa etc)
; (Not necessary if WEIGHT_DISTR_EXT.PRO is called.)
IF (N_ELEMENTS(calc_sigmad) eq 0) THEN BEGIN
  binsize= sthollim(0)^2
  sigobsbin= FLTARR(no_bins)
  IF (N_ELEMENTS(deconvolute) gt 0) THEN BEGIN
    s2= 1/(2*d_calc(ref.m,par=par))^2
    FOR i=0,no_bins-1 DO BEGIN 
      ind= WHERE((s2 gt i*binsize) and (s2 le (i+1)*binsize), ct)
      IF (ct gt 0) THEN BEGIN
        sigobsbin(i)= sqrt(total(sigobs(ind)^2)/ct)
      ENDIF
    ENDFOR
  ENDIF
ENDIF

; =======================================
; Call WEIGHT_DISTR_EXT if the proper keyword is set:
IF (N_ELEMENTS(calc_sigmad) gt 0) THEN BEGIN
  IF (N_ELEMENTS(no_bins) eq 0) THEN no_bins=10
  sr= WEIGHT_DISTR_EXT(ref.m,ref.entry(if1),ref.entry(if2), $
	sig=sigobs, $
	centric=centric,epsilon=epsilon, $
	par=par,no_bins=no_bins,count=ctr)
  sthollim= REFORM(sqrt(sr(0,*))/2)
  sthol= sthollim-sthollim(0)/2
  binsize= sthollim(0)^2
  ; These will be used later to calculate sigmad. Maybe would
  ; be clearer to calculate sigmad directly.
  sigmaa=FLTARR((size(sthollim))(1))
  ; 2/7/96: Used to be "sqrt(2*sr(1,*)^2)" but there shouldn't be 
  ;         a factor 2 (?):
  sigmap= REFORM(sr(1,*))
  IF (N_ELEMENTS(deconvolute) gt 0) THEN sigobsbin= REFORM(sr(2,*)) $
  ELSE sigobsbin= FLTARR(no_bins)
ENDIF

; =======================================
; Calculate sigma_d2= sigma_d^2 = epsilon * sigma_N^2 (1-sigma_A^2)
; (for general non-centrosymmetric reflections).
; and adjust for special and centric set of reflections:
; First a linear interpolation (in Sin(Theta)/Lambda, maybe should
; be linear in (Sin(Theta)/Lambda)^2?):
; (a little bit too complicated...but maybe possible to follow:)
; 1
; We set s to between sthollim(0) and sthollim(no_bins-1) for "practical
; reasons" (see "lbin" and "ubin" below):
s= (s>sthollim(0))<sthollim(no_bins-1)
; 2
; Each "s" is between the middle of two bins. "lbin" is the lower
; one and "ubin" is the upper one. For reflections with "s" in the
; first half of the first bin and the second half of the last bin 
; there will only be one value of SIGMAD (i.e. no interpolation):
bin = FIX(s^2/binsize) < (no_bins-1)

lbin = (bin + (s ge sthol(bin)) - 1)
ubin = (lbin + 1) < (no_bins-1)
ind= WHERE(lbin eq ubin, ct)
IF (ct gt 0) THEN lbin(ind) = lbin(ind)-1
; Now lbin should point to the next lower sthol limit 
; and ubin be lbin + 1 except for the reflections with "s" in the
; first half of the first bin and the second half of the last bin . 
; (Remember that e.g. sthol(0) is somewhere between 0 and 
; sqrt(binsize) etc.) 
fr1 = (s-sthol(lbin))/(sthol(ubin)-sthol(lbin))
sigmap1 = fr1 * sigmap(lbin) +    (1-fr1) * sigmap(ubin)
sigmaa1 = fr1 * sigmaa(lbin) +    (1-fr1) * sigmaa(ubin)
sigobs1 = fr1 * sigobsbin(lbin) + (1-fr1) * sigobsbin(ubin)
sigma_d2 = sigmap1^2 * (1-sigmaa1^2)
; Deconvolution (if DECONVOLUTE is not set then sigobs1=0):
; (The factor 2 is there since sigma_d2 is two times the distribution
;  parameter of a Gaussian distribution.)
sigma_d2= (sigma_d2-2*sigobs1^2)>(sigma_d2/2)
; This is only for output of SIGMAD:
sigmad = sqrt(sigmap^2 * (1-sigmaa^2))
sigmad = sqrt((sigmad^2-2*sigobsbin^2)>(sigmad^2/2))

; Special set of reflections:
FOR i=0,6 DO BEGIN
  IF (epsilon(i) gt 1) THEN BEGIN 
    ind= WHERE(((classes(0,i) xor (abs(ref.m(0)) gt 0))+ $
	(classes(1,i) xor (abs(ref.m(1)) gt 0))+ $
	(classes(2,i) xor (abs(ref.m(2)) gt 0))) eq 0, ct)
    IF (ct gt 0) THEN sigma_d2(ind)= epsilon(i)*sigma_d2(ind)
  ENDIF
ENDFOR
; Centric reflections:
FOR i=0,6 DO BEGIN
  IF (centric(i) gt 0) THEN BEGIN 
    ind= WHERE(((classes(0,i) xor (abs(ref.m(0)) gt 0))+ $
	(classes(1,i) xor (abs(ref.m(1)) gt 0))+ $
	(classes(2,i) xor (abs(ref.m(2)) gt 0))) eq 0, ct)
    IF (ct gt 0) THEN sigma_d2(ind)= 2.0*sigma_d2(ind)
  ENDIF
ENDFOR

; =======================================
; Get the weighting factor (NOTE that the parameter sigma_d2 has been
; modified above so that the same expression below apply for both 
; centric and non-centric reflections.)

sigobs2=sigobs^2
q= (sigma_d2/2.0)/(sigobs2+(sigma_d2/2.0))
delfopt= q*(ref.entry(if2)-ref.entry(if1))
relopt= w1*delfopt/(0.5*(ref.entry(if2)+ref.entry(if1)))
delfmap= w2*delfopt-(w1-w2)*ref.entry(if1)

; =======================================
; Reject outliers according to the maximum probability criterium
; ie PROBA_CUTOFF
; The maximum probability occurs exactly at delfopt
; so we have to calculate p(delfopt)
IF N_ELEMENTS(proba_cutoff) THEN BEGIN
maxp=EXP(-delfopt^2/sigma_d2 -((1-q)*(ref.entry(if2)-ref.entry(if1)))^2/(2*sigobs2))
w_proba_ok = WHERE(maxp GT proba_cutoff,ct_ok)
w_proba_rejected = WHERE(maxp LE proba_cutoff,ct_rejected)
IF KEYWORD_SET(plot) THEN BEGIN
PLOT_OO,ABS(ref.entry(if2)-ref.entry(if1)),sigobs,XTITLE=('Absolute Nonoptimized DeltaF'),YTITLE='(Sigma (DeltaF))',PSYM=3,CHARTHICK=1.5
IF (SIZE(w_proba_rejected))(0) GT 1 THEN BEGIN
 IF (SIZE(w_proba_rejected))(1) GT 1 THEN $
  OPLOT,ABS(ref(w_proba_rejected).entry(if2)-ref(w_proba_rejected).entry(if1)),sigobs(w_proba_rejected),PSYM=2 ELSE $
  XYOUTS,ABS(ref(w_proba_rejected).entry(if2)-ref(w_proba_rejected).entry(if1)),sigobs(w_proba_rejected),'*',/DATA
ENDIF 
ENDIF

q=q(w_proba_ok)
ref=ref(w_proba_ok)
delfopt=delfopt(w_proba_ok)
relopt=relopt(w_proba_ok)
delfmap=delfmap(w_proba_ok)
PRINT, 'Number of ref rejected according to maximum probability criteria : ',ct_rejected
ENDIF

; =======================================
; Reject outliers having f1 and f2 both equal to 0 !
w_not_both_zero = WHERE((ref.entry(if2) GT 0) OR (ref.entry(if1) GT 0), ct_not_both_zero)
IF ct_not_both_zero GT 0 THEN BEGIN
 tmp_s1=(SIZE(ref))(1)
 q=q(w_not_both_zero)
 ref=ref(w_not_both_zero)
 delfopt=delfopt(w_not_both_zero)
 relopt=relopt(w_not_both_zero)
 delfmap=delfmap(w_not_both_zero)
 PRINT, 'Number of ref rejected because both structure factor are 0 : ',tmp_s1-(SIZE(ref))(1)
ENDIF ELSE MESSAGE,'Not a single meaningful difference !'

; =======================================

PRINT, 'Average Q:', total(q)/(size(q))(1)

; Edit labels for output:
IF (N_ELEMENTS(labout) gt 0) THEN BEGIN
  labout= STRUPCASE(labout)
  IF ((size(labout))(0) eq 1) THEN no_labout=1 ELSE no_labout=(size(labout))(2)
  FOR i=0,no_labout-1 DO BEGIN
    index= WHERE(labels eq labout(0,i), ct)
    IF (ct eq 0) THEN PRINT, 'Output label '+labout(0,i)+' not found.' $
    ELSE labels(index)=labout(1,i)
  ENDFOR
ENDIF

; Add the labels corresponding to the DeltaFopt, weight, the modified 
; weight , Relopt and the modified F: 
tmp=labels
labels= STRARR((size(labels))(1)+5)
labels(0:(size(labels))(1)-6)=tmp
labels((size(labels))(1)-5:(size(labels))(1)-1)= ['Q','DELFOPT','DELFMAP','QF','RELOPT']
index= WHERE(labels eq 'F', ct)
IF (ct eq 0) THEN $
  data= DBLARR((size(ref.entry))(1)+4,(size(ref.entry))(2)) $
ELSE data= DBLARR((size(ref.entry))(1)+5,(size(ref.entry))(2))
data(0:(size(labels))(1)-6-3,*)= ref.entry(*)
data((size(labels))(1)-5-3,*)= q
data((size(labels))(1)-4-3,*)= delfopt
data((size(labels))(1)-3-3,*)= delfmap
data((size(labels))(1)-2-3,*)= relopt
IF (ct gt 0) THEN data((size(labels))(1)-1-3,*)= q*data(index-3,*)

; =======================================
; Determine which labels to output:
IF (N_ELEMENTS(output) gt 0) THEN BEGIN
  output= STRUPCASE(output)
  outindex= INTARR((size(output))(1))
  FOR i=0,(size(output))(1)-1 DO BEGIN
    index= WHERE(labels eq output(i), ct)
    IF (ct eq 0) THEN BEGIN
      PRINT, 'Output '+output(i)+' not found.'
      outindex(i)=-1
    ENDIF ELSE outindex(i)=index-3
  ENDFOR    
  outindex= outindex(WHERE(outindex ne -1))
ENDIF ELSE outindex= INDGEN((size(labels))(1))

; =======================================
; Write out the new file:
pos= RSTRPOS(filename,'.')
IF (pos eq -1) THEN pos= STRLEN(filename)
filename_out= STRCOMPRESS(STRMID(filename,0,pos)+'.txt',/REMOVE_ALL)
OPENW, file, filename_out, /GET_LUN
no_col= (size(outindex))(1)
IF (N_ELEMENTS(format) eq 0) THEN $
	format='(3I4,'+STRCOMPRESS(STRING(no_col),/remove_all)+'F9.3)'

FOR i=0L,(size(ref))(1)-1 DO $
	PRINTF, format=format, file, $
		ref(i).m, data(outindex,i)
FREE_LUN, file

END


