pro WEIGHT, filename, sigma_wilson, dstar2, $
		centric=centric, special=special, epsilon=epsilon, $
		resolution=resolution, par=par, sdfac=sdfac, $
		labin=labin, labout=labout, output=output
;+
; NAME:
;	WEIGHT
;
; PURPOSE:
;	Modify weights so that reflections with large sigma gets smaller
;	weight. The input weights (label W) are modified with the factor 
;		w= sigma_x^2/(sigma^2+sigma_x^2) 
;	where 
;		sigma= sqrt( sig1^2 + sig2^2) 
;	and
;		sigma_x^2= epsilon 1/2 SIGMA_WILSON
;	for non-centric reflections and 
;		sigma_x^2= epsilon SIGMA_WILSON
;	for centric reflections. 
;	See SIGMA_WILSON below for further details.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	WEIGHT, filename , sigma_wilson, dstar2 [, $
;		centric=centric, special=special, epsilon=epsilon, $
;		resolution=resolution, par=par, sdfac=sdfac, $
;		labin=labin, labout=labout, output=output]
;
; INPUTS:
;	FILENAME: Name of the input MTZ file (or MTZ dump file; see
;		READ_MTZ.PRO).
;		The program requires the labels: W, SIG1, SIG2
;		(see LABIN).
;	SIGMA_WILSON: A vector containing the average value of (deltaF)^2
;		as a function of resolution. DeltaF is the amplitude of
;		the difference between the two states and thus not the
;		average value of (deltaFx)^2 where deltaFx is the projection
;		of the vector deltaF along a arbitrary direction (which
;		is actually half as large for non-centrosymmetric reflections).
;		The value refers to general reflections for which epsilon is 
;		one in:
;			<(deltaF)^2>= epsilon Sum(f^2)
;		where f is the atomic scattering factor for the "missing"
;		atoms. Reflections for which epsilon is not one can still
;		be used to calculate SIGMA_WILSON but (deltaF)^2 should
;		be divided by epsilon for these reflection before adding
;		them to the sum (and systematic absences should be counted
;		in the total number of reflections).
;	DSTAR2: A vector with elements giving the maximum value of dstar^2=
;		1/d^2 in the corresponding bin for SIGMA_WILSON. The first
;		bin should be between 0 and "binsize" and all the bins of 
;		the same size.
;
; KEYWORDED PARAMETERS:
;	CENTRIC: An array BYTARR(n,3) containing n triplets defining the
;		set of reflections that are centrosymmetric. The elements
;		should be set to one for those indices that are zero for
;		the set of reflections that are centrosymmetric. For example
;		if the reflections (hk0) are centric then CENTRIC=[0,0,1].
;		Default to no centric reflections.
;	SPECIAL: In the same format as CENTRIC but defining the reflections
;		that have epsilon different from one, BYTARR(m,3).
;		Default to no special reflections.
;	EPSILON: Vector BYTARR(m) giving the epsilon for the set of reflections
;		in SPECIAL.
;	PAR: The cell parameters. Structure containing the tags a,b,c,alpha, $
;		beta,gamma. Angles in degrees. Used for caclulating the d-
;		spacing to assign right bin for SIGMA_WILSON (see D_CALC). 
;		Defaults to cubic with a=50.0.
;	SDFAC: The e.s.d.:s of the measurements are multiplied with this
;		factor prior to calculating the weighting factor.
;		The output is also adjusted (the labels SIG1 and SIG2).
;	RESOLUTION: FLTARR(2) with the limits in Angstroms (either order). 
;		Reflections outside these limits are removed. In any case
;		reflections of higher resolution than max(DSTAR2) are removed.
;		Default [1000,0.1]
;	LABIN: To assign the columns required by the program to the right
;		columns in the input file.
;		STRARR(2,n) where LABIN(0,i) contains the label to be used
;		in the program and LABIN(1,i) contains the label found in 
;		the input MTZ file. LABIN is converted to upper case before 
;		comparing with the MTZ file. 'n' can be any number so that 
;		not all labels have to be edited. They can be in any order.
;		The program requires the labels: W, SIG1, SIG2.
;	LABOUT: To change name of the columns for the output. Since the labels
;		are not output this keyword is superfluous at the moment.
;		STRARR(2,n) where LABIN(0,i) contains the label from the input
;		and LABOUT(1,i) contains the label that will be output. 
;	OUTPUT: If given then only those columns whose names (after LABOUT has 
;		been applied) matches OUTPUT will be output and in the order
;		as they appear in OUTPUT. (STRARR(n)) 
;
; OUTPUTS:
;	Writes a file with the input name (FILENAME) after having removed a
;	possible extension and added '.txt'. This file contains the indices
;	and the output columns as specified by OUTPUT in free format.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, November 1995
;
;-

; Read in the MTZ-file:
ref= READ_MTZ(filename,labels)
; Edit labels:
labels_int= labels
IF (N_ELEMENTS(labin) gt 0) THEN BEGIN
  labin= STRUPCASE(labin)
  FOR i=0,(size(labin))(2)-1 DO BEGIN
    index= WHERE(labels eq labin(1,i), ct)
    IF (ct eq 0) THEN PRINT, 'Input label '+labin(1,i)+' not found.' $
    ELSE labels_int(index)=labin(0,i)
  ENDFOR
ENDIF

; Find the right columns and calculate weighted sigma:
iw= 	WHERE(labels_int eq 'W' 	,ct1)-3
isig1= 	WHERE(labels_int eq 'SIG1'	,ct2)-3
isig2= 	WHERE(labels_int eq 'SIG2'	,ct3)-3
;IF ((ct1 eq 0)or(ct2 eq 0)or(ct3 eq 0)) $
;	THEN MESSAGE,'All required labels not found.'

; Definitions:
no_bins= (size(dstar2))(1)
binsize= dstar2(0)
IF (N_ELEMENTS(centric) eq 0) THEN no_centric=0 $
ELSE no_centric=(size(centric))(1)
IF (N_ELEMENTS(special) eq 0) THEN no_special=0 $
ELSE no_special=(size(special))(1)
IF (N_ELEMENTS(resolution) eq 0) THEN resolution=[1000.0,0.1] $
ELSE resolution= FLOAT(resolution(REVERSE(SORT(resolution))))
IF (N_ELEMENTS(sdfac) eq 0) THEN sdfac=1.0

; Get 1/d^2 limits:
slim= 1/resolution^2
slim(1)= slim(1) < dstar2(no_bins-1)
s= 1/d_calc(ref.m,par=par)^2

; Remove reflections outside the resolution limits:
ind= WHERE((s ge slim(0)) and (s le slim(1)),ct)
IF (ct gt 0) THEN ref= ref(ind)

; sdfac * sqrt( sig1^2 + sig2^2) :
sno= ct2+ct3*2
CASE (sno) OF
 1: sigma= sdfac * ref.entry(isig1)
 2: sigma= sdfac * ref.entry(isig2)
 3: sigma= sdfac * sqrt(ref.entry(isig1)^2 + ref.entry(isig2)^2)
 ELSE: Message, 'Help'
ENDCASE

; Calculate sigma_x2= sigma_x^2 = standard deviation of the distribution
; of the deltaFx (for general non-centrosymmetric reflections).
sigma_x2= 0.5*sigma_wilson(FIX(s/binsize)<(no_bins-1))
FOR i=0,no_special-1 DO BEGIN
  ind= WHERE((special(i,0)*abs(ref.m(0))+ $
	special(i,1)*abs(ref.m(1)) + special(i,2)*abs(ref.m(2))) eq 0, ct)
  IF (ct gt 0) THEN sigma_x2(ind)= epsilon(i)*sigma_x2(ind)
ENDFOR
FOR i=0,no_centric-1 DO BEGIN
  ind= WHERE((centric(i,0)*abs(ref.m(0))+ $
	centric(i,1)*abs(ref.m(1)) + centric(i,2)*abs(ref.m(2))) eq 0, ct)
  IF (ct gt 0) THEN sigma_x2(ind)= 2.0*sigma_x2(ind)
ENDFOR


; Get the weighting factor:
factor= sigma_x2/(sigma^2+sigma_x2)

; Adjust the old weighting factors:
wsigma= ref.entry(iw)
ref.entry(iw)= factor*ref.entry(iw)

; Edit labels:
IF (N_ELEMENTS(labout) gt 0) THEN BEGIN
  labout= STRUPCASE(labout)
  FOR i=0,(size(labout))(2)-1 DO BEGIN
    index= WHERE(labels eq labout(0,i))
    IF (index eq -1) THEN PRINT, 'Output label '+labout(0,i)+' not found.' $
    ELSE labels(index)=labout(1,i)
  ENDFOR
ENDIF

; Determine which labels to output:
IF (N_ELEMENTS(output) gt 0) THEN BEGIN
  outindex= INTARR((size(output))(1))
  FOR i=0,(size(output))(1)-1 DO BEGIN
    index= WHERE(labels eq output(i), ct)
    IF (ct eq 0) THEN BEGIN
      PRINT, 'Output '+output(i)+' not found.'
      outindex(i)=-1
    ENDIF ELSE outindex(i)=index-3
  ENDFOR    
  outindex= outindex(WHERE(outindex ne -1))
ENDIF ELSE outindex= INDGEN((size(labels))(1))

; Write out the new file:
pos= RSTRPOS(filename,'.')
IF (pos eq -1) THEN pos= STRLEN(filename)
filename_out= STRCOMPRESS(STRMID(filename,0,pos)+'.txt',/REMOVE_ALL)
OPENW, file, filename_out, /GET_LUN
form='(3I4,6F9.3)'
FOR i=0,(size(ref))(1)-1 DO $
	PRINTF, format=form, file, ref(i).m, ref(i).entry(outindex), wsigma(i)
FREE_LUN, file

END
