FUNCTION RMSD_PDB,pdbfile1,pdbfile2,main_rmsd=main_rmsd,side_rmsd=side_rmsd
;+
; NAME:
;	RMSD_PDB
;
; PURPOSE:
;	Computes the RMS deviation between 2 pdb files.
;
; CATEGORY:
;	Crystallography, Input
;
; CALLING SEQUENCE:
;   Result = RMSD_PDB(pdbfile1,pdbfile2 [,main_rmsd=main_rmsd,side_rmsd=side_rmsd])
;
; INPUTS:
;	NAME: String containing the name of the pdb files.
;
; KEYWORDED PARAMETERS:
;	MAIN_RMSD: An array containing the rmsd for main chain per residue.
;		MAIN_RMSD(0,*) contains the RMSD
;		MAIN_RMSD(1,*) contains the residue number
;	SIDE_RMSD: An array containing the rmsd for SIDE chain per residue.
;		SIDE_RMSD(0,*) contains the RMSD
;		SIDE_RMSD(1,*) contains the residue number
;
; OUTPUTS:
;	RMSD_PDB returns a FLTARR(2) with 
;		the first element being the overall rmsd for main chain
;		the second element being the overall rmsd for side chain
;
; SIDE EFFECTS:
;	None.
;
; RETRICTIONS:
;	The PDB files should be consistent !
;
; MODIFICATION HISTORY:
;       Written by Dominique Bourgeois, May 2002
;-


;calculations of RMSD deviations between two structures.
;residue numbers should be unique in each pdb file !

;End of input

pdb1 = READ_PDB(pdbfile1,/CNS,header=header1)
pdb2 = READ_PDB(pdbfile2,/CNS,header=header2)

;remove waters for PDB1
w_wat1 = WHERE(pdb1.r NE 'WAT',ct_wat1)
IF ct_wat1 GT 0 THEN pdb1 = pdb1(w_wat1)

;select main chain
w_main_chain1 = WHERE(pdb1.t EQ 'CA' OR pdb1.t EQ 'N' OR pdb1.t EQ 'C' OR pdb1.t EQ 'O' OR pdb1.t EQ 'OT1' OR pdb1.t EQ 'OT2', ct_main_chain1)
IF ct_main_chain1 GT 0 THEN main_chain1 = pdb1(w_main_chain1) ELSE MESSAGE,'No main chain atom in first structure !'
PRINT,'Number of main chain atom in structure 1: ',ct_main_chain1

w_main_chain2 = WHERE(pdb2.t EQ 'CA' OR pdb2.t EQ 'N' OR pdb2.t EQ 'C' OR pdb2.t EQ 'O' OR pdb2.t EQ 'OT1' OR pdb2.t EQ 'OT2', ct_main_chain2)
IF ct_main_chain2 GT 0 THEN main_chain2 = pdb2(w_main_chain2) ELSE MESSAGE,'No main chain atom in second structure !'
PRINT,'Number of main chain atom in structure 2: ',ct_main_chain2

;select side chain
w_side_chain1 = WHERE(pdb1.t NE 'CA' AND pdb1.t NE 'N' AND pdb1.t NE 'C' AND pdb1.t NE 'O' AND pdb1.t NE 'OT1' AND pdb1.t NE 'OT2', ct_side_chain1)
IF ct_side_chain1 GT 0 THEN side_chain1 = pdb1(w_side_chain1) ELSE MESSAGE,'No side chain atom in first structure !'
PRINT,'Number of side chain atom in structure 1: ',ct_side_chain1

w_side_chain2 = WHERE(pdb2.t NE 'CA' AND pdb2.t NE 'N' AND pdb2.t NE 'C' AND pdb2.t NE 'O' AND pdb2.t NE 'OT1' AND pdb2.t NE 'OT2', ct_side_chain2)
IF ct_side_chain2 GT 0 THEN side_chain2 = pdb2(w_side_chain2) ELSE MESSAGE,'No side chain atom in second structure !'
PRINT,'Number of side chain atom in structure 2: ',ct_side_chain2

;calculate the rmsd residue by residue
n_res1 = MAX(pdb1.rn)
n_res2 = MAX(pdb2.rn)
n_res = n_res1>n_res2
 
main_rmsd = FLTARR(2,n_res+1)
side_rmsd= FLTARR(2,n_res+1)
res_number = FLTARR(n_res+1)
total_rmsd = FLTARR(2)

FOR i=0,n_res DO BEGIN

 w_residue1 = WHERE(main_chain1.rn EQ i,ct_residue1)
 w_residue2 = WHERE(main_chain2.rn EQ i,ct_residue2)
 IF (ct_residue1 EQ ct_residue2) AND ct_residue1 GT 0 THEN BEGIN
  ;a residue has been found
  ;the ca atoms of that residue are
  main_res1 = main_chain1(w_residue1)
  main_res2 = main_chain2(w_residue2)
  ;check that the residue is the same
  IF main_res1(0).r NE main_res2(0).r THEN MESSAGE,'Residue numbers are incorrectly matched between the two structures !'
  d2 = (main_res1.x-main_res2.x)^2+(main_res1.y-main_res2.y)^2+(main_res1.z-main_res2.z)^2
  main_rmsd(0,i) = SQRT(TOTAL(d2)/ct_residue1)
  main_rmsd(1,i) = i
 ENDIF 

 w_residue1 = WHERE(side_chain1.rn EQ i,ct_residue1)
 w_residue2 = WHERE(side_chain2.rn EQ i,ct_residue2)
 IF (ct_residue1 EQ ct_residue2) AND ct_residue1 GT 0 THEN BEGIN
  ;a residue has been found
  ;the ca atoms of that residue are
  side_res1 = side_chain1(w_residue1)
  side_res2 = side_chain2(w_residue2)
  IF side_res1(0).r NE side_res2(0).r THEN MESSAGE,'Residue numbers are incorrectly matched between the two structures !'
  d2 = (side_res1.x-side_res2.x)^2+(side_res1.y-side_res2.y)^2+(side_res1.z-side_res2.z)^2
  side_rmsd(0,i) = SQRT(TOTAL(d2)/ct_residue1)
  side_rmsd(1,i) = i
 ENDIF 

  res_number(i) = i

ENDFOR

w_main = WHERE(main_rmsd(0,*) GT 0,ct_main)
PRINT,'Number of matched residues for main chain: ',ct_main
main_rmsd = main_rmsd(*,w_main)
main_res_number = res_number(w_main)

title = 'RMSD Main Chain'
xtitle = 'Residue #'
ytitle = 'RMS deviation for main chain [A]'
xrange = [MIN(main_res_number),MAX(main_res_number)]

PLOT,main_res_number,main_rmsd(0,*),XTITLE=xtitle,YTITLE=ytitle,CHARSIZE=1,XRANGE=xrange,XSTYLE=1

total_rmsd(0)=TOTAL(main_rmsd(0,*))/FLOAT((SIZE(main_rmsd(0,*)))(2))

w_side = WHERE(side_rmsd(0,*) GT 0,ct_side)
PRINT,'Number of matched residues for side chain: ',ct_side
side_rmsd = side_rmsd(*,w_side)
side_res_number = res_number(w_side)

title = 'RMSD Side Chain'
xtitle = 'Residue #'
ytitle = 'RMS deviation for side chain [A]'
xrange = [MIN(side_res_number),MAX(side_res_number)]

PLOT,side_res_number,side_rmsd(0,*),XTITLE=xtitle,YTITLE=ytitle,CHARSIZE=1,XRANGE=xrange,XSTYLE=1

total_rmsd(1)=TOTAL(side_rmsd(0,*))/FLOAT((SIZE(side_rmsd(0,*)))(2))

RETURN,total_rmsd

END

