pro RFACTOR_S, arr1, arr2, ssq, w=w, type=type, sbins=sbins
;+
; NAME:
;	RFACTOR_S
;
; PURPOSE:
;	Calculate R-factors in s^2-bins. Normally called from
;	RFACTOR. s=d*/2=1/2d.
;
; CATEGORY:
;	Crystallography, data analysis.
;
; CALLING SEQUENCE:
;	RFACTOR_S, arr1, arr2, ssq [, w=w, type=type, sbins=sbins]
;
; INPUTS:
;	ARR1,ARR2: The arrays containing the data to be scaled.
;	SSQ: The square of s=d*/2=1/2d. Array of same size as the data. 
;
; KEYWORDED PARAMETERS:
;	W: Weighting. Array of the same dimension as ARR1 and ARR2.
;		The weighting is used for the
;		R-factor calculations if weighting is requested.
;	TYPE: Indicates what type of R-factor (default 1):
;		1: R= Sum(abs(arr1-arr2))/Sum(arr1)
;		2: R= Sum(abs(arr1-arr2))/Sum((arr1+arr2)/2)
;		3: R= Sum(w*abs(arr1-arr2))/ Sum(w*arr1)
;		4: R= Sum(w*abs(arr1-arr2))/ Sum(w*(arr1+arr2)/2)
;		5: R= Sum(w*(arr1-arr2)^2) / Sum(w*arr1^2)
;		6: R= Sum(w*(arr1-arr2)^2) / Sum(w*(arr1+arr2)^2/4)
;	SBINS:  The R-factor is calculated in s^2-bins
;		where s=d*/2=1/2d. SBINS indicates how many bins.
;		The range is s=[0,max(s^2)]. Default 10.
;
; OUTPUTS:
;	Prints a table of R-factors in s^2-bins.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1995
;-

ON_ERROR, 2

IF NOT KEYWORD_SET(w) THEN w=1.0
IF NOT KEYWORD_SET(type) THEN type=1
IF NOT KEYWORD_SET(sbins) THEN sbins=10
r= dblarr(sbins)
racc=dblarr(sbins)
no = intarr(sbins)
noacc = intarr(sbins)

  ; Calculate R-factors in s^2-bins:
  dmin= FIX(10.0*0.5/sqrt(max(ssq)))/10.0 
  sbinsize= (1/(2*dmin))^2/sbins
  racc(0)=0.0 & noacc(0)=0
  FOR i=0,sbins-1 DO BEGIN
    ind= WHERE((ssq gt i*sbinsize) and $
		(ssq le (i+1)*sbinsize), count)
    IF (count ne 0) THEN BEGIN
      no(i) =(size(ind))(1)
      CASE (type) OF
        1: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / TOTAL(arr1(ind))
        2: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(arr1(ind)+arr2(ind))/2)
        3: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / TOTAL(w*arr1(ind))
        4: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(w*(arr1(ind)+arr2(ind)))/2)
        5: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / TOTAL(w*arr1(ind)^2)
        6: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / $
		(TOTAL(w*(arr1(ind)+arr2(ind))^2)/4)
        ELSE: MESSAGE,'Illegal type'
      ENDCASE
    ENDIF $
    ELSE BEGIN
      r(i)= 0.0
      no(i)= 0 
    ENDELSE
    indacc= WHERE(ssq le (i+1)*sbinsize, count)
    IF (count ne 0) THEN BEGIN
      noacc(i) =(size(indacc))(1)
      CASE (type) OF
        1: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(arr1(indacc))
        2: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
		(TOTAL(arr1(indacc)+arr2(indacc))/2)
        3: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(w*arr1(indacc))
        4: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc)))/2)
        5: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			TOTAL(w*arr1(indacc)^2)
        6: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc))^2)/4)
        ELSE: MESSAGE,'Illegal type'
      ENDCASE
    ENDIF $
    ELSE BEGIN
      racc(i)= racc((i-1)>0) 
      noacc(i)= noacc((i-1)>0)
    ENDELSE
  ENDFOR

  ; Print R-factors
  PRINT, '      s^2max	      Dmin             R          No         Racc       No,acc'
  FOR i=0,sbins-1 DO PRINT,(i+1)*sbinsize, ' ',$
		0.5/sqrt((i+1)*sbinsize),' :', r(i), no(i), racc(i), $
			noacc(i)

END
