pro READ_LSCALE, filename, bins, lambda, coeff, ranges, polorder, lmin, lmax, $
		plot=plot, oplot=oplot, last_plot=last_plot, $
		x=x, y=y, $
		rfactor=rfactor, scale=scale, k=k, b=b,refinement_no=refinement_no
;+
; NAME:
;	READ_LSCALE
;
; PURPOSE:
;	Read log file from LSCALE and extract:
;		*The normalisation bin scale and b factors
;		*The fitted chebytchev normalization curve
;	All iterations are read and the last one is returned.
;	If the PLOT keyword is set, the inverse of the normalization 
;	curve will be plotted.
;
; CATEGORY:
;	Input, Laue processing
;
; CALLING SEQUENCE:
; 	READ_LSCALE, filename [, lambda, coeff, ranges, polorder, lmin, lmax, $
;			/plot, /oplot, /last_plot$
;			x=x, y=y, $
;			/rfactor, /scale, k=k, b=b, $
;			refinement_no=refinement_no]
;
; INPUTS:
;	FILENAME: Name of the Lauenorm log file.
;
; KEYWORDED PARAMETERS:
;	PLOT: If set then the bin factors and the fit are plotted
;		(in the current graphics window). NOTE: What is
;		plotted are the inverted scale factors and polynomial(s).
;		The plots are normalised to maximum 1.0
;	OPLOT: Same as PLOT but the curves are overplotted on the current
;		graph.
;	LAST_PLOT: Only in combination with PLOT or OPLOT. If set then
;		only the last curve is plotted.
;	RFACTOR: If set then the file is searched for all R-factors
;		(lines starting with  RFACT) and the rfactors are
;		printed.
;	SCALE: If set then the scale factors and the B-factors of the 
;		different film packs will be output to K and B. 
;	REFINEMENT_NO : Get the parameters from LSCALE refinement number : refinement_no.
;		Default = the Last refinement.
; OUTPUTS:
;	Optional:
;	LAMBDAREF: The reference wavelength which gets a scale of 1.
;	COEFF: The polynomial coefficients. DBLARR(ranges,max(polorder)+1)
;	RANGES: The number of lambda ranges.
;	LMIN,LMAX: The minimum and maximum lambda in the ranges.
;		FLTARR(ranges)
;	X,Y: Two vectors with x,y-coordinates for the last plot.
;		Supply as arguments for PLOT if you want to redo the plot.
;		They are DBLARR(number_of_ranges,plot_points). 
;		(plot_points=100)
;	K,B: The scale and B-factors of the different film packs (see SCALE).
;		k,b=FLTARR(film_packs) where (the number of) film_packs is
;		read from the file. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	The parsing of the logfile heavily depends on the output format
;	of Lauenorm: That it was correctly interpreted and that it hasn't
;	changed (or is different on another machine).
;	The routine will probably also have problems in some
;	other unusual(?) configurations. 
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, August 98.
;-

ON_ERROR, 2
plot_points=100
ok='' & dummy='' & ranges=0 & dummyint=0
IF (KEYWORD_SET(plot) or KEYWORD_SET(oplot)) THEN first_plot=1 $
ELSE first_plot=0

OPENR, file, filename, /GET_LUN

refinement=0
IF N_ELEMENTS(refinement_no) EQ 0 THEN refinement_no=10000

REPEAT BEGIN
; Find and read the number of film packs:
;  searchstr = 'NUMPACK'
  searchstr = 'NUMIMAGES'
  start= 0 & npacks=0
  length = STRLEN(searchstr)
  WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
    READF, file, dummy
  ENDWHILE
  IF EOF(file) THEN GOTO, finish
  ;IF EOF(file) THEN RETURN

  PRINT,''
  PRINT,'*******************************'
  PRINT,'Values for Refinement No : ',refinement+1
  PRINT,'*******************************'

  dummy=STRMID(dummy,length,STRLEN(dummy))
  READS,dummy,npacks
  print,'Number of Images to scale : ',npacks


;  searchstr = 'PACK_SCALE['
;  searchstr = 'IMAGE_SCALE['
;  start=0 & found=0 
;  length = STRLEN(searchstr)
;  REPEAT BEGIN
;    READF,file,dummy
;    IF (STRMID(dummy,start,length) EQ searchstr) THEN BEGIN
;      PRINT,dummy
;      dummy=STRMID(dummy,length,STRLEN(dummy))
;      READS,dummy,no_iter
;    ENDIF
;  ENDREP UNTIL (EOF(file) or (STRPOS(dummy,STRCOMPRESS('PACK_SCALE['+STRING(npacks)+']',/REMOVE_ALL)) NE -1))
;  ENDREP UNTIL (EOF(file) or (STRPOS(dummy,STRCOMPRESS('IMAGE_SCALE['+STRING(npacks)+']',/REMOVE_ALL)) NE -1))

;  searchstr = 'PACK_BFAC['
;  searchstr = 'IMAGE_BFAC['
;  start=0 & found=0 
;  length = STRLEN(searchstr)
;  REPEAT BEGIN
;    READF,file,dummy
;    IF (STRMID(dummy,start,length) EQ searchstr) THEN BEGIN
;      PRINT,dummy
;      dummy=STRMID(dummy,length,STRLEN(dummy))
;      READS,dummy,no_iter
;    ENDIF
;  ENDREP UNTIL (EOF(file) or (STRPOS(dummy,STRCOMPRESS('PACK_BFAC['+STRING(npacks)+']',/REMOVE_ALL)) NE -1))
;  ENDREP UNTIL (EOF(file) or (STRPOS(dummy,STRCOMPRESS('IMAGE_BFAC['+STRING(npacks)+']',/REMOVE_ALL)) NE -1))
  searchstr = 'NORM_LAMREF'
  start=0 & found=0 
  length = STRLEN(searchstr)
  REPEAT BEGIN
    READF,file,dummy
    IF (STRMID(dummy,start,length) EQ searchstr) THEN BEGIN
      found=1
      PRINT,dummy
;      dummy=STRMID(dummy,length,STRLEN(dummy))
;      READS,dummy,no_iter
    ENDIF
  ENDREP UNTIL (EOF(file) or (found EQ 1))



  searchstr = 'NORM_COEFF'
  start=0 & found=0 
  length = STRLEN(searchstr)
  ntot=0
  REPEAT BEGIN
    READF,file,dummy
    IF (STRMID(dummy,start,length) EQ searchstr) OR (found EQ 1) THEN BEGIN
;      PRINT,dummy
      IF found EQ 0 THEN BEGIN
      dummy2=strmid(dummy,length+2,strlen(dummy)-length-3)
      dummy2=STRTRIM(dummy2,2)
      dummy2=STR_SEP(dummy2,' ')
      nn=(SIZE(dummy2))(1)
       n_coeff=dummy2(0)
       coeff=FLTARR(n_coeff+1)
       PRINT,'Number of chebytchev coefficients : ',n_coeff+1
       lmin = dummy2(1) & lmax=dummy2(2)
       PRINT,'Lmin : ',lmin,' ,Lmax : ',lmax
       FOR ii=3,nn-1 DO coeff(ii-3)=dummy2(ii)
       ntot=nn-3
      ENDIF ELSE BEGIN
      dummy2=STRTRIM(dummy,2)
      dummy2=STR_SEP(dummy2,' ')
      nn=(SIZE(dummy2))(1)
       FOR ii=ntot,ntot+nn-1 DO coeff(ii)=dummy2(ii-ntot)
       ntot=nn+ntot
      ENDELSE
      found=1
    ENDIF
  ENDREP UNTIL (EOF(file) or (STRPOS(dummy,'&') NE (STRLEN(dummy)-1)))
  print,'Tchebytchev coefficients :',coeff

refinement=refinement+1

ENDREP UNTIL ((refinement EQ refinement_no) OR (EOF(file)))

finish: 
data = INDGEN(512)*(FLOAT(lmax)-FLOAT(lmin))/512. + FLOAT(lmin)

norm_curve = CHEBYCHEV(coeff,data)
PLOT,data,1/norm_curve,XTITLE='Wavelength [A]', YTITLE='NORM COEFF [AU]',CHARSIZE=1.5,TITLE='Normalization Curve : Refinement # :'+STRING(refinement)
FREE_LUN,file
RETURN
;MESSAGE, 'Error reading file.'

END

; Read and print R-factors:
IF KEYWORD_SET(rfactor) THEN BEGIN
  OPENR, file, filename, /GET_LUN
  searchstr = ' RFACT'
  start= 0
  length = STRLEN(searchstr)
  REPEAT BEGIN
    READF,file,dummy
    WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
      READF,file,dummy
    ENDWHILE
    IF NOT EOF(file) THEN PRINT, STRMID(dummy,0,15)
    IF (STRMID(dummy,start,length+1) EQ ' RFACT3') THEN PRINT, ''
  ENDREP UNTIL EOF(file)
  FREE_LUN, file
ENDIF

END
