function MAP_INTEG2, map, coordinates, radius, par, $
		average=average, index=index, sigma_cut=sigma_cut, $
		map_index=map_index, mask_index=mask_index,only_above=only_above, $
                force_integration=force_integration
;+
; NAME:
;	MAP_INTEG2
;
; PURPOSE:
;	Do box (or rather sphere) integration around a point in a density
;	map. 
;
; CATEGORY:
;	Crystallography (and...)
;
; CALLING SEQUENCE:
; 	integrated_density= MAP_INTEG2(map,coordinates,radius,par [, $
;	, average=averag, index=index, map_index=map_index, 
;       mask_index=mask_index, sigma_cut=sigma_cut, /only_above, /force_integration)
;
; INPUTS:
;	MAP: The map as read by READ_MAP (FLTARR with three dimensions).
;	COORDINATES: Orthogonal coordinates of the point around which to
;		integrate (as for the atoms in the PDB file and assuming
;		that SKEW=0 for the map file).
;	RADIUS: The search procedure (search3D) will be carried out in a box
;		with size=2*radius. 
;	PAR: The parameters returned by READ_MAP (in the keyword PAR).
;	MAP_INDEX: supplied as input to disallow integration of grid points already
;		used in a previous integration.
;	MASK_INDEX: supplied as input to restrict integration of grid points to a mask.
;
; KEYWORDED PARAMETERS:
;	SIGMA_CUT: Sigma cut for the SEARCH3D routine. All grid points in the map
;		within the sigma level at "coodinates" and the "sigma_cut"*par.sigma 
;		level will be considered. Default=3
;	ONLY_ABOVE: If set, only the part of the density which is ABOVE the threhold
;		"sigma_cut"*par.sigma will be considered. So that a peak "just" above 
;		the threshold will give almost a zero intensity, as if the peak was 
;		not there.		
;       FORCE_INTEGRATION: force integration even if no pixel is above
;               sigma_cut. Useful only when using a mask.
;
; OUTPUTS:
;	Returns the integrated density.
;	AVERAGE: Contains the averaged density within the sphere.
;	INDEX: Indices of the map (one-dimensional) of the points in
;		the maps that were included in the integration (i.e. the
;		points in the map within the specified distance from
;		COORDINATES).
;	MAP_INDEX: supplies the coordinates of the map that were counted in the integration. 
;		Can also be supplied as input to define a mask or to disallow integration of grid points already
;		used in a previous integration.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	MAP_INTEG can't handle  par.skew not equal 0 (see READ_MAP
;	    and the definition of the CCP4 map format), i.e.
;	    MAP_INTEG is relying on that the axis are as the PDB-standard
;	    (It is using PDBSCALE to get the transformation matrix
;	     between orthogonal and fractional coordinates. If these
;	     fractional coordinates doesn't correspond to the axis
;	     of the maps then everyhing is wrong.)
;	If the map is not containing the full sphere around the integration
;	     point, then it will integrate only those points available
;	     but not give any error message. (I think...)
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, June 2001.
;	Major changes, TU, Apr, 2000 (Selecting part of map to
;		speed up calculations).
;-

ON_ERROR, 2

IF NOT KEYWORD_SET(sigma_cut) THEN sigma_cut = 3
IF NOT KEYWORD_SET(map_index) THEN BEGIN
 map_index = map
 map_index(*) = 0
ENDIF

IF KEYWORD_SET(map_index) THEN BEGIN
 w_map_index = WHERE (map_index EQ 1, ct_map_index)
 PRINT,'Disallowing integration on: ',ct_map_index,' pixels'
ENDIF

 ;keep a copy of map_index
 map_index_saved = map_index

IF KEYWORD_SET(mask_index) THEN BEGIN
 w_mask = WHERE (mask_index EQ 1, ct_mask)
 PRINT,'Applying a mask on: ',ct_mask,' pixels'
 map_index = 1 - ((mask_index EQ 1) AND (map_index EQ 0))
 w_map_index = WHERE (map_index EQ 0, ct_map_index)
 IF ct_mask NE ct_map_index THEN PRINT,'Reducing mask to: ',ct_map_index,' pixels'
ENDIF


; Inital checks:
IF (par.skew ne 0) THEN $
  MESSAGE, 'Skew not equal zero. MAP_INTEG cant handle this!'

; Transformation from orthogonal coordinates to fractional: 
m0= PDBSCALE(par)
; Transformation from fractional coordinates to orthogonal: 
m= INVERT(m0)

; Some parameters from par that will be used:
;  (The axes can be shifted so we must use mapindex)
;  (After transformation, [0] will correspond to a-axis etc)
inverted_mapindex= [WHERE(par.mapindex eq 0), $
		    WHERE(par.mapindex eq 1), $
		    WHERE(par.mapindex eq 2)]
mapstart=  par.mapstart[inverted_mapindex]
mapsize=   par.mapsize[inverted_mapindex]
intervals= par.intervals[inverted_mapindex]

; Reduce the size of the map to the around the centre coordinate.
; This can increase the speed a lot!
; Calculate the fractional coordinates of all eight corners of a box
; with side 2*radius around "coordinates" to find minimum and maximum
; of the fractional coordinates => Part of map to use.
ss= radius ; Note that one extra step is added in all directions below.
minfc= FLOAT(mapstart+mapsize)/intervals
maxfc=  FLOAT(mapstart)/intervals 
FOR i=-1,1,2 DO BEGIN
  FOR j=-1,1,2 DO BEGIN
    FOR k=-1,1,2 DO BEGIN
      fc= m0 # (coordinates + [i*ss, j*ss, k*ss])
      minfc= [fc[0]<minfc[0], fc[1]<minfc[1], fc[2]<minfc[2]]
      maxfc= [fc[0]>maxfc[0], fc[1]>maxfc[1], fc[2]>maxfc[2]]
    ENDFOR
  ENDFOR
ENDFOR
; Get the array limits of the small selected part of the map:
i1= FLOOR(minfc*intervals) - [1,1,1] - mapstart
i2=  CEIL(maxfc*intervals) + [1,1,1] - mapstart
; Make sure the selected limits are inside the map:
i1= i1> [0,0,0]
i2= i2< (mapsize - [1,1,1])
; Note that the small selected part of the map, smap, keeps
; the column,row,section ordering of map (i.e. we use mapindex
; to transform back):
smap= map[i1[par.mapindex[0]]:i2[par.mapindex[0]], $
	  i1[par.mapindex[1]]:i2[par.mapindex[1]], $
	  i1[par.mapindex[2]]:i2[par.mapindex[2]]]
; We also now work with the following variables expressed in
; coloumns,row,sections ordering:
smapstart= (i1+mapstart)[par.mapindex]
smapsize=  (i2-i1 + [1,1,1])[par.mapindex]
sintervals= intervals[par.mapindex]

nc= smapsize(0) & nr=smapsize(1) & ns=smapsize(2)
ic= FLOAT(sintervals(0)) & ir= FLOAT(sintervals(1)) & is= FLOAT(sintervals(2))
; Array of the map size 
smappos= LINDGEN(long(nc)*nr*ns)
; xx[0,*] will step in 1/ic steps from mapstart(0)/ic to 
;        (mapstart(0)+mapsize(0))/ic and then start from 0,
; xx[1,*] will step in 1/ir steps from mapstart(1)/ir to
;        (mapstart(1)+mapsize(1))/ir and then start from 0
;        but each number is repeated nc times,
; xx[2,*] will step in 1/is steps from mapstart(2) to
;	(mapstart(2)+mapsize(2))/is but each number is 
;        repeated nc*nr times (so it will arrive at the "mapend" at the
;       last position and therefore not start again from 0)
xx=TRANSPOSE( $
	[[(smapstart(0)+((smappos mod (nc*nr)) mod nc))/ic],$
	[(smapstart(1)+FIX((smappos mod (nc*nr))/nc))/ir], $
	[(smapstart(2)+FIX(smappos/(nc*nr)))/is]])
; The axes can be shifted so we must use mapindex:
xx= xx(inverted_mapindex,*)
; Transform to orthogonal coordinates:
xx= m # TEMPORARY(xx)
; Now set x to the distance to the required position 

;xx are all coordinates of the small map
s_smap = SIZE(smap)
xpos=s_smap(1)/2
ypos=s_smap(2)/2
zpos=s_smap(3)/2

;sigma level of the central point
sig0=smap(xpos,ypos,zpos)/par.sigma
box=smap(xpos-1:xpos+1,ypos-1:ypos+1,zpos-1:zpos+1)
IF sig0 GT 0 THEN w=WHERE(box EQ MAX(box))
IF sig0 LT 0 THEN w=WHERE(box EQ MIN(box))
w=w(0)
new_c=WHERESUB3(w,box)
xpos=xpos+new_c(0)-1
ypos=ypos+new_c(1)-1
zpos=zpos+new_c(2)-1

PRINT,'Sigma level of the central point: ',sig0
sig0 = smap(xpos,ypos,zpos)/par.sigma 
PRINT,'Sigma level of the maximum central point: ',sig0
ok=0
IF sig0 GE sigma_cut THEN BEGIN
 index=SEARCH3D(smap,xpos,ypos,zpos,sigma_cut*par.sigma,par.max)
 ok=1
ENDIF
IF sig0 LE -sigma_cut THEN BEGIN
 index=SEARCH3D(smap,xpos,ypos,zpos,par.min,-sigma_cut*par.sigma)
 ok=1
ENDIF
IF KEYWORD_SET(force_integration) THEN BEGIN
 index=SEARCH3D(smap,xpos,ypos,zpos,par.min,par.max) ; select everything
 IF KEYWORD_SET(mask_index) THEN ok=1
ENDIF

;VIEW_MAP,BYTSCL(smap),/NO_SLICER

IF ok EQ 1 THEN BEGIN
 index_z = index/(s_smap(1)*s_smap(2))
 index_y = (index - (index_z*s_smap(1)*s_smap(2)))/s_smap(1)
 index_x = (index - (index_z*s_smap(1)*s_smap(2))) - (index_y*s_smap(1))

 IF NOT KEYWORD_SET(force_integration) THEN BEGIN
 IF (MIN(index_x) EQ 0) OR (MIN(index_y) EQ 0) OR (MIN(index_z) EQ 0) THEN PRINT,'Increase radius !'
 IF (MAX(index_x) EQ (s_smap(1)-1)) OR (MAX(index_y) EQ (s_smap(2)-1)) OR (MAX(index_z) EQ (s_smap(3)-1)) THEN PRINT,'Increase radius !'
 ENDIF
 
 ;IF (ct eq 0) THEN MESSAGE, $
 ;	'No grid points of the map within specified distance.'
 ;integ= TOTAL(smap(index))

;below, only the level above the threshold is taken into account
IF KEYWORD_SET(only_above) THEN BEGIN
 IF sig0 GE sigma_cut THEN BEGIN
  integ= TOTAL((1-map_index[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z])*(map[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z]-sigma_cut*par.sigma))
 ENDIF
 IF sig0 LE -sigma_cut THEN BEGIN
  integ= TOTAL((1-map_index[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z])*(map[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z]+sigma_cut*par.sigma))
 ENDIF
ENDIF ELSE BEGIN

 integ= TOTAL((1-map_index[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z])*map[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z])
ENDELSE

; map_index[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z] = 1

;modif introduced on 3 august 2004
 map_index_saved[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z] = ((1-map_index[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z])+(map_index_saved[i1[par.mapindex[0]]+index_x, i1[par.mapindex[1]]+index_y,i1[par.mapindex[2]]+index_z]))

 map_index = map_index_saved
;end modif

 average= integ/(SIZE(index))(1)
 ;PRINT,'Integrated intensity = ',integ
ENDIF ELSE BEGIN
 integ=0
 average=0
ENDELSE

RETURN, integ

END

