pro ESTIMATE_EXCL,par

;+
; NAME:
;	ESTIMATE_EXCL
;
; PURPOSE:
;	Estimates exclusion parameters (a B-factor and a Threshold factor) which are subsequently used to
;	apply the RDBW (or WDRL) techniques.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	ESTIMATE_EXCL,par
;
; INPUTS:
;	par : a structure containing :
;		par.journalfile --> The journal file (log file with all the infos)
;		par.idl_outfile --> IDL file containing the results
;		par.noex_file --> The .PROW file containing a Laue pattern integrated with hard limits
;		par.b_start --> The initial b_factor for testing exclusion parameters (will be tuned anyway)
;		par.b_end --> The endig b_factor for testing exclusion parameters (will be tuned anyway)
;		par.b_nstep  --> The number of steps for testing B-factor exclusion
;		par.th_start  --> The initial threshold  for testing exclusion parameters (will be tuned anyway)
;		par.th_end  --> The ending threshold for testing exclusion parameters (will be tuned anyway)
;		par.th_nstep  --> The number of steps for testing threshold exclusion 
;		par.isigi_limit  --> The initial i/sigi for testing exclusion parameters (will be tuned anyway)
;		par.uisigi_limit  --> The upper limit for I/SIGI (not used)
;		par.lisigi_limit  --> The lower limit for I/SIGI (not used)
;		par.lscale_file  --> An lscale output file containing the reference normalization curve
;		par.use_mono  --> Set if monochromatic data are to be used
;		par.mono_file  --> If use_mono is set, filename of the monochromatic data
;		par.laue_group  --> The Laue group number
;
; KEYWORDED PARAMETERS:
;	None
;
; OUTPUTS:
;	A file is written with the results.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;
;	
; MODIFICATION HISTORY:
;	Written by Dominique Bourgeois, December 1999.
;
;-

journalfile=par.journalfile
idl_outfile=par.idl_outfile
noex_file=par.noex_file
b_start=par.b_start
b_end=par.b_end
b_nstep=par.b_nstep
th_start=par.th_start
th_end=par.th_end
th_nstep=par.th_nstep
isigi_limit=par.isigi_limit
uisigi_limit=par.uisigi_limit
lisigi_limit=par.lisigi_limit
lscale_file=par.lscale_file
use_mono=par.use_mono
mono_file=par.mono_file
laue_group=par.laue_group

IF !JOURNAL NE 0 THEN JOURNAL

dum = FINDFILE(journalfile,count=count)
IF count NE 0 THEN BEGIN
 SPAWN,'/bin/rm -f '+journalfile 
ENDIF

JOURNAL,journalfile

;Recapitulation of parameters:
PRINT,'Starting B value : ',b_start
PRINT,'Ending B value : ',b_end
PRINT,'Number of steps for B scan : ',b_nstep

PRINT,'Starting THRESHOLD value : ',th_start
PRINT,'Ending THRESHOLD value : ',th_end
PRINT,'Number of steps for THRESHOLD scan : ',th_nstep

PRINT,'Initial I/SigI limit for distinction between spots that are allowed to be excluded : ',isigi_limit

;PRINT,'Initial Upper I/SigI limit for evaluation : ',uisigi_limit
;PRINT,'Lower I/SigI limit for evaluation : ',lisigi_limit

PRINT,'LSCALE output file containing normalization : ',lscale_file

IF use_mono EQ 1 THEN BEGIN
 PRINT,'A monochromatic data set will be used !'
 PRINT,'Its name is : ', mono_file
 PRINT,'Laue symmetry is : ',laue_group
ENDIF ELSE $
 PRINT,'No monochromatic data set will be used !'
;End recapitulation of parameters:

;First read in the normalization curve
READ_LSCALE,lscale_file,bins,lambda,lambdapol,ranges,polorder,lmin,lmax

;Is it ok ?
ok=''

PRINT,''
PRINT,'*********************************************'
PRINT,'STARTING THE CALCULATION (ON SINGLES ONLY)...'
PRINT,'*********************************************'
PRINT,''

;Set the window correctly
!P.MULTI = [0,1,2,0,0]

;Open the prow file without exclusion.
RESTORE,noex_file
;we do the calculation only on singles with meaningful sigmas.
all=integ(WHERE((integ.mul EQ 1) AND (integ.isigp(0) GT 0)))

r_array=FLTARR(b_nstep,th_nstep)
c_array=FLTARR(b_nstep,th_nstep)

kept_corr=0.0
kept_rfactor=10.^10
kept_b = 0.
kept_th = 0.
kept_b2 = 0.
kept_th2 = 0.
kept_n = 0.
kept_number_selected = 0.0

IF b_nstep LT 2 THEN MESSAGE,'Need more than one step for B factor evaluation !'
IF th_nstep LT 2 THEN MESSAGE,'Need more than one step for threshold evaluation !'

;We need to estimate the I/SigI limit so that the right percentage of reflexions will be selected.
;The method is the following : we select the wavelength range where the signal is highest and at least
;5% of the singles are selected. We then look at how many of the selected reflexions have a I/sigI greater
;than 3. Then we sweep through the whole nomalization curve, and, considering the decrease in the X-ray spectrum
;intensity we assess how many of these reflexions would have the sufficient I/sigI, would they have been recorded
;with the decrease incident intensity. If the reduction in intensity is alpha, the decrease in I is alpha and SigI is 
;modified by sigI^2 --> sigI^2+gI(alpha-1)  so only the reflections with I > 1/alphax3.0xSQRT(SigI^2+gI(alpha-1)) are acceptable.

n_points=100L
l=FLOAT(lmin)+indgen(n_points)/FLOAT(n_points)*(FLOAT(lmax)-FLOAT(lmin))

;the factor 1/l^2 comes from the Lorentz factor. I(lambda) = Inorm(lambda)/lambda^2*f(resolution_only).
max_lambdacurve = MAX(1/l^2*1/CHEBYCHEV(lambdapol,l,lmin=FLOAT(lmin),lmax=FLOAT(lmax)))
max_lambda_index=WHERE((1/l^2*1/CHEBYCHEV(lambdapol,l,lmin=FLOAT(lmin),lmax=FLOAT(lmax))) EQ max_lambdacurve,ct_max_lambda)
max_lambda=l(max_lambda_index(0))

w_pos=WHERE(1/CHEBYCHEV(lambdapol,l,lmin=FLOAT(lmin),lmax=FLOAT(lmax)) GT 0,ct_pos)
PRINT,'Maximum at wavelength [A] : ',max_lambda
range=(FLOAT(lmax)-FLOAT(lmin))/FLOAT(ct_pos)
best_range=[max_lambda-range/2.,max_lambda+range/2.]

;increase the range until 5% of the reflexions are included
enlarge=1.
REPEAT BEGIN
 best_range=[max_lambda-enlarge*range/2.,max_lambda+enlarge*range/2.]
 enlarge=enlarge*1.2
 w_best=WHERE(all.l LE best_range(1) AND all.l GE best_range(0),n_best)
ENDREP UNTIL n_best GT 0.05*(SIZE(all))(1)
enlarge=enlarge/1.2

PRINT,'Best range is : ',best_range
PRINT,'Number of reflections taken into account : ',n_best

best_ref=all(w_best)

;in the best range, estimate how many reflections are indeed measurable
isigi_cutoff=3.0

;a correction must be applied if the lambda curve already decays over this best range.
;we must estimate what the i/sigi would be if all the reflection in the best range would be measured
;at the top wavelength
;first the integrated intensities of these reflections would be :
ratio=CHEBYCHEV(lambdapol,best_ref.l,lmin=FLOAT(lmin),lmax=FLOAT(lmax),lnorm=max_lambda)
best_ii=best_ref.intp(0)*ratio
best_sig=SQRT(0>(best_ref.isigp(0)^2+best_ref.intp(0)*par.gain*1.5418*(ratio/max_lambda-1/best_ref.l)))
w_high_isigi = WHERE(best_ii GT isigi_cutoff*ABS(best_sig),n_high_isigi)
fraction_measurable = n_high_isigi/FLOAT(n_best)

PRINT,''
PRINT,'************************************************************************'
PRINT,'The fraction of reflection with I/SIGI > 3 in the best range should be [%] : ',100*fraction_measurable
PRINT,'************************************************************************'
PRINT,''

;Now run through all the wavelength range
l_ok=l(w_pos)
loc_lambdacurve = 1/l_ok^2*1/CHEBYCHEV(lambdapol,l_ok,lmin=FLOAT(lmin),lmax=FLOAT(lmax))
tot_n=0L
FOR loc_i=0,ct_pos-1 DO BEGIN
 loc_ratio=loc_lambdacurve(loc_i)/max_lambdacurve
; w_loc_high_isigi = WHERE(best_ref.intp(0) GT (1/loc_ratio)*isigi_cutoff*SQRT(0>(best_ref.isigp(0)^2+best_ref.intp(0)*par.gain*1.5418*(loc_ratio/l_ok(loc_i)-1/best_ref.l))),n_loc_high_isigi)
 w_loc_high_isigi = WHERE(best_ii GT (1/loc_ratio)*isigi_cutoff*SQRT(0>(best_sig^2+best_ii*par.gain*1.5418*(loc_ratio/l_ok(loc_i)-1/max_lambda))),n_loc_high_isigi)
 tot_n=tot_n+n_loc_high_isigi
ENDFOR

;Multiply the obtained percentage by 1.25 for safety margin and to account for variability in Fmono.
percentage=100.0<(((fraction_measurable)*2<100.0)*1.25*100*tot_n/FLOAT(ct_pos*n_high_isigi))
;percentage=100.0<(fraction_measurable*1.25*100*tot_n/FLOAT(ct_pos*n_high_isigi))
PRINT,''
PRINT,'************************************************************************'
PRINT,'The percentage of selected reflexions should be around [%] : ',percentage
PRINT,'************************************************************************'
PRINT,''

;***** OLD TECHNIQUE : NOT USED ANYMORE ! ********
;We need to estimate the I/SigI limit so that the right percentage of reflexions will be selected.
;First estimate this percentage as being proportional to the surface area of the normalization curve.
;l=FLOAT(lmin)+indgen(100)/100.*(FLOAT(lmax)-FLOAT(lmin))
;max_lambdacurve = MAX(1/CHEBYCHEV(lambdapol,l,lmin=FLOAT(lmin),lmax=FLOAT(lmax)))
;w_pos=WHERE(1/CHEBYCHEV(lambdapol,l,lmin=FLOAT(lmin),lmax=FLOAT(lmax)) GT 0,ct_pos)
;square_area=max_lambdacurve*(FLOAT(lmax)-FLOAT(lmin))*ct_pos/100.
;lambdacurve_area=TOTAL(1/CHEBYCHEV(lambdapol,l(w_pos),lmin=FLOAT(lmin),lmax=FLOAT(lmax)))*(FLOAT(lmax)-FLOAT(lmin))/100.0
;percentage=1.5*100*lambdacurve_area/square_area
;PRINT,'The percentage of selected reflexions should be around [%] : ',percentage
;***************************************************

;Estimation of the optimal i/sigi cutoff
ct_all = FLOAT((SIZE(all))(1))
isigi_limit=0
REPEAT BEGIN
 ;w1_all=WHERE((all.intp(0)/all.isigp(0) GE isigi_limit) AND (all.intp(0)/all.isigp(0) LE uisigi_limit) ,ct1_all)
 ;w2_all=WHERE((all.intp(0)/all.isigp(0) LT isigi_limit) AND (all.intp(0)/all.isigp(0) GE lisigi_limit) ,ct2_all)
 w1_all=WHERE((all.intp(0)/all.isigp(0) GE isigi_limit) ,ct1_all)
 isigi_limit=isigi_limit+1
ENDREP UNTIL (100*ct1_all/ct_all LT percentage)
isigi_limit=0>(isigi_limit-2)

REPEAT BEGIN
;w1_all=WHERE((all.intp(0)/all.isigp(0) GE isigi_limit) AND (all.intp(0)/all.isigp(0) LE uisigi_limit) ,ct1_all)
;w2_all=WHERE((all.intp(0)/all.isigp(0) LT isigi_limit) AND (all.intp(0)/all.isigp(0) GE lisigi_limit) ,ct2_all)
 w1_all=WHERE((all.intp(0)/all.isigp(0) GE isigi_limit)  ,ct1_all)
 isigi_limit=isigi_limit+0.1
ENDREP UNTIL (100*ct1_all/ct_all LT percentage)
isigi_limit=0>(isigi_limit-0.1)

PRINT,''
PRINT,'***************************************'
PRINT,'Selected I/SigI cutoff is : ',isigi_limit
PRINT,'The fraction of the reflexions with I/SigI > ',isigi_limit,' is : ',100*ct1_all/FLOAT((SIZE(all))(1)),' %'
PRINT,'***************************************'
PRINT,''

;uisigi_limit=5*isigi_limit
uisigi_limit=MAX(all.intp(0)/all.isigp(0))
lisigi_limit=MIN(all.intp(0)/all.isigp(0))
;PRINT,'Selected upper I/SigI cutoff is : ',uisigi_limit
;PRINT,'Selected lower I/SigI cutoff is : ',lisigi_limit

w1_all=WHERE((all.intp(0)/all.isigp(0) GE isigi_limit) AND (all.intp(0)/all.isigp(0) LE uisigi_limit) ,ct1_all)
w2_all=WHERE((all.intp(0)/all.isigp(0) LT isigi_limit) AND (all.intp(0)/all.isigp(0) GE lisigi_limit) ,ct2_all)


IF ct1_all LT 2 THEN MESSAGE,'Too few strong spots !'
all_strong_isigi=all(w1_all).intp(0)/all(w1_all).isigp(0)

IF ct2_all LT 2 THEN MESSAGE,'Too few weak spots !'
all_weak_isigi=all(w2_all).intp(0)/all(w2_all).isigp(0)

;produce an histogram of the I/SIGI of the all the strong reflexions
min_val=isigi_limit
max_val=MAX(all_strong_isigi)
binsize=(max_val-min_val)/100.0

h_all_strong=HISTOGRAM(all_strong_isigi,binsize=binsize,min=min_val,max=max_val)
x_all_strong=min_val+INDGEN((SIZE(h_all_strong))(1))*binsize
n_all_strong=TOTAL(h_all_strong)

;produce an histogram of the I/SIGI of the all the weak reflexions
min_val2=MIN(all_weak_isigi)
max_val2=isigi_limit
binsize2=(max_val2-min_val2)/100.0


h_all_weak=HISTOGRAM(all_weak_isigi,binsize=binsize2,min=min_val2,max=max_val2)
x_all_weak=min_val2+INDGEN((SIZE(h_all_weak))(1))*binsize2
n_all_weak=TOTAL(h_all_weak)


IF use_mono EQ 1 THEN BEGIN
 ; First read in the monochromatic data
 PRINT,'Reading in monochromatic data ...'
 mono=READ_MTZ(mono_file)
 w_ok=WHERE(mono.entry(0) GT 0,ct_ok)
 IF ct_ok GT 0 THEN mono=mono(w_ok) ELSE MESSAGE,'All monochromatic SF are zero ! ...'
 ;try to normalize the mono SF's
 mono.entry(0)=mono.entry(0)/MEDIAN(mono.entry(0))
 sig_mono=STDEV(mono.entry(0),mean_mono)
 PRINT,'Standard deviation and mean of normalized mono data : ', sig_mono,mean_mono
 ;Then combine with monochromatic F's
 all2=GET_MONOF(all,mono,mult=mult,laue_group=laue_group)
ENDIF

;Check the start values for b_factor and threshold
PRINT,'Adjusting starting B and THRESHOLD values ...'
check_start:
b=b_start & th=th_start
IF use_mono EQ 1 THEN BEGIN
 IF N_ELEMENTS(norm_coeff) EQ 0 THEN $
  strong=PRED_II(all2, lambdapol, b, scale=1,threshold=th,weak=weak,mult=mult,/USE_RDBW,/LSCALE,USE_MONO=use_mono,lmin=par.lmin,lmax=par.lmax,/GET_NORM_COEFF,norm_coeff=norm_coeff,/GET_CORRELATION) $
 ELSE $
  strong=PRED_II(all2, lambdapol, b, scale=1,threshold=th,weak=weak,mult=mult,/USE_RDBW,/LSCALE,USE_MONO=use_mono,lmin=par.lmin,lmax=par.lmax,set_norm_coeff=norm_coeff,/GET_CORRELATION)
ENDIF ELSE BEGIN
 IF N_ELEMENTS(norm_coeff) EQ 0 THEN $
  strong=PRED_II(all, lambdapol, b, scale=1,threshold=th,weak=weak,/USE_RDBW,/LSCALE,USE_MONO=0,lmin=par.lmin,lmax=par.lmax,/GET_NORM_COEFF,norm_coeff=norm_coeff,/GET_CORRELATION) $
 ELSE $
 strong=PRED_II(all, lambdapol, b, scale=1,threshold=th,weak=weak,/USE_RDBW,/LSCALE,USE_MONO=0,lmin=par.lmin,lmax=par.lmax,set_norm_coeff=norm_coeff,/GET_CORRELATION)
ENDELSE

IF (SIZE(weak))(0) EQ 0 THEN n_weak=0 ELSE n_weak=(SIZE(weak))(1)
IF (SIZE(strong))(0) EQ 0 THEN n_strong=0 ELSE n_strong=(SIZE(strong))(1)

IF 100*n_strong/FLOAT(n_weak+n_strong) LT 95L<(2*percentage) THEN BEGIN
 b_start = b_start/2.0
 th_start = th_start/2.0
 b_end = b_end/2.0
 th_end = th_end/2.0
 PRINT,'Starting (and ending) values decreased to [b_start,th_start,b_end,th_end] : ',b_start,th_start,b_end,th_end
 GOTO,check_start
ENDIF

;Check the end values for b_factor and threshold
PRINT,'Adjusting ending B and THRESHOLD values ...'
check_end:
b=b_end & th=th_end
IF use_mono EQ 1 THEN BEGIN
 strong=PRED_II(all2, lambdapol, b, scale=1,threshold=th,weak=weak,mult=mult,/USE_RDBW,/LSCALE,USE_MONO=use_mono,lmin=par.lmin,lmax=par.lmax,set_norm_coeff=norm_coeff)
ENDIF ELSE BEGIN
 strong=PRED_II(all, lambdapol, b, scale=1,threshold=th,weak=weak,/USE_RDBW,/LSCALE,USE_MONO=0,lmin=par.lmin,lmax=par.lmax,set_norm_coeff=norm_coeff)
ENDELSE

IF (SIZE(weak))(0) EQ 0 THEN n_weak=0 ELSE n_weak=(SIZE(weak))(1)
IF (SIZE(strong))(0) EQ 0 THEN n_strong=0 ELSE n_strong=(SIZE(strong))(1)

IF 100*n_weak/FLOAT(n_weak+n_strong) LT (100.-percentage/2.) THEN BEGIN
 b_end = b_end*2.0
 th_end = th_end*2.0
 PRINT,'Ending values increased to [b_end,th_end] : ',b_end,th_end
 GOTO,check_end
ENDIF

PRINT,''
PRINT,'***************************************'
PRINT,'Selected B_factor range : ',b_start,b_end
PRINT,'Selected th_factor range : ',th_start,th_end
PRINT,'***************************************'
PRINT,''

FOR i=0,b_nstep-1 DO BEGIN
 FOR j=0,th_nstep-1 DO BEGIN
  b_step = (b_end-b_start)/(FLOAT(b_nstep)-1)
  b=b_start+i*b_step
  th_step = (th_end-th_start)/(FLOAT(th_nstep)-1)
  th=th_start+j*th_step

  PRINT,''
  PRINT,'******************************************'
  PRINT,'B factor : ',b
  PRINT,'Threshold : ',th
  PRINT,'******************************************'

  IF use_mono EQ 1 THEN BEGIN
   strong=PRED_II(all2, lambdapol, b, scale=1,threshold=th,weak=weak,mult=mult,/USE_RDBW,/LSCALE,USE_MONO=use_mono,lmin=par.lmin,lmax=par.lmax,ii_strong=ii_strong,ii_weak=ii_weak,set_norm_coeff=norm_coeff)
  ENDIF ELSE BEGIN
   strong=PRED_II(all, lambdapol, b, scale=1,threshold=th,weak=weak,/USE_RDBW,/LSCALE,USE_MONO=0,lmin=par.lmin,lmax=par.lmax,ii_strong=ii_strong,set_norm_coeff=norm_coeff)
  ENDELSE
  IF (SIZE(weak))(0) EQ 0 THEN BEGIN
   PRINT,'There are no predicted weak reflexions ...'
   PRINT,'Going to next step ...'
   GOTO,jump
  ENDIF

  IF (SIZE(strong))(0) EQ 0 THEN BEGIN
   PRINT,'There are no predicted strong reflexions ...'
   PRINT,'Going to next step ...'
   GOTO,jump
  ENDIF

  number_selected=100*(SIZE(strong))(1)/FLOAT((SIZE(all))(1))
  strong_isigi=strong.intp(0)/strong.isigp(0)
  weak_isigi=weak.intp(0)/weak.isigp(0)
  ;correlate strong to all for those reflexions above I/SigI limit
  w1=WHERE((strong_isigi GE isigi_limit) AND (strong_isigi LE uisigi_limit) ,ct1)
  IF ct1 GT 1 THEN BEGIN
   strong_isigi=strong_isigi(w1)
   ;produce an histogram of the I/SIGI of the strong reflexions
   h_strong=HISTOGRAM(strong_isigi,binsize=binsize,min=min_val,max=max_val+0.001)
   ;Do a plot
   PLOT_IO,x_all_strong,h_all_strong,xrange=[min_val,max_val],xstyle=1,yrange=[0.1,MAX(h_all_strong)],linestyle=0,Title='Histogram of STRONG I/SigI',xtitle='I/SIGI',ytitle='# of reflexions',charsize=1.5
   oplot,x_all_strong,h_strong,linestyle=1
   ;Get the first correlation
   rfactor1=RFACTOR(h_strong,h_all_strong,TYPE=2)
   corr1=CORRELATE(h_strong,h_all_strong)
   PRINT,'Percentage of recovered strong spots [%] : ',100*ct1/ct1_all
 ;  PRINT,'Number of predicted strong spots : ',ct1
 ;  PRINT,'Number of observed strong spots : ',ct1_all
   PRINT,'Rfactor between strong spots for b, th = ',b,th,' is :',rfactor1
 ;  PRINT,'Correlation between strong spots for b, th = ',b,th,' is :',corr1
   sigma_strong = STDEV(strong_isigi,mean_strong)
 ;  PRINT,'RMS deviation and mean value for strong reflexions : ',sigma_strong,mean_strong

  ENDIF ELSE BEGIN
   PRINT,'Too few strong spots, skipping b_factor and threshold : ',b,th
   rfactor1=10.^10
   corr1=0
  ENDELSE

  ;correlate weak to all for those reflexions below I/SigI limit
  w2=WHERE((weak_isigi LT isigi_limit) AND (weak_isigi GE lisigi_limit),ct2)
  IF ct2 GT 1 THEN BEGIN
   weak_isigi=weak_isigi(w2)
   ;produce an histogram of the I/SIGI of the weak reflexions
   h_weak=HISTOGRAM(weak_isigi,binsize=binsize2,min=min_val2,max=max_val2)
   ;Do a plot
   PLOT_IO,x_all_weak,h_all_weak,xrange=[min_val2,max_val2],xstyle=1,yrange=[0.1,MAX(h_all_weak)],linestyle=0,Title='Histogram of WEAK I/SigI',xtitle='I/SIGI',ytitle='# of reflexions',charsize=1.5
   oplot,x_all_weak,h_weak,linestyle=1
   ;Get the second correlation
   rfactor2=RFACTOR(h_weak,h_all_weak,TYPE=2)
   corr2=CORRELATE(h_weak,h_all_weak)
   PRINT,'Percentage of recovered weak spots [%] : ',100*ct2/ct2_all
  ; PRINT,'Number of predicted weak spots : ',ct2
  ; PRINT,'Number of observed weak spots : ',ct2_all
   PRINT,'Rfactor between weak spots for b, th = ',b,th,' is :',rfactor2
  ; PRINT,'Correlation between weak spots for b, th = ',b,th,' is :',corr2
   sigma_weak = STDEV(weak_isigi,mean_weak)
  ; PRINT,'RMS deviation and mean value for weak reflexions : ',sigma_weak,mean_weak


  ENDIF ELSE BEGIN
   PRINT,'Too few weak spots, skipping b_factor and threshold : ',b,th
   rfactor2=10.^10
   corr2=0
  ENDELSE

  main_rfactor=ABS(rfactor1)>ABS(rfactor2)
  main_n=SQRT(LONG(ct1)*LONG(ct2))
  main_corr=corr1*corr2
  r_array(i,j)=main_rfactor
  c_array(i,j)=main_corr
  PRINT,'Main R factor for b, th = ',b,th,' is :',main_rfactor
 ; PRINT,'Main correlation factor for b, th = ',b,th,' is :',main_corr
  PRINT,'Main correlation n1*n2 for b, th = ',b,th,' is :',main_n

  IF main_corr GT kept_corr THEN BEGIN
  ; PRINT,'Correlation better than before :  for b, th = ',b,th,'
   kept_corr = main_corr
   kept_b=b
   kept_th=th
  ENDIF

  IF main_rfactor LT kept_rfactor THEN BEGIN
;  IF main_n GT kept_n THEN BEGIN
   PRINT,'Rfactor better than before :  for b, th = ',b,th,'
;   PRINT,'Number of common reflections higher than before :  for b, th = ',b,th,'
   kept_n=main_n
   kept_number_selected=number_selected
   kept_rfactor = main_rfactor
   kept_b2=b
   kept_th2=th
  ENDIF

  PRINT,'***********************************************'
  PRINT,'Best Rfactor at this point is : ',kept_rfactor
  PRINT,'Best Bfactor and threshold at this point is : ',kept_b2,kept_th2
  PRINT,'Best number of common reflections at this point is : ',kept_n
  PRINT,'Best percentage of selected reflections at this point is : ',kept_number_selected
  PRINT,'***********************************************'

  jump:

 ENDFOR
ENDFOR


PRINT,''
PRINT,'&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&'
PRINT,'Optimal B-factor by rfactor : ',kept_b2
PRINT,'Optimal threshold by rfactor : ',kept_th2
PRINT,'Optimal rfactor : ',kept_rfactor
PRINT,'Best percentage of selected reflections is : ',kept_number_selected
PRINT,'&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&'


;End journaling session
JOURNAL

!P.MULTI = [0,1,1,0,0]

b_step = (b_end-b_start)/(FLOAT(b_nstep)-1)
b_x=b_start+INDGEN(b_nstep)*b_step
th_step = (th_end-th_start)/(FLOAT(th_nstep)-1)
th_y=th_start+INDGEN(th_nstep)*th_step


SURFACE,r_array,b_x,th_y,XTITLE='B-factor',YTITLE='Threshold',TITLE='R array results',LINESTYLE=0,CHARSIZE=2.5

PRINT,'Saving results ...'
excl_result={$
	correlation_b:kept_b, $
	correlation_th:kept_th, $
	correlation_opt:kept_corr, $
	number_opt:kept_number_selected, $
	rfactor_b:kept_b2, $
	rfactor_th:kept_th2, $
	rfactor_opt:kept_rfactor, $
	r_array:r_array, $
	b_x:b_x, $
	th_y:th_y $
	}
SAVE,excl_result,file=idl_outfile

escape:

END

