function MERGE_REFSPOTS,new_ref_spots,old_ref_spots,par,MAX_OSC_DISTANCE=MAX_OSC_DISTANCE,MAX_REF_SPOTS=MAX_REF_SPOTS,SHOW=SHOW
;+
; NAME:
;	MERGE_REFSPOTS
;
; PURPOSE:
;
;	Selects out reference spots in order to optimize the distribution of their
;	positions on the image.
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	ref_spots = MERGE_REFSPOTS(new_ref_spots,old_ref_spots,par,MAX_OSC_DISTANCE=MAX_OSC_DISTANCE,$
;						MAX_REF_SPOTS=MAX_REF_SPOTS,/SHOW)
;
; INPUTS:
;	new_ref_spots : a set a potential reference spots, usually corresponding to the new potential reference spots
;			extracted from the image under process
;	old_ref_spots : a set a potential reference spots, usually corresponding to the old potential reference spots
;			extracted from the images previously processed.
;	max_osc_distance : [optional] the maximum oscillation distance [in degree] for which reference spots can be selected.
;			default : 0
;	max_ref_spots : [optional] the maximum number of reference spots that can be selected. default : (SIZE(new_ref_spots))(1)
;	par : the parameters
;	show : [optional] to show the distribution of the reference spots.
;
; OUTPUTS:
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;
; PROCEDURE:
;	Quite straightforward
;
; MODIFICATION HISTORY:
;	D.Bourgeois, July 1997.
;-

;set the defaults
IF N_ELEMENTS(max_osc_distance) EQ 0 THEN max_osc_distance=0
IF N_ELEMENTS(max_ref_spots) EQ 0 THEN max_ref_spots=(SIZE(new_ref_spots))(1)

;get general image parameters
 ;position of the center of the image [raster]
 centerX = par.cenx-1
 centerY = par.ceny-1

 ;y scale
 scaleY = par.yscale

 ;size of "image"
 imageX = par.rastx
 imageY = par.rasty

 ;original pixel size [micrometers]
 pix_size = par.raster

;how many new spots
n_new_ref=(SIZE(new_ref_spots))(1)

;select the spots with acceptable oscillation distance :
IF (SIZE(old_ref_spots))(0) EQ 1 THEN BEGIN
 w_old_ref = WHERE(ABS(old_ref_spots.phi - 0.5*(par.osc_start+par.osc_end)) LE max_osc_distance,n_old_ref)
 IF KEYWORD_SET(show) THEN PRINT,'Number of profiles with acceptable oscillation distance : ',n_old_ref+n_new_ref
ENDIF ELSE n_old_ref=0

IF n_old_ref GT 0 THEN old=old_ref_spots(w_old_ref)

;merge the two sets
spots = REPLICATE(new_ref_spots(0),n_new_ref+n_old_ref)
spots(0:n_new_ref-1)=new_ref_spots
IF n_old_ref GT 0 THEN BEGIN
 FOR loc_i = 0, n_tags(spots(0))-1 DO BEGIN
  result=EXECUTE('spots(n_new_ref:n_new_ref+n_old_ref-1).'+(tag_names(spots(0)))(loc_i)+' = old.'+(tag_names(spots(0)))(loc_i))
 ENDFOR
ENDIF

loc_max_ref_spots = max_ref_spots < (n_new_ref+n_old_ref)

;do some initialization
number_sectors = par.n_sectors
start_indices=LONARR(number_sectors)
number_indices=LONARR(number_sectors)
current_indices=LONARR(number_sectors)
good_indices = LONARR(loc_max_ref_spots)
n_ref=0

;separate into the various sectors
;sort along sector
s=SORT(spots.sector)
spots=spots(s)
;in each sector, sort in I/SigI
FOR i=0,number_sectors-1 DO BEGIN
 w_sector=WHERE(spots.sector EQ i,ct_sector)
 IF ct_sector GT 0 THEN BEGIN
  number_indices(i)=ct_sector
  start_indices(i)=w_sector(0)
  spots(w_sector) = spots(w_sector(REVERSE(SORT(spots(w_sector).intp(0)/spots(w_sector).bc_coeff(2)))))
 ENDIF
ENDFOR

;now run the selection
REPEAT BEGIN
 FOR i=0,number_sectors-1 DO BEGIN
  IF (current_indices(i) LT number_indices(i)) AND (n_ref LT loc_max_ref_spots) THEN BEGIN
   good_indices(n_ref)=start_indices(i)+current_indices(i)
   n_ref=n_ref+1
   current_indices(i)=current_indices(i)+1
  ENDIF
 ENDFOR
ENDREP UNTIL n_ref EQ loc_max_ref_spots

ref_spots = spots(good_indices)

PRINT,'Total number of selected reference profiles : ',(SIZE(ref_spots))(1)

;get the regions and print coarse info
 wait_sec=BYTARR(number_sectors)
 region_levels=FLTARR(number_sectors)
 fudge_factor=5
 max_region_level=15./100. & min_region_level=0.1/100.

 FOR i=0,number_sectors-1 DO BEGIN
  w_sector=WHERE(ref_spots.sector EQ i,ct_sector)

  IF ct_sector GT 0 THEN BEGIN

   ;******
   ;select optimal restrained region in the sector
   ;******
   tot_spot=LONARR(par.boxsize.x,par.boxsize.y)
   max_bc=0
   ;get the average spot

   FOR loc_i=0,(SIZE(ref_spots(w_sector)))(1)-1 DO BEGIN
    tot_spot=tot_spot + ref_spots(w_sector(loc_i)).sp - ref_spots(w_sector(loc_i)).bc
    max_bc=max_bc+MAX(ref_spots(w_sector(loc_i)).bc)
   ENDFOR

   max_tot_spot=MAX(tot_spot(par.boxsize.x/2-par.boxsize.x/4:par.boxsize.x/2+par.boxsize.x/4,par.boxsize.y/2-par.boxsize.y/4:par.boxsize.y/2+par.boxsize.y/4), w_max_tot_spot)
   xy_max_tot_spot=WHERESUB(w_max_tot_spot(0),tot_spot(par.boxsize.x/2-par.boxsize.x/4:par.boxsize.x/2+par.boxsize.x/4,par.boxsize.y/2-par.boxsize.y/4:par.boxsize.y/2+par.boxsize.y/4))+par.boxsize.x/2-par.boxsize.x/4

   ;SURFACE,tot_spot
   ;get the optimal region, corresponding to approximately the 0.1% level of this average spot.
   ;The area for 5.0% of a gaussian corresponds to twice the one for 47 % [check ?]
   ;The area for 2.0% of a gaussian corresponds to twice the one for 38 % [check ?]
   ;The area for 1.0% of a gaussian corresponds to twice the one for 32 % [check ?]
   ;The area for 0.5% of a gaussian corresponds to twice the one for 27 % [check ?]
   ;The area for 0.25% of a gaussian corresponds to twice the one for 22 % [check ?]
   ;The area for 0.1% of a gaussian corresponds to twice the one for 18 % [check ?]
   ;we get the correspondance by a linear polynom fit.
   ;pol = [14.7148,35.3438,-26.2188,9.25000,-1.02930]
   ;rest_area = GET_BOXREG(tot_spot,ROUND(POLY(par.region_level,pol)),par.boxsize,/FINE_SEARCH)

   region_levels(i)=min_region_level>(fudge_factor*SQRT(par.bg_gain*max_bc)/FLOAT(max_tot_spot))<max_region_level

   ;rest_area = SEARCH2D(tot_spot,par.boxsize.x/2,par.boxsize.y/2,max_tot_spot*region_levels(i),max_tot_spot)
   rest_area = SEARCH2D(tot_spot,xy_max_tot_spot(0),xy_max_tot_spot(1),max_tot_spot*region_levels(i),max_tot_spot)
   ;rest_area = SEARCH2D(tot_spot,par.boxsize.x/2,par.boxsize.y/2,max_tot_spot*par.region_level,max_tot_spot)
   ;rest_area=GET_ENLARGED_REGION(rest_area,par.boxsize.x,par.boxsize.y,extend=1)

   tmp_spot=LONARR(par.boxsize.x,par.boxsize.y)
   tmp_spot(rest_area)=1
   IF MAX(par.overall_region) NE 0 THEN tmp_spot(par.overall_region)=0 ; intersec with the overall region if it is defined
   rest_area=WHERE(tmp_spot EQ 0,ct_rest)
   ;this is unlikely, but if it happens, just remove the four corners
   IF ct_rest EQ 0 THEN rest_area=[0,par.boxsize.x-1,(par.boxsize.y-1)*(par.boxsize.x),(par.boxsize.x*par.boxsize.y)-1]

   ;SURFACE,tot_spot
   ;trunc_spot=tot_spot
   ;trunc_spot(*)=0
   ;trunc_spot(rest_area)=tot_spot(rest_area)
   ;SURFACE,trunc_spot,COLOR=0.8*!D.N_COLORS  
   ;ok='ok'
   ;read,'ok?',ok

   ;now load the restrained region
   PTR_FREE,par.regions(i)
   par.regions(i)=PTR_NEW(rest_area)

   ;dummy_spot=LONARR(par.boxsize.x,par.boxsize.y)
   ;dummy_spot(*par.regions(sec_i))=1
   ;SURFACE,dummy_spot

   IF MAX(wait_sec) GE 1 THEN BEGIN
    w_wait = WHERE(wait_sec GE 1,ct_wait)
    FOR loc_i=0,ct_wait-1 DO BEGIN
     PTR_FREE,par.regions(w_wait(loc_i))
     par.regions(w_wait(loc_i))=PTR_NEW(*par.regions(i))
     wait_sec(w_wait(loc_i))=0
     region_levels(w_wait(loc_i))=region_levels(i)
    ENDFOR
   ENDIF
  ENDIF ELSE BEGIN ; there is no ref spot in this sector
    wait_sec(i) = 1
    IF i EQ number_sectors-1 THEN BEGIN
     ;fill all the empty sectors
     w_ok = WHERE(wait_sec LT 1,ct_ok)
     last_ok=w_ok(ct_ok-1)
     w_wait = WHERE(wait_sec GE 1,ct_wait)
     FOR loc_i=0,ct_wait-1 DO BEGIN
      PTR_FREE,par.regions(w_wait(loc_i))
      par.regions(w_wait(loc_i))=PTR_NEW(*par.regions(last_ok))
      wait_sec(w_wait(loc_i))=0
      region_levels(w_wait(loc_i))=region_levels(last_ok)
     ENDFOR
    ENDIF
  ENDELSE

  number_indices(i)=ct_sector
 ENDFOR 
 PRINT,'Distribution of reference spots [#] and contour levels [%] (by sector) : '
 FOR i=0,number_sectors-1 DO $
   PRINT,FORMAT="('Sector #:',I2,' ; Number of ref spots:',I3, ' ; Contour level:',F4.1)", i+1,number_indices(i),region_levels(i)*100

RETURN,ref_spots

END








