FUNCTION fit_voigt,x,y,r,Tol=tol,tmax=tmax,fwhm=fwhm_fit,$
  Gaussian=gaussian, Lorentzian=lorentzian, verbose=verbose

;+
; NAME:
;	FIT_VOIGT
; PURPOSE:
;	This function returns a fitted pseudo-voit function
;	to the x,y arrays.
; CATEGORY:
;	Math 
; CALLING SEQUENCE:
;	yfit=voigt_fit(x,y,r)
; INPUTS:
;	x: the abscissas array
;	y: the ordinates array
;
; OUTPUT:
;	yfit: the array with the fitted values
;	
; OPTIONAL OUTPUT:
;	r: a named variable an array with the parameters of the voigt function
;
; KEYWORD PARAMETERS:
;	Tol: The tolerance value
;	Tmax: 
;	fwhm: a named variable to receive the FWHM value
;	Gaussian: If set, the pseudovoigt is set to Gaussian (Factor=0)
;	Lorentzian: If set, the pseudovoigt is set to Lorentzian (Factor=1)
;	Verbose: flag for verbose output (default: verbose=1)
;
;
; MODOFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, April 2006
;	2006/10/17 srio@esrf.fr adds Lorentzian and Gaussian keywords. 
;	2009/01/15 srio@esrf.eu puts constraints: fwhm>0, ratio in [0,1]
;
;-

IF N_Elements(verbose) EQ 0 THEN verbose=1
IF N_Elements(y) EQ 0 THEN BEGIN
  a=rascii('HISTO1',/Double)
  x = Reform(a[0,*])
  y = Reform(a[1,*])
ENDIF

IF verbose EQ 1 THEN quiet=0 else quiet=1

; calculate initial values
fwhm = getfwhm(make_set(x,y))
IF fwhm EQ 0 THEN fwhm = abs(max(x,ii,min=amin)-amin)/4.0
IF verbose THEN print,'FIT_VOIGT: Starting FWHM is : '+strcompress(fwhm,/rem)

pars = [ max(y,ii,min=amin)-amin,x[ii],fwhm,0.5]
parinfo = replicate({value:0.D, fixed:0, limited:[0,0], $
                                  limits:[0.D,0]}, 4)
;
; width/fwhm is always positive
;
tmp = parinfo.limited
tmpv = parinfo.limits
tmp[0,2]=1
tmpv[0,2]=0
parinfo.limited=tmp
parinfo.limits=tmpv

;
; ratio is in [0,1]
;
tmp = parinfo.limited
tmpv = parinfo.limits
tmp[0,3]=1
tmp[1,3]=1
tmpv[0,3]=0
tmpv[1,3]=1
parinfo.limited=tmp
parinfo.limits=tmpv


parinfo.value=pars
IF Keyword_Set(Gaussian) THEN BEGIN
  parinfo.fixed=[0,0,0,1]
  pars[3]=0.0
  parinfo.value=pars
ENDIF
IF Keyword_Set(Lorentzian) THEN BEGIN
  parinfo.fixed=[0,0,0,1]
  pars[3]=1.0
  parinfo.value=pars
ENDIF


; call MPFIT

IF N_ELEMENTS(tol) EQ 0 THEN tol = 1d-10
IF N_ELEMENTS(tmax) EQ 0 THEN tmax = 20

r = mpfitexpr('voigt1(x,p[0:3])', X, Y, Y*0+1, pars, $
  parinfo=parinfo, $
  FTol=tol, XTol=tol, GTol=tol, $
  NPrint=20, MaxIter=tMax, Quiet=quiet, $
  Status=status, $      ; outputs
  PError=pError, $      ; output
  nIter=nIter, $        ; output
  yFit=yFit $   ; output
   )

fwhm_fit = r[2]

IF verbose THEN BEGIN
  Print,'FIT_VOIGT: Fit status,niter: ',status,niter
  Print,'FIT_VOIGT: pars: ',vect2string(r)
  Print,'FIT_VOIGT: Fit fwhm: ',fwhm_fit
ENDIF

;; plot
;erase
;plot,x,y
;oplot,x,voigt1(x,pars),color=2
;oplot,x,yfit,color=3
;pause


Return,yfit


END

