;+
;
;==============================  Xyaup   =======================================
;
;  Xyaup is a widget based graphical interface to calculate spectra of a 
;  tapered undulator insertion devices.
;
;  It interfaces the program YAUP 1.3.1 written by B. Boyanov (see Help/Yaup)
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    XYaup input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;    Yaup file: This option allows to load/write the current parameter
;		and result files for further calculations. The used file
;		formats are those used by YAUP. Therefore, they
;		can be used to run YAUP without the interface or to
;		import YAUP files created outside the XYAUP application.
;    Write Files for XOP/Optics: Saves the current undulator
;		spectrum in the SRCOMPE (flux) and SRCOMPW (Power) files
;		to be used for the other XOP application from Optics
;		menu.
;  Quit and preserve files: Quits without cleaning current directory
;		(i.e., preserving the created data files.)
;  Quit: to exit from the program
;
; BField:
;         	from ASCII file: 
;		Yaup needs a file with tha magnetic field. This may be 
;		obtain from a user's file (column-formatted, containing
;		three colums: z, B(z), and phi(z), where the z s are 
;		equidistant with step PERIOD/NPTS. 
;		See Help/Yaup for definitions of PERIOD and NPTS.  
;		There should be NPTS*NPER+1 lines in the ASCII file.
;
;		from XBFIELD Preprocessor: calls XBFIELD, an interface for the 
;		program BFIELD distributed in the YAUP package. This program
;		creates the magnetic field starting from the undulator
;		oparameters.
;
;		from Linear B field: Creates a magnetic field modulated with a 
;		straight line with the given B or K extrema.
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XYAUP window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, YAUP starts running.
;  Set Defaults: Sets the default parameters.
;
;  Show: Display results
;		Plot B field: Plots the computed B field.
;		Plot Trajectory: Plots the electron trajectory in the 
;		defined magnetic field.
;		Show Trajectory Numerical Data: display the trajectory 
;		output file.
;		Plot Results: Makes 2-D or 3-D plots of the results,
;		depending on the mode and other defined parameters.
;		Show Results Numerical Data: display the yaup result file.
;
;  Help:   Shows the YAUP help and the XYAUP help (this text).
;
; COPYRIGHT:
;	XYAUP and XBFIELD belongs to XOP package and it is distributed 
;       within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
;
; REFERENCE:
;	Published calculations made with XOP should refer:
;
;	M. Sanchez del Rio and R. J. Dejus 
;        "Status of XOP: an x-ray optics software toolkit"
;        SPIE Proceedings Vol. 5536 (2004) pp.171-174
;
;        http://dx.doi.org/10.1117/12.560903
;
;
;	In addition, published calculations using the XYAUP package 
;	should also cite:
;
;	  B. I. Boyanov, G. Bunker, J. M. Lee, and T. I. Morrison
;	  "Numerical Modeling of Tapered Undulators"
;	  Nucl. Instr. Meth. A339,  596-603, 1994
;
;
;
; LAST MODIFICATION: srio@esrf.eu 2008-02-01
;
;
;
;-
; -----------------------------------------------------------------------
;	Author: F. Legrand (CEA Bruyeres le Chatel 91 France) , Oct 94.
;         and M. Sanchez del Rio (ESRF srio@esrf.fr)
;	Modification history:
;         94/11/18 MSR terminates the version to be integrated in xop 1.1
;         97/01/23 MSR ports to Windows95. 
;         97/10/01 MSR renames read_ascii by rascii
;         97/10/02 MSR uses sdep(), xop_ws, and much more cosmetics.
;         97/10/27 MSR updates doc.
;	  98/11/26 srio@esrf.fr adapts menus and input files to xop2.0.
;		Use of Xop_Spawn.
;	  99/03/16 srio@esrf.fr updates doc.
;	  00/07/21 srio@esrf.fr prepared for xop2.0.  Version 1.2
;
;=======================================================================
;
FUNCTION XYaup_Version
RETURN,'1.2'
END ; XYaup_Version
;
;
;=======================================================================
;
FUNCTION XYAUP_DESCRIPTION
;
; give description of the input structure
;
on_error,2
;
res = [ ' Title:  ' 			$
,'PERIOD - magnet period (cm)'		$
,'NPER - number of periods'		$
,'NPTS - number of point/period'	$
,'EMIN - minimum energy (eV)'		$
,'EMAX - maximum energy (eV)'		$
,'NE - number of energy points'		$
,'ENERGY - e energy (GeV)'		$
,'CUR - e current (A)'		$
,'SIGX - H rms e beam (mm)'	$
,'SIGY - V rms e beam (mm)'	$
,'SIGX1 - rms H e div (mrad)'	$
,'SIGY1 - rms V e div (mrad)'	$
,'D - dist und-observator (m)'	$
,'XPC - H obs position (mm)'	$	
,'YPC - V obs position (mm)'	$
,'XPS - H acceptance (mm\mrad)'	$
,'YPS - V acceptance (mm\mrad)'	$
,'NXP - no acceptance pts (H)'	$
,'NYP - no acceptance pts (V)'	$
,'MODE - (see help)'			$	
,'NSIG - (see help)'			$
,'TRAJECTORY - calculation flag'	$
,'XSYM - horizontal symmetry'	$
,'HANNING - (see help)'	$
,'BFILE - B filename'	$
,'TFILE - Traj filename'	$
]
;
return,res
end

;
;======================================================================
;
FUNCTION  XYAUP_INP_READ,file,struct, Group=group
;
; read an Yaup input file, ans stores the result in struct
;
;on_error,2
;catch,error_status
;if error_status NE 0 then begin
;  print,'XYAUP_INP_READ: Error reading yaup input file.'
;  return,error_status
;endif

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'Error reading yaup input file: '+file
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XYAUP_INP_READ: Error reading yaup input file: '+file)
   Catch, /Cancel
   On_Error,2
   RETURN,error_status
ENDIF
;
; reads the file and stotes its contents in a buffer
;
openr,Unit,file,/get_lun
line = ''
buffer = ''
repeat begin
  READF,unit,line
  ;print,'>>>> line: ',line
  if line NE '' and strmid(line,0,1) NE ';' then buffer = [buffer,line]
endrep until (strcompress(strupcase(line),/rem) EQ 'END')
free_lun,Unit
;
; rewrites the buffer in an idl readable  form
;
; line 1
tmp = buffer(1)
strput,tmp,'&',strpos(tmp,'NPER')-1
strput,tmp,'&',strpos(tmp,'NPTS')-1
buffer(1) = tmp
;print,'>>>',tmp

; line 2
tmp = buffer(2)
strput,tmp,'&',strpos(tmp,'EMAX')-1
strput,tmp,'&',strpos(tmp,'NE')-1
strput,tmp,'N',strpos(tmp,'E=')
buffer(2) = tmp
;print,'>>>',tmp

; line 3
tmp = buffer(3)
strput,tmp,'&',strpos(tmp,'CURRENT')-1
buffer(3) = tmp
;print,'>>>',tmp

; line 4
tmp = buffer(4)
strput,tmp,'&',strpos(tmp,'SIGY')-1
buffer(4) = tmp
;print,'>>>',tmp

; line 5
tmp = buffer(5)
strput,tmp,'&',strpos(tmp,'SIGY1')-1
buffer(5) = tmp
;print,'>>>',tmp

; line 7
tmp = buffer(7)
strput,tmp,'&',strpos(tmp,'XPS')-1
strput,tmp,'&',strpos(tmp,'NXP')-1
buffer(7) = tmp
;print,'>>>',tmp

; line 8
tmp = buffer(8)
strput,tmp,'&',strpos(tmp,'YPS')-1
strput,tmp,'&',strpos(tmp,'NYP')-1
buffer(8) = tmp
;print,'>>>',tmp

; line 9
tmp = buffer(9)
strput,tmp,'&',strpos(tmp,'NSIG')-1
ss = strpos(tmp,'TRAJECTORY')-1
strput,tmp,'&',ss
ss2=strpos(tmp,'=',ss)+1
tmp = strmid(tmp,0,ss2)+"'"+strmid(tmp,ss2,strlen(tmp))+"'"
buffer(9) = tmp
;print,'>>>',tmp

; line 10
tmp = buffer(10)
ss=strpos(tmp,'M=')+2
tmp = strmid(tmp,0,ss)+"'"+strmid(tmp,ss,strlen(tmp))
strput,tmp,"'",strpos(tmp,'HANNING')-2
strput,tmp,'&',strpos(tmp,'HANNING')-1
buffer(10) = strcompress(tmp,/rem)
;print,'>>>',tmp
;
; now, define the variables and set their values
;
print,'>>>> defining...'
period = 3.3  &  nper=75  &  npts=40
emin = 100.  &  emax = 30000.0  &  nn = 100
energy = 6.0  &  current = 0.1
sigx = 0.43  &  sigy = 0.089
sigx1 = 0.016  &  sigy1 = 0.008
distance = 30.0
xpc = 0.0  &  xps = 2.0  &  nxp = 25
ypc = 0.0  &  yps = 2.0  &  nyp = 25
mode = 4  &  nsig = 2  &  trajectory='new+keep'
xsym = 'yes' & hanning = 0
CASE sdep() OF 
  'WINDOWS': bfile = 'undulbf' 
  'UNIX':    bfile = 'undul.bf'
  else:      bfile = 'undulbf' 
ENDCASE
tfile = 'undul.traj'

for i=1,n_elements(buffer)-2 do itmp=execute(buffer(i))
;for i=1,n_elements(buffer)-2 do print,buffer(i)
;
; writes the output structure
;
struct=xop_defaults('xyaup')
struct.TITLE= file
struct.PERIOD= PERIOD
struct.NPER= NPER
struct.NPTS= NPTS
struct.EMIN= EMIN
struct.EMAX= EMAX
struct.NENERGY= NN

struct.ENERGY= ENERGY
struct.CUR= CURRENT
struct.SIGX= SIGX
struct.SIGY= SIGY
struct.SIGX1= SIGX1
struct.SIGY1= SIGY1

struct.D= DISTANCE
struct.XPC= XPC
struct.YPC= YPC
struct.XPS= XPS
struct.YPS= YPS
struct.NXP= NXP
struct.NYP= NYP

struct.MODE= MODE
struct.NSIG= NSIG
struct.TRAJECTORY = TRAJECTORY
struct.XSYM = XSYM
struct.HANNING= HANNING
struct.BFILE= BFILE
struct.TFILE= TFILE

return,1
end 
;
;======================================================================
;
PRO XYAUP_INP_WRITE,yainpstr,NAME=name
;
; write the contents of the structure with parameters for running YAUP
;
IF NOT(KEYWORD_SET(name)) THEN name='yaup.inp'
;
PRINT,'*** Writing input parameters into file: '+name+' ***'

OPENW,unit,name,/GET_LUN

PRINTF,unit,''
PRINTF,unit,';Magnet parameters'
PRINTF,unit,'PERIOD='+ STRCOMPRESS(yainpstr.PERIOD,/REMOVE_ALL)+$
' NPER='+ STRCOMPRESS(yainpstr.NPER,/REMOVE_ALL)+$
' NPTS='+ STRCOMPRESS(yainpstr.NPTS,/REMOVE_ALL)

PRINTF,unit,''
PRINTF,unit,';Photon energy'
PRINTF,unit,'EMIN='+STRCOMPRESS(yainpstr.EMIN,/REMOVE_ALL)+$
' EMAX='+ STRCOMPRESS(yainpstr.EMAX,/REMOVE_ALL)+$
' NE='+ STRCOMPRESS(yainpstr.NENERGY,/REMOVE_ALL)

PRINTF,unit,''
PRINTF,unit,';Storage ring'
PRINTF,unit,'ENERGY='+STRCOMPRESS(yainpstr.ENERGY,/REMOVE_ALL)+$
' CURRENT='+ STRCOMPRESS(yainpstr.CUR,/REMOVE_ALL)
PRINTF,unit,' SIGX='+STRCOMPRESS(yainpstr.SIGX,/REMOVE_ALL)+$
' SIGY='+ STRCOMPRESS(yainpstr.SIGY,/REMOVE_ALL)
PRINTF,unit,'SIGX1='+STRCOMPRESS(yainpstr.SIGX1,/REMOVE_ALL)+$
' SIGY1='+STRCOMPRESS(yainpstr.SIGY1,/REMOVE_ALL)

PRINTF,unit,''
PRINTF,unit,';Pinhole (mm or mrad)'
PRINTF,unit,'DISTANCE='+STRCOMPRESS(yainpstr.D,/REMOVE_ALL)
PRINTF,unit,'XPC='+STRCOMPRESS(yainpstr.XPC,/REMOVE_ALL)+ $
' XPS='+STRCOMPRESS(yainpstr.XPS,/REMOVE_ALL)+ $
' NXP='+STRCOMPRESS(yainpstr.NXP,/REMOVE_ALL)
PRINTF,unit,'YPC='+STRCOMPRESS(yainpstr.YPC,/REMOVE_ALL)+ $
' YPS='+STRCOMPRESS(yainpstr.YPS,/REMOVE_ALL)+ $
' NYP='+STRCOMPRESS(yainpstr.NYP,/REMOVE_ALL)

PRINTF,unit,''
PRINTF,unit,';Calculation parameter'
PRINTF,unit,'MODE='+STRCOMPRESS(yainpstr.MODE,/REMOVE_ALL)+ $
' NSIG='+STRCOMPRESS(yainpstr.NSIG,/REMOVE_ALL)+' '+ $
'  TRAJECTORY='+STRCOMPRESS(yainpstr.TRAJECTORY,/REMOVE_ALL)
PRINTF,unit,'XSYM='+STRCOMPRESS(yainpstr.XSYM,/REMOVE_ALL)+$
	'  HANNING='+STRCOMPRESS(yainpstr.HANNING,/REMOVE_ALL)

PRINTF,unit,''
PRINTF,unit,';Filenames'
PRINTF,unit,'BFILE="'+STRCOMPRESS(yainpstr.BFILE,/REMOVE_ALL)+'"'
PRINTF,unit,'TFILE="'+STRCOMPRESS(yainpstr.TFILE,/REMOVE_ALL)+'"'

PRINTF,unit,''
PRINTF,unit,'END'
PRINTF,unit,''

FREE_LUN,unit
;
END
;
;======================================================================
;
FUNCTION  XYAUP_OUT_READ,file,struct, Group=group
;
; read an Yaup input file, ans stores the result in struct
;
;on_error,2
;catch,error_status
;if error_status NE 0 then begin
;  print,'XYAUP_OUT_READ: Error reading yaup input file.'
;  return,error_status
;endif

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'Error reading yaup input file: '+file
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XYAUP_OUT_READ: Error reading yaup input file: '+file)
   Catch, /Cancel
   On_Error,2
   RETURN,error_status
ENDIF
;
; reads the file and stores its contents in a buffer
;
openr,Unit,file,/get_lun
line = ''
buffer = ''
repeat begin
  READF,unit,line
  ;print,'>>>> line: ',line
  if line NE '' then buffer = [buffer,line]
endrep until (strcompress(strupcase(line),/rem) EQ 'UNITS:')
free_lun,Unit
;
; rewrites the buffer in an idl readable  form
;
; line 2
tmp = buffer(2)
s1 = strpos(tmp,':')+1
s2 = strlen(tmp) - s1
tmp= strmid(tmp, s1, s2)
strput,tmp,'    ',strpos(tmp,'(cm)')
strput,tmp,'&',strpos(tmp,'nper')-1
strput,tmp,'&',strpos(tmp,'npts')-1
buffer(2) = tmp
;print,'>>>',tmp

; line 3
tmp = buffer(3)
s1 = strpos(tmp,':')+1
s2 = strlen(tmp) - s1
tmp= strmid(tmp, s1, s2)
strput,tmp,'&',strpos(tmp,'emax')-1
strput,tmp,'&',strpos(tmp,'ne')-1
strput,tmp,'nn',strpos(tmp,'ne')
buffer(3) = tmp
;print,'>>>',tmp


; line 4
tmp = buffer(4)
s1 = strpos(tmp,':')+1
s2 = strlen(tmp) - s1
tmp= strmid(tmp, s1, s2)
strput,tmp,'     ',strpos(tmp,'(GeV)')
strput,tmp,'   ',strpos(tmp,'(A)')
strput,tmp,'&',strpos(tmp,'current')-1
buffer(4) = tmp
;print,'>>>',tmp
;
; line 5
tmp = buffer(5)
strput,tmp,'    ',strpos(tmp,'(mm)')
strput,tmp,'    ',strpos(tmp,'(mm)')
strput,tmp,'&',strpos(tmp,'sigy')-1
buffer(5) = tmp
;print,'>>>',tmp
;
; line 6
tmp = buffer(6)
strput,tmp,'      ',strpos(tmp,'(mrad)')
strput,tmp,'      ',strpos(tmp,'(mrad)')
strput,tmp,'&',strpos(tmp,'sigy1')-1
buffer(6) = tmp
;print,'>>>',tmp

; line 7
tmp = buffer(7)
s1 = strpos(tmp,':')+1
s2 = strlen(tmp) - s1
tmp= strmid(tmp, s1, s2)
strput,tmp,'   ',strpos(tmp,'(m)')
buffer(7) = tmp
;print,'>>>',tmp
;
; line 8
tmp = buffer(8)
strput,tmp,'         ',strpos(tmp,'(mrad/mm)')
strput,tmp,'         ',strpos(tmp,'(mrad/mm)')
strput,tmp,'&',strpos(tmp,'xps')-1
strput,tmp,'&',strpos(tmp,'nxp')-1
buffer(8) = tmp
;print,'>>>',tmp
;
;
; line 8
tmp = buffer(9)
strput,tmp,'         ',strpos(tmp,'(mrad/mm)')
strput,tmp,'         ',strpos(tmp,'(mrad/mm)')
strput,tmp,'&',strpos(tmp,'yps')-1
strput,tmp,'&',strpos(tmp,'nyp')-1
buffer(9) = tmp
;print,'>>>',tmp

; line 10
tmp = buffer(10)
s1 = strpos(tmp,':')+1
s2 = strlen(tmp) - s1
tmp= strmid(tmp, s1, s2)
strput,tmp,'&',strpos(tmp,'nsig')-1
strput,tmp,'&',strpos(tmp,'itraj')-1
buffer(10) = tmp
;print,'>>>',tmp
;
; line 11
tmp = buffer(11)
strput,tmp,'&',strpos(tmp,'nhan')-1
buffer(11) = tmp
;print,'>>>',tmp
;
; line 12
tmp = buffer(12)
strput,tmp,'&',strpos(tmp,'nhan')-1
ss=strpos(tmp,'file')+4
tmp = "bfile = '"+strcompress(strmid(tmp,ss,strlen(tmp)),/rem)+"'"
buffer(12) = tmp
;print,'>>>',tmp
;
; line 13
tmp = buffer(13)
strput,tmp,'&',strpos(tmp,'nhan')-1
ss=strpos(tmp,'file')+4
tmp = "tfile = '"+strcompress(strmid(tmp,ss,strlen(tmp)),/rem)+"'"
buffer(13) = strcompress(tmp,/rem)
;print,'>>>',tmp
;
; now, define the variables and set their values
;
;print,'>>>> defining...'
period = 3.3  &  nper=75  &  npts=40
emin = 100.  &  emax = 30000.0  &  nn = 100
energy = 6.0  &  current = 0.1
sigx = 0.43  &  sigy = 0.089
sigx1 = 0.016  &  sigy1 = 0.008
distance = 30.0
xpc = 0.0  &  xps = 2.0  &  nxp = 25
ypc = 0.0  &  yps = 2.0  &  nyp = 25
mode = 4  &  nsig = 2  &  itraj = 2
ixsym = 1 & nhan = 0
bfile = 'undul.bf'
tfile = 'undul.traj'

for i=2,n_elements(buffer)-1 do itmp=execute(buffer(i))
;for i=2,n_elements(buffer)-1 do print,buffer(i)
;
; writes the output structure
;
struct=xop_defaults('xyaup')
struct.TITLE= file
struct.PERIOD= PERIOD
struct.NPER= NPER
struct.NPTS= NPTS
struct.EMIN= EMIN
struct.EMAX= EMAX
struct.NENERGY= NN

struct.ENERGY= ENERGY
struct.CUR= CURRENT
struct.SIGX= SIGX
struct.SIGY= SIGY
struct.SIGX1= SIGX1
struct.SIGY1= SIGY1

struct.D= DISTANCE
struct.XPC= XPC
struct.YPC= YPC
struct.XPS= XPS
struct.YPS= YPS
struct.NXP= NXP
struct.NYP= NYP

struct.MODE= MODE
struct.NSIG= NSIG
;struct.TRAJECTORY = TRAJECTORY
;struct.XSYM = XSYM
struct.HANNING= nhan
struct.BFILE= BFILE
struct.TFILE= TFILE

return,1
end 
;
;======================================================================
;
PRO xyaup_surfplot,file,zz,x,y,POLARIZATION=POLARIZATION,Group=group
;
; make surface plots of a yaup output file (mode 2,3,6 and 7)
; MSR/94/11/04
;
on_error,2
;
; load file
;
input_type = (size(FILE)) ((size(FILE)) (0)+1)
print,'input_type: ',input_type
if (input_type eq 7) then begin
  print,'XYAUP_SURFPLOT: readinf file...'
  a=rascii(FILE)
  itmp = xyaup_out_read(FILE,yainpstr,Group=group)
  print,'XYAUP_SURFPLOT: Done'
endif else begin
  a=file
endelse
;
; reform data array
;
nplots = yainpstr.nenergy
if yainpstr.mode EQ 7 then nplots = 1
data = fltarr(nplots,3,n_elements(a(0,*))/nplots)
ener = fltarr(nplots)
nn = n_elements(a(0,*))/nplots
for ii=0,nplots-1 do begin
  i0 = ii*nn
  i1 = i0 + nn - 1
  ;print,'***',i0,i1
  if yainpstr.mode NE 7 then data(ii,0:2,*)  = a(1:3,i0:i1) else $
       data(ii,0:2,*)  = a(0:2,i0:i1)
  ener(ii) = a(0,i0)
  if keyword_set(polarization) then data(ii,2,*)  = a(4,i0:i1)
endfor
;help,a
;help,data
;
; prepare titles
;
if (yainpstr.d eq 0) then begin
  xtitle='x [mrad]'
  ytitle='x [mrad]'
endif else begin
  xtitle='x [mm]'
  ytitle='x [mm]'
endelse

case yainpstr.mode of
  2: begin
     if keyword_set(polarization) then begin
       title='polarization at E='+strcompress(yainpstr.emin)+' eV' 
       ztitle='Polarization degree'
     endif else begin 
       title='brightness at E='+strcompress(yainpstr.emin)+' eV'
       if yainpstr.d EQ 0 then ztitle='photons/sec/0.1%b.w./mrad^2' else $
                               ztitle='photons/sec/0.1%b.w./mm^2'
     endelse
     end
  3: begin
     if keyword_set(polarization) then begin
       title='polarization'
       ztitle='Polarization degree'
     endif else begin
       title='brightness'
       if yainpstr.d EQ 0 then ztitle='photons/sec/0.1%b.w./mrad^2' else $
                               ztitle='photons/sec/0.1%b.w./mm^2'
     endelse
     end
  6: begin
     title='spectral power'
     if yainpstr.d EQ 0 then ztitle='Watts/eV/mrad^2' else $
                              ztitle='Watts/eV/mm^2'
     end
  7: begin
     title='integrated spectral power'
     if yainpstr.d EQ 0 then ztitle='Watts/mrad^2' else $
                              ztitle='Watts/mm^2'
     end
  else: begin
     print,'XYAUP_SURFPLOT: Error: Selected mode do not represent a surface.'
     return
     end
endcase
;
; make surface plots
;
charold = !p.charsize
!p.charsize = 2.0
ss = sqrt(nplots)
if ss GT 1.0 then $
!p.multi=[0,fix(ss)+1 > 2,fix(ss) > 2,0] else !p.multi=0
;print,!p.multi

for ii=0,nplots-1 do begin
  coor2matr,reform(data(ii,*,*)),z1,x,y
  if ii EQ 0 then $
    zz = fltarr(nplots,n_elements(z1(0,*)),n_elements(z1(1,*)))
  zz(ii,*,*) = z1
  if ((yainpstr.mode EQ 3) or (yainpstr.mode EQ 4) or $
      (yainpstr.mode EQ 6)) then $
     titletmp=title+' E= '+strcompress(ener(ii),/rem)+' eV' else $
     titletmp=title
  surface,z1,x,y,xtitle=xtitle,ytitle=ytitle,ztitle=ztitle,title=titletmp
endfor
!p.multi=0

end
;
;======================================================================
;
PRO xyaup_blinear,yaupstr,GROUP=group
;
;creates a liner b file as wanted by yaup
;
if n_params() lt 1 then begin
  print,'XYAUPBLINER: Usage xyaup_linear,yaupstr'
  return
endif

aa       =  {  PERIOD: yaupstr.period, NPER: yaupstr.nper, $
NPTS: yaupstr.npts, $
ITYPE: ['0','Magnetic field B [Tesla]','Deflection parameter K'], $
a1: 0.5, a2: 1.0, FILE:yaupstr.bfile}

titles = [  'PERIOD - magnet period (cm)','N - number of periods'  $
,'NPTS - nb of point / period','Input parameter: '                 $
,'From:','To:','FILE - Output (binary) file name']

XSCRMENU,aa,ACTION=action, /INTERP, TITLES=TITLES,/NOTYPE,wtitle= $
 'Linear B generation Parameters'

IF (action EQ 'DONT') THEN RETURN

; updates yaup structure
yaupstr.period = aa.period
yaupstr.nper   = aa.NPER
yaupstr.npts   = aa.NPTS
yaupstr.bfile  = aa.FILE

; makes the calculation
n = aa.npts
lambdau = aa.period
npts_per = aa.nper

if strcompress(aa.itype(0),/rem) EQ '0' then begin
  b1 = aa.a1
  b2 = aa.a2
endif else begin
  b1 = aa.a1/0.934/aa.period
  b2 = aa.a2/0.934/aa.period
endelse

und_len = lambdau*npts_per
z = dindgen(n*npts_per+1)/double(n*npts_per)
z = z*und_len

bmod = dindgen(n*npts_per+1)/float(n*npts_per) * (b2-b1) + b1
berr =  dindgen(n*npts_per+1) * 0.0
btot = bmod * sin(2.0D0*!pi/lambdau*z)
;plot,z,btot
;oplot,z,bmod

; writes the ascii file
openw,unit,'xyaup.B.ascii',/get_lun
printf,unit, '# Columns: z(cm), ampl(tesla), phserr, total(tesla)'
printf,unit, '# total = ampl * sin ( twopi/period*z + phserr ) '
printf,unit, '# period= '+strcompress(lambdau,/rem)+' nper= '+ $
  strcompress(n,/rem)+' npts=',strcompress(npts_per,/rem)
for i=0,n_elements(z)-1 do printf,unit,z(i),bmod(i),berr(i),btot(i)
free_lun,unit

; runs txt2u to create the binary file
CASE SDep() OF
   'WINDOWS': tmpfile = 'xyaupB.inp'
   'UNIX':    tmpfile = 'xyaup.B.inp' 
   else:      tmpfile = 'xyaupB.inp'  
ENDCASE
openw,unit,tmpfile,/GET_LUN
  printf,unit,'xyaup.B.ascii'
  printf,unit,aa.file
  printf,unit,aa.period
  printf,unit,aa.nper
  printf,unit,aa.npts
  free_lun,unit
CASE sdep() OF
  'WINDOWS':   command = 'txt2u < xyaupB.inp'
  'UNIX':      command = 'txt2u < xyaup.B.inp'
  else:   command = 'txt2u < xyaupB.inp'
ENDCASE
Xop_Spawn,command

; shows the B
if keyword_set(group) then widget_control,/hourglass
xplot,'xyaup.B.ascii',ycol=4, xtitle='z(cm)',ytitle='B [Tesla]', $
 group=group,Wtitle='xyaup: Magnetic Field'

; clean unwanted files
CASE SDep() OF
  'WINDOWS':    files=['xyaupB.asc','xyaupB.inp'] 
  'UNIX':       files=['xyaup.B.ascii','xyaup.B.inp']
  else:         files=['xyaupB.asc','xyaupB.inp'] 
ENDCASE
Delete_Files,files

end
;
;======================================================================
;
PRO XYAUP_SET,yainpstr,GROUP=group
;
; set the input parameters for YAUP and run it
;
;
CASE sdep() OF
  'WINDOWS': bfile = 'undulbf' 
  'UNIX':    bfile = 'undul.bf'
  else: bfile = 'undulbf' 
ENDCASE
if (checkfile(bfile) EQ 0) then begin
  itmp = widget_message(dialog_parent=group,$
     /error,['File '+bfile, $
     'with the magnetic field not','found.', $
     'Please run "Bfield from..." before'])
  return
endif

sep = sdep(/ds)
helpcmd = "xdisplayfile1,'"+Xop_GetEnv('XOP_HOME')+$
 sep+'doc'+sep+'txt'+sep+'xyaup_par.txt'+"'"
;helpcmd = "xdisplayfile1,'"+Xop_GetEnv('XOP_HOME')+$
; sep+'doc'+sep+'xyaup.par'+"',group=event.top,/NoMenuBar"
XSCRMENU,yainpstr,GROUP=group,nrow=9,ACTION=action, $
  TITLES=XYAUP_DESCRIPTION(),/NOTYPE,HELP=helpcmd, $
  WTitle='YAUP parameters'
IF (action EQ 'DO') THEN BEGIN
  XYAUP_INP_WRITE,yainpstr
  if sdep() eq "UNIX" then begin 
     command = '/bin/rm -f  yaup1.out yaup.cfg yaup.s0 yaup.s1 yaup-*.out'
     PRINT,command  
     spawn,command,/sh 
  endif else begin
     command = 'del  yaup1.out yaup.cfg yaup.s0 yaup.s1 yaup-*.out'
     PRINT,command  
     spawn,command
  endelse
  ;PRINT,'********* Start running YAUP: Please wait few minutes *************'
  widget_control,/hourglass
  ;if sdep() EQ 'WINDOWS' then command = 'yaup' else command = $
  ;	'source $XOP_RC ; yaup'
  ;message,/info,'Executing: '+command  
  ; SPAWN,command
  command = 'yaup'
  Xop_Spawn,command,CleanFiles=['yaup-0.out']
  ;PRINT,'******************* End running YAUP ******************************'
  ;PRINT,'successful completation of work'
ENDIF

end
;
;==========================================================================
;
PRO xyaup_event,event
;

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XYAUP_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

WIDGET_CONTROL, event.id, GET_UVALUE=eventuval, HOURGLASS=1
widget_control, event.top, GET_UVALUE=yainpstr, HOURGLASS=1

CASE eventuval OF


  "FILEINPUT": BEGIN
     action=''
     Widget_Control,event.id, Get_Value=action
     CASE action OF
       'Load from file...': BEGIN
         ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
	 filter='*.xop'
         str_par = Xop_Input_Load(Title='Select XYaup input file...',$
           /NoConf,Filter=filter,Group=event.top)
         IF Type(str_par) EQ 8 THEN BEGIN
           tmp = yainpstr
           Copy_Structure,str_par,tmp,Group=event.top ;,/OnlyFirstField
           yainpstr = tmp
           Widget_Control,event.top,Set_UVal=yaInpStr,/No_Copy
         ENDIF
       END
       'Save to file...': BEGIN
         str_par = yainpstr
         Xop_Input_Save,str_par,File='xyaup.xop',$
           /Write, Group=event.top, Comment='; xop/xyaup(v'+$
         xyaup_version()+') input file on '+SysTime()
       END
       'Save as default': BEGIN
         str_par = yainpstr
         Xop_Input_Save,str_par,Group=event.top, $
           Default='xyaup.xop',Comment='; xop/xyaup(v'+$
           xyaup_version()+') input file on '+SysTime()
       END
     ENDCASE
  END

  "XYALOAD": BEGIN
     infile_old = ''
     infile=Dialog_PICKFILE(/read,GROUP=event.top)
     IF (infile_old NE infile) THEN BEGIN
       k_err = XYAUP_INP_READ(infile,tmp, Group=event.top)
      print,'>>> infile***'+infile+'***'
      print,'>>> k_err : ',k_err
       IF k_err LT 0 THEN BEGIN
         itmp = widget_message(dialog_parent=event.top,$
	 /error,['',' Error reading file...',$
         ' This seems not an Yaup ', ' input file  '])
         return
       ENDIF ELSE BEGIN
         widget_control,event.top,set_uvalue=tmp,/NO_COPY
       ENDELSE
     ENDIF
     END
  "XYASAVE": BEGIN
     file = Dialog_pickfile(/WRITE,FILE='xyaup.inp',GROUP=Event.Top)
     if file NE '' then xyaup_inp_write,yainpstr,NAME=file
     END
  'XYAOUTLD': BEGIN
     infile=Dialog_PICKFILE(/read,GROUP=event.top)
     IF infile EQ '' then return
     CASE SDep() OF
	'WINDOWS': command = 'copy '+infile+' yaup-0.out'
	'UNIX':    command = '/bin/cp '+infile+' yaup-0.out'
	else:      Message,'Not implemented platform'
     ENDCASE
     itmp = xyaup_out_read('yaup-0.out',yainpstr,Group=event.top)
     widget_control,event.top,set_uval=yainpstr,/NO_COPY
     message,/info,'Executing: '+command  
     if sdep() eq "UNIX" then spawn,command,/sh else spawn,command
     END
  'XYAOUTSV': BEGIN
     file = Dialog_pickfile(/WRITE,FILE='xyaup.out',GROUP=Event.Top)
     if file NE '' then begin
       CASE SDep() OF
	'WINDOWS':  command = 'copy yaup-0.out '+file
        'UNIX':     command = '/bin/cp yaup-0.out '+file
        else:       Message,'Platform not implemented'
       ENDCASE
       message,/info,'Executing: '+command  
       if sdep() eq "UNIX" then spawn,command,/sh else spawn,command
     endif
     END

  'XYATRAN': BEGIN
    if checkfile('yaup-0.out') EQ 0 then begin
      itmp = widget_message(dialog_parent=event.top,$
	 /error, ['Yaup output file', $
	'"yaup-0.out" not found', 'Please run yaup by using the',$
	'"Set Parameters" option'])
      RETURN
    endif
    if yainpstr.mode NE 4 then begin
      itmp = widget_message(dialog_parent=event.top,$
	 /error,['Option MODE not valid.',$
	'Please use option MODE=4','in Set Parameters'])
      return
    endif
    tmp = xyaup_out_read('yaup-0.out',yainpstr,Group=event.top)
    widget_control,event.top,set_uvalue=yainpstr
    a = rascii('yaup-0.out')
    openw,Unit,'SRCOMPE',/GET_LUN
    printf,Unit,a
    free_lun,Unit
    a(1,*) = a(1,*) * 1.602e-16  ; pass to watts
    openw,Unit,'SRCOMPW',/GET_LUN
    printf,Unit,a
    free_lun,Unit
    itmp = widget_message(dialog_parent=event.top,$
	/info,['','  Files SRCOMPE & SRCOMPW  ',$ 
	' written to current directory  '])
    END
  'XYAQUIT': BEGIN
     Delete_Files,['yaup.inp','yaup1.out','yaup.cfg','yaup.s0', $
          'yaup.s1','yaup-*.out','undul.bf','undul.traj']
     WIDGET_CONTROL,event.top,/DESTROY
     END
  'XYASOFTQUIT': WIDGET_CONTROL,event.top,/DESTROY
  "XYASCII": BEGIN
     itmp = widget_message(dialog_parent=event.top,/info, $
      ['This option takes an ASCII file and convert it to YAUP format.',$
       'The text file should be column-formatted, and contain three colums:', $
       ' z, B(z), and phi(z), where the z s are equidistant with step ', $
       ' PERIOD/NPTS. See HELP/YAUP for definitions of PERIOD and NPTS.' , $
       ' There should be NPTS*NPER+1 lines in the ASCII file.'])
     infile=Dialog_PICKFILE(GROUP=event.top)
     IF (infile NE '') THEN BEGIN
        tmp = { BFILE:yainpstr.BFILE,  PERIOD:yainpstr.PERIOD, $
        NPER:yainpstr.NPER, NPTS:yainpstr.NPTS } 
        titles = ['B output filename', 'magnet period (cm)', $
         'number of periods','number of point/period' ]
        XscrMenu,tmp,/NOTYPE,ACTION=ACTION,TITLES=titles
        IF action EQ 'DONT' then return
        openw,unit,'xyaup.tmp',/get_lun
          printf,unit,infile
          printf,unit,tmp.BFILE    &   yainpstr.BFILE = tmp.BFILE
          printf,unit,tmp.PERIOD   &   yainpstr.PERIOD = tmp.PERIOD
          printf,unit,tmp.NPER   &   yainpstr.NPER = tmp.NPER
          printf,unit,tmp.NPTS   &   yainpstr.NPTS = tmp.NPTS
        free_lun,unit
        command='txt2u < xyaup.tmp'
        Xop_Spawn,command
        widget_control,event.top,set_uvalue=yainpstr,/NO_COPY
     ENDIF
     END
  'XYABF'  : xbfield,GROUP=event.top
  'XYALINB'  : BEGIN
     xyaup_blinear,yainpstr,GROUP=event.top
     widget_control,event.top,set_uvalue=yainpstr,/NO_COPY
     END
  'XYADATA':	BEGIN
     XYAUP_SET,yainpstr,GROUP=event.top
     widget_control,event.top,set_uvalue=yainpstr,/NO_COPY
     END
  'SETDEF':	BEGIN
     itmp = Widget_Message(Dialog_Parent=event.top, $
       /Question,['This option initializes the',$
       'xyaup parameters to their default values.',$
       'Then you must click Set_parameters to run the program.',$
       'Please confirm'],title='xyaup: set defaults')
     IF itmp EQ 'No' THEN RETURN
     tmp = Xop_Defaults('xyaup')
     yainpstr = tmp
     Widget_Control,event.top,Set_Uvalue=yainpstr,/No_Copy
     END
  'XYAHELP':	BEGIN
     WIDGET_CONTROL,event.id,GET_VALUE=eventval,HOURGLASS=1
     CASE eventval OF
       "Xyaup": Xhelp,'xyaup',Group=event.top
       "Yaup":  BEGIN
	 sep = sdep(/ds)
	 hfile = DosPath(Xop_GetEnv('XOP_HOME'))+sep+'doc'+sep+'txt'+sep+'yaup.txt'
         XDISPLAYFILE1, hfile, GROUP=event.top
       END
       ENDCASE
     END
  'XYASHOW':	BEGIN
     WIDGET_CONTROL,event.id,GET_VALUE=eventval
     CASE eventval OF
       "Plot B field": BEGIN
          if (checkfile(yainpstr.bfile) EQ 0) then begin
            itmp = widget_message(dialog_parent=event.top,$
	 	/error,['File '+yainpstr.bfile+$
		' not found', 'Please Set Parameters', 'and run YAUP'])
            return
          endif
	  CASE SDep() OF
	    'WINDOWS': tmpfile = 'xyaupB.inp'
	    'UNIX':    tmpfile = 'xyaup.B.inp'
	    else:      tmpfile = 'xyaupB.inp'
	  ENDCASE
          openw,unit,tmpfile,/GET_LUN
            printf,unit,'1'
            printf,unit,yainpstr.bfile
            CASE sdep() OF
		'WINDOWS': printf,unit,'xyaupB.asc' 
                'UNIX':    printf,unit,'xyaup.B.ascii'
                else:      printf,unit,'xyaupB.asc'
	    ENDCASE
          free_lun,unit
          CASE sdep() OF 
		'WINDOWS': command = 'u2txt < xyaupB.inp' 
                'UNIX':    command = 'u2txt < xyaup.B.inp'
                else:    Message,'Platform not implamented.'
	  ENDCASE
	  Xop_Spawn,command
	  
	  CASE sdep() OF 
		'WINDOWS': tmpfile = 'xyaupB.asc'
		'UNIX':    tmpfile = 'xyaup.B.ascii'
		else:      tmpfile = 'xyaupB.as'
	  ENDCASE
          XPlot,tmpfile,group=event.top,ycol=4, xtitle='z(cm)', $
	    ytitle='B [Tesla]',wtitle='xyaup: Magnetic Field'
          CASE sdep() OF 
	    'WINDOWS': Delete_Files,['xyaupB.asc','xyaupB.inp']
            'UNIX':    Delete_Files,['xyaup.B.ascii','xyaup.B.inp']
            else:      Delete_Files,['xyaupB.asc','xyaupB.inp']
	  ENDCASE
	  END
       "Plot Trajectory": BEGIN
          if (checkfile('undul.traj') EQ 0) then begin
            itmp = widget_message(dialog_parent=event.top,$
	 	/ERROR, ['File undul.traj not found',$
		'Please Set Parameters', 'and run YAUP'])
            return
          endif
	  openw,unit,'xyaup.traj.inp',/GET_LUN
	    printf,unit,'2'
	    printf,unit,'undul.traj'
	    printf,unit,'xyaup.traj.ascii'
	  free_lun,unit
          ;spawn,'arch',arch
	  ;command = '$XOP_HOME/bin/'+arch+'/u2txt < xyaup.traj.inp'
	  command = 'u2txt < xyaup.traj.inp'
          ;print,command  &  spawn,command
          Xop_Spawn,command
          xplot,'xyaup.traj.ascii',group=event.top,ycol=3, $
           xtitle='z(cm)',ytitle='x(cm)',wtitle='xyaup: trajectory'
	  Delete_Files,['xyaup.traj.ascii','xyaup.traj.inp']
	END
       "Show Trajectory Numerical Data": BEGIN
	  if (checkfile('undul.traj') EQ 0) then begin
            itmp = widget_message(dialog_parent=event.top,$
	 	/error, ['File undul.traj not found',$
		'Please Set Parameters', 'and run YAUP'])
            return
          endif
          openw,unit,'xyaup.traj.inp',/GET_LUN
            printf,unit,'2'
            printf,unit,'undul.traj'
            printf,unit,'xyaup.traj.ascii'
          free_lun,unit
          ;spawn,'arch',arch
          ;command = '$XOP_HOME/bin/'+arch+'/u2txt < xyaup.traj.inp'
          ;command = 'source $XOP_RC ; u2txt < xyaup.traj.inp'
          command = 'u2txt < xyaup.traj.inp'
          ;print,command  &  spawn,command
          Xop_Spawn,command
	  Delete_Files,['xyaup.traj.inp']
	  xdisplayfile1,'xyaup.traj.ascii',Group=event.top,/REMOVE,$
		title='trajectory data'
	  END
       "Plot Results": BEGIN
         if checkfile('yaup-0.out') EQ 0 then begin
           itmp = widget_message(dialog_parent=event.top,$
	 	/error,['Yaup output file', $
		'"yaup-0.out" not found','Please run yaup by using the',$
		'"Set Parameters" option'])
	   return
         endif
         tmp = xyaup_out_read('yaup-0.out',yainpstr,Group=event.top)
         widget_control,event.top,set_uvalue=yainpstr

         if (yainpstr.mode EQ 0) then begin
           itmp = widget_message(dialog_parent=event.top,$
	 	/error, ['Option not available','with MODE=0'])
           return
         endif

         if (yainpstr.mode EQ 1) then begin
           xtitle='Energy [eV]'
           widget_control,event.top,sensitive=0
           itmp = wmenu2(['Plot Brightness','Plot Polarization'], $
		Dialog_Parent=event.top)
           widget_control,event.top,sensitive=1
           if itmp EQ 1 then begin
             title='Polarization'
             ytitle='Polarization'
           endif else begin
             title='Brightness spectrum'
             if (yainpstr.d eq 0) then ytitle='ph/sec/0.1%b.w./mrad^2' $
                else ytitle='ph/sec/0.1%b.w./mm^2'
           endelse
           XPLOT,'yaup-0.out',WTITLE='xyaup results: '+yainpstr.TITLE, $
              xtitle=xtitle,ytitle=ytitle,title=title,ycol=4+itmp
           return
         endif

         if (yainpstr.mode EQ 2) or (yainpstr.mode EQ 3) or $
           (yainpstr.mode EQ 6) or (yainpstr.mode EQ 7) then begin
           widget_control,event.top,sensitive=0
           IF ((yainpstr.mode EQ 2) OR (yainpstr.mode EQ 3)) THEN BEGIN
             polarization = wmenu2(['Plot Brightness/Power',$
               'Plot Polarization'], Dialog_Parent=event.top) 
	   ENDIF ELSE polarization = 0
           widget_control,event.top,sensitive=1
           widget_control,/hourglass
           ;window,/free
	   data = {polarization:polarization}
	   XWindow,Buffer="xyaup_surfplot,'yaup-0.out'"+$
		",polarization=data.polarization",Data=data,/Edit,$
		WTitle='YAUP results'
           return
         endif

         if (yainpstr.mode EQ 4) then begin
           xtitle='Energy [eV]'
           ytitle='ph/sec/0.1%b.w.'
           title='Flux spectrum'
           XPLOT,'yaup-0.out', $
              xtitle=xtitle,ytitle=ytitle,title=title, $
	      Wtitle='xyaup results: '+yainpstr.TITLE
           return
         endif

         if (yainpstr.mode EQ 5) then begin
           xtitle='Energy [eV]'
           title='Spectral radiated power'
           if (yainpstr.d eq 0) then ytitle='Watts/eV/mrad^2' $
               else ytitle='Watts/eV/mm^2'
           XPLOT,'yaup-0.out',WTITLE='xyaup results: '+yainpstr.TITLE, $
              xtitle=xtitle,ytitle=ytitle,title=title
           return
         endif
       end
    
       "Show Results Numerical Data": begin
         if checkfile('yaup-0.out') EQ 0 then begin
           itmp = widget_message(dialog_parent=event.top,$
	 	/error,['Yaup output file', $
		'"yaup-0.out" not found','Please run yaup by using the',$
		'"Set Parameters" option'])
	   return
         endif
	 XDISPLAYFILE1,'yaup-0.out', GROUP=event.top, TITLE= $
	  'YAUP results: yaup-0.out'
	end
     ENDCASE
     END
  ENDCASE
END
;
;==============================================================================
;
PRO XYAUP, GROUP=group, InputFile=inputFile

Forward_Function xop_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XYAUP: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

if xregistered('xyaup') then return

;now in xop_ini.pro:  if not(keyword_set(group)) then xop_wd,/default

;
; create the base (xyabase) widget
;
xyabase=WIDGET_BASE(/COLUMN,TITLE='XYaup '+xyaup_version(),MBAR=wMenuBar)

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='XYaup input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')

  wtmp0 = widget_button(wFile,VALUE='Yaup file', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load input file...',UValue='XYALOAD')
  wtmp = widget_button(wtmp0,VALUE='Load output file...',UValue='XYAOUTLD')
  wtmp = widget_button(wtmp0,VALUE='Write input file...',UValue='XYASAVE')
  wtmp = widget_button(wtmp0,VALUE='Write output file...',UValue='XYAOUTSV')

  wtmp = widget_button(wFile,VALUE='Save files for XOP/Optics', $
	UVALUE='XYATRAN')
  wtmp = widget_button(wFile,VALUE='Quit and preserve files', $
	UVALUE='XYASOFTQUIT',/SEPARATOR)
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='XYAQUIT')


wBField = widget_button(wMenuBar,VALUE='BField', /MENU)
  wtmp = widget_button(wBField,VALUE='from ASCII file', UVALUE='XYASCII')
  wtmp = widget_button(wBField,VALUE='from XBFIELD preprocessor', $
	UVALUE='XYABF')
  wtmp = widget_button(wBField,VALUE='from Liner B field', UVALUE='XYALINB')

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='XYADATA')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wtmp = widget_button(wResults,VALUE='Plot B field',UVALUE='XYASHOW')
  wtmp = widget_button(wResults,VALUE='Plot Trajectory',UVALUE='XYASHOW')
  wtmp = widget_button(wResults,VALUE='Show Trajectory Numerical Data',$
	UVALUE='XYASHOW')
  wtmp = widget_button(wResults,VALUE='Plot Results',UVALUE='XYASHOW')
  wtmp = widget_button(wResults,VALUE='Show Results Numerical Data',$
	UVALUE='XYASHOW')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='Yaup', UVALUE='XYAHELP')
  wtmp = widget_button(wHelp,VALUE='Xyaup', UVALUE='XYAHELP')
;

wtmp = widget_button(xyabase,VALUE='Set Parameters', UVALUE='XYADATA')

if sdep() EQ 'WINDOWS' then font = 'VERDANA*BOLD*ITALIC*24' else $
  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( xyabase, FONT=font, VALUE='YAUP')

junk = WIDGET_LABEL( xyabase, FONT=font, VALUE=$
	'     Tapered Undulator Radiation     ')

WIDGET_CONTROL,/REALIZE,xyabase

str = xop_defaults('xyaup')

IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile, Group=xyabase)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str
    Copy_Structure,str_par, tmp, Group=xyabase
    str = tmp
  ENDIF
ENDIF

Widget_Control, xyabase, Set_Uvalue=str, /NO_COPY
XMANAGER,'xyaup',xyabase,GROUP_LEADER=group
end

