;
; NAME:
;	XTC_SET
; PURPOSE:
;	Pop-up dialog widget for entering information for progam TC.
; CATEGORY:
;	GAMS Index N1: Data handling; input, output
; CALLING SEQUENCE:
;	XTC_SET, data [,ACTION=action, GROUP=group]
; INPUTS:
;	data:	Anonymous structure. Parameters are contained in data.dat1
; KEYWORD PARAMETERS:
;	ACTION:	Indicates whether data changed. ACTION='DO' if data
;		are changed, and ACTION='DONT' if data are unchanged
;		(user canceled the action).
;	GROUP:	Group leader for this widget.  If the group leader dies,
;		then this widget dies as well.
; OUTPUTS:
;	data:	Anonymous structure.  Result are overwritten in data.dat1
; COMMON BLOCKS:
;	None
; PROCEDURE:
;	The routine uses compound widgets and anonymous structures to pass
;	the data to the pop-up event handler.  The user is not forced to
;	press the carriage return after each entry.  The input fields are
;	still checked for valid entries and valid ranges.
; EXAMPLE:
;	XTC_SET, data, ACTION=action, GROUP=event.top
;	IF action eq 'DONT' THEN GOTO, LBL900 ; cancel
;	<save input file (tc.inp) and execute TC>
; MODIFICATION HISTORY:
; 	Written by:	Roger J. Dejus, XFD/APS, May, 1996.
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 09-MAY-1996     | RJD   | First version.
; ----------------+-------+-----------------------------------------------------
; 21-MAY-1996     | RJD   | First official release.
; ----------------+-------+-----------------------------------------------------
; 15-NOV-1997     | RJD   | Modified to run version 1.9 of the tuning curve
;                 |       | program.
; ----------------+-------+-----------------------------------------------------
; 26-NOV-1998     | MSR   | XTC_SET under IDL5.1 crashes
;                 |       | at the second "cancel". The problem was
;                 |       | related to the "handlers" and I did not
;                 |       | understand why. Solved by using pointers
;                 |       | instead. se /Modal in Widget_Base. Added catch.
; ----------------+-------+-----------------------------------------------------
; 04-JAN-2003     | MSR   | Reuse XTC structure to avoid strange conflict.
; ----------------+-------+-----------------------------------------------------
; 24-JAN-2008     | MSR   | Remove /NO_COPY keywords to avoid crashed in
;                 |       | case of error.
; ----------------+-------+-----------------------------------------------------
; 14-AUG-2008     | RJD   | Replaced all statements related to "HANDLES" with
;                 |       | pointers. Reintroduced the /No_Copy keyword.
;                 |       | Placed the /FLOATING keywords on the Widget_Base call.
;                 |       | The cleanup routine is currently not used
;                 |       | for the modal widget. It's kept for future reference.
;                 |       | Converted "ik" from integer array of one element to integer.
;                 |       | The named structure "Xtc" would otherwise assume incorrect
;                 |       | data structure and generate an error. This error was
;                 |       | present in earlier version of IDL. However, don't know
;                 |       | when it started to fail.
;                 |       | Developed and tested with IDL v7.01 on Sun Solaris 10,
;                 |       | and v7.03 on Windows XP.
; ----------------+-------+-----------------------------------------------------
; 30-JAN-2014     | RJD   | Changed format of sigx and sigy to use 4 decimal places.
;                 |       | Removed trailing white space.
;                 |       | Removed /Modal keyword from base widget because it will
;                 |       | block all other widgets including the help widget.
;                 |       | Added group keyword to help event.
;                 |       | Prepared for XOP v2.4 embedded with IDL v8.3.
; ----------------+-------+-----------------------------------------------------
;-

FORWARD_FUNCTION sdep

;
;====================================================================
;
PRO XTC_SET_EVENT, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XTC_SET_EVENT: Error caught: '+!error_state.msg)
   IF N_Elements(info) NE 0 THEN Widget_Control, event.top, Set_UValue=info, /No_Copy
   RETURN
ENDIF

WIDGET_CONTROL, event.top, GET_UVALUE=info
name = TAG_NAMES(event, /STRUCTURE_NAME)

ds = sdep(/ds)		; system dependent directory separator

IF name eq 'WIDGET_BUTTON' THEN BEGIN
  WIDGET_CONTROL, event.id, GET_UVALUE=uval
  CASE uval OF
    'HELP': BEGIN
    hfile = Xop_GetEnv('XOP_HOME') +ds +'doc' +ds +'txt'+ds+'tc_par.txt'
    XDISPLAYFILE1, hfile, GROUP=event.top
    END

    'CANCEL': BEGIN
    *(info.ptr_action) = 'DONT'
    WIDGET_CONTROL, event.top, /DESTROY
    END

    'ACCEPT': BEGIN
;  Title
    WIDGET_CONTROL, info.id_title, GET_VALUE=title & title = title(0)
    rtxt = ' is out of range.'

;  Machine parameters
;  Use on_ioerror to trap conversion errors
    ON_IOERROR, ENERGY_ERROR
    WIDGET_CONTROL, info.id_energy, GET_VALUE=energy & energy = energy(0)
    attl = 'Edit Energy' & atxt = 'Energy (GeV): New Value'
    LBL_ENERGY: energy = FLOAT(energy)
    IF (energy le 0.0) or (energy gt 100.0) THEN BEGIN ; GeV
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Energy Error',STRING(energy, FORMAT='(f8.2)') +rtxt])
      xedit, energy, TITLE=attl, TEXT=atxt
      GOTO, LBL_ENERGY
    ENDIF
    ON_IOERROR, CUR_ERROR
    WIDGET_CONTROL, info.id_cur, GET_VALUE=cur & cur = cur(0)
    attl = 'Edit Current' & atxt = 'Current (mA): New Value'
    LBL_CUR: cur = FLOAT(cur)
    IF (cur le 0.0) or (cur gt 10000.0) THEN BEGIN ; mA
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Current Error',STRING(cur, FORMAT='(f8.2)') +rtxt])
      xedit, cur, TITLE=attl, TEXT=atxt
      GOTO, LBL_CUR
    ENDIF
    ON_IOERROR, SIGE_ERROR
    WIDGET_CONTROL, info.id_sige, GET_VALUE=sige & sige = sige(0)
    attl = 'Edit Energy Spread' & atxt = 'Energy Spread: New Value'
    LBL_SIGE: sige = FLOAT(sige)
    IF (sige le 0.0) or (sige gt 1.0) THEN BEGIN ;
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Energy Spread Error',STRING(sige, FORMAT='(f8.5)') +rtxt])
      xedit, sige, TITLE=attl, TEXT=atxt
      GOTO, LBL_SIGE
    ENDIF

;  Beam Parameters
    ON_IOERROR, SIGX_ERROR
    WIDGET_CONTROL, info.id_sigx, GET_VALUE=sigx & sigx = sigx(0)
    attl = 'Edit Sigmax' & atxt = 'Sigx (mm): New Value'
    LBL_SIGX: sigx = FLOAT(sigx)
    IF (sigx le 0.0) or (sigx gt 10.0) THEN BEGIN ; mm
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Sigmax Error',STRING(sigx, FORMAT='(f8.4)') +rtxt])
      xedit, sigx, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX
    ENDIF
    ON_IOERROR, SIGY_ERROR
    WIDGET_CONTROL, info.id_sigy, GET_VALUE=sigy & sigy = sigy(0)
    attl = 'Edit Sigmay' & atxt = 'Sigy (mm): New Value'
    LBL_SIGY: sigy = FLOAT(sigy)
    IF (sigy le 0.0) or (sigy gt 10.0) THEN BEGIN ; mm
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Sigmay Error',STRING(sigy, FORMAT='(f8.4)') +rtxt])
      xedit, sigy, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY
    ENDIF
    ON_IOERROR, SIGX1_ERROR
    WIDGET_CONTROL, info.id_sigx1, GET_VALUE=sigx1 & sigx1 = sigx1(0)
    attl = "Edit Sigmax'" & atxt = 'Sigx1 (mrad): New Value'
    LBL_SIGX1: sigx1 = FLOAT(sigx1)
    IF (sigx1 le 0.0) or (sigx1 gt 10.0) THEN BEGIN ; mrad
      junk = WIDGET_MESSAGE(/ERROR, $
        ["Sigmax' Error",STRING(sigx1, FORMAT='(f8.4)') +rtxt])
      xedit, sigx1, TITLE=attl, TEXT=atxt, FORMAT='(f10.4)' & GOTO, LBL_SIGX1
    ENDIF
    ON_IOERROR, SIGY1_ERROR
    WIDGET_CONTROL, info.id_sigy1, GET_VALUE=sigy1 & sigy1 = sigy1(0)
    attl = "Edit Sigmay'" & atxt = 'Sigy1 (mrad): New Value'
    LBL_SIGY1: sigy1 = FLOAT(sigy1)
    IF (sigy1 le 0.0) or (sigy1 gt 10.0) THEN BEGIN ; mrad
      junk = WIDGET_MESSAGE(/ERROR, $
        ["Sigmay' Error",STRING(sigy1, FORMAT='(f8.4)') +rtxt])
      xedit, sigy1, TITLE=attl, TEXT=atxt, FORMAT='(f10.4)' & GOTO, LBL_SIGY1
    ENDIF

;  Undulator Parameters
    ON_IOERROR, PERIOD_ERROR
    WIDGET_CONTROL, info.id_period, GET_VALUE=period & period = period(0)
    attl = 'Edit Period Length' & atxt = 'Period (cm): New Value'
    LBL_PERIOD: period = FLOAT(period)
    IF (period le 0.0) or (period gt 100.0) THEN BEGIN ; cm
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Period Length Error',STRING(period, FORMAT='(f8.2)') +rtxt])
      xedit, period, TITLE=attl, TEXT=atxt & GOTO, LBL_PERIOD
    ENDIF
    ON_IOERROR, NP_ERROR
    WIDGET_CONTROL, info.id_np, GET_VALUE=np & np = np(0)
    attl = 'Edit Number of Periods' & atxt = 'N: New Value'
    LBL_NP: np = FIX(np)
    IF (np lt 1) or (np gt 1000) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Number of Periods Error',STRING(np, FORMAT='(i8)') +rtxt])
      xedit, np, TITLE=attl, TEXT=atxt & GOTO, LBL_NP
    ENDIF

;  Scan Parameters
    ON_IOERROR, EMIN_ERROR
    WIDGET_CONTROL, info.id_emin, GET_VALUE=emin & emin = emin(0)
    attl = 'Edit Emin' & atxt = 'Emin (eV): New Value'
    LBL_EMIN: emin = FLOAT(emin)
    IF (emin le 0.0) or (emin gt 1.0e5) THEN BEGIN ; eV
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Emin Error',STRING(emin, FORMAT='(f10.1)') +rtxt])
      xedit, emin, TITLE=attl, TEXT=atxt & GOTO, LBL_EMIN
    ENDIF
    ON_IOERROR, EMAX_ERROR
    WIDGET_CONTROL, info.id_emax, GET_VALUE=emax & emax = emax(0)
    attl = 'Edit Emax' & atxt = 'Emax (eV): New Value'
    LBL_EMAX: emax = FLOAT(emax)
    IF (emax le 0.0) or (emax gt 1.0e6) THEN BEGIN ; eV
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Emax Error',STRING(emax, FORMAT='(f10.1)') +rtxt])
      xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    ENDIF
    IF (emax le emin) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Emax Error',STRING(emax, FORMAT='(f10.1)') +' less than emin.'])
      xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    ENDIF
    ON_IOERROR, N_ERROR
    WIDGET_CONTROL, info.id_n, GET_VALUE=n & n = n(0)
    attl = 'Edit Number of Energy-Points' & atxt = 'N: New Value'
    LBL_N: n = FIX(n)
    IF (n lt 1) or (n gt 1000) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Number of Energy-Points Error',STRING(n, FORMAT='(i8)') +rtxt])
      xedit, n, TITLE=attl, TEXT=atxt & GOTO, LBL_N
    ENDIF

    ON_IOERROR, IHMIN_ERROR
    WIDGET_CONTROL, info.id_ihmin, GET_VALUE=ihmin & ihmin = ihmin(0)
    attl = 'Edit Hmin' & atxt = 'hmin: New Value'
    LBL_IHMIN: ihmin = FIX(ihmin)
    IF (ihmin lt 1) or (ihmin gt 100) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Ihmin Error',STRING(ihmin, FORMAT='(i8)') +rtxt])
      xedit, ihmin, TITLE=attl, TEXT=atxt & GOTO, LBL_IHMIN
    ENDIF
    ON_IOERROR, IHMAX_ERROR
    WIDGET_CONTROL, info.id_ihmax, GET_VALUE=ihmax & ihmax = ihmax(0)
    attl = 'Edit Hmax' & atxt = 'hmax: New Value'
    LBL_IHMAX: ihmax = FIX(ihmax)
    IF (ihmax lt 1) or (ihmax gt 100) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Ihmax Error',STRING(ihmax, FORMAT='(i8)') +rtxt])
      xedit, ihmax, TITLE=attl, TEXT=atxt & GOTO, LBL_IHMAX
    ENDIF
    IF (ihmax lt ihmin) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
        ['Hmax Error',STRING(ihmax, FORMAT='(i8)') +' less than ihmin.'])
      xedit, ihmax, TITLE=attl, TEXT=atxt & GOTO, LBL_IHMAX
    ENDIF
    ON_IOERROR, IHSTEP_ERROR
    WIDGET_CONTROL, info.id_ihstep, GET_VALUE=ihstep & ihstep = ihstep(0)
    attl = 'Edit Hstep' & atxt = 'hstep: New Value'
    LBL_IHSTEP: ihstep = FIX(ihstep)
    IF (ihstep lt 1) or (ihstep gt 100) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
      ['Hstep Error',STRING(ihstep, FORMAT='(i8)') +rtxt])
      xedit, ihstep, TITLE=attl, TEXT=atxt & GOTO, LBL_IHSTEP
    ENDIF

;  Number of energy points for peak search
    ON_IOERROR, NEKS_ERROR
    WIDGET_CONTROL, info.id_neks, GET_VALUE=neks & neks = neks(0)
    attl = 'Edit Neks' & atxt = 'neks: New Value'
    LBL_NEKS: neks = FIX(neks)
    IF (neks ne 0) and (neks lt 100) or (neks gt 10000) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
      ['Neks Error',STRING(neks, FORMAT='(i8)') +rtxt])
      xedit, neks, TITLE=attl, TEXT=atxt & GOTO, LBL_NEKS
    ENDIF

;  Cancel conversion errors
    ON_IOERROR, NULL

;  Mode
    WIDGET_CONTROL, info.id_mode, GET_VALUE=ihel
;  Method
    WIDGET_CONTROL, info.id_method, GET_VALUE=method & method = method -1
    if method eq -1 then method = 3
;  Printing
    WIDGET_CONTROL, info.id_print, GET_VALUE=ik & ik = ik(0)

;  Run mode
    WIDGET_CONTROL, info.id_run_mode, GET_VALUE=run_mode
    IF run_mode eq 0 THEN run_mode_name = 'foreground' ELSE run_mode_name = 'background'

   dat1 = {Xtc, title:title, energy:energy, cur:cur, sige:sige, text_machine:'', $
	    sigx:sigx, sigy:sigy, sigx1:sigx1, sigy1:sigy1, text_beam:'', $
	    period:period, np:np, text_undulator:'', $
	    emin:emin, emax:emax, n:n, text_energy:'', $
	    ihmin:ihmin, ihmax:ihmax, ihstep:ihstep, text_harm:'', $
	    ihel:ihel, method:method, ik:ik, neks:neks, text_parm:'', $
	    run_mode_name:run_mode_name}
    *(info.ptr_action) = 'DO'
    *(info.ptr_data) = dat1
    WIDGET_CONTROL, event.top, /DESTROY
    RETURN

;  Error Handling
;  Machine Parameters
    ENERGY_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Energy Error',!err_string])
    xedit, energy, TITLE=attl, TEXT=atxt & GOTO, LBL_ENERGY
    END
    CUR_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Current Error',!err_string])
    xedit, cur, TITLE=attl, TEXT=atxt & GOTO, LBL_CUR
    END
    SIGE_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Energy Spread Error',!err_string])
    xedit, sige, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGE
    END

;  Beam Parameters
    SIGX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Sigmax Error',!err_string])
    xedit, sigx, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX
    END
    SIGY_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Sigmay Error',!err_string])
    xedit, sigy, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY
    END
    SIGX1_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ["Sigmax' Error",!err_string])
    xedit, sigx1, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX1
    END
    SIGY1_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ["Sigmay' Error",!err_string])
    xedit, sigy1, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY1
    END

;  Undulator Parameters
    PERIOD_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Period Length Error',!err_string])
    xedit, period, TITLE=attl, TEXT=atxt & GOTO, LBL_PERIOD
    END
    NP_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Number of Periods Error',!err_string])
    xedit, np, TITLE=attl, TEXT=atxt & GOTO, LBL_NP
    END

;  Scan Parameters
    EMIN_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Emin Error',!err_string])
    xedit, emin, TITLE=attl, TEXT=atxt & GOTO, LBL_EMIN
    END
    EMAX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Emax Error',!err_string])
    xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    END
    N_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Number of Energy-Points Error',!err_string])
    xedit, n, TITLE=attl, TEXT=atxt & GOTO, LBL_N
    END

    IHMIN_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Hmin Error',!err_string])
    xedit, ihmin, TITLE=attl, TEXT=atxt & GOTO, LBL_IHMIN
    END
    IHMAX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Hmax Error',!err_string])
    xedit, ihmax, TITLE=attl, TEXT=atxt & GOTO, LBL_IHMAX
    END
    IHSTEP_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Hstep Error',!err_string])
    xedit, ihstep, TITLE=attl, TEXT=atxt & GOTO, LBL_IHSTEP
    END

;  Number of energy points for peak search
    NEKS_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Neks Error',!err_string])
    xedit, neks, TITLE=attl, TEXT=atxt & GOTO, LBL_NEKS
    END

    END ; Accept
  ENDCASE

ENDIF ELSE BEGIN
  WIDGET_CONTROL, event.id, GET_UVALUE=uval
  CASE uval OF
    'MODE': BEGIN ; nothing
    END

    'METHOD': BEGIN ; nothing
    END

    'PRINTING': BEGIN ; nothing
    END

    'RUN_MODE': BEGIN ; nothing
    END

  ENDCASE ; uval
ENDELSE

END ; XTC_SET_EVENT

;
;====================================================================
;
PRO XTC_SET, data, ACTION=action, GROUP=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XTC_SET: Error caught: '+!error_state.msg)
   RETURN
ENDIF

;  Determine screen size (for future reference)
DEVICE, GET_SCREEN_SIZE=ssz
xsz = ssz(0)
ysz = ssz(1)

;  Definitions and labels
XTC_SET_LBL1='Xtc input parameters'

;setbase = WIDGET_BASE(TITLE=XTC_SET_LBL1, COLUMN=1, /Modal, /Floating, Group=group)	; do not use /Modal because it will block all other widgets
setbase = WIDGET_BASE(TITLE=XTC_SET_LBL1, COLUMN=1, /Floating, Group=group)

dat1 = data.dat1
;  Control buttons
b40 = WIDGET_BASE(setbase, ROW=1, /FRAME)
bt  = WIDGET_BUTTON(b40, VALUE='Accept', UVALUE='ACCEPT')
bt  = WIDGET_BUTTON(b40, VALUE='Cancel', UVALUE='CANCEL')
bt  = WIDGET_BUTTON(b40, VALUE='Help', UVALUE='HELP')

;  Title information
b0 = WIDGET_BASE(setbase, ROW=1, /FRAME)
bs = WIDGET_BASE(b0, COLUMN=1)
fd_title = CW_FIELD(bs, TITLE='Title', VALUE=dat1.title, /ROW, XSIZE=80)

;  Machine Parameters
b1 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b1, VALUE='Machine Parameters  ')
bs = WIDGET_BASE(b1, COLUMN=3)
fd_energy = CW_FIELD(bs, TITLE='Energy (GeV)', /COLUMN, $
	    VALUE=STRING(dat1.energy, FORMAT='(f10.2)'))
fd_cur    = CW_FIELD(bs, TITLE='Current (mA)', /COLUMN, $
	    VALUE=STRING(dat1.cur, FORMAT='(f10.2)'))
fd_sige   = CW_FIELD(bs, TITLE='Energy Spread', /COLUMN, $
	    VALUE=STRING(dat1.sige, FORMAT='(f10.5)'))

;  Beam Parameters
b2 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b2, VALUE='Beam Parameters     ')
bs = WIDGET_BASE(b2, COLUMN=4)
fd_sigx  = CW_FIELD(bs, TITLE='Sigx (mm)', /COLUMN, $
	   VALUE=STRING(dat1.sigx, FORMAT='(f10.4)'))
fd_sigy  = CW_FIELD(bs, TITLE='Sigy (mm)', /COLUMN, $
	   VALUE=STRING(dat1.sigy, FORMAT='(f10.4)'))
fd_sigx1 = CW_FIELD(bs, TITLE='Sigx1 (mrad)', /COLUMN, $
	   VALUE=STRING(dat1.sigx1, FORMAT='(f10.4)'))
fd_sigy1 = CW_FIELD(bs, TITLE='Sigy1 (mrad)', /COLUMN, $
	   VALUE=STRING(dat1.sigy1, FORMAT='(f10.4)'))

;  Undulator Parameters
b3 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b3, VALUE='Undulator Parameters')
bs = WIDGET_BASE(b3, COLUMN=2)
fd_period = CW_FIELD(bs, TITLE='Period length (cm)', /COLUMN, $
	    VALUE=STRING(dat1.period, FORMAT='(f10.2)'))
fd_np     = CW_FIELD(bs, TITLE='Number of periods', /COLUMN, $
            VALUE=STRING(dat1.np, FORMAT='(i10)'))

;  Scan parameters
;  Energies
b4 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b4, VALUE='Scan Parameters     ')
bs = WIDGET_BASE(b4, ROW=2)
fd_emin = CW_FIELD(bs, TITLE='E1 minimum energy (eV)', /COLUMN, $
	  VALUE=STRING(dat1.emin, FORMAT='(f10.1)'))
fd_emax = CW_FIELD(bs, TITLE='E1 maximum energy (eV)', /COLUMN, $
	  VALUE=STRING(dat1.emax, FORMAT='(f10.1)'))
fd_n    = CW_FIELD(bs, TITLE='Number of energy-points', /COLUMN, $
	  VALUE=STRING(dat1.n, FORMAT='(i10)'))
;  Harmonics
bs = WIDGET_BASE(bs, COLUMN=4)
fd_ihmin  = CW_FIELD(bs, TITLE='Minimum harmonic #', /COLUMN, $
	    VALUE=STRING(dat1.ihmin, FORMAT='(i10)'))
fd_ihmax  = CW_FIELD(bs, TITLE='Maximum harmonic #', /COLUMN, $
	    VALUE=STRING(dat1.ihmax, FORMAT='(i10)'))
fd_ihstep = CW_FIELD(bs, TITLE='Harmonic step size', /COLUMN, $
	    VALUE=STRING(dat1.ihstep, FORMAT='(i10)'))

;  Menus
b8 = WIDGET_BASE(setbase, ROW=1, /FRAME)

;  Mode
initial_state = dat1.ihel
mode_names = ['Regular planar undulator', $
              'Helical undulator']
bg_mode = CW_BGROUP(b8, mode_names, LABEL_TOP='Mode', SET_VALUE=initial_state, $
	            UVALUE='MODE', IDS=mode_ids, /COLUMN, /EXCLUSIVE, /FRAME, /NO_RELEASE)

;  Method
if ((dat1.method eq 0) or (dat1.method eq 1)) then begin
  initial_state = 1
endif else begin
  initial_state = 0
endelse
method_names = ['Finite-N', $
	        'Infinite-N w/ convolution']
bg_method = CW_BGROUP(b8, method_names, LABEL_TOP='Method', SET_VALUE=initial_state, $
	              UVALUE='METHOD', IDS=method_ids, /COLUMN, /EXCLUSIVE, /FRAME, $
		      /NO_RELEASE)

;  Run mode
run_mode_names = ['foreground', $
                  'background']
index = WHERE(run_mode_names eq dat1.run_mode_name)
initial_state = index(0)
bg_run_mode = CW_BGROUP(b8, run_mode_names, LABEL_TOP='Run Mode', $
	      SET_VALUE=initial_state, UVALUE='RUN_MODE', $
	      IDS=run_mode_ids, /COLUMN, /EXCLUSIVE, /FRAME, /NO_RELEASE)

;  Printing
print_names = ['Save K-value/power?']
print_label = 'Optional Output'
if dat1.ik eq 0 then $
  bg_print = CW_BGROUP(b8, print_names, LABEL_TOP=print_label, $
	     SET_VALUE=[0],UVALUE='PRINTING', /COLUMN, /NONEXCLUSIVE, /FRAME) $
else $
  bg_print = CW_BGROUP(b8, print_names, LABEL_TOP=print_label, $
	     SET_VALUE=[1], UVALUE='PRINTING', /COLUMN, /NONEXCLUSIVE, /FRAME)

;  Number of energy points for peak search
fd_neks = CW_FIELD(b8, TITLE='Intrinsic NEKS', /COLUMN, $
          VALUE=STRING(dat1.neks, FORMAT='(i10)'))

ids = {mode_ids:mode_ids, method_ids:method_ids}	; for future use
ptr_action = ptr_new('DONT')
ptr_data = ptr_new(/Allocate_Heap)

info = {id_title:fd_title, id_energy:fd_energy, id_cur:fd_cur, id_sige:fd_sige, $
	id_sigx:fd_sigx, id_sigy:fd_sigy, id_sigx1:fd_sigx1, id_sigy1:fd_sigy1, $
	id_period:fd_period, id_np:fd_np, $
	id_emin:fd_emin, id_emax:fd_emax, id_n:fd_n, $
	id_ihmin:fd_ihmin, id_ihmax:fd_ihmax, id_ihstep:fd_ihstep, $
	id_mode:bg_mode, id_method:bg_method, id_print:bg_print, id_neks:fd_neks, $
	id_run_mode:bg_run_mode, $
	ptr_action:ptr_action, ptr_data:ptr_data}

WIDGET_CONTROL, setbase, /REALIZE

WIDGET_CONTROL, setbase, SET_UVALUE=info, /NO_COPY

XMANAGER, 'XTC_SET', setbase, GROUP_LEADER=group	; use blocking, not necessary to call a cleanup routine as all cleanup and assignments follows here

action = *ptr_action
IF action eq 'DO' THEN BEGIN
  dat1 = *ptr_data
  data = {dat1:dat1}
ENDIF

; Free pointers
ptr_free, ptr_action, ptr_data

END ; XTC_SET
