;+
;====================================  Xtc   ===================================
;
;   Xtc is an widget based graphical user interface to calculate on-axis
;   brilliance tuning curves for an ideal-field undulator insertion device using
;   the Bessel function approximation (regular planar device or helical device).
;   The calculation is performed by the program TC ("tuning curves") which is
;   executed from the XTC interface. The effects of the particle beam
;   emittance and beam energy spread are taken into account.
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xtc input parameters: This option allows to save the current
;               input parameters to a file for later use. It also allows
;               to save the current parameters as defaults for being
;               used when the application is initialized. For this
;               case the filename is "application".xop (where
;               "application" is the name of the current XOP
;               application) and it is written in the directory given
;               by the XOP_DEFAULTS_DIR environment variable (which
;               must be set). The parameter file is an ASCII file and
;               it may be edited with care.
;
;    TC file: This option allows to load and write the current parameter
;               and result files for further calculations. The file formats
;               are those used by the program TC. Therefore the input files
;               may be used to run the program TC without this interface.
;               Alternatively one may load TC input files created outside
;               of the XTC application to be executed herein.
;
;    Diplay file: Select to display an arbitrary ASCII file.
;
;    Quit: Exit from the program and remove any unsaved files created by TC.
;
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;               The same function is obtained by pressing the "Set Parameters"
;               button in the main XTC window.
;               Please refer to the information under the HELP button for
;               a complete description of all the parameters. After pressing
;               the ACCEPT button the program TC runs with the current parameters.
;
;  Set Defaults: Sets the default parameters.
;
;
;  Show: Display results
;    Plot ...: Plots different results from the program TC.
;               Plot "All" interconnects data points between different harmonics.
;               Plot "<Quantity>" shows individual harmonics without interconnects.
;               The x- and y-ranges are chosen to display the region of interest.
;               All quantities (except the energy down shift) are plotted versus
;               the emittance down shifted energy.
;
;    View Results: Displays the TC results file "tc.out," in a scrollable window.
;               The file may be printed from this window.
;
;    View Logfile: Displays the TC log file "tc.log," which shows the progress of
;               of the calculations.
;
;
;  Help:   Shows the TC help and the XTC help (this text).
;
;
;  COPYRIGHT:
;       XTC belongs to the XOP package and it is distributed as part therof.
;       PLEASE REFER TO THE XOP COPYRIGHT NOTICE FOR MORE DETAILS.
;
;  REFERENCE:
;       Published calculations made with XOP should reference:
;
;       M. Sanchez del Rio and R. J. Dejus
;       "Status of XOP: an x-ray optics software toolkit"
;       SPIE Proceedings Vol. 5536 (2004) pp.171-174
;
;       http://dx.doi.org/10.1117/12.560903
;
;
;  LAST MODIFICATION: RJD 2014-01-30
;
;-
; -----------------------------------------------------------------------
;       Author: Roger J. Dejus (dejus@aps.anl.gov), XFD/APS, May, 1996.
;       MODIFICATION HISTORY:
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 09-MAY-1996     | RJD   | First version.
; ----------------+-------+-----------------------------------------------------
; 21-MAY-1996     | RJD   | First official release, v1.2.
; ----------------+-------+-----------------------------------------------------
; 08-SEPT-1997    | RJD   | Modified to run version 1.6 of the tuning curve
;                 |       | program. Current release is v1.3.
; ----------------+-------+-----------------------------------------------------
; 18-NOV-1997     | RJD   | Modified to run version 1.9 of the tuning curve
;                 |       | program. Changed FILTER keyword in PICKFILE so that
;                 |       | it is undefined for operating systems other than
;                 |       | UNIX. Added the keyword DIALOG_PARENT to
;                 |       | WIDGET_MESSAGE. Sets working directory if the GROUP
;                 |       | keyword is not set.
;                 |       | Current release is v1.4.
; ----------------+-------+-----------------------------------------------------
; 26-NOV-1998     | MSR   | Using the new (xop2.0) input file system.
;                 |       | Changed the menu look to adapt to xop "standards".
;                 |       | Added xtc_version function. Use of Xop_Spawn.
;                 |       | Backgroud task in a new window.
;                 |       | "Advanced" use of Xplot instead of XWindow.
;                 |       | Current release is v1.5Beta1.
; ----------------+-------+-----------------------------------------------------
; 28-JAN-2008     | MSR   | Remove /NO_COPY keywords to avoid crashed in
;                 |       | case of error. Updated doc.
; ----------------+-------+-----------------------------------------------------
; 14-AUG-2008     | RJD   | Replaced all statements related to "HANDLES" with
;                 |       | pointers. Reintroduced the /No_Copy keyword as all
;                 |       | errors are caught properly. Added the cleanup routine.
;                 |       | Defined window size for xplot window.
;                 |       | Developed and tested with IDL v7.01 on Sun Solaris 10,
;                 |       | and v7.03 on Windows XP.
; ----------------+-------+-----------------------------------------------------
; 30-JAN-2014     | RJD   | Minor language edits. Added display of logfile.
;                 |       | Added the tee command to save logfile when executed.
;                 |       | Changed default size of plot windows. Use 4 decimal
;                 |       | in xtc_write_dat for sigx and sigy.
;                 |       | Changed calculation of screen size.
;                 |       | Added menu item "Display file" to display an arbitrary
;                 |       | ASCII file.
;                 |       | Use /No_block in call to XMANAGER so that blocking
;                 |       | in xtc_set.pro works correctly (with No_block not set).
;                 |       | Prepared for XOP v2.4 embedded with IDL v8.3.
;                 |       | Current release is v1.6.
; ----------------+-------+-----------------------------------------------------
;
;-

FORWARD_FUNCTION sdep

;
;==============================================================================
;
FUNCTION xtc_version
RETURN,'1.6'
END ; xtc_version

;
;==============================================================================
;
FUNCTION xtc_read_dat, struct, FNAME=fname

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: Error reading TC input data file'
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XTC_READ_DAT: Error caught: Error reading TC input data file')
   RETURN,error_status
ENDIF

if n_elements(fname) eq 0 then fname = 'tc.inp'

inp = XOP_DEFAULTS('xtc') ; define structure
openr, unit, fname, /get_lun
readf, unit, inp
free_lun, unit
struct = inp ; make assignment

return, error_status
END ; xtc_read_dat

;
;==============================================================================
;
FUNCTION xtc_read2d_plt, a, nd, nh, np, title, FNAME=fname

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XTC_READ2D_PLT: Error caught: Error reading TC plot file')
   RETURN,error_status
ENDIF

if n_elements(fname) eq 0 then fname = 'tc.out'

a = rascii(fname, npoints=nd, header=head, error_status=error_status)
if error_status ne 0 then begin
  Message, 'Error reading TC plot file'         ; need Message to jump to the cacth error handler
  ;print, 'XTC_READ2D_PLT: Error reading TC plot file'
endif

; find number of harmonics and number of points/harmonic
head = reform(head)
i    = strmatch1(' Nharm',head,6)
hharm= head(i)
nv   = strparse(hharm,':',hlist)
nh   = fix(hlist(1))    ; number of harmonics
asize= size(a)
n    = fix(asize(2))    ; total number of points
np   = n/nh             ; number of points/harmonic
title= head(2)          ; title

return, error_status
END ; xtc_read2d_plt

;
;==============================================================================
;
FUNCTION xtc_write_dat, struct, FNAME=fname

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XTC_WRITE_DAT: Error caught: '+!error_state.msg)
   RETURN,error_status
ENDIF

if n_elements(fname) eq 0 then fname = 'tc.inp'

;  Define text strings
text_machine   = '                       ring-energy(GeV) current(mA) sige'
text_beam      = '               sx(mm) sy(mm) sx1(mrad) sy1(mrad)'
text_undulator = '                               period(cm) N'
text_energy    = '                   emin emax ne'
text_harm      = '                       hmin hmax hstep'
text_parm      = '               helical method print_k neks'

openw, unit, fname, /get_lun

printf, unit, struct.title, format='(a)'
printf, unit, struct.energy, struct.cur, struct.sige, text_machine, format='(2f8.2,f8.5,a)'
printf, unit, struct.sigx, struct.sigy, struct.sigx1, struct.sigy1, text_beam, $
              format='(2f8.4,2f8.4,a)'
printf, unit, struct.period, struct.np, text_undulator, $
              format='(f8.2,i8,a)'
printf, unit, struct.emin, struct.emax, struct.n, text_energy, $
              format='(2f10.1,i8,a)'
printf, unit, struct.ihmin, struct.ihmax, struct.ihstep, text_harm, $
              format='(3i8,a)'
printf, unit, struct.ihel, struct.method, struct.ik, struct.neks, text_parm, $
              format='(4i8,a)'
printf, unit, struct.run_mode_name, $
              format='(a)'
free_lun, unit
END ; xtc_write_dat

;
;==============================================================================
;
PRO xtc_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XTC_EVENT: Error caught: '+!error_state.msg)
   IF N_Elements(info) NE 0 THEN Widget_Control, event.top, Set_UValue=info, /No_Copy
   RETURN
ENDIF

;  Retrieve anonymous structure
WIDGET_CONTROL, event.top, GET_UVALUE=info, /No_Copy    ; use the No_Copy as all errors are handled properly

;  Get structures
control = *(info.pc)
data = *(info.pd)
IF ptr_valid(info.pp) THEN plt = *(info.pp)

delimiter = control.delimiter
fname_res = control.fname_res
fname_log = control.fname_log
osversion = control.osversion
ds        = control.ds
ps        = control.ps
os        = control.os
ssz       = control.ssz

IF N_ELEMENTS(data) eq 0 THEN BEGIN
  l1d = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('dat1',TAG_NAMES(data)) ne -1 THEN l1d = 1b ELSE l1d = 0b
ENDELSE
IF N_ELEMENTS(plt) eq 0 THEN BEGIN
  l2p = 0b
ENDIF ELSE BEGIN
  IF STRMATCH1('plt2',TAG_NAMES(plt)) ne -1 THEN l2p = 1b ELSE l2p = 0b
ENDELSE
;print,'l1d l2p',l1d,l2p

;  Definitions
C_EV_KEV = 1.0e-3               ; conversion eV -> keV
C_KEV_EV = 1.0e+3               ; conversion keV -> eV
C_W_KW   = 1.0e-3               ; conversion W  -> kW

; srio
IF (Where(StrUpCase(Tag_Names(event)) EQ 'VALUE'))(0) NE -1 THEN BEGIN
  eVal = event.value
ENDIF ELSE BEGIN
  Widget_Control,event.id, Get_UValue=eVal
ENDELSE

nv = strparse(eVal, delimiter, elist)

;print,tag_names(event)
;print,'nv elist',nv,' ',elist
;print, event.value
;print,'************'+eVal+'***********'

CASE eval OF

  'FILEINPUT': BEGIN
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
        filter='*.xop'
        str_par = Xop_Input_Load(Title=$
          'Select XTC input file...',$
          /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = data.dat1
          Copy_Structure,str_par,tmp,Group=event.top ; ,/OnlyFirstField
          data.dat1 = tmp
        ENDIF
      END
      'Save to file...': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,File='xtc.xop',$
          /Write, Group=event.top, Comment='; xop/xtc(v'+$
          xtc_version()+') input file on '+SysTime()
      END
      'Save as default': BEGIN
        str_par = data.dat1
        Xop_Input_Save,str_par,Group=event.top, $
          Default='xtc.xop',Comment='; xop/xtc(v'+$
          xtc_version()+') input file on '+SysTime()
      END
    ENDCASE
  END

  'File:Load  TC:Input      File': BEGIN
    CD, CURRENT=cwd & cwd = cwd +ds
    CASE osversion OF
      'UNIX':    filter = '*.inp'
      'WINDOWS': filter = 0
      'MACOS':   filter = 0 ; 'MACOS command here'
    ENDCASE
    fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
    IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
    ff = FINDFILE(fp)
    fname = ff(0)
    IF fname eq '' THEN BEGIN
      stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
      GOTO, LBL900
    ENDIF
    error_status = XTC_READ_DAT(dat1, FNAME=fname)
    IF error_status NE 0 THEN BEGIN
      stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['','XTC_EVENT: Error reading file: '+ stxt,'',$
      'This seems not a TC ','input file  '])
      GOTO, LBL900
    ENDIF
    data = {dat1:dat1}
  END
  'File:Load  TC:Output     File': BEGIN
    CD, CURRENT=cwd & cwd = cwd +ds
    CASE osversion OF
      'UNIX':    filter = '*.out'
      'WINDOWS': filter = 0
      'MACOS':   filter = 0 ; 'MACOS command here'
    ENDCASE
    fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, GROUP=event.top)
    IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
    ff = FINDFILE(fp)
    fname = ff(0)
    IF fname eq '' THEN BEGIN
      stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['',stxt,'File not found'])
      GOTO, LBL900
    ENDIF
    error_status = XTC_READ2D_PLT(a2, nd, nh, np, title, FNAME=fname)
    IF error_status NE 0 THEN BEGIN
      stxt = STRMID(fname, STRLEN(path), STRLEN(fname) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['','XTC_EVENT: Error reading file: '+stxt, $
      'This seems not a TC ','output file  '])
      GOTO, LBL900
    ENDIF
    control.fname_res = fname
    nc = nd(0)
    plt2 = {a2:a2, nc:nc, nh:nh, np:np, title:title}
    plt  = {plt2:plt2}
  END
  'File:Load  TC:Default    File': BEGIN
    itmp = Dialog_Message(Dialog_Parent=event.top, $
      /Question,['This option initializes the xtc parameters to their default values.',$
      'Then click "Set Parameters" to run the program.',$
      'Please confirm.'],Title='xtc: set defaults')
    IF itmp EQ 'No' THEN GoTo,LBL900
    dat1 = XOP_DEFAULTS('xtc')
    data = {dat1:dat1}
  END
  'File:Write TC:Input  File': BEGIN
    CD, CURRENT=cwd & cwd = cwd +ds
    CASE osversion OF
      'UNIX':    filter = '*.inp'
      'WINDOWS': filter = 0
      'MACOS':   filter = 0 ; 'MACOS command here'
    ENDCASE
    fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
      TITLE='Save File As', GROUP=event.top)
    IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
    stxt = 'tc.inp'             ; check for existence of tc.inp
    ff = FINDFILE(path +stxt)
    fname = ff(0)
    IF fname eq '' THEN BEGIN ; file does not exist
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['','Error copying file ' +stxt, $
      'File not found','Run TC first and try again!'])
      GOTO, LBL900
    ENDIF
    ff = FINDFILE(fp)
    fname = ff(0)
    IF fname ne '' THEN BEGIN ; file already exist
      stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
      IF junk eq 'No' THEN GOTO, LBL900
    ENDIF
    CASE osversion OF
      'UNIX':    command = '/bin/cp ' +path +'tc.inp ' +fp
      'WINDOWS': command = 'copy '    +path +'tc.inp ' +fp
      'MACOS':   command = 'MACOS command here'
    ENDCASE
    message,/info,'Executing: '+command
    if sdep() eq "UNIX" then spawn,command,/sh else spawn,command
  END
  'File:Write TC:Output File': BEGIN
    CD, CURRENT=cwd & cwd = cwd +ds
    CASE osversion OF
      'UNIX':    filter = '*.out'
      'WINDOWS': filter = 0
      'MACOS':   filter = 0  ; 'MACOS command here'
    ENDCASE
    fp = Dialog_PICKFILE(PATH=cwd, FILTER=filter, GET_PATH=path, $
      TITLE='Save File As', GROUP=event.top)
    IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
    stxt = 'tc.out'             ; check for existence of tc.out
    ff = FINDFILE(path +stxt)
    fname = ff(0)
    IF fname eq '' THEN BEGIN ; file does not exist
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
      ['','Error copying file ' +stxt, $
      'File not found','Run TC first and try again!'])
      GOTO, LBL900
    ENDIF
    ff = FINDFILE(fp)
    fname = ff(0)
    IF fname ne '' THEN BEGIN ; file already exist
      stxt = STRMID(fp, STRLEN(path), STRLEN(fp) -STRLEN(path)) ; get filename
      junk = Dialog_Message(/QUESTION, DIALOG_PARENT=event.top, $
      ['',stxt,'Already exist; Overwrite File ?'])
      IF junk eq 'No' THEN GOTO, LBL900
    ENDIF
    CASE osversion OF
      'UNIX':    command = '/bin/cp ' +path +'tc.out ' +fp
      'WINDOWS': command = 'copy '    +path +'tc.out ' +fp
      'MACOS':   command = 'MACOS command here'
    ENDCASE
    message,/info,'Executing: '+command
    if sdep() eq "UNIX" then spawn,command,/sh else spawn,command
  END

  'File:Display File': BEGIN
    CD, CURRENT=cwd & cwd = cwd +ds
    fp = Dialog_PICKFILE(PATH=cwd, GET_PATH=path, GROUP=event.top)
    IF fp eq '' or fp eq cwd THEN GOTO, LBL900 ; cancel
    XDISPLAYFILE1, fp, GROUP=event.top
  END

  'File:Quit': BEGIN
    ptr_free,info.pc,info.pd
    IF ptr_valid(info.pp) THEN ptr_free,info.pp
    WIDGET_CONTROL, event.top, /DESTROY
    Delete_Files,['tc.inp','tc.out','tc.log'],Group=event.top
  END

  'Set & Run': BEGIN
    Widget_Control,/HourGlass

;    IF NOT l1d THEN BEGIN
;      ff = findfile('tc.inp')
;      fname = ff(0)
;      IF fname eq '' THEN BEGIN
;        dat1 = XOP_DEFAULTS('xtc') ; not found, define new structure
;      ENDIF ELSE BEGIN ; use file tc.inp
;        error_status = XTC_READ_DAT(dat1)
;        IF error_status NE 0 THEN BEGIN
;          junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
;          ['','Error reading file', $
;          'tc.inp','This seems not a TC ','input file  '])
;          GOTO, LBL900
;        ENDIF
;      ENDELSE
;      data = {dat1:dat1}
;    END

    XTC_SET, data, ACTION=action, GROUP=event.top
    IF action eq 'DONT' THEN GOTO, LBL900

;  Save tc.inp
    error_status = XTC_WRITE_DAT(data.dat1)
    IF error_status NE 0 THEN BEGIN
      junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
        'Error writing file tc.inp')
      GOTO, LBL900
    ENDIF

;  Run TC
    IF data.dat1.run_mode_name eq 'foreground' THEN BEGIN ; foreground execution
      PRINT, ' Running TC in the foreground; PLEASE WAIT!'
      CASE osversion OF
        'UNIX':           command = 'tc | tee tc.log'
        'WINDOWS':        command = 'tc | tee tc.log'
        'MACOS':          command = 'tc'
      ENDCASE
      Xop_Spawn,command,CleanFiles=['tc.out']
      bell = STRING(7b)
      FOR i=0,1 DO BEGIN
        PRINT, bell
        WAIT, 0.2
      ENDFOR
    ENDIF ELSE BEGIN ; background execution
      CASE osversion OF
        'UNIX': BEGIN
          PRINT, ' Running TC in the background'
          command='tc | tee tc.log'
          Xop_Spawn,command,onlyGetCommand=command,CleanFiles=['tc.out']
          command = 'xterm -e ' +"'" +command +"'" +' &'  ; need to quote command
          message,/info,'Executing: '+command
          SPAWN, command,/sh
        END
        'WINDOWS': BEGIN
          itmp = Dialog_Message(/INFO, DIALOG_PARENT=event.top, $
            ['Background execution ','not implemented in Windows'])
        END
        'MACOS': BEGIN
          itmp = Dialog_Message(/INFO, DIALOG_PARENT=event.top, $
            ['Background execution ','not implemented in MacOS'])
        END
      ENDCASE
    ENDELSE
  END

  'Help:TC': BEGIN
    hfile = Xop_GetEnv('XOP_HOME') +ds +'doc' +ds+'txt'+ds+'tc.txt'
    XDISPLAYFILE1, hfile, GROUP=event.top
  END

  'Help:Xtc': Xhelp,'xtc',GROUP=event.top

  ELSE: BEGIN ; Show
    CASE elist(1) OF
      'Plot Results': BEGIN
        IF l2p THEN BEGIN ; display data saved in anonymous structure
          a2 = plt.plt2.a2 ; data
          nc = plt.plt2.nc ; # of columns
          nh = plt.plt2.nh ; # of harmonics
          np = plt.plt2.np ; # of points/harmonic
          title = plt.plt2.title
        ENDIF ELSE BEGIN ; read data and save in anonymous structure
          error_status = XTC_READ2D_PLT(a2, nd, nh, np, title)
          IF error_status NE 0 THEN BEGIN
            stxt = 'tc.out'
            junk = Dialog_Message(/ERROR, DIALOG_PARENT=event.top, $
              ['','XTC_EVENT: Error reading file: '+stxt,'',$
            'Please use "Set Parameters" button to set and run TC'] )
            GOTO, LBL900
          ENDIF
          nc = nd(0)
          plt2 = {a2:a2, nc:nc, nh:nh, np:np, title:title}
          plt  = {plt2:plt2}
        ENDELSE
        ptxt = ['',' data not found...', $
          ' Rerun TC and set "Save K-Value/power?"']

        CASE elist(2) OF
          'Xplot': BEGIN
            CASE elist(3) OF
              'On-Axis Brilliance': BEGIN         ; currently not used
                ap = a2(1:2,*)                  ; emittance shifted energy = second column
                ap(0,*) = ap(0,*)*C_EV_KEV      ; emittance shifted energy = first column
                xtitle = 'Energy (keV)'
                ytitle = 'On-Axis Brilliance (ph/s/mrad^2/mm^2/0.1%bw)'
                ip = where(a2[2,*])               ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,ip])
                ymin = 0.1*min(ap[1,ip])          ; use only non-zero elements of on-axis brilliance and factor 0.1 of min value
                ymax = max(ap(1,*))
                xplot,ap,GROUP=event.top,xrange=[xmin,xmax],yrange=[ymin,ymax],/ylog, $
                  coltitles=[xtitle,ytitle], xtitle='-1',ytitle='-1',title=title, $
                  WINDOW_SIZE=ssz
              END
              'All': BEGIN
                ap = fltarr(nc+1,np*nh)         ; make room for energy shift -> use nc+1
                ap(0,*) = a2(0,*)*C_EV_KEV      ; zero emittance energy (keV)
                ap(1,*) = a2(1,*)*C_EV_KEV      ; emittance down shifted energy (keV)
                ap(2,*) = (ap(0,*)-ap(1,*))*C_KEV_EV       ; energy down shift (eV)
                ap(3:nc,*) = a2(2:nc-1,*)
                if nc eq 6 then begin           ; regular device
                  ap(5,*) = ap(5,*)*C_W_KW      ; power (kW)
                  ap(6,*) = ap(6,*)*C_W_KW      ; on-axis power density (kW/mrad^2)
                endif else if nc eq 7 then begin; helical device
                  ap(6,*) = ap(6,*)*C_W_KW      ; power (kW)
                  ap(7,*) = ap(7,*)*C_W_KW      ; on-axis power density (kW/mrad^2)
                endif
                coltitles = ['Zero Emittance Energy (keV)', $
                  'Energy (keV)', 'Energy Down Shift (eV)', $
                  'On-Axis Brilliance (ph/s/mrad^2/mm^2/0.1%bw)']
                coltitles_extra = ['Ky', 'Ptot (kW)', 'On-Axis Power Density (kW/mrad^2)']
                IF nc EQ 7 THEN coltitles_extra = ['Kx', coltitles_extra] ; helical
                IF nc EQ 6 or nc EQ 7 THEN coltitles = [coltitles, coltitles_extra]
  ;           xplot,ap,GROUP=event.top,xrange=[xmin,xmax],yrange=[ymin,ymax],/ylog, $ ; for future use, need to set x- and y-range
                xplot,ap,GROUP=event.top, $
                  coltitles=coltitles, xtitle='-1',ytitle='-1',title=title, $
                  xcol=2,ycol=4, WTitle='XTC results', $
                  WINDOW_SIZE=ssz
              END
            ENDCASE ; elist(3)
          END ; Xplot

          'Xwindow': BEGIN
            CASE elist(3) OF
              'On-Axis Brilliance': BEGIN
                ap = a2(1:2,*)                  ; emittance shifted energy = second column
                ap(0,*) = ap(0,*)*C_EV_KEV      ; emittance shifted energy = first column
                xtitle = 'Energy (keV)'
                ytitle = 'On-Axis Brilliance (ph/s/mrad^2/mm^2/0.1%bw)'
                ip = where(a2[2,*])               ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,ip])
                ymin = 0.1*min(ap[1,ip])          ; use only non-zero elements of on-axis brilliance and factor 0.1 of min value
                ymax = max(ap(1,*))

                p=0L
                ip = where(ap[1,0:np-1])          ; pick non-zero elements of on-axis brilliance
                Xplot,Parent=p,ap[*,ip],/No_Block, XRange=[xmin,xmax],$
                  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
                  /YLog, Wtitle='XTC results ('+elist(3)+')', $
                  WINDOW_SIZE=ssz
                FOR i=1,nh-1 DO BEGIN
                  ip = np*i +where(ap[1,np*i:np*(i+1)-1]) ; pick non-zero elements of on-axis brilliance
                  Xplot_SaveCurrent,p
                  Xplot_LoadFile,p,ap[*,ip]
                ENDFOR
              END

              'Energy Shift': BEGIN
                ap = fltarr(2,np*nh)
                ap(0,*) = a2(0,*)*C_EV_KEV        ; zero emittance energy (keV)
                ap(1,*) = (a2(0,*)-a2(1,*))       ; energy shift (eV)
                xtitle = 'Zero Emittance Energy (keV)'
                ytitle = 'Harmonic Energy Down Shift (eV)'
                i = where(a2[2,*])                ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,i])
                ymin = 0.0
                ymax = max(ap(1,*))

                p=0L
                ip = where(ap[1,0:np-1])          ; pick non-zero elements of energy shift
                Xplot,Parent=p,ap[*,ip],/No_Block, XRange=[xmin,xmax],$
                  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
                  Wtitle='XTC results ('+elist(3)+')', $
                  WINDOW_SIZE=ssz
                Xplot_Controls_Action,p,PSymbol=2
                FOR i=1,nh-1 DO BEGIN
                  ip = np*i +where(ap[1,np*i:np*(i+1)-1]) ; pick non-zero elements of energy shift
                  Xplot_SaveCurrent,p
                  Xplot_LoadFile,p,ap[*,ip]
                ENDFOR
              END

              'K-Value': BEGIN
                IF (nc NE 6) AND (nc NE 7) THEN BEGIN
                  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
                  GOTO, LBL900
                ENDIF
                ap = fltarr(2,np*nh)
                ap(0,*) = a2(1,*)*C_EV_KEV      ; emittance shifted energy (keV)
                ap(1,*) = a2(3,*)               ; if helical Kx = Ky
                xtitle = 'Energy (keV)'
                IF nc EQ 6 THEN ytitle = 'K-Value (Ky)' ELSE ytitle = 'K-Value (Kx=Ky)'
                ip = where(a2[2,*])               ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,ip])
                ymin = 0.0
                ymax = max(ap(1,*))

                p=0L
                Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
                  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
                  Wtitle='XTC results ('+elist(3)+')', $
                  WINDOW_SIZE=ssz
                FOR i=1,nh-1 DO BEGIN
                  Xplot_SaveCurrent,p
                  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
                ENDFOR
              END

              'Total Power': BEGIN
                IF (nc NE 6) AND (nc NE 7) THEN BEGIN
                  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
                  GOTO, LBL900
                ENDIF
                ap = fltarr(2,np*nh)
                ap(0,*) = a2(1,*)*C_EV_KEV      ; emittance shifted energy (keV)
                IF nc EQ 6 THEN ap(1,*) = a2(4,*) ELSE ap(1,*) = a2(5,*)
                ap(1,*) = ap(1,*)*C_W_KW        ; convert W -> kW
                xtitle = 'Energy (keV)'
                ytitle = 'Total Power (kW)'
                ip = where(a2[2,*])               ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,ip])
                ymin = 0.0
                ymax = max(ap(1,*))

                p=0L
                Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
                  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
                  Wtitle='XTC results ('+elist(3)+')', $
                  WINDOW_SIZE=ssz
                FOR i=1,nh-1 DO BEGIN
                  Xplot_SaveCurrent,p
                  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
                ENDFOR
              END

              'Power Density': BEGIN
                IF (nc NE 6) AND (nc NE 7) THEN BEGIN
                  junk = Dialog_Message(/INFO, DIALOG_PARENT=event.top, ptxt)
                  GOTO, LBL900
                ENDIF
                ap = fltarr(2,np*nh)
                ap(0,*) = a2(1,*)*C_EV_KEV      ; emittance shifted energy (keV)
                IF nc EQ 6 THEN ap(1,*) = a2(5,*) ELSE ap(1,*) = a2(6,*)
                ap(1,*) = ap(1,*)*C_W_KW        ; convert W/mrad^2 -> kW/mrad^2
                xtitle = 'Energy (keV)'
                ytitle = 'On-Axis Power Density (kW/mrad^2)'
                ip = where(a2[2,*])               ; pick non-zero elements of on-axis brilliance
                xmin = min(ap(0,*))
                xmax = max(ap[0,ip])
                ymin = 0.0
                ymax = max(ap(1,*))

                p=0L
                Xplot,Parent=p,ap[*,0:np-1],/No_Block, XRange=[xmin,xmax],$
                  YRange=[ymin,ymax],XTitle=xtitle,YTitle=ytitle, Title=title, $
                  Wtitle='XTC results ('+elist(3)+')', $
                  WINDOW_SIZE=ssz
                FOR i=1,nh-1 DO BEGIN
                  Xplot_SaveCurrent,p
                  Xplot_LoadFile,p,ap[*,np*i:np*(i+1)-1]
                ENDFOR
              END
            ENDCASE ; elist(3)
          END ; Xwindow
        ENDCASE ; elist(2)
      END ; Plot Results

      'View  Results': BEGIN
        XDISPLAYFILE1, fname_res, GROUP=event.top
      END

      'View  Log': BEGIN
        XDISPLAYFILE1, fname_log, GROUP=event.top
      END

    ENDCASE ; elist(1)
  ENDELSE ; event.value
ENDCASE ; event.value

LBL900:
IF WIDGET_INFO(event.top, /VALID) THEN BEGIN $
  *(info.pc) = control
  *(info.pd) = data
;  save structure plt if not a new run
  IF ptr_valid(info.pp) THEN ptr_free,info.pp
  IF eVal ne 'Set & Run' and n_elements(plt) ne 0 THEN info.pp = ptr_new(plt)
  WIDGET_CONTROL, event.top, SET_UVALUE=info, /No_Copy
ENDIF

RETURN

END ; xtc_event

;
;===============================================================================
;
PRO XTC_Cleanup, tcbase         ; called if widget is closed by mouse click or destroyed (as in Quit)

;print, '*** IN XTC_Cleanup ***'
WIDGET_CONTROL, tcbase, GET_UVALUE=info, /No_Copy
IF n_elements(info) eq 0 THEN RETURN

; Free pointers
ptr_free,info.pc,info.pd
IF ptr_valid(info.pp) THEN ptr_free,info.pp
RETURN
END

;
;==============================================================================
;
PRO xtc, GROUP=group, InputFile=inputFile, No_Block=no_block

Forward_Function xop_defaults, screensize

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Catch, /Cancel
  Message,/Info,'Error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error,Dialog_Parent=group, $
    'XTC: Error caught: '+!error_state.msg)
  RETURN
ENDIF

;  Definitions and labels
XTC_TITLE1 = 'Xtc '+xtc_version()
XTC_LBL1   = 'TC'
XTC_LBL2   = 'Undulator Tuning Curves'

ssz = screensize()      ; get screen size

delimiter = ':'
osversion = sdep()      ; system dependent operating system
ds = sdep(/ds)          ; system dependent directory separator
ps = sdep(/ps)          ; system dependent path separator
os = sdep(/os)          ; system dependent flavor of operating system

; set keyword no_block
IF (~ KEYWORD_SET(No_Block)) THEN no_block = 1  ; set default to not block

; set working directory
IF N_Elements(group) EQ 0 THEN xop_wd,/default

CD, CURRENT=cwd & cwd = cwd +ds
fname_res = cwd +'tc.out'
fname_log = cwd +'tc.log'

;  Structure for pull-down menu
;junk = {CW_PDMENU_S, flags:0, name:''}

tcbase = WIDGET_BASE(COLUMN=1,TITLE=XTC_TITLE1,MBar=wMenuBar)

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='Xtc input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')

  wtmp0 = widget_button(wFile,VALUE='TC file', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load input file...',$
        UValue='File:Load  TC:Input      File')
  wtmp = widget_button(wtmp0,VALUE='Load output file...',$
        UValue='File:Load  TC:Output     File')
  wtmp = widget_button(wtmp0,VALUE='Write input file...',$
        UValue='File:Write TC:Input  File')
  wtmp = widget_button(wtmp0,VALUE='Write output file...',$
        UValue='File:Write TC:Output File')

; Not yet implemented
;  wtmp = widget_button(wFile,VALUE='Save files for XOP/Optics', $
;       UVALUE='OPTICSFILES')

  wtmp = widget_button(wFile,VALUE='Display file', UVALUE='File:Display File')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='File:Quit',/Separator)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', $
        UVALUE='Set & Run')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', $
        UVALUE='File:Load  TC:Default    File')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
;  wtmp = widget_button(wResults,VALUE='Plot/Xplot/OnAxisBrilliance',$
;       UValue='Show:Plot Results:Xplot:On-Axis Brilliance')
  wtmp = widget_button(wResults,VALUE='Plot All',$
        UValue='Show:Plot Results:Xplot:All')
  wtmp = widget_button(wResults,VALUE='Plot On-Axis Brilliance',$
        UValue='Show:Plot Results:Xwindow:On-Axis Brilliance')
  wtmp = widget_button(wResults,VALUE='Plot Energy Down Shift',$
        UValue='Show:Plot Results:Xwindow:Energy Shift')
  wtmp = widget_button(wResults,VALUE='Plot K-Value',$
        UValue='Show:Plot Results:Xwindow:K-Value')
  wtmp = widget_button(wResults,VALUE='Plot Total Power',$
        UValue='Show:Plot Results:Xwindow:Total Power')
  wtmp = widget_button(wResults,VALUE='Plot On-Axis Power Density',$
        UValue='Show:Plot Results:Xwindow:Power Density')
  wtmp = widget_button(wResults,VALUE='View Results',$
        UValue='Show:View  Results')
  wtmp = widget_button(wResults,VALUE='View Logfile',$
        UValue='Show:View  Log')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='TC', $
        UValue='Help:TC')
  wtmp = widget_button(wHelp,VALUE='Xtc', $
        UValue='Help:Xtc')
;

wtmp = widget_button(tcbase,VALUE='Set Parameters', UVALUE='Set & Run')

;=======================================
;md = [ $
;  {CW_PDMENU_S, 1, 'File'}, $
;    {CW_PDMENU_S, 1, 'Load  TC'}, $
;      {CW_PDMENU_S, 0, 'Input      File'}, $
;      {CW_PDMENU_S, 0, 'Output     File'}, $
;      {CW_PDMENU_S, 2, 'Default    File'}, $
;    {CW_PDMENU_S, 1, 'Write TC'}, $
;      {CW_PDMENU_S, 0, 'Input  File'}, $
;      {CW_PDMENU_S, 2, 'Output File'}, $
;    {CW_PDMENU_S, 0, 'Display File'}, $
;    {CW_PDMENU_S, 2, 'Quit'}, $
;
;  {CW_PDMENU_S, 0, 'Set & Run'}, $
;
;  {CW_PDMENU_S, 1, 'Show'}, $
;    {CW_PDMENU_S, 1, 'Plot Results'}, $
;      {CW_PDMENU_S, 1, 'Xplot'}, $
;        {CW_PDMENU_S, 0, 'On-Axis Brilliance'}, $
;        {CW_PDMENU_S, 2, 'All'}, $
;      {CW_PDMENU_S, 3, 'Xwindow'}, $
;        {CW_PDMENU_S, 0, 'On-Axis Brilliance'}, $
;        {CW_PDMENU_S, 0, 'Energy Shift'}, $
;        {CW_PDMENU_S, 0, 'K-value'}, $
;        {CW_PDMENU_S, 0, 'Total Power'}, $
;        {CW_PDMENU_S, 2, 'Power Density'}, $
;    {CW_PDMENU_S, 2, 'View  Results'}, $
;
;  {CW_PDMENU_S, 3, 'Help'}, $
;    {CW_PDMENU_S, 0, 'TC'}, $
;    {CW_PDMENU_S, 2, 'Xtc'} ]
;
;pd = CW_PDMENU(tcbase, md, DELIMITER=delimiter, /RETURN_FULL_NAME)
;
;=======================================

CASE osversion OF
  'UNIX': BEGIN
    ;font='-b&h-lucida-bold-i-normal-sans-24-240-75-75-p-151-iso8859-1'
    font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
  END
  'WINDOWS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24'
  END
  'MACOS': BEGIN
    font = 'VERDANA*BOLD*ITALIC*24' ; 'MACOS command here'
  END
ENDCASE

lb = WIDGET_LABEL(tcbase, FONT=font, VALUE=XTC_LBL1)
lb = WIDGET_LABEL(tcbase, FONT=font, VALUE=XTC_LBL2)

control = {delimiter:delimiter, fname_res:fname_res, fname_log:fname_log, osversion:osversion, $
           ds:ds, ps:ps, os:os, ssz:ssz}

xtc_str = Xop_Defaults('xtc')

IF N_Elements(inputFile) NE 0 THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = xtc_str
    Copy_Structure,str_par, tmp, Group=group
    xtc_str = tmp
  ENDIF
ENDIF

; define necessary pointers
pc = ptr_new(control)
pd = ptr_new({dat1:xtc_str})
pp = ptr_new()
info = {pc:pc, pd:pd, pp:pp}

WIDGET_CONTROL, tcbase, /REALIZE

WIDGET_CONTROL, tcbase, SET_UVALUE=info, /No_Copy

XMANAGER, 'xtc', tcbase, CLEANUP='XTC_Cleanup', GROUP_LEADER=group, No_Block=no_block

END ; xtc
