;+
;=================================   xcrystal   ================================
;
;
;   Xcrystal is a widget based graphical interface to calculate diffraction
;   profiles of perfect and mosaic crystals. It allows Bragg and Laue
;   geometries and (in the case of perfect crystals) diffracted and
;   transmitted intensity calculations.  The diffracted intensity can be
;   calculated as a function of either rock angle or photon energy.
;   It uses the DABAX database to define the crystal structure and to 
;   retrieve the scattering factors to build the crystal structure factors.
;
;   It interfaces diff_pat, a computer program written by M. Sanchez del Rio
;   (srio@esrf.fr) derived from the SHADOW application bragg.
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xcrystal input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XCrystal window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, diff_pat starts running.
;  Set Defaults: Sets the default parameters.
;
;  Show: Display results
;   Diffraction Curve: starts a graphical window with the calculation results.
;   Crystal Parameters: shows output parameters of the calculation.
;   Crystallographic Parameters: shows numerical values used to build the 
;		crystal structure factor.
;
;  Help:   Shows the xcrystal help (this text).
;
;  COPYRIGHT:
; 	XCRYSTAL belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
;
;	M. Sanchez del Rio, C. Ferrero and V. Mocella. 
;	  "Computer simulations of bent perfect crystal diffraction profiles" 
;	  SPIE proceedings vol. 3151, pp.312-323, 1997. 
;
; 
;  LAST MODIFICATION: srio@esrf.eu 2014-03-24
; 
; 
;-
; ----------------------------------------------------------------------
;
;       Modification history:
;		97/01/03 srio@esrf.fr creates the initial version.
;			This version obsoletes xreflex and xpenpol.
;		97/01/21 srio@esrf.fr adapts for Windows95.
;		97/10/07 srio@esrf.fr version for xop 1.9, now v 1.1
;			Many bugs and cosmetic fixes. 
;		98/12/10 srio@esrf.fr version for xop 2.0, now v 1.2
;		99/03/16 srio@esrf.fr updates doc.
;		2013/01/29 srio@esrf.fr updates for diff_pat for bent crystals.
;		                        Scan-variable added.
;		2014/03/24 srio@esrf.fr new version integrating bent 
;                          crystals (ML & PP models).
;-
;
;========================================================================
;
FUNCTION XCRYSTAL_version
return,'1.3'
end
;		
;========================================================================
;
PRO XCrystal_Inp_Write,state, Group=group, error=error


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XCRYSTAL_INP_WRITE: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
error = 0
;
; write the contents of the structure with parameters for running reflex
;
inp = state.str.parameters
name='xcrystal.inp'
;
Openw,unit,name,/GET_LUN
;
;
Printf,unit,state.str.bragg.parameters.outfil
Printf,unit,inp.mosaic(0)

Printf,unit,state.str.parameters.geometry(0)

case fix(inp.mosaic(0))  of
  1:    begin ; mosaic
          Printf,unit,StrCompress(state.str.parameters.Mosaic_fwhm,/Remove_All)
          Printf,unit,StrCompress(state.str.parameters.Thickness,/Remove_All)
        end
  else: begin ; perfect (0) of bent (>1)
          Printf,unit,StrCompress(state.str.parameters.Thickness,/Remove_All)
          Printf,unit,StrCompress(state.str.parameters.asymmetry_angle, $
                 /Remove_All)
        end
endcase
;if strcompress(inp.mosaic(0),/rem) EQ 0 then begin ; perfect
;  Printf,unit,StrCompress(state.str.parameters.Thickness,/Remove_All)
;  Printf,unit,StrCompress(state.str.parameters.asymmetry_angle,/Remove_All)
;endif else begin  ; mosaic
;  Printf,unit,StrCompress(state.str.parameters.Mosaic_fwhm,/Remove_All)
;  Printf,unit,StrCompress(state.str.parameters.Thickness,/Remove_All)
;endelse

scanFlag= 1+fix(state.str.parameters.scan(0))
Printf,unit,StrCompress(scanFlag)

;Printf,unit,StrCompress(state.str.parameters.Energy,/Remove_All)
Printf,unit,StrCompress(string(state.str.parameters.Energy,$
  Format='(F19.9)'),/Remove_All)

if scanFlag LE 3 then $
  Printf,unit,StrCompress(state.str.parameters.Unit(0),/Remove_All)
;
Printf,unit,StrCompress(state.str.parameters.ScanFrom,/Remove_All)
Printf,unit,StrCompress(state.str.parameters.ScanTo,/Remove_All)
Printf,unit,StrCompress(state.str.parameters.ScanPoints,/Remove_All)
;
; bent crystal inputs
if (fix(inp.mosaic(0)) GT 1) then begin ; bent
  Printf,unit,StrCompress(state.str.parameters.Rsag,/Remove_All)
  Printf,unit,StrCompress(state.str.parameters.Rmer,/Remove_All)
  ;Printf,unit,StrCompress(state.str.parameters.poisson,/Remove_All)
  Printf,unit,'0' ; StrCompress(state.str.parameters.polarization[0],/Remove_All)
  ;elasticity

;=============================================================
;help,/str,state.str.parameters

  icrystal_material = fix( (state.str.parameters.crystal_material)[0]) 
  ianisotropy = fix( (state.str.parameters.anisotropy)[0] )

  if icrystal_material GE 5 then begin ; not Si Ge Diamond
     if ((ianisotropy GE 1) AND (ianisotropy LE 2)) then begin 
       itmp = dialog_message(/Error, $
           ['Anisotropy data not available for this crystal', $
            'Either use isotropic or use external compliance file',$
            'Please change and run again'] )
       ianisotropy = 0 ; to finish calculations
       error=1
       return
     endif
  endif
  ;Printf,unit,'2' ; input directions
  Printf,unit,ianisotropy

  case ianisotropy of
     0: Printf,unit,state.str.parameters.poisson
     1: begin
           ;Printf,unit,'0' ; Si
           Printf,unit,icrystal_material
           ;alpha
           Printf,unit,StrCompress(state.str.parameters.asymmetry_angle, /Remove_All)
           ; hkl
           Printf,unit,state.str.parameters.miller_index_h
           Printf,unit,state.str.parameters.miller_index_k
           Printf,unit,state.str.parameters.miller_index_l
        end
     2: begin
           ;Printf,unit,'0' ; Si
           Printf,unit,icrystal_material
           ;alpha
           Printf,unit,StrCompress(state.str.parameters.asymmetry_angle, /Remove_All)
           itmp = strparse(state.str.parameters.cut,';',mylist)
           if (itmp NE 2) then begin
             itmp = dialog_message(/Error, $
             ['Cut syntax is: ',$
             'valong_X valong_Y valong_Z ; vnorm_X vnorm_Y vnorm_Z ; vperp_x vperp_Y vperp_Z', $
             'Please change and run again'] )
           endif
           Printf,unit,mylist[0]
           Printf,unit,mylist[1]
           Printf,unit,mylist[2]
           ;; valong
           ;Printf,unit,'2' ;h
           ;Printf,unit,'-1' ;k
           ;Printf,unit,'-1' ;l
           ;; vnorm
           ;Printf,unit,'1' ;h
           ;Printf,unit,'1' ;k
           ;Printf,unit,'1' ;l
           ;; vperp
           ;Printf,unit,'0' ;h
           ;Printf,unit,'0' ;k
           ;Printf,unit,'0' ;l
        end
     3: Printf,unit,state.str.parameters.filecompliance
  endcase
;;    ; ; TEST1 (meridional bending)
;;    ; valong
;;    Printf,unit,'0' ;h
;;    Printf,unit,'0' ;k
;;    Printf,unit,'0' ;l
;;    ; vnorm
;;    Printf,unit,'1' ;h
;;    Printf,unit,'1' ;k
;;    Printf,unit,'1' ;l
;;    ; vperp
;;    Printf,unit,'-1' ;h
;;    Printf,unit,'-1' ;k
;;    Printf,unit,'2' ;l

;;  ; ; TEST2 (Xianbo)
;;  ; valong
;;  Printf,unit,'2' ;h
;;  Printf,unit,'-1' ;k
;;  Printf,unit,'-1' ;l
;;  ; vnorm
;;  Printf,unit,'1' ;h
;;  Printf,unit,'1' ;k
;;  Printf,unit,'1' ;l
;;  ; vperp
;;  Printf,unit,'0' ;h
;;  Printf,unit,'0' ;k
;;  Printf,unit,'0' ;l
;=============================================================

endif ; bent crystal


;
Free_Lun,unit
;
END



;		
;========================================================================
;
PRO XCrystal_spawn,state, Group=group


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XCRYSTAL_SPAWN: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF


;CASE 1 OF ; fix(state.str.parameters.binary[0]) OF
;0:  begin 
    command = 'diff_pat < xcrystal.inp'
    Xop_Spawn,command, CleanFiles=['diff_pat.par','diff_pat.dat']
;    end
;1:  begin
;    print,'Executing: /bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat'
;    spawn,'/bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat',/sh
;    print,'Executing: ./diff_pat < xcrystal.inp'
;    spawn,'./diff_pat < xcrystal.inp',/sh
;    end
;2:  begin
;    print,'Executing: /bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat'
;    spawn,'/bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat',/sh
;    print,'Executing: ./CRYSTAL_final_backup20131008/diff_pat < xcrystal.inp'
;    spawn,'./CRYSTAL_final_backup20131008/diff_pat < xcrystal.inp',/sh
;endelse
;    end
;ENDCASE

;
END ; xcrystal_spawn


;=======================================================================
;
PRO Xcrystal_EVENT,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XCRYSTAL_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   If Type(stateid) EQ 3 THEN $
     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
     Widget_Control,stateid,Set_UValue=state,/No_Copy
   On_Error,2
   RETURN
ENDIF
;
; register the events
;
Widget_control, event.id, Get_uvalue=eventuval

stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uvalue=state ; ,/No_Copy

;inp = state.str

CASE eventuval OF

       'FILEINPUT': BEGIN
         action=''
         Widget_Control,event.id, Get_Value=action
         CASE action OF
           'Load from file...': BEGIN
             ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
	     filter='*.xop'
             str_par = Xop_Input_Load(Title=$
             'Select Xcrystal input file...',$
             /NoConf,Filter=filter,Group=event.top)
             IF Type(str_par) EQ 8 THEN BEGIN
               tmp = state.str.parameters
               Copy_Structure,str_par, tmp, Group=event.top ; , /OnlyFirstField
               state.str.parameters = tmp
             ENDIF
           END
           'Save to file...': BEGIN
             str_par = state.str.parameters
             Xop_Input_Save,str_par,File='xcrystal.xop',$
               /Write, Group=event.top, Comment='; xop/xcrystal(v'+$
             xcrystal_version()+') input file on '+SysTime()
           END
           'Save as default': BEGIN
             str_par = state.str.parameters
             Xop_Input_Save,str_par,Group=event.top, $
               Default='xcrystal.xop',Comment='; xop/xcrystal(v'+$
             xcrystal_version()+') input file on '+SysTime()
           END
         ENDCASE
       END


	'QUIT': BEGIN
		files = ['xcrystal.bra','diff_pat.dat','diff_pat.par', $
		  'xcrystal.inp'] 
		Delete_Files,files,Group=event.top
		Widget_control,event.top,/Destroy
		RETURN
		END
	'SETDEF': BEGIN
		itmp = Dialog_Message(dialog_parent=event.top,$
		  /Question,['This option initializes the',$
		  'xcrystal parameters to their default values.',$
		  'Then you must click Set_parameters to run the program.',$
		  'Please confirm'],title='xcrystal')
		if itmp eq 'No' then goto,out
		;tmp = xop_defaults('xcrystal')
		state.str.parameters = state.str_defaults
		END
	'SETPAR': BEGIN
		str_par = state.str.parameters

		sep = sdep(/ds)
                helpcmd = "xdisplayfile1,'"+Xop_GetEnv('XOP_HOME')+$
                  sep+'doc'+sep+'txt'+sep+"xcrystal_par.txt'"
		Xscrmenu,str_par,Group=event.top,Ncol=3,Action=action,$
		Titles=state.str.titles, Flags=state.str.flags, $
		/NoType, /Interp, Wtitle='Xcrystal input parameters',$
		help=helpcmd
		IF (action EQ 'DONT') THEN Goto,out
		; test for correct laue setting
		IF (Fix((str_par.geometry)[0]) EQ 1 OR $ ; Laue
		    Fix((str_par.geometry)[0]) EQ 3) AND $ ; Laue
		   str_par.asymmetry_angle EQ 0 THEN BEGIN
		   itmp = Dialog_Message(Dialog_Parent=event.top,/Question, $
			['In xcrystal the asymmetry angle is the angle',$
			'between Bragg planes and crystal surface, ',$
			'in BOTH Bragg and Laue geometries.','',$
			'It seems you want to use Laue-symmetrical ',$
			'configuration so the asymmetry angle must be',$
			'90 deg instead of 0','Correct it?'])
		   IF itmp EQ 'Yes' THEN str_par.asymmetry_angle=90.0 
		ENDIF


		Widget_control,/Hourglass
		state.str.parameters = str_par
		Xcrystal_Inp_Write,state, Group=event.top, error=error
                IF error THEN goto,out
		; bragg
		strbragg = state.str.bragg
		strbragg.parameters.filef0 = str_par.filef0
		strbragg.parameters.filef1f2 = str_par.filef1f2
		if strcompress(str_par.Scan(0),/rem) EQ '3' then begin 
							; energy scan
		  strbragg.parameters.estep = $
			(str_par.ScanTo-str_par.ScanFrom)/100
		  strbragg.parameters.emin = str_par.ScanFrom -1 
		  strbragg.parameters.emax = $
			str_par.ScanTo + strbragg.parameters.estep + 1
		endif else begin
		  strbragg.parameters.emin = str_par.energy - $
			state.str.energy_shift
		  strbragg.parameters.emax = str_par.energy + $
			state.str.energy_shift
		  strbragg.parameters.estep = state.str.energy_step
		endelse
		strbragg.parameters.ilattice = str_par.Crystal_material
		strbragg.parameters.hmiller = str_par.Miller_index_h
		strbragg.parameters.kmiller = str_par.Miller_index_k
		strbragg.parameters.lmiller = str_par.Miller_index_l
		strbragg.parameters.i_absorp = str_par.i_absorp
		strbragg.parameters.temper = str_par.temper
		Message,/Info,'Running bragg_calc'
		tmp = bragg_calc(strbragg)
		Message,/Info,'Writing results (bragg_out)'
		bragg_out,tmp
                ;if (strcompress(str_par.binary[0],/rem) EQ 0) then begin
		;    command = 'diff_pat < xcrystal.inp'
		;    Xop_Spawn,command, CleanFiles=['diff_pat.par','diff_pat.dat']
                ;endif else begin ; debugging version in current directory
		;    spawn,'/bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat' 
		;    spawn,'./diff_pat < xcrystal.inp'
                ;    ;spawn,'gle diff_pat.gle'
                ;endelse
                xcrystal_spawn,state,group=event.top
		END




	'SETSCAN': BEGIN
		hlp =  ['This option permits to scan a parameter (e.g., thickness). ',$
			'The resulting diffraction profiles are shown (3D) ',$
			'together the FWHM, pean and integrated reflectivities.',$
			'',$
			'']
                ;scan = {what:what, from:8000d0,to:10000d0,npoints:5L }
                scan = state.scan
		Titles=['Variable to scan','From: ','To: ','Number of points']
                Flags=Replicate('1',n_elements(Titles))
		Xscrmenu,scan,Group=event.top,Ncol=1,Action=action,$
		Titles=titles, Flags=flags, $
		/NoType, /Interp, Wtitle='Xcrystal scan parameter',help=hlp
		IF (action EQ 'DONT') THEN Goto,out
                state.scan=scan

		Widget_control,/Hourglass
                ;
		; bragg (prepare input file xcrystal.bra)
                ;
		strbragg = state.str.bragg
		strbragg.parameters.filef0 = state.str.parameters.filef0
		strbragg.parameters.filef1f2 = state.str.parameters.filef1f2
		if strcompress(state.str.parameters.Scan(0),/rem) EQ '3' then begin 
							; energy scan
		  strbragg.parameters.estep = $
			(state.str.parameters.ScanTo-state.str.parameters.ScanFrom)/100
		  strbragg.parameters.emin = state.str.parameters.ScanFrom -1 
		  strbragg.parameters.emax = $
			state.str.parameters.ScanTo + strbragg.parameters.estep + 1
		endif else begin
                  if fix(scan.what[0]) eq 0 then begin ; scan energy, set limits accordingly
		    strbragg.parameters.emin = scan.from-100d0
		    strbragg.parameters.emax = scan.to+100d0
		    strbragg.parameters.estep = state.str.energy_step
                  endif else begin
		    strbragg.parameters.emin = state.str.parameters.energy - $
			state.str.energy_shift
		    strbragg.parameters.emax = state.str.parameters.energy + $
			state.str.energy_shift
		    strbragg.parameters.estep = state.str.energy_step
                  endelse
		endelse
		strbragg.parameters.ilattice = state.str.parameters.Crystal_material
		strbragg.parameters.hmiller = state.str.parameters.Miller_index_h
		strbragg.parameters.kmiller = state.str.parameters.Miller_index_k
		strbragg.parameters.lmiller = state.str.parameters.Miller_index_l
		strbragg.parameters.i_absorp = state.str.parameters.i_absorp
		strbragg.parameters.temper = state.str.parameters.temper
		Message,/Info,'Running bragg_calc'
		tmp = bragg_calc(strbragg)
		Message,/Info,'Writing results (bragg_out)'
		bragg_out,tmp
                ; 
                ; run  external diff_pat
                ;
		str_par_old = state.str.parameters
                str_par = str_par_old

                ; 
                ; loop over the scanned variable
                ; 
                scani = makearray1(scan.npoints,scan.from,scan.to)
                out = dblarr(7,scan.npoints)
                for i=0,scan.npoints-1 do begin
                   case fix(scan.what[0]) of
                     0: str_par.energy=scani[i]
                     1: str_par.asymmetry_angle=scani[i]
                     2: str_par.thickness=scani[i]
                     3: str_par.rsag=scani[i]
                     4: str_par.rmer=scani[i]
                     else: begin
                           print,'Error: case not implemented: ',scan.what[0]
                           goto,out
                           end
                   endcase
		   state.str.parameters = str_par
		   Xcrystal_Inp_Write,state, Group=event.top
		   Xcrystal_spawn,state, Group=event.top
                   ;if (strcompress(state.str.parameters.binary[0],/rem) EQ 0) then begin
		   ;    command = 'diff_pat < xcrystal.inp'
		   ;    Xop_Spawn,command, CleanFiles=['diff_pat.par','diff_pat.dat']
                   ;endif else begin ; debugging version in current directory
		   ;    spawn,'/bin/rm diff_pat.par diff_pat.dat diff_pat.gle diff_pat.xop compliance.dat' 
		   ;    spawn,'./diff_pat < xcrystal.inp'
                   ;    ;spawn,'gle diff_pat.gle'
                   ;endelse

                   ;
                   ; analyze result
                   ;
                   a=rascii('diff_pat.dat')
                   data = a[[0,5],* ] ; get columns 1 and 6 (pi-pol)
                   data_s = a[[0,6],* ] ; get columns 1 and 7 (sigma-pol)
                   out[0,i] = scani[i]
                   out[1,i] = getfwhm(data)
                   out[2,i] = integset(data)
                   out[3,i] = max(data[1,*])
                   out[4,i] = getfwhm(data_s)
                   out[5,i] = integset(data_s)
                   out[6,i] = max(data_s[1,*])
                   ;print,'scanval, fwhm, integ, peak: ',out[*,i]
                   if i eq 0 then begin
                     curves = dblarr(scan.npoints,N_elements(a[0,*]))
                   endif
                   curves[i,*]=data_s[1,*]
                endfor
                ; back to initial state and run it
		state.str.parameters = str_par_old 
		Xcrystal_Inp_Write,state, Group=event.top
		Xcrystal_spawn,state, Group=event.top
                ;
                ; display results
                ;
                title = scan.what[1+scan.what[0]]
                xplot,out,coltitles=[title, 'fwhm-p','integral-p','peak-p','fwhm-s','integral-s','peak-s'],wtitle='xcrystal scan'
                xsurface1,curves,scani,reform(data[0,*]),wtitle='xcrystal scan s-pol',xtitle=title
		END


	'SETCOMPLIANCE': BEGIN
		hlp =  ['This option permits to scan the elements of the ',$
                        'compliance tensor against asymmetry angle.',$
			'',$
			'']
                compliance = state.compliance
		Titles=['From: ','To: ','Number of points']
                Flags=Replicate('1',n_elements(Titles))
		Xscrmenu,compliance,Group=event.top,Ncol=1,Action=action,$
		Titles=titles, Flags=flags, $
		/NoType, /Interp, Wtitle='Xcrystal compliance tensor',help=hlp
		IF (action EQ 'DONT') THEN Goto,out
                state.compliance=compliance

		Widget_control,/Hourglass
                ; 
                ; run  external compliance
                ;

                name='compliance.inp'
                ;
                Openw,unit,name,/GET_LUN
                ;
                ;
                icrystal_material = fix( (state.str.parameters.crystal_material)[0]) 
                ianisotropy = fix( (state.str.parameters.anisotropy)[0] )
                
                if icrystal_material GE 5 then begin ; not Si Ge Diamond
                  if ((ianisotropy GE 1) AND (ianisotropy LE 2)) then begin 
                       itmp = dialog_message(/Error, $
                           ['Anisotropy data not available for this crystal', $
                            'Either use isotropic or use external compliance file',$
                            'Please change and run again'] )
                       ianisotropy = 0 ; to finish calculations
                       goto,out
                  endif
                endif
                
                Printf,unit,'2'
                Printf,unit,icrystal_material
                Printf,unit,'0.0'
                itmp = strparse(state.str.parameters.cut,';',mylist)
                if (itmp NE 2) then begin
                  itmp = dialog_message(/Error, $
                  ['Cut syntax is: ',$
                  'valong_X valong_Y valong_Z ; vnorm_X vnorm_Y vnorm_Z ; vperp_x vperp_Y vperp_Z', $
                  'Please change and run again'] )
                endif
                Printf,unit,mylist[0]
                Printf,unit,mylist[1]
                Printf,unit,mylist[2]
                wtitle='valong(x2): '+mylist[0]+', vnorm(x3): '+mylist[1]+', vperp(x1): '+mylist[2]
                itmp = execute("valong=["+strsubstitute(mylist[0],' ',',')+"]")
                itmp = execute("vnorm=["+strsubstitute(mylist[1],' ',',')+"]")
                itmp = execute("vperp=["+strsubstitute(mylist[2],' ',',')+"]")
                Printf,unit,'1' ; yes scan
                Printf,unit,StrCompress(compliance.From,/Remove_All)
                Printf,unit,StrCompress(compliance.To,/Remove_All)
                Printf,unit,StrCompress(compliance.nPoints,/Remove_All)
                
                ;
                Free_Lun,unit
                ;
                command = 'compliance < compliance.inp'
                Xop_Spawn,command, CleanFiles=['compliance.spec']
                ;spawn,'./compliance < compliance.inp',/sh
                ;
                ;
                ; display results
                ;
                xplot,spec='compliance.spec',group=event.top,title=wtitle
                ; 
                ; use xianbo's code
                ;
                ;scan_calc_s,valong=valong,vnorm=vnorm,from=compliance.from,to=compliance.to,$
                ;   npoints=compliance.npoints
		END
        'HELP': Xhelp,'xcrystal',Group=event.top
	'SHOW':	BEGIN
		WIDGET_CONTROL,event.id,GET_VALUE=eventval
		root='diff_pat' 
		ffile = root+'.dat'
		size_inp = SIZE(state.str.parameters)
		IF (checkfile(ffile) EQ 0 OR  size_inp(1) EQ 0) THEN BEGIN
		  message,/info,'File not found. '
		  itmp = Dialog_Message(dialog_parent=event.top,$
			/ERROR,['Crystal data file not found...',$ 
			'Set Parameters before...  '])
		  Goto,Out
		ENDIF 
		  title='Crystal '+xcrystal_version()+' '+ $
state.str.parameters.Crystal_material( fix(strcompress(state.str.parameters.Crystal_material(0),/rem)) + 1) +$
' '+STRCOMPRESS(state.str.parameters.Miller_index_h,/REM)+STRCOMPRESS(state.str.parameters.Miller_index_k,/REM)+STRCOMPRESS(state.str.parameters.Miller_index_l,/REM)+' E='+string(abs(state.str.parameters.Energy),format='(I7)')+' eV, t='+string(state.str.parameters.Thickness,format='(g8.4)')+' cm'
                  if (fix(state.str.parameters.mosaic[0]) gt 1) then begin
                    if (abs(state.str.parameters.rmer) GT 1e-6) then $
                        title=title+', Rm='+string(state.str.parameters.rmer,format='(g9.7)')+' cm'
                    if (abs(state.str.parameters.rsag) GT 1e-6) then $
                        title=title+', Rs='+string(state.str.parameters.rsag,format='(g9.7)')+' cm'
                  endif
		  CASE eventval OF
		  "Diffraction curve": $
			XPLOT,SPEC=root+'.dat',$
			wtitle='xcrystal results', TITLE=title,GROUP=event.top,$
			xtitle='-1',ytitle='-1'
		  "Crystal Parameters":XDisplayFile1,root+'.par',Group=event.top
		  "Crystallographic Parameters":XdisplayFile1,$
		     state.str.bragg.parameters.outfil,Group=event.top
		  "Schematic draw": begin
                     xwindow,window_size=[600,600] , $
                     buffer= read_textfile('diff_pat.xop') ,$
                     edit=2,wtitle="Schematic draw",Group=event.top
                     end
		  else: message,/info,'case item not found'
		  ENDCASE
		  END  ;END SHOW
  ENDCASE
out:
Widget_Control, stateid, Set_Uvalue=state, /No_Copy
END
;
;==============================================================================
;
PRO Xcrystal, GROUP=group, InputFile=inputFile

Forward_Function xop_defaults

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XCRYSTAL: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

if xregistered('xcrystal') gt 1 then return
; if not(keyword_set(group)) then xop_wd,/default
;
; widget definition
;

base=WIDGET_BASE(/COLUMN,TITLE='Xcrystal '+xcrystal_version(),$
	MBAR=wMenuBar)

wButtons = widget_base(base,/Column) ; also to store state

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
;  wtmp = widget_button(wFile,VALUE='Load Xcrystal input file...', $
;		UVALUE='FILELOAD')
;  wtmp = widget_button(wFile,VALUE='Write Xcrystal input file...', $
;		UVALUE='FILEWRITE')

  wtmp0 = widget_button(wFile,VALUE='Xcrystal input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/Separator)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SETPAR')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')
  wtmp = widget_button(wSetParameters,VALUE='Scan Parameter', UVALUE='SETSCAN')
  wtmp = widget_button(wSetParameters,VALUE='Compliance tensor', UVALUE='SETCOMPLIANCE')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wtmp = widget_button(wResults,VALUE='Diffraction curve', UVALUE='SHOW')
  wtmp = widget_button(wResults,VALUE='Crystal Parameters', UVALUE='SHOW')
  wtmp = widget_button(wResults,VALUE='Crystallographic Parameters', UVALUE='SHOW')
  wtmp = widget_button(wResults,VALUE='Schematic draw', UVALUE='SHOW')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='xcrystal', UVALUE='HELP')
;
wtmp = widget_button(BASE,VALUE='Set Parameters', UVALUE='SETPAR')


if sdep() EQ 'WINDOWS' then font = 'VERDANA*BOLD*ITALIC*24' else $
  FONT='-b&h-lucida-bold-i-normal-sans-24-240-75-75-p-151-iso8859-1'
;  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'


wtmp = WIDGET_LABEL( BASE,  VALUE='Crystal' , FONT=font)
wtmp = WIDGET_LABEL( BASE,  VALUE='   (Crystal Diffraction)   ',FONT=font)

;
;state
;
str = xop_defaults('xcrystal')
str_defaults = str.parameters


IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group
    str.parameters = tmp
  ENDIF
ENDIF
wids = {dummy:0L}
scan = {what: ['0','Fixed value (E[eV] or Theta[deg])',$
             'Asymmetry angle [deg] (to surf.)','Crystal Thickness [cm]',$
             'R sagittal [cm]','R meridional [cm]'], $
        from:8000d0,to:10000d0,npoints:5L }
compliance = { from:-360d0,to:360d0,npoints:361L }
state = { str:str, str_defaults:str_defaults, scan:scan, compliance:compliance, wids:wids}
;
;actions
;
Widget_control,Widget_Info(base,/Child),Set_uvalue=state,/No_Copy
Widget_control,base,/Realize
Xmanager,'xcrystal',base,Group_leader=group,/No_Block
end
;
