FUNCTION Read_EDF, input, BLOCK=block, FORGET=forget, BYTSCL=bytscl, $
  Verbose = verbose

;+
; NAME:
;	READ_EDF
; PURPOSE:
;	this function reads a file in the EDF format (ESRF Data Format)
; CATEGORY:
;
; CALLING SEQUENCE:
;	a=read_edf(input)
; INPUTS:
;	input:  name of the file with data (between quotes). 
; 		Alternately, input may be the file name 
;               (may be gzipped with the .gz file extension) or
; 		the mapping structure (from edf_analyzer).
;
; KEYWORD PARAMETERS:
;
;	BLOCK: the block of data where to find the image [default='0']
;		(unsed only for files containing multiple images)
;
;	VERBOSE: set this flag to print some working information (default=1)
;
;	These other keywords are obsolete and are kept for compatibility
;		reasons only:
;
;	FORGET: Sometimes. specially with files generated by bad
;	  written codes, the block has one byte more than the image size.
;	  One (the first or last) characted has been added and must to be
;	  removed for a correct processing. This flag inicates if the
;	  character to be removed is the last one [FORGET='last' , default
;	  option] or the first [FORGET='first'].
;
;	BYTSCL: when set to 1, data is scaled to a byte array.
;
;	
; MODIFICATION HISTORY:
;	by M. Sanchez del Rio. ESRF. Grenoble, March 1994
;	2004-06-10 Almost completely rewritten.
;	2006-10-19 srio@esrf fixes a bug with data types
;       2008/07/31 srio@esrf.eu adapted for compressed files
;-	

on_error,2
IF N_Elements(verbose) EQ 0 THEN verbose = 0
if not(keyword_set(block)) then block='0' else block=strcompress(block,/rem)
if not(keyword_set(forget)) then forget='last' else $
  forget=strcompress(forget,/rem)
;
IF N_Elements(input) EQ 0 THEN BEGIN
  file = Dialog_Pickfile()
  IF file EQ '' THEN Message,'No input file'
  input = file
ENDIF

input_type = (size(input))((size(input))(0)+1)
if input_type eq 7 then edf_analyzer,input,filemap1,sizeShit=sizeShit  ; filename
if input_type eq 8 then filemap1 = input ; structure

;
; get the useful keywords
;
IF Keyword_Set(sizeShit) THEN BEGIN
  Size=String(sizeShit)
ENDIF ELSE BEGIN
  Size = EDF_GetKeyword(filemap1,'Size',header=block)
ENDELSE
IF verbose THEN print,'READ_EDF: Size keyword is = ',Size
Dim_1 = EDF_GetKeyword(filemap1,'Dim_1',header=block)
IF verbose THEN print,'READ_EDF: Dim_1 keyword is = ',Dim_1
Dim_2 = EDF_GetKeyword(filemap1,'Dim_2',header=block)
IF verbose THEN print,'READ_EDF: Dim_2 keyword is = ',Dim_2

if (  (Size eq 'ErrorFindingKeyword') or  $
  (Dim_1 eq 'ErrorFindingKeyword') or  $
  (Dim_2 eq 'ErrorFindingKeyword') ) then begin
  itmp = Dialog_Message(/Error, $
  ['Keywords Size or/and', $
  'Dim_1 or/and Dim_2', $
  'not recognized in', $
  'the selected block...', $
  'Are you sure it is ', $
  'an image or 3D plot?', $
  'Action not considered. '])
  Return,0
endif 

;
; swapping bytes information
;

ByteOrder = EDF_GetKeyword(filemap1, $
  'ByteOrder',header=block)
ByteOrder = strcompress(ByteOrder,/REMOVE_ALL)
IF verbose THEN print,'READ_EDF: ByteOrder keyword is = ',ByteOrder

LITTLE_ENDIAN = (BYTE (1, 0, 1))[0]
CASE byteorder OF
  'LowByteFirst':  IF LITTLE_ENDIAN EQ 1 THEN swap_flag = 0 ELSE swap_flag =1
  'HighByteFirst': IF LITTLE_ENDIAN EQ 1 THEN swap_flag = 1 ELSE swap_flag =0
  else: swap_flag =0
ENDCASE
IF swap_flag THEN IF verbose THEN print,'READ_EDF: swapping bytes needed'

;
DataType = EDF_GetKeyword(filemap1, $
    'DataType',header=block)
DataType = strcompress(DataType,/REMOVE_ALL)
IF verbose THEN print,'READ_EDF: DataType keyword is = ',DataType



StartBlock = filemap1.blockpos(0,fix(block))
EndBlock = filemap1.blockpos(1,fix(block))
SizeBlock=1+ EndBlock - StartBlock


IF filemap1.compress EQ 0 THEN BEGIN
If (Size ne SizeBlock) then begin
  print,'READ_EDF: ********** WARNING **********'
  print,'READ_EDF: Different sizes found by analyzer and Size keyword'
  print,'READ_EDF:      Size = ',Size
  print,'READ_EDF:      SizeBlock = ',SizeBlock
  if forget EQ 'last' then begin
    EndBlock = EndBlock - 1
    SizeBlock = SizeBlock - 1
    print,'READ_EDF: Last Byte elliminated.'
  endif else if forget EQ 'first' then begin
    StartBlock = StartBlock + 1
    SizeBlock = SizeBlock - 1
    print,'READ_EDF: First Byte elliminated.'
  endif else print,'READ_EDF: keyword FORGET not understood.'
  print,'READ_EDF: ********** ******* **********'
endif 
ENDIF

IF verbose THEN print,'READ_EDF: SizeBlock = ',long(SizeBlock)
  
bytesperpixel = long(SizeBlock)/long(Dim_1)/long(Dim_2)

IF verbose THEN print,'READ_EDF: '+StrCompress(bytesperpixel,/Rem)+ $
  ' calculated bytes per pixel'

if bytesperpixel eq 0 then begin
  itmp = Dialog_Message(/Error, ['0 bytes per pixel found...', $
		' Wrong file'])
  return,0
endif
;
CASE  datatype of
	'UnsignedByte': 	idltype='Byte'		; 1byte 
	'SignedShort': 		idltype='Integer'	; 2 
	'UnsignedShort': 	idltype='Unsigned Integer'
	'SignedInteger': 	idltype='Longword integer' ; 4
	'UnsignedInteger': 	idltype='Unsigned Longword Integer'
	;'SignedLong': 		idltype='64-bit Integer'
	'SignedLong': 		idltype='Longword Integer'
	;'UnsignedLong': 	idltype='Unsigned 64-bit Integer'
	'UnsignedLong': 	idltype='Unsigned Longword Integer'
	'FloatValue': 		idltype='Floating point'
        ; added srio@esrf.eu 2009-12-02 to read fit2d files
	'FLOATVALUE': 		idltype='Floating point'
	'FLOAT': 		idltype='Floating point'  ; fit2d
	'DoubleValue':		idltype='Double-precision floating'	; 8 -> 8
	else:			Message,'Not implemented datatype: '+datatype
ENDCASE

IF verbose THEN print,'READ_EDF: EDF datatype: ',datatype
IF verbose THEN print,'READ_EDF: IDL datatype: ',idltype

;
; initialize image to its new value
;
case idltype of
	'Byte': 	        image = bytarr(Dim_1,Dim_2)
	'Integer': 	        image = intarr(Dim_1,Dim_2)
	'Unsigned Integer':     image = uintarr(Dim_1,Dim_2)
	'Longword integer': 	image = lonarr(Dim_1,Dim_2)
	'Unsigned Longword Integer': 	image = ulonarr(Dim_1,Dim_2)
	'Floating point': 	        image = fltarr(Dim_1,Dim_2)
	'Double-precision floating': 	image = dblarr(Dim_1,Dim_2)
	'64-bit Integer':		image = lon64arr(Dim_1,Dim_2)
	'Unsigned 64-bit Integer':	image = ulon64arr(Dim_1,Dim_2)
	ELSE: Message,'Error, case not found:'+ idltype
endcase  


;IF verbose THEN print,'READ_EDF: reading from file '+filemap1.name

openr,Unit,filemap1.name,/get_lun,compress=filemap1.compress ; ,/Swap_If_Little_Endian
point_lun,Unit,StartBlock
readu,Unit,image
free_lun,Unit

;IF verbose THEN print,'READ_EDF: Done reading file '+filemap1.name+'.'

if swap_flag then image = swap_endian(image)

if keyword_set(bytscl) then image = bytscl(image)

Return,Image

end
