FUNCTION periodictable_data,z

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       PERIODICTABLE_DATA
;
; PURPOSE:
;	Returns a text array with the atomic properties
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	txt = periodictable_data(z)
; INPUT:
;	z: an integer with the z value of the desired element
;	
; OUTPUT:
;	txt: a string array with the info. 
;
; PROCEDURE:
;	It uses DABAX files and applications
;
; EXAMPLES:
;       ; for iron:
;	xdisplayfile1,text=periodictable_data(26)
;
;
; MODIFICATION HISTORY:
;       2007-10-23 srio@esrf.eu Created
;-

on_error,2
;
; returns P.T. data for a fiven element with Z
;

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(Dialog_Parent=group, $
   /Error,'PERIODICTABLE_DATA: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   Return,''
endif
symbol=StrCompress(atomic_symbols(z))
;
; Print Name 
; 
txt=symbol+' Z='+StrCompress(z,/Rem)+'  '+atomic_names(z)
text=['','','', $
'*********************************************************************************',$ 
'Data for: ','                 '+txt,$
'*********************************************************************************' ]

;
; atomic weights
;
;dxFile = 'AtomicWeights.dat'
;h = Dabax_access(dxFile)
;scans = dabax_select(h,atomic_symbols(z),Return=1)
;nScans = N_Elements(scans)
;txt = StrArr(nScans+1)
;for j=0,nscans-1 do begin
;  tmp = spec_data(h,scans[j],index=1,/string)
;  ncol = spec_columns(h,scans[j],index=1)
;  labels = spec_labels(h,scans[j],index=1)
;  name = spec_name(h,scans[j],index=1)
;  tmp1='----'+name
;  txt[j]=string(name,tmp[0],tmp[1],Format='(A14,2G14.4)')
;endfor
;text = [text, ' ', $
;        '------  Atomic weights  ------',$
;        '        (Data from DABAX file: '+dxFile+')  ', $
;        string('Isotope','Atomic Mass','Abundance[%]',Format='(3A14)'),$
;        txt]


files = [$
  'AtomicConstants.dat', $
;  'AtomicDensities.dat', $
;  'Econfiguration.dat', $
  'EBindEner2.dat' ] ;, $
  ;'XREmission_NIST.dat' ] 
titles = [$
  'Some atomic constants',$
;  'Density from several sources [g/cm^3]',$
;  'Electronic configuration [electrons]',$
  'Binding energies [keV]' ] ; ,$
  ;'X-ray emission lines [eV]']

FOR i=0L,N_Elements(files)-1 DO BEGIN
  dxFile = files[i]
  h = Dabax_access(dxFile)
  IF (spec_scan(h,z))[0] NE -1 THEN BEGIN
  tmp = spec_data(h,z,index=0,/string)
  ncol = spec_columns(h,z,index=0)
  labels = spec_labels(h,z,index=0)
  tmp1=' '
  tmp1='----  '+titles[i]+'  ----'
  txt = StrArr(ncol)
  for j=0,ncol-1 do begin
    tmp1=[tmp1,labels[j]+': '+strcompress(tmp[j],/rem)]
    txt[j] = String(labels[j],tmp[j],Format='(A35,G14.7)')
    ;print,tmp1[j]
  endfor
  text = [text, ' ', $
        '------  '+titles[i]+'  ------',$
        '        (Data from DABAX file: '+dxFile+')  ',txt]
  ENDIF ELSE BEGIN
  text = [text, ' ', $
        '------  '+titles[i]+'  ------',$
        '        (No Data for Z='+StrCompress(Z,/rem)+' in DABAX file: '+dxFile+')  ']
  ENDELSE
ENDFOR

;
; XrayEmissionLines
;
dxFile = 'XREmission_NIST.dat'
title = 'X-ray emission lines and radiation rates'

h = Dabax_access(dxFile)
IF (spec_scan(h,z))[0] NE -1 THEN BEGIN
  tmp = spec_data(h,z,index=0,/string)
  ncol = spec_columns(h,z,index=0)
  labels = spec_labels(h,z,index=0)
  tmp1=' '
  tmp1='----  '+title+'  ----'
  txt = StrArr(ncol)
  txt0 = String('Line(IUPAC)','Line(Siegbahn)','Energy[eV]','Wavelength[A]','Rad rate',$
       Format='(5A16)')
  for j=0,ncol-1 do begin
    tmp1=[tmp1,labels[j]+': '+strcompress(tmp[j],/rem)]
    tmpsieg = xrf_iupac2siegbahn(labels[j])
    IF N_Elements(tmpsieg) EQ 2 THEN tmpsieg=tmpsieg[0]+' '+tmpsieg[1]
    IF tmpsieg[0] EQ '-1' THEN tmpsieg=''
    tmprad = xrf_radrate(z,labels[j])
    IF tmprad[0] EQ 0 THEN tmprad='' ELSE tmprad=String(tmprad[0],Format='(G16.7)')
    txt[j] = String(labels[j],tmpsieg[0],tmp[j],physical_constants('hc')/tmp[j],tmprad,$
             Format='(2A16,2G16.7,A16)')
    ;print,tmp1[j]
  endfor
  text = [text, ' ', $
        '------  '+title+'  ------',$
        '        (X-ray emission data from DABAX file: '+dxFile+')  ',$
        '        (Radiation rates from DABAX file: RadiativeRates_KrauseSchofield.dat)',$
        ;'        (Data from DABAX file: '+dxFile+')  ',$
        txt0,txt]
  ENDIF ELSE BEGIN
  text = [text, ' ', $
        '------  '+title+'  ------',$
        '        (No Data for Z='+StrCompress(Z,/rem)+' in DABAX file: '+dxFile+')  ']
  ENDELSE








dxFile='f0_xop.dat'
ratio=FindGen(7)
tmp = f0_calc(dxFile,atomic_symbols(z),ratio)
txt = StrArr(N_Elements(ratio))
FOR i=0L,N_Elements(ratio)-1 DO BEGIN
  txt[i]=String(ratio[i],tmp[i],Format='(2G20.3)')
ENDFOR
  text = [text, ' ', $
        '------  Elastic scattering f0 at ratio=sin(theta)/lambda  ------',$
        '        (Data from DABAX file: '+dxFile+')  ',$ 
        String('ratio','f0[electron units]',  $
           Format='(2A20)'), $
        txt]



energy=[100.,1000,5000,10000,15000,20000,30000,50000,100000]
nEnergy = N_Elements(energy)

dxFile='f1f2_Windt.dat'
f1f2=f1f2_calc(dxFile,atomic_symbols(z),energy)
delta=f1f2_calc(dxFile,atomic_symbols(z),energy,F=3)
beta1=f1f2_calc(dxFile,atomic_symbols(z),energy,F=4)
txt = StrArr(nEnergy)
FOR i=0L,nEnergy-1 DO BEGIN
  txt[i] = string(energy[i],f1f2[0,i],f1f2[1,i], $
    delta[i],beta1[i],delta[i]/beta1[i],Format='(6G12.5)')
ENDFOR
  text = [text, ' ', $
        '------  Scattering data at several x-ray energies  ------',$
        '        (Data from DABAX file: '+dxFile+')  ',$
        String('Energy[eV]',"f'","f''",'Delta','Beta','Delta/Beta', $
           Format='(6A12)'), $
        txt]


dxFile='CrossSec_XCOM.dat'
tmp1 = cross_calc(dxFile,atomic_symbols(z),energy,colTitles1,$
   Partial='TotalCrossSection[',Unit=1)
tmp2 = cross_calc(dxFile,atomic_symbols(z),energy,colTitles2,$
   Partial='TotalCrossSection[',Unit=2)
tmp3 = cross_calc(dxFile,atomic_symbols(z),energy,colTitles3,$
   Partial='TotalCrossSection[',Unit=3)
FOR i=0L,N_Elements(energy)-1 DO BEGIN
  txt[i]=String(energy[i],tmp1[i],tmp2[i],tmp3[i],Format='(4G15.5)')
ENDFOR
colTitles1 = StrSubstitute(colTitles1,'TotalCrossSection','CS')
colTitles2 = StrSubstitute(colTitles2,'TotalCrossSection','CS')
colTitles3 = StrSubstitute(colTitles3,'TotalCrossSection','CS')

  text = [text, ' ', $
        '------  X-ray attenuation at several x-ray energies  ------',$
        '        (Data from DABAX file: '+dxFile+')  ',$
        String('Energy[eV]',colTitles1,colTitles2,colTitles3, $
           Format='(4A15)'), $
        txt]

RETURN,text

END



