function bragg_calc,inp,group=group,verbose=verbose, $
  anomalous=anomalous,  $ 
  crystalCell=cell,crystalStr=str,m0=m0,         $   ; input/output 
  f0=f0coeff,f1=f1inp,f2=f2inp                       ; input/output

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       BRAGG_CALC
;
; PURPOSE:
;	Preprocessor for Structure Factor (FH) calculations.
;	It calculates the basic ingredients of FH.
;	
; CATEGORY:
;       X-Ray optics.
;
; CALLING SEQUENCE:
;	out = bragg_calc(inp)
; INPUTS:
;	inp: a structure with the input data. This structure may
;		be created by either dabax_defaults('bragg') or
;		bragg_inp() and modified by bragg_inp(ask={1,2}.
;
;       verbose: set this keyword to 1 to display info messages
;		(d-spacing, etc.)
;
;	anomalous: flag to consider (anomalous=1=default) or 
;		   do not consider (anomalous=0) anomalous scattering
;                  factors (i.e., if anomalous=0 then set f'=f''=0)
;
;	
; KEYWORDS:
;
; OUTPUT:
;	out: a structure with the output data. This structure is the
;		input of other programs (as crystal_fh) and can be 
;		dumped to disk by using bragg_out.
;
;	group = The widget ID of the widget that calls the routine.
;		This is passed to Dialog_Message in its dialog_parent
;		keyword to center the window message.
;
; INPUT/OUTPUT
;	crystalCell:
;	crystalStr:
;	f0:
;	f1:
;	f2:
;	M0:
;	       Set this keywords to named variables with value zero 
;	       to get the corresponding values load from DABAX files. 
;              This is useful for making multiple calls to bragg_calc()
;              for the same crystal structure, changing for instance the
;              miller indices, without re-reading the information from
;              DABAX, thus accelerating the execution time
;
;       Note that M0 is used for calculating the DebyeWaller = exp[-M0*ratio^2].
;	Eventually the user may want to set this to zero for the calculation and 
;       not for retrieving the calculated M0. In this case, set M0=0. or M0=0D,
;       and will be used for the calculation. If set to M0=0, then M0 will be 
;       loaded with the internally calculated value. 
;
; PROCEDURE:
;	Reads the crystal structure, f0, f' and f" from the
;	DABAX library and returns some crystal parameters, the 
;	geometrical part of FH and the atomic part F' and F".
;	It has been adapted from bragg 2.2 (SHADOW ray-tracing package).
;	It uses the following DABAX files:
;	  CrystalStructures.dat, CrystalCell.dat, f1f2_*.dat, f0_*.dat
;	  (note that not all datafiles f0_* are used, only the ones
;	  having f0 parametrizated, not tabulated).
;
; EXAMPLE:
;	inp = bragg_defaults()
;	out = bragg_calc(inp)
;
; EXAMPLE2: 
;       ; Suppose we want to compute the Structure Factor for 
;       ; miller indices [h,1,1] , h=1,2,3,...10
;     
;       ; first call to bragg_inp for reading files for subsequential calls
;	inp = bragg_defaults(ask=2)
;       crystalCell=0 ; must be defined to zero and will hold the loaded values
;       crystalStr=0
;       f0=0
;       f1=0
;       f2=0
;       ; first call to fill the crystal and f values...
;       inp2 = bragg_calc1(inp,crystalCell=crystalCell,crystalStr=crystalStr, $
;              f0=f0,f1=f1,f2=f2)
;       and now the loop:
;       FOR h=1,10 DO BEGIN
;         inp.parameters.hmiller=h
;         inp2 = bragg_calc1(inp,crystalCell=crystalCell,crystalStr=crystalStr, $
;                f0=f0,f1=f1,f2=f2,verbose=0)
;         crystal_fh,inp2,1000
;       ENDFOR
;
;
; SEE ALSO:
;       bragg_inp, bragg_out, dabax_defaults.
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-12-03 srio@esrf.fr: adapt to new bragg structure. It allows
;		the selection of f1f2 file. It users now f1f2_calc.pro
;       97-01-14 srio@esrf.fr: adapts to Windows.
;       97-10-08 srio@esrf.fr: fixes no-absorption case.
;       97-10-16 srio@esrf.fr: Avoids to stop the program when a scan
;		(usually H) is not found. Uses sdep, adds group kw.
;	98-10-13 srio@esrf.fr uses catch. 
;	02-10-09 srio@esrf.fr adds calculation of DebyeWaller factor. 
;	06-10-09 srio@esrf.fr bugg fixed for 1 point calculation
;	06-11-23 srio@esrf.fr adds crystalCell, crystalStr, f0, f1, f2
;                keywords for avoiding accessing many times the DABAX
;                files in multiple calculations with the same structure.
;		 Calculates volume and d-spacing using bragg_metrictensor()
;       07-01-31 srio@esrf.fr adds the M0 keyword
;       07-09-24 srio@esrf.eu bug fixed: the occupation number was not
;		correctly used. It failed when using different occupations 
;		for the same element. Now occupation number is ised to
;		compute G, and FRACT is always set to 1.
;       13-01-30 srio@esrf.eu adds compton correction. Exit a new tag
;               fcompton that contains the correction factor:
;                  if i_absorp=0,1 : 1.0
;                  if i_absorp=2 : (mu_ph+mu_compton)/mu_ph 
;                  if i_absorp=3 : (mu_ph+mu_compton+mu_rayleigh)/mu_ph 
;
;-
;on_error,2

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'BRAGG_CALC: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

if n_elements(verbose) EQ 0 then verbose=1
if n_elements(anomalous) EQ 0 then anomalous=1

if sdep(/w) then ask=2 else ask=1
if n_params() NE 1 then begin
  inp = bragg_inp(ask=ask)
endif


ifilef0 = inp.parameters.filef0(0) + 1
filef0 = inp.parameters.filef0(ifilef0)
ifilef1f2 = inp.parameters.filef1f2(0) + 1
filef1f2 = inp.parameters.filef1f2(ifilef1f2)
ifileCrossSec = inp.parameters.fileCrossSec(0) + 1
fileCrossSec = inp.parameters.fileCrossSec(ifileCrossSec)
ilattice = inp.parameters.ilattice(0) + 1
hmiller=inp.parameters.hmiller 
 kmiller=inp.parameters.kmiller 
  lmiller=inp.parameters.lmiller
I_ABSORP = fix(inp.parameters.i_absorp(0))
;; TEMPER1 = inp.parameters.temper ; moved below
EMIN = inp.parameters.emin 
 EMAX = inp.parameters.emax 
  ESTEP = inp.parameters.estep
OUTFIL = inp.parameters.outfil


IF keyword_Set(cell) AND keyword_Set(str) THEN reLoad=0 else reLoad=1

IF reload THEN BEGIN
  IF verbose THEN message,/info,'BRAGG_CALC Loading crystal files...'
  ;
  ; load crystal structure and cell parameters from files
  ;
  index = ilattice
  dabax_crystals,index-1,str=str,cell=cell
ENDIF 

;
; Calculate the volume of the unit cell and interlattice distance d
; following, for example, "X-ray diffraction", B. E. Warren, Dover, 1990
;
	M_REF = double([0,hmiller,kmiller,lmiller])
	D_LATT_A = cell(0)*1.0D-8
	D_LATT_B = cell(1)*1.0D-8
	D_LATT_C = cell(2)*1.0D-8
	D_LATT_ALPHA  = cell(3)*!dpi/180.
	D_LATT_BETA  =  cell(4)*!dpi/180.
	D_LATT_GAMMA  = cell(5)*!dpi/180.



	;d_latt_volume = D_LATT_A*D_LATT_B*D_LATT_C * SQRT(1+ $
	;    2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)*cos(D_LATT_GAMMA) $
	;    - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	;    - cos(D_LATT_GAMMA)^2               )

        d_latt_volume = bragg_metrictensor(cell,/return_volume)*1D-24

	;oneoverd2 = 1.0D0 + 2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)* $
	;	  cos(D_LATT_GAMMA) - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	;    - cos(D_LATT_GAMMA)^2
	;oneoverd2 = 1.0D0/oneoverd2       
	;oneoverd2 = oneoverd2  * ( $
	; (M_REF(1)*sin(D_LATT_ALPHA)/D_LATT_A)^2 + $
	; (M_REF(2)*sin(D_LATT_BETA)/D_LATT_B)^2 + $
	; (M_REF(3)*sin(D_LATT_GAMMA)/D_LATT_C)^2 + $
	; 2.0D0*M_REF(1)*M_REF(2)/D_LATT_A/D_LATT_B* $
	;     (cos(D_LATT_ALPHA)*cos(D_LATT_BETA)-cos(D_LATT_GAMMA)) + $
	; 2.0D0*M_REF(2)*M_REF(3)/D_LATT_B/D_LATT_C* $
	;     (cos(D_LATT_BETA)*cos(D_LATT_GAMMA)-cos(D_LATT_ALPHA)) + $
	; 2.0D0*M_REF(3)*M_REF(1)/D_LATT_A/D_LATT_C* $
	;     (cos(D_LATT_GAMMA)*cos(D_LATT_ALPHA)-cos(D_LATT_BETA)) $
	;	  			  )      
	;SP_HKL = 1.0D0/SQRT(oneoverd2)
	SP_HKL = bragg_metrictensor(cell,hkl=[M_REF(1),M_REF(2),M_REF(3)])*1D-8
        SP_HKL = SP_HKL[0]
	R_NATOM = 1.0D0/d_latt_volume

	IF verbose THEN BEGIN
           message,/info,'Interplanar distances (Angs) '+$
                'for '+vect2String(Fix(M_Ref[1:3]))+' :' +$
		strcompress(SP_HKL*1.0D8,/rem)
	   message,/info,'d_latt_volume (A^3) : ' +$
		strcompress(d_latt_volume*1D24,/rem)
	   message,/info,'d_latt_volume (cm^3) : ' +$
		strcompress(d_latt_volume,/rem)
	   message,/info,'1/V=r_natom (A^-3) :' +$
		strcompress(R_NATOM*1D-24,/rem)
	   message,/info,'1/V=r_natom (cm^-3) :' +$
		strcompress(R_NATOM,/rem)
	ENDIF

	E2_MC2  =  2.817939D-13    ; e^2/mc^2 classical e- radius in cm
	RN = E2_MC2*R_NATOM

;
; fo the non-dispersive part of the atomic scattering factor 
;
;
;        print, '************************************************'
;        print, 'Atomic scattering factor is defined by '
;        print, '            fo + f'' + if"'
;        print, ' where fo = fo(SIN(theta)/Lambda) '
;        print, ' is the non-dispersive part'
;        print, ' and f'', f" (Lambda) are the dispersive part.'
;        print, '***********************************************'
;

	zcol = str(0,*)
	fcol = str(1,*)
	batom =  zcol(uniq(zcol,sort(zcol)) )
	nbatom = n_elements( batom )

;
; access to f0, fp and fpp from database
;

IF keyword_Set(f0coeff) AND keyword_Set(f1inp) AND keyword_Set(f2inp) THEN BEGIN
  reLoadF=0 
  fp=f1inp
  fpp=f2inp
ENDIF ELSE BEGIN 
  reLoadF=1
ENDELSE

NPOINT  = (EMAX - EMIN)/ESTEP + 1
IF NPOINT LE 0 THEN NPOINT=1
energy = dblarr(npoint)
for i=0,npoint-1 do energy(i) = emin+estep*i

IF reLoadF THEN BEGIN
  IF verbose THEN message,/info,'BRAGG_CALC: reading atomic factors from DABAX files...'
  ;        print,' fo is obtained by the parametrization values'
  ;        print,' read from file $SHADOW_DATA/TABLE0, which '
  ;        print,' stores data from elements and some ionized'
  ;        print,' states.'

	;tmp = spec_access(h,filef0)
	h = dabax_access(filef0)
	for ii=0,nbatom-1 do begin
	  tmp = double(spec_data(h,batom(ii)))
	  if ii EQ 0 then f0coeff= dblarr(nbatom,n_elements(tmp))
	  f0coeff(ii,*) = tmp
	endfor


  ;
  ; Retrieve the corrections to the atomic scattering factor
  ; from the SHADOW optical constant library.  Note that it is assumed
  ; that the corrections are independent of k, Also note that READLIB
  ; obtains the real part of f(k=0) = f0(k=0) + f1(k=0) =  Z  + f1(k=0)
  ; and so Z is subtracted from f1 to obtain f1(k=0).
  ; We call readlib for any element obtaining : an array with the 
  ; 420 energy points in eV and logarithmic scale, the atomic number 
  ; atnum, the atomic weight atwt, the absorption coefficient rmu, 
  ; and 420 values for the f1 and f2, the real and imaginary (respectively)
  ; dispersive parts of the atomic scattering factor. 
  ;

	fp = dblarr(nbatom,npoint)
	fpp = dblarr(nbatom,npoint)
	fcompton = dblarr(nbatom,npoint)+1.0d0
        IF i_absorp GT 0 THEN BEGIN
          for ii=0,nbatom-1 do begin
	    sym = atomic_symbols(batom(ii))
	    f1f2 = f1f2_calc(filef1f2,sym,energy,group=group)
	    if ((n_elements(f1f2) EQ 1) and (f1f2(0) EQ 0)) then begin
	      if ask EQ 2 then begin
	        itmp = Dialog_Message(dialog_parent=group,/Question,$
	        ['Error calculating f1f2 values for '+sym,'Continue?'])
	      endif else begin
	        itmp=0
	        message,/info,'Error calculating f1f2 values for '+sym+'.'
	        read,'>>> Continue? [1=Yes,0=No] ',itmp
	        if itmp EQ 1 then itmp='Yes' else itmp = 'No'
	      endelse
	      if itmp EQ 'No' then return,0
	      fp(ii,*) = 0.0+BATOM(ii)
	      fpp(ii,*) = 0.0
	    endif else begin
	      fp(ii,*) = f1f2(0,*)
	      fpp(ii,*) = f1f2(1,*)
	    endelse
          endfor
        ENDIF
        IF i_absorp GT 1 THEN BEGIN ; need compton etc corrections
          for ii=0,nbatom-1 do begin
	    sym = atomic_symbols(batom(ii))
	    mu = cross_calc(fileCrossSec,sym,energy,partial='all',col_title,unit=2,group=group)
	    if ((n_elements(mu) EQ 1) and (mu(0) EQ 0)) then begin
	      if ask EQ 2 then begin
	        itmp = Dialog_Message(dialog_parent=group,/Question,$
	        ['Error calculating CrossSec values for '+sym,'Continue?'])
	      endif else begin
	        itmp=0
	        message,/info,'Error calculating CrossSec values for '+sym+'.'
	      endelse
	      return,0
	    endif 
            tmp = strpos(col_title,'Rayleigh(coherent)[') 
            ir = (where(tmp GE 0))[0]
            tmp = strpos(col_title,'PhotoElectric[') 
            ip = (where(tmp GE 0))[0]
            tmp = strpos(col_title,'Compton(incoherent)[') 
            ic = (where(tmp GE 0))[0]
            IF i_absorp EQ 2 THEN BEGIN ; compton correction
	       fcompton(ii,*) = (mu[ip,*]+mu[ic,*])/mu[ip,*]
            ENDIF ELSE BEGIN ; (compton+rayleigh) correction
	       fcompton(ii,*) = (mu[ip,*]+mu[ic,*]+mu[ir,*])/mu[ip,*]
            ENDELSE
          endfor
        ENDIF
ENDIF

;
; Compute the geometrical part G's of the structure factor
;
	
	CI = dcomplex(0.0,1.0)
	G = replicate(dcomplex(0.0,0.0),nbatom)
	G_BAR = G
        for i = 0, nbatom-1 do begin
		ibatom = batom(i)
		tmp = where(zcol EQ ibatom)
		for j=0,n_elements(tmp)-1 do begin
			X = str(2,tmp(j))
			Y = str(3,tmp(j))
			Z = str(4,tmp(j))
			FCOEF = str(1,tmp(j))
			ttt = exp(ci*2*!dpi*  $
			  (X*M_REF(1) + Y*M_REF(2) + Z*M_REF(3)))
                	G(i) = G(i) + ttt*FCOEF
;			ttt = exp(ci*2*!dpi*  $
;			  (-X*M_REF(1)  -Y*M_REF(2)  -Z*M_REF(3)))
;                	G_BAR(i) = G_BAR(i) + ttt
		endfor
        G_BAR(i) = CONJ(G(i))
	endfor
;
; Crystal absorption and Temperature factor
;
;

; If NO crystal absorption is selected, set to zero the imaginary
; part of all the atomic form factor
; if i_absorp EQ 0 then fpp = fpp*0.0

;
; Now prepare the file for SHADOW.
;

ATNUM= fix(batom)

;FRACT = dblarr(nbatom)
;for i=0,nbatom-1 do FRACT(i)=(fcol( where(Zcol EQ batom(i))))(0)
;
; srio 07/09/24 set FRACT always to 1 because fcol (FCOEF) is already 
; considered in the Geometric part of the structure factor. 
; This fixed a bug with different occupation numbers for atoms corresponding
; to the same element 
FRACT = dblarr(nbatom)+1D0

G_0 = dblarr(nbatom)
for i=0,nbatom-1 do G_0(i)=n_elements( where(Zcol EQ batom(i)))

NPOINT = long(NPOINT)
F2 = FPP
F1 = dblarr(nbatom,npoint)

; substract Z only if anomalous scattering factors are considered
IF Total(ABS(FP)) EQ 0 THEN BEGIN
  for j=0,nbatom-1 do F1(j,*) = FP(j,*)
ENDIF ELSE BEGIN
  for j=0,nbatom-1 do F1(j,*) = FP(j,*)-BATOM(j)
ENDELSE


;
; temperature parameter
;

IF Keyword_set(M0) THEN BEGIN
  ratio = 1D/ (2D*SP_HKL*1.0D8)
  temper=exp(-m0*ratio^2)
ENDIF ELSE BEGIN
  ; 
  ; If M0=0. (float or double) use it for calculations, do not recalculate
  ;
  iCalcDW=1
  IF N_Elements(M0) EQ 1 THEN BEGIN
    IF type(M0) EQ 4 OR type(M0) EQ 5 THEN BEGIN 
      iCalcDW=0
      ratio = 1D/ (2D*SP_HKL*1.0D8)
      temper=exp(-m0*ratio^2)
    ENDIF
  ENDIF 
  IF iCalcDW THEN BEGIN
    TEMPER1 = inp.parameters.temper
    IF StrCompress(temper1,/Rem) EQ '?' THEN BEGIN
      temper1 = -300.0
      editFlag=1
    ENDIF ELSE BEGIN
      temper1 = Float(temper1)
      editFlag=0
    ENDELSE
    
    IF temper1 LT 0.0 THEN BEGIN
      ratio = 1D/ (2D*SP_HKL*1.0D8)
      temper=DebyeWaller(batom,Ratio=ratio, Temper=Abs(temper1), M0=M0, Edit=editFlag )
    ENDIF ELSE BEGIN
      ratio = 1D/ (2D*SP_HKL*1.0D8)
      TEMPER = replicate(TEMPER1,nbatom)
      M0 = -alog(TEMPER)/(ratio^2)
    ENDELSE
  ENDIF
ENDELSE

IF anomalous EQ 0 THEN BEGIN
  Message,/Info,'Anomalous scattering factors neglected.'
  f1=f1*0
  f2=f2*0
  fp=fp*0
  fpp=fpp*0
  fcompton=fcompton*0d0+1d0
ENDIF


D_SPACING = sp_hkl

; Return values in keyword
IF reLoadF THEN BEGIN
  f1inp = fp
  f2inp = fpp
  fcomptoninp = fcompton
ENDIF

out = { outfil:outfil, RN:rn, d_spacing:d_spacing, nbatom:nbatom, $
 atnum:atnum, fract:fract, TEMPER:temper, g_0:g_0, $
 G:G, G_BAR:G_BAR, f0coeff:f0coeff, $
 NPOINT:NPOINT, energy:ENERGY, f1:f1, f2:f2, fcompton:fcompton }

return,out

end

