;+ 
;      ============================= XPeakFit  ======================
;
; 
; XPeakFit is a graphical interface that allows to fit (x,y) data
; with a number of pseudo=-voigt functions and a linear background.
;
; The interface allows to:
;    Get automatic initial values (guess)
;    Get initial values with cursor.
;    Overplot data and model
;    Performs non-linear fit (using the MPFit routine of Craig B.  Markwardt, 
;      NASA/GSFC Code 662, Greenbelt, MD 20770, craigm@lheamail.gsfc.nasa.gov
;      http://astrog.physics.wisc.edu/~craigm/idl.html based on the 
;      Levenberg-Marquardt least-squares minimization (MINPACK-1)
;    Any parameter can be fixed during the fitting process
;    
;
;  This interface is included in the XPlot application of the XOP package.
;
;  Written by Manuel Sanchez del Rio, January 1999.
;  Acknowledments: 
;	1) The main idea of this interface is based on a similar
;	   application developed by Sean Brennan in MatLab.
;	2) Special thanks to Craig B. Markwardt for allowing the 
;	   use and redistribution of the MPFit routines.
;
;
;
;
;
; =================  Non-Interactive use of XpeakFit =======================
;	
;	NAME:
;		XPeakFit
;
;	PURPOSE:
;		Widget interface to non-linear fits with peudo-Voigt
;		functions (a linear combination of a Gaussian and a
;		Lorentzian).
;
;	CALLING SEQUENCE:
;		XPeakFit,set 
;
;	INPUT PARAMETERS:
;		set = a data set with the data (FltArr(2,nPts)) to be
;			fitted. If undefined, a sample data is created.
;
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls XPeakFIt.
;			When this ID is specified, a death of the caller 
;			results in a death of PowellFit.
;		No_Block = A boolean indicating if the No_Block keyword is 
;			passed to Xmanager (default: No_Block=1).
;		NPeaks = The initial number of peaks (default: NPeaks=3)
;		MaxPeaks = The maximum number of peaks allowed. This
;			value can only be defined at initialization time. 
;		
;	OUTPUTS:
;		Performs a fit. 
;
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;		Loads the Textronix colot table (tek_colors)
;
;	PROCEDURE:
;		Used PeakFinder to guess the starting parameters and 
;		MPFit for the non-linear fit.
;	
;	MODIFICATION HISTORY:
;       	writtem by  Manuel Sanchez del Rio. ESRF. 99-01-28
;		99/03/18 srio@esrf.fr bug fixed when freezing straight line.
;		01/01/25 srio@esrf.fr gaussian/lrentzian parameter
;			forced to be in the [0.0,1.0] interval.
;-
;
;============================================================================
;
FUNCTION XPeakFit_Eval, _expr, x, p

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error,$
    'XPeakFit_Eval: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN, x*0
ENDIF

_cmd = '_f = '+_expr
_err = Execute(_cmd)
RETURN, _f

END ; XPeakFit_Eval
;
;============================================================================
;
FUNCTION XPeakFit_Get, nPeaks, $
  Expression=expression,Start=start, Flags=flags, $  ; keywords
  strCoeff=strCoeff
;
; Gets the expression of the fitting function . Also returns 
; the initial values and fixed-par flags.
;
; Usage:
; expr = XPeakFit_Get(npeaks, /Expression)
; start = XPeakFit_Get(npeaks, /Start, StrCoeff=strCoeff)
; flags = XPeakFit_Get(npeaks, /Flags, StrCoeff=strCoeff)
;

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'XPeakFit_Get: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF
IF N_Params() EQ 0 THEN Message,'Usage expr = XPeakFit_Get(npeaks, ...) '

IF Not(keyword_Set(expression)) AND Not(keyword_Set(start)) AND $
  Not(keyword_Set(flags)) THEN expression=1

IF Keyword_Set(expression) THEN BEGIN
  expr='P(1)+P(0)*x '
  FOR i=1,nPeaks DO BEGIN
     expr = expr + ' + voigt1(x, P('+StrCompress(4*(i-1)+2,/Rem)+':'+$
	StrCompress(4*(i-1)+5,/Rem)+'))'
  ENDFOR
  RETURN,expr
ENDIF


IF Keyword_Set(start) THEN BEGIN
  IF Not(Keyword_Set(strCoeff)) THEN Message,$
	'Usage: expr = XPeakFit_Get(/start,StrCoeff=strCoeff)'
  start =  FltArr(4*nPeaks+2)
  start[0:1] = (strCoeff[0].(0))[0:1]
  IF nPeaks GE 1 THEN BEGIN
    istart = 2
    FOR i=1,nPeaks DO BEGIN
      start[istart:istart+3] = (strCoeff[i].(0))
      istart=istart+4
    ENDFOR
  ENDIF
  RETURN,start
ENDIF

IF Keyword_Set(flags) THEN BEGIN
  IF Not(Keyword_Set(strCoeff)) THEN Message,$
	'Usage: expr = XPeakFit_Get(/flags,StrCoeff=strCoeff)'
  flags =  IntArr(4*nPeaks+2)
  ;flags[0:1] = (strCoeff[0].(1))[0:1]
  tmp = (strCoeff[0].(1))
  flags[0:1] = tmp[0:1]
  IF nPeaks GE 1 THEN BEGIN
    iflags = 2
    FOR i=1,nPeaks DO BEGIN
      flags[iflags:iflags+3] = (strCoeff[i].(1))
      iflags=iflags+4
    ENDFOR
  ENDIF
  RETURN,flags
ENDIF
END ;  XPeakFit_Get

;
;============================================================================
;

PRO XPeakFit_MPFit_Display, fcn, x, iter, FunctArgs=fcnargs, FMT=fmt, $
         Quiet=quiet, _Extra=iterargs

Forward_Function mpfit_enorm
; this routine is called by MPFit

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'XPeakFit_MPFit_Display: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Keyword_Set(quiet) THEN RETURN
fvec = Call_Function(fcn, x, _Extra=fcnargs)
fnorm = mpfit_enorm(fvec)

XDisplayfile1_Append, [String(iter, fnorm^2, $
  Format='("Iter ",I6,"   CHI-SQUARE = ",G20.8)'), $
  '           Timing [sec]:  '+$
	String(SysTime(1)-iterargs.t0,Format='(G6.2)')]
IF N_Elements(fmt) GT 0 THEN BEGIN
    XDisplayfile1_Append, String(p, format=fmt)
ENDIF ELSE BEGIN
    p = '  P('+StrTrim(LIndGen(N_Elements(x)),2)+') = ' $
      + StrTrim(String(x,Format='(G20.6)'),2) + '  '
    XDisplayfile1_Append, String('         '+p, Format='(A)')
ENDELSE

IF (SysTime(1)-iterargs.t0) GE iterargs.tmax THEN BEGIN
  itmp = Widget_Message(Dialog_Parent=iterargs.parent, $
	['Maximum time ('+StrCompress(Long(iterargs.tmax))+' sec) reached.',$
	 'Fitting process aborted.'])
  !err=-1
ENDIF
  
RETURN
END ; XPeakFit_MPFit_Display

;
;============================================================================
;

PRO XPeakFit_UpdateCoeff,strCoeff,wids

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'XPeakFit_UpdateCoeff: '+'error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

tmp=0
FOR i=0L,N_Elements(strCoeff)-1 DO BEGIN
  Widget_Control,wids.coeff[i],Get_Value=tmp
  IF i EQ 0 THEN BEGIN
    tmp1=(strCoeff[i].(0))
    tmp1[0:1]=tmp.(0)
    strCoeff[i].(0)=tmp1

    tmp1=(strCoeff[i].(1))
    tmp1[0:1]=tmp.(1)
    strCoeff[i].(1)=tmp1

    ;(strCoeff[i].(0))[0:1]=tmp.(0)
    ;(strCoeff[i].(1))[0:1]=tmp.(1)
  ENDIF ELSE BEGIN
    strCoeff[i]=tmp
  ENDELSE
ENDFOR
END ; XPeakFit_UpdateCoeff

;
;============================================================================
;

PRO XPeakFit_UpdatePanels,strCoeff,wids

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'XPeakFit_UpdatePanels: '+'error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

tmp=0
FOR i=0L,N_Elements(strCoeff)-1 DO BEGIN
  Widget_Control,wids.coeff[i],Set_Value=strCoeff[i]
ENDFOR
END ; XPeakFit_UpdatePanels

;
;============================================================================
;
PRO XPeakFit_Event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPEAKFIT_EVENT: '+'error caught: '+!error_state.msg)
  Catch, /Cancel
  GoTo,out
ENDIF

;
; register the events
;
stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uValue=state,/No_copy

Widget_Control,event.id,GET_UValue=uval

wset,state.window

CASE StrMid(uval,0,4) OF
  'Quit': BEGIN
	IF Ptr_Valid(state.ptrData) THEN Ptr_Free,state.ptrData
	IF Ptr_Valid(state.ptrStrCoeff) THEN Ptr_Free,state.ptrStrCoeff
	Widget_Control,event.top,/Destroy
	END
  'Pref': BEGIN ; Preferences
	str0=state.str
	str = {tol:str0.tol, maxIter:str0.maxIter, errors:str0.errors, $
	  nPrint:str0.nPrint, timeLimit:str0.timeLimit}
	action=''
	flags=Replicate('1',N_tags(str))
	titles=['Fitting tolerance','Max of fitting iterations',$
	  'Data errors', 'printing frequency','time limit [sec]']
	XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
	  Flags=flags, Titles=titles, WTitle='XPeakFit preferences'
	IF action EQ 'DONT' THEN GoTo,out
	Copy_Structure,str,str0
	state.str = str0
	END
  'Over': BEGIN
	set = *state.ptrData
	WSet,state.window
	x = set[0,*]
	y = set[1,*]
	nn=N_elements(x)
	plot,x,y,color=1
	;
	strCoeff = *(state.ptrStrCoeff)
	expr = XPeakFit_Get(state.str.nPeaks,/Expression)
	r = XPeakFit_Get(state.str.nPeaks,/Start,strCoeff=strCoeff)
	OPlot,x,XpeakFit_Eval(expr, x, r),Color=2
	END

  'Expo': BEGIN
	set = *state.ptrData
	x = Reform(set[0,*])
	y = Reform(set[1,*])
	;
	strCoeff = *(state.ptrStrCoeff)
	expr = XPeakFit_Get(state.str.nPeaks,/Expression)
	r = XPeakFit_Get(state.str.nPeaks,/Start,strCoeff=strCoeff)
	yf = XPeakFit_eval(expr,x,r)
	set = make_Set(x,y,yf)
	Widget_Control,event.id,Get_Value=tmp
	CASE tmp[0] OF
	  'Export to Xplot': BEGIN
		Widget_Control,/HourGlass
		Xplot,set,Group=event.top,ColTitles=['X','Y','Model'],$
		  Ycol=2,WTitle='data from XPlot/XPeakFit', $
		  No_Block=state.no_Block
		END
	  'Save data to a file...': BEGIN
		file = ''
		WHILE file EQ '' DO BEGIN
		  file = Dialog_PickFile(Dialog_Parent=event.top, /Write, $
		    file='peaks.dat')
		  IF file EQ '' THEN GoTo,out
		  IF CheckFile(file) THEN BEGIN
		    itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
		      ['File '+file+' already exists.','Overwrite it?'])
		    IF itmp EQ 'No' THEN file = ''
		  ENDIF
		ENDWHILE
		OpenW,unit,file,/get_Lun
		Printf,unit,'#F '+file
		Printf,unit,'#S 1 data from Xplot/XPeakFit'
		Printf,unit,'#C '
		Printf,unit,'#C File created by XPeakFit on '+SysTime()
		Printf,unit,'#C '
		Printf,unit,'#C Model function: '+expr
		Printf,unit,'#C P parameters: '+Vect2String(r)
		Printf,unit,'#N 3'
		Printf,unit,'#C '
		Printf,unit,'#L X  Y  YModel'
		FOR i=0L,N_Elements(x)-1 DO $
		  PrintF,unit,x[i],y[i],yf[i]
		Free_Lun,unit
		Message,/Info,'File '+file+' written to disk.'
		  
		END
	ENDCASE
	END
  'SetA': BEGIN
	tmp=''
	Widget_Control,event.id,Get_Value=tmp
	strCoeff = *(state.ptrStrCoeff)
	CASE tmp[0] OF
	  'Set all Gaussians': BEGIN
		tmp0=(strCoeff(*).(0))
		tmp0[3,*]=0.0
		FOR i=0,N_Elements(strCoeff)-1 DO strCoeff(i).(0)=tmp0[*,i]
		END
	  'Set all Lorentzians': BEGIN
		tmp0=(strCoeff(*).(0))
		tmp0[3,*]=1.0
		FOR i=0,N_Elements(strCoeff)-1 DO strCoeff(i).(0)=tmp0[*,i]
		END
	  else: Print,'Error.......'
	ENDCASE
	XPeakFit_UpdatePanels,strCoeff,state.wids
	*(state.ptrStrCoeff) = strCoeff
	END
  'PFLD': BEGIN
	set = *state.ptrData
	WSet,state.window
	x = set[0,*]
	y = set[1,*]
	nn=N_elements(x)
	plot,x,y,color=1
	;
	; update variable
	;
	strCoeff = *(state.ptrStrCoeff)
	wids = state.wids
	XPeakFit_UpdateCoeff,strCoeff,wids
	; 
	; in case that a number is changed, update plot
	;
	IF event.type EQ '' THEN BEGIN
	  ; 
	  expr = XPeakFit_Get(state.str.nPeaks,/Expression)
	  r = XPeakFit_Get(state.str.nPeaks,/Start,strCoeff=strCoeff)
	  OPlot,x,XpeakFit_Eval(expr, x, r),Color=2
	ENDIF
	; 
	; in case that "G" is pressed, get values with mouse
	;
	IF event.type EQ 'G' THEN BEGIN
	  fieldnum=Fix(Strmid(uval,4,Strlen(uval)-1))
	  fieldpos=event.index
	  Widget_Control,state.wids.coeff[fieldnum],Get_value=val
	  IF fieldnum EQ 0 THEN BEGIN
	    CASE fieldpos OF
	      0:BEGIN
		tmp = (val.(0))[0]
		action=''
		XEdit,tmp,Text='New pendent value ',$
		Action=action,Title='Change value', $
		Dialog_Parent=event.top
		IF action EQ 'CANCEL' THEN GoTo,out
		tmp0=(val.(0))
		tmp0[0]=tmp
		val.(0)=tmp0
		END
	      1:BEGIN
		tmp = (val.(0))[1]
		action=''
		XEdit,tmp,Text='New value (intersection with y axis) ',$
		Action=action,Title='Change value', $
		Dialog_Parent=event.top
		IF action EQ 'CANCEL' THEN GoTo,out
		tmp0=(val.(0))
		tmp0[1]=tmp
		val.(0)=tmp0
		END
	    ENDCASE
	    Widget_Control,state.wids.coeff[fieldnum],Set_value=val
	  ENDIF ELSE BEGIN
	    IF fieldpos EQ 0 THEN fieldpos=1
	    CASE fieldpos OF
	      1:BEGIN
		IF state.ask EQ 1 THEN BEGIN
		  itmp = Widget_Message(/Question,Dialog_parent=event.top,$
		   ['Please click One point to define the maximum and center',$
		    'of your peak. ',$
		    '(In other cases click twice to get the FWHM.)','',$
		    'Do you want to present this message next time?'])
		  IF itmp EQ 'No' THEN state.ask=0
		ENDIF
		Cursor,xx,yy,Wait=3,/Data
		OPLot,[xx,xx],[yy,yy],Color=2,Psym=2
		tmp0=(val.(0))
		tmp0[0:1]=[yy,xx]
		val.(0)=tmp0
		Widget_Control,state.wids.coeff[fieldnum],Set_value=val
		END
	      2:BEGIN
		IF state.ask EQ 1 THEN BEGIN
		  itmp = Widget_Message(/Question,Dialog_parent=event.top,$
		   ['Please click twice to get the FWHM.','',$
		    '(In other cases single-click to define the max and center.)',$
		    '(In other cases click twice to get the FWHM.)','',$
		    'Do you want to present this message next time?'])
		  IF itmp EQ 'No' THEN state.ask=0
		ENDIF
		Cursor,xx1,yy1,Wait=3,/Data
		OPLot,[xx1,xx1],[yy1,yy1],Color=2,Psym=2
		Wait,0.2
		Cursor,xx2,yy2,Wait=3,/Data
		OPLot,[xx2,xx2],[yy2,yy2],Color=2,Psym=2
		fwhm = Abs(xx1-xx2)
		OPLot,[xx1,xx2],[yy1,yy2],Color=3
		tmp0=(val.(0))
		tmp0[2]=fwhm
		val.(0)=tmp0
		Widget_Control,state.wids.coeff[fieldnum],Set_value=val
		END
	      3:BEGIN
		tmp = (val.(0))[3]
		action=''
		XEdit,tmp,Text='New value (0=Gaussian, 1=Lorentzian)',$
		  Action=action,Title='Change value', $
		  Dialog_Parent=event.top
		IF action EQ 'CANCEL' THEN GoTo,out
		tmp0=(val.(0))
		tmp0[3]=tmp
		val.(0)=tmp0
		Widget_Control,state.wids.coeff[fieldnum],Set_value=val
		END
	    else:
	    ENDCASE
	  ENDELSE
	ENDIF
	
	XpeakFit_UpdateCoeff,strCoeff,state.wids
	*(state.ptrStrCoeff) = strCoeff
	END
  'PlDt': BEGIN
	set = *state.ptrData
	WSet,state.window
	Plot,set[0,*],set[1,*],Color=1
	END
  'SFit': BEGIN
	set = *state.ptrData
	WSet,state.window
	x = set[0,*]
	y = set[1,*]
	nn=N_elements(x)
	Plot,x,y,Color=1
	strCoeff = *state.ptrStrCoeff
	; 

	expr=XPeakFit_Get(state.str.nPeaks,/Expression)
	start=XPeakFit_Get(state.str.nPeaks,/Start,strCoeff=strCoeff)
	startFlags=XPeakFit_Get(state.str.nPeaks,/Flags,strCoeff=strCoeff)
	CASE StrCompress(state.str.errors[0],/Remove_All) OF
	  '0': BEGIN
		errtxt = 'Using constant errors'
		Message,/Info,errtxt
		rerr=y*0+1
		END
	  '1': BEGIN
		errtxt = 'Using statistical errors'
		Message,/Info,errtxt
		rerr = Sqrt(y)  ; 1*sigma = sqrt(y)
		END
	  else:
	ENDCASE
	parinfo = replicate({value:0.D, fixed:0, limited:[0,0], $
	limits:[0.D,0]}, state.str.nPeaks*4+2)
	parinfo(*).value = start
	parinfo(*).fixed = not(startFlags)+2
	; set boundaries of the gauss/lorentzian parameter in [0.0,1.0]
	FOR iii=5,(state.str.nPeaks*4+2)-1,4 DO BEGIN
	  parinfo[iii].limited=[1,1]
	  parinfo[iii].limits=[0.0,1.0]
        ENDFOR
	tol=state.str.tol
	tmax=state.str.timeLimit
	covar=0
	status=0
	
	Widget_Control,/HourGlass
	XDisplayfile1,Text=['','Running MPFit on '+SysTime(),'',$
	  'Function Expression: '+expr, errtxt, $
	  'Initial parameters: '+Vect2String(start),$
	  'Fixed params (0=free,1=fix):'+Vect2String(parinfo(*).fixed),'',''],$
	  Group=event.top, Title='Peak fitting results'
	r = mpfitexpr(expr, x, y, rerr, ParInfo=parInfo, $
	  FTol=tol, XTol=tol, GTol=tol, $
	  NPrint=state.str.nprint, MaxIter=state.str.maxIter, $
	  IterProc='xpeakfit_mpfit_display',Status=status, $
	  IterArgs={t0:systime(1), tmax:tmax, parent:event.top} )
	CASE status OF
	  0:  text='improper input parameters.'
	  1:  text=['both actual and predicted relative reductions',$
		'in the sum of squares are at most FTOL(=TOL).']
	  2:  text=['relative error between two consecutive iterates',$
		'is at most XTOL(=TOL)']
	  3:  text=['both actual and predicted relative reductions',$
		'in the sum of squares are at most FTOL(=TOL). And, ',$
	  	'relative error between two consecutive iterates',$
		'is at most XTOL(=TOL)']
	  4:  text=['the cosine of the angle between fvec and any column',$
		'of the jacobian is at most GTOL(=TOL) in absolute value.']
	  5:  text='the maximum number of iterations has been reached'
	  6:  text=['FTOL(=TOL) is too small. no further reduction in the ',$
		'sum of squares is possible.']
	  7:  text=['XTOL(=TOL) is too small. no further improvement in the ',$
		'approximate solution x is possible.']
	  8:  text=['GTOL(=TOL) is too small. fvec is orthogonal to the ',$
		'columns of the jacobian to machine precision.']
	  else: text='Unknown.'
	ENDCASE


	; moved before XDisplayFile1 msr 010125
	OPlot,x,XpeakFit_Eval(expr, x, r),Color=2
	
	XDisplayFile1_Append,['','MPFit termination status: ',text,$
	  '','Resulting params: '+Vect2String(r),$
	  '']


	;
	; update results
	;
	itmp = Dialog_Message(Dialog_Parent=event.top, /Question,$
	  'Update parameters with fit results?' )
	IF itmp EQ 'No' THEN GoTo,out
	;
	tmp = (strCoeff[0].(0))
	tmp[0:1]=r[0:1]
	strCoeff[0].(0)=tmp
	;
	FOR i=1,state.str.npeaks DO BEGIN
	  tmp = (strCoeff[i].(0))
	  tmp=r[(4*(i-1)+2):(4*(i-1)+5)]
	  strCoeff[i].(0)=tmp
	ENDFOR

	;FOR i=0,N_Elements(strCoeff)-1 DO BEGIN
	  ;tmp =  strCoeff[i]
	  ;Widget_Control,state.wids.coeff[i],Set_Value=tmp
	;ENDFOR
	XpeakFit_UpdatePanels,strCoeff,state.wids
	*(state.ptrStrCoeff)=strCoeff
	END
  'Gues': BEGIN
	set = *state.ptrData
	WSet,state.window
	x = set[0,*]
	y = set[1,*]
	nn=N_elements(x)
	Plot,x,y,Color=1
	;
	; calls PeakFinder
	;
	pcutoff=0 & CLimits=0 & npeaks=0
	a=PeakFinder(y,x,PCutoff=pcutoff,CLimits=CLimits,$
	  NPeaks=npeaks,/Sort,/Opt)
	strCoeff = *state.ptrStrCoeff
	nmax = nPeaks < state.str.nPeaks
	; take only nmax peaks and sort it as a function of abscissas
	a=a[*,0:nmax-1]
	itmp = Sort(a[1,*])
	a=a[*,itmp]
	;
	; linear part
	;
	itmp = LIndGen(nn)
	itmp = Where(itmp LE 10 OR itmp GT (nn-11))
	xtmp=x[itmp]
	ytmp=y[itmp]
	lcoeff = LinFit(xtmp,ytmp)
	OPlot,x,x*lcoeff[1]+lcoeff[0],color=3
	strCoeff(0).(0) = [lcoeff[1],lcoeff[0],0,0]
	;
	; This part overplots Lorentzian functions estimated from 
	; the PeakFinder results
	;
	FOR i=0,nmax-1 DO BEGIN
	  ypeak = a[2,i]
	  xpeak = a[1,i]
	  imin = a[0,i]+(a[5,i]/2)
	  imax = a[0,i]-(a[5,i]/2)
	  xfwhm = Abs(x[imax]-x[imin])
	  OPlot,x,Voigt1(x,[ypeak,xpeak,xfwhm,0]),Color=4+i
	  strCoeff(i+1).(0) = [ypeak,xpeak,xfwhm,0]
	ENDFOR

	FOR i=0,N_Elements(strCoeff)-1 DO BEGIN
	  tmp =  strCoeff[i]
	  Widget_Control,state.wids.coeff[i],Set_Value=tmp
	ENDFOR
	*(state.ptrStrCoeff)=strCoeff
	END

  'NPks': BEGIN
	;
	nn_old = state.str.nPeaks
	nn_new=0
	Widget_Control,state.wids.nPeaks,Get_Value=nn_new

	IF nn_new GT state.str.maxPeaks THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    ['Maximum number of peaks: '+StrCompress(state.str.maxPeaks),$
	     '',$
	     '(to increase it, use the maxPeaks keyword when starting '+$
		'XPeakFit)'] )
	  Widget_Control,state.wids.nPeaks,Set_Value=nn_old
	  GoTo,out
	ENDIF 
	state.str.nPeaks=nn_new
	FOR i=0,state.str.maxPeaks DO $
		Widget_Control,state.wids.coeffBase[i],Map=1
	FOR i=state.str.nPeaks+1,state.str.maxPeaks DO $
		Widget_Control,state.wids.coeffBase[i],Map=0
	
	END
  'Help': BEGIN
	tmp=''
	Widget_Control,event.id,Get_Value=tmp
	Xhelp,StrLowCase(tmp[0]),group=event.top
	END
  else:  Print,'IN ELSE'
ENDCASE

out:
If Type(stateid) EQ 3 THEN $
  IF Widget_Info(stateid,/Valid_Id) EQ 1 THEN $
  Widget_Control, stateid, Set_UValue=state,/No_Copy

END ; XPeakFit_event
;
;============================================================================
;


PRO XPeakFit, set, Group = group, $
 XplotMain=xplotmain, No_Block=no_block, NPeaks=nPeaks, $
 MaxPeaks=maxPeaks

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error,Dialog_Parent=group,$
    'XPeakFit: '+'error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

;
; loads color table with tektronix colors
;
Tek_Color

IF N_Elements(No_Block) EQ 0 THEN no_block=1
IF N_Elements(nPeaks) EQ 0 THEN nPeaks=3
IF N_Elements(maxPeaks) EQ 0 THEN maxPeaks=nPeaks

wTop = Widget_Base(Title='XPeakFit',MBAR=wMenu)
wtmp = Widget_Base(wTop) ; to store state

;
; define data set 
;
IF N_Elements(set) EQ 0 THEN BEGIN
  nn=200
  x=FIndGen(nn)/Float(nn-1)*4
  x=x-2
  y0 = 0.11*x + 1.0
  y1 = Voigt1(x,[10,-0.5,0.1,0.5])
  y2 = Voigt1(x,[10,0,0.2,0.3])
  y3 = Voigt1(x,[10,0.5,0.4,0.0])
  y=y0+y1+y2+y3
  yran=max(y) - min(y)
  yran = (yran * 0.05)*randomu(seed,nn)
  y=y+yran
  set=Make_Set(x,y)
ENDIF

;
; create structure with data
;
tmp = {coeff:Replicate(0.0,4), flags:Replicate(1,4)}
strcoeff = Replicate(tmp,maxPeaks+1)  ; first index is for background

;
; define menu bar
;
wtmp0 = Widget_Button(wMenu, Value='File', /MENU)
 wtmp1 = Widget_Button(wtmp0,Value='Export to Xplot',UValue='Export')
 wtmp1 = Widget_Button(wtmp0,Value='Save data to a file...',UValue='Export')
 wtmp1 = Widget_Button(wtmp0,Value='Preferences...',UValue='Pref')
 wtmp1 = Widget_Button(wtmp0, Value='Quit', UValue='Quit',/Separator)
wtmp0 = Widget_Button(wMenu, Value='Help', /HELP)
 wtmp1 = Widget_Button(wtmp0, Value='XPeakFit', UValue='Help')
 wtmp1 = Widget_Button(wtmp0, Value='MPFit', UValue='Help')


wBase = Widget_Base(wTop,/Column)

if not(keyword_set(nodraw)) then $
wDraw = widget_draw(wBase, XSize=525, YSize=300, Retain=2)

wtmp0 = widget_base(wBase,/ROW)
  wtmp = Widget_Button(wtmp0,Value='Quit',UValue='Quit')
  wtmp = Widget_Button(wtmp0,Value='**Start Fit**',UValue='SFit')
  wtmp = Widget_Button(wtmp0,Value='Guess',UValue='Guess')
  wNPeaks = CW_Field(wtmp0,Value=nPeaks,UValue='NPks',/Int,$
    Title='Number of Peaks',XSize=2,/Return_Events)
wtmp0 = widget_base(wBase,/ROW)
  wtmp = Widget_Button(wtmp0,Value='Refresh Screen',UValue='PlDt')
  wtmp = Widget_Button(wtmp0,Value='Oplot model',UValue='Over')
  wtmp = Widget_Button(wtmp0,Value='Set all Gaussians',UValue='SetA')
  wtmp = Widget_Button(wtmp0,Value='Set all Lorentzians',UValue='SetA')

wCoeff=LonArr(maxPeaks+1)
wCoeffbase=LonArr(maxPeaks+1)
wtmp = Widget_Label(wbase,Value='Hints: "G" get value with mouse, "F" Fix parameter')
wtmp0 = widget_base(wBase,/Col)
  tmp = { coeff:(strCoeff[0].(0))[0:1], flags:(strCoeff[0].(1))[0:1]}
  ;tmp = { coeff:[1.0,2.0],flags:[1,0] }
  wtmpbase = Widget_Base(wtmp0)
  wtmp = CW_PField(wtmpbase,UValue='PFLD0',Value=tmp,Title=$
	'y = A x + B (background)',PTitles=['A:','B:'])
  wCoeffbase[0]=wtmpbase
  wCoeff[0]=wtmp
  wtmp = Widget_Label(wtmp0,value='========== Peak ======== Center ======='+$
    ' FWHM  ===== 0:Gauss;1:Lor')
  FOR i=0,maxPeaks-1 DO BEGIN
    tmp = strCoeff[i+1]
    wtmpbase = Widget_Base(wtmp0)
    wtmp = CW_PField(wtmpBase,UValue='PFLD'+StrCompress(i+1,/Rem),Value=tmp,$
	Title='Peak #'+StrCompress(i+1,/Rem))
    wCoeffBase[i+1]=wtmpbase
    wCoeff[i+1]=wtmp
  ENDFOR


Widget_Control,wTop,/Realize 
;
; map 
;
FOR i=nPeaks+1,maxPeaks DO Widget_Control,wCoeffBase[i],Map=0

Widget_Control,wDraw,Get_Value=wnum

ptrData=Ptr_New(0)
ptrStrCoeff=Ptr_New(strCoeff)

str = {maxPeaks:maxPeaks, nPeaks:npeaks, tol:1D-10, maxIter:200, $
  errors:['0','Constant','Statistical'], nPrint:5, timeLimit:30.0  }

wids = { coeff:wCoeff, coeffBase:wCoeffBase, nPeaks:wNPeaks }
state= { window:wnum, wids:wids, ptrData:ptrData, ptrStrCoeff:ptrStrCoeff,$
  str:str, ask:1, no_Block:no_Block }

Widget_control,Widget_Info(wTop,/CHILD),set_uValue=state,/No_copy

; if xplotmain EQ 0 then plotfile,set
Plot,set[0,*],set[1,*],Color=1
*ptrData=set

XManager, 'XPeakFit', wTop, GROUP_LEADER=group,No_Block=no_block

END ; XpeakFit
