;______________________________________________________________________________
;+
; NAME:
;       SPEC_SCAN()
;
; PURPOSE:
;       This function returns a vector loaded with the numbers of a group of
;       scans in a SPEC file.
;       The values returned correspond to either the numbers ('#S' lines) of
;       the scans or their indexes (relative position in the file).
;       This function can be used to check if a scan or group of scans exist
;       in a SPEC data file before calling any other function of the library.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_SCAN(Handle, ScanList [, ErrMsg])
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       ScanList - List of scans in the SPEC data file. ScanList can be a 
;                  scalar or an array. In the case of arrays all the elements 
;                  are included in the list. ScanList can also be of numeric
;                  or string type.
;                  When using character strings one can specify a group of
;                  consecutive scans in compressed form using the syntax
;                  `first-last'. If `first' or `last' are not specified they
;                  are defaulted to the first and last scan in the file. It is
;                  also possible to include several scans or groups of scans
;                  in a single string by using commas or white spaces as
;                  separators.
;                  The values in ScanList follow the same convention used for
;                  scan identifiers (see the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). By default, numeric values are
;                  interpreted as SPEC scan numbers, but this behaviour is
;                  changed by the INDEX keyword.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret the numeric values in ScanList
;               as the indexes of the scans instead of as scan numbers.
;               Care must be taken when using indexes in strings that specify
;               groups of scans. In order to allow the use of negative indexes
;               the sign of the index must be explicitly included when the
;               expression can be ambiguous. As an example compare the
;               following expressions when INDEX is set: 
;                  '-2'    represents the scan before the last one (index=-2).
;                  '--2'   represents the scans from the first to the one
;                           before the last (index=1 to index=-2).
;                  '-+2'   represents the first two scans in the file
;                           (index=1 to index=2).
;
;       RETURN_INDEX: Set this keyword to return the index of the scans instead
;               of the scan numbers.
;
;       OCCURRENCE: Set this keyword to return the scan number and the scan
;               occurrence in the format SSSS.OO as explained in the `SCAN
;               IDENTIFIERS' section at the beginning of this file). This 
;               keyword has not any effect if RETURN_INDEX is also set.
;
;       STRING: Set this keyword to force the output as a string vector instead
;               of float.
;
;       NO_EMPTY: Set this keyword to make the output value of the function
;               refer only to those scans that actually contain standard data
;               columns, excluding the empty ones.
;
;       MCDATA: Set this keyword to make the output value of the function
;               refer only to those scans that contain multichannel data.
;
;       NO_RANGE: Set this keyword to force the function to accept only 
;               single scans as input instead of lists.
;
; OUTPUT: 
;       If there is no error this function returns a vector that contains the
;       list of scans specified in ScanList. If any of the scans is not found
;       or any other error occurs, the function return -1.
;
; OPTIONAL OUTPUTS:
;
;       ErrMsg  - Error message if an error occurs.
;    
; SIDE EFFECTS:
;       The `current' scan is set to the last scan in the list.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;
; EXAMPLE:
;       To print the number of most recently accessed scan, enter:
;
;       PRINT, SPEC_SCAN(OldHandle, 'current')
;______________________________________________________________________________
;-
function spec_scan, handle, scan_list, errmsg, INDEX=idx, RETURN_INDEX=rindex,$
                                               OCCURRENCE=occurr, STRING=str, $
                                               NO_EMPTY=noemp, MCDATA=mca,    $
                                               NO_RANGE=no_range
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   ; Check arguments
   if N_PARAMS() ne 2 and N_PARAMS() ne 3 then begin
      message, 'Incorrect number of arguments.'
   endif
   errmsg = ''

   __speccheck, handle

   aux = size(scan_list)
   nelem = n_elements(scan_list)
   isstring = (aux(aux(0)+1) eq 7)
   if keyword_set(no_range) then begin
      if nelem gt 1 then return, -1
      if isstring then begin
         if strpos(scan_list(0),',') ne -1 or strpos(scan_list(0),'-') ne -1 $
                                                   then return, -1
      endif
   endif

   if not isstring then begin
      for i=0, nelem - 1 do begin
         __specload, handle, scan_list(i), errmsg, INDEX=idx
         if !ERR then return, -1
         if i eq 0 then begin
            indexlist = handle.currscan
         endif else begin
            indexlist = [indexlist, handle.currscan]
         endelse
      endfor
   endif else begin
      _scan_list = scan_list(0)
      for i=1, nelem - 1 do _scan_list = _scan_list + ',' + scan_list(i)
      _scan_list = strcompress(_scan_list, /remove_all)
      p = strpos(_scan_list, ',')
      while p ge 0 do begin
         strput, _scan_list, ' ', p
         p = strpos(_scan_list, ',')
      endwhile
      _scan_list = __linesplit(strcompress(_scan_list), 1, 0, 0)
      ngroups = n_elements(_scan_list)
      if ngroups eq 1 and _scan_list(0) eq '' then return, -1
      for i=0, ngroups-1 do begin
         p = strpos(_scan_list(i), '-')
         if p eq 0 and keyword_set(idx) then begin
            nextchar = strmid(_scan_list(i), 1, 1)
            if nextchar ne '-' and nextchar ne '+' and nextchar ne '' then begin
               p = strpos(_scan_list(i), '-', 1) 
            endif
         endif
         if p ge 0 then begin
            scan0 = strmid(_scan_list(i), 0, p)
            if scan0 eq '' then scan0 = 'FIRST'
            scan1 = strmid(_scan_list(i), p + 1, strlen(_scan_list(i)) - p - 1)
            if scan1 eq '' then scan1 = 'LAST'
         endif else begin
            scan0 = _scan_list(i)
            scan1 = ''
         endelse
         __specload, handle, scan0, errmsg, INDEX=idx
         if !ERR then return, -1 
         firstscan = handle.currscan
         if i eq 0 then begin
            indexlist = firstscan
         endif else begin
            indexlist = [indexlist, firstscan]
         endelse
         if scan1 ne '' then begin
            __specload, handle, scan1, errmsg, INDEX=idx
            if !ERR then return, -1 
            if handle.currscan ge firstscan then delta = 1 else delta = -1
            for j = firstscan + delta, handle.currscan, delta do begin
               indexlist = [indexlist, j]
            endfor
         end
      endfor
   endelse

   if keyword_set(mca) then begin
      dpoints = handle.scan(indexlist).a_datapoints
   endif else if keyword_set(noemp) then begin
      dpoints = handle.scan(indexlist).n_datapoints
   endif else goto, NoFilter
   aux = where(dpoints gt 0)
   if aux(0) eq -1 then return, -1
   indexlist = indexlist(aux)

NoFilter:
   if keyword_set(rindex) then begin
      result = indexlist + 1
   endif else if keyword_set(occurr) then begin
      result = handle.scan(indexlist).scan_n +     $
               handle.scan(indexlist).scan_ap/100.
   endif else begin
      result = handle.scan(indexlist).scan_n 
   endelse

   return, result
end
