PRO Wiggler_Trajectory, $
  inp1, $    ; input structute 
  b_from=b_from, $  	; input keyword
  inFile=inFile, $  	; input keyword
  nPer=nPer, $  	; input keyword
  ener_gev=ener_gev, $  ; input keyword
  nTrajPoints=nTrajPoints,$	; input keyword
  TrajFile=trajFile, $	; input keyword
  ParFile=parFile, $	; input keyword
  Xplot=xplot, $	; input keyword
  kValue=kValue, $	; input keyword
  traj=traj, Pars=pars  ; output keywords

;+
; NAME:
;       WIGGLER_TRAJECTORY
;
; PURPOSE:
;       Calculates the trajectory of the electrons under a magnetic
;	field in Z direction. 
;
; CATEGORY:
;       Synchrotron radiation
;
; CALLING SEQUENCE:
;       Wiggler_Trajectory,inp
;	Wiggler_Trajectory [,Keywords]
;
; INPUTS:
;       inp:  A structure of the type "xwiggler" contaning (for example):
;	** Structure XWIGGLER, 12 tags, length=156, data length=156:
;	   FIELD           STRING    Array[4]
;	   NPERIODS        LONG                 1
;	   ULAMBDA         DOUBLE          0.12500000
;	   K               DOUBLE           14.000000
;	   ENERGY          DOUBLE           6.0400000
;	   [ PHOT_ENERGY_MIN DOUBLE           100.00000 ] 
;	   [ PHOT_ENERGY_MAX DOUBLE           100100.00 ] 
;	   [ NPOINTS         LONG               100 ] 
;	   [ LOGPLOT         STRING    Array[3] ] 
;	   NTRAJPOINTS     LONG               500
;	   [ CURRENT         DOUBLE           200.00000 ] 
;	   FILE            STRING    '?'

;		(tags between brackets are not used). 
;	Alternatively, if not input structure is used, keywords should 
;	be used to define input parameters. If an input structure is
;	used, then input keywords are ignored. 
;
; INPUT KEYWORD PARAMETERS (if no input structure is chosen):
;       b_from:      A Flag for the type of inpyt magnetic field: 
;		       0: kValue (deflecting parameters) is given
;		       1: A file with the magnetic field (y[m] B[T]) is given
;		       2: A file with the magnetic field harmonics (n Bn[T]) is given
;       infile:      The file with the field information (for b_from:1,2)
;       nPer:        Number of periods  (for b_from:1,2,3)
;       per:         Wiggler period in meters  (for b_from:1,3)
;       ener_gev:    The electron energy in GeV
;       nTrajPoints: Number of trajectory points (101 if input structure is used)
;       kValue:      The K (deflecting parameter) value
;
; KEYWORD PARAMETERS 
;       TrajFile:    The name of a file where the resut is written
;	ParFile:     The name of a file where the text info is written
;
; OUTPUT KEYWORD PARAMETERS:
;       TRAJ=traj a named variable to store the output matrix (8 colums with:
;	   x[m]  y[m]  z[m]  BetaX  BetaY  BetaZ  Curvature  B[T] )
;       PARS=pars a named varianle to store text info
;
; OUTPUTS:
;       returns the value in output keywords or in disk files. 
;
; PROCEDURE:
;	Based on btraj.f, a utility writtem in 10/91 by M. Sanchez del Rio
;	  and C. Vettier to input asymmetric wigglers in SHADOW. 
;
;	See formulae in ESRF red book	pag CIV-297
;
; MODIFICATION HISTORY:
;       Written by:     M. Sanchez del Rio, srio@esrf.fr, 2002-07-17
;	2002-07-17 srio@esrf.fr 
;
;-

On_Error,2
;
; Get inputs
;
IF N_Params() EQ 0 THEN BEGIN
	inp = {b_from:0, infile:'', nPer:0L, nTrajPoints:0L, $
		ener_gev:0.0D0, per:0.0D0, kValue:0.0D0 } 
	IF N_Elements(b_from) NE 0 THEN inp.b_from=b_from
	IF N_Elements(infile) NE 0 THEN inp.infile=infile
	IF N_Elements(nPer) NE 0 THEN inp.nPer=nPer ; b_from=0,1,3
	IF N_Elements(per) NE 0 THEN inp.per=per    ; b_from=1,3
	IF N_Elements(ener_gev) NE 0 THEN inp.ener_gev=ener_gev
	IF N_Elements(nTrajPoints) NE 0 THEN inp.nTrajPoints=nTrajPoints ; b_from=1
	IF N_Elements(kValue) NE 0 THEN inp.kValue=kValue ; b_from=3
ENDIF ELSE BEGIN
	inp = {b_from:Fix(inp1.field[0]), infile:inp1.file, $
	nPer:inp1.nPeriods, nTrajPoints:inp1.nTrajPoints, $
	ener_gev:inp1.energy, per:inp1.uLambda, kValue:inp1.k } 
ENDELSE

;
; Read data files or define a single harmonic (if K entered)
;
	IF inp.b_from GT 0 THEN BEGIN
	  IF StrCompress(inp.infile,/Rem) EQ '?' THEN BEGIN
	    file = Dialog_Pickfile(Title='Load file with B data',$
		Dialog_Parent=Dialog_Parent)
	    IF file EQ '' THEN BEGIN
		Message,'Calculation aborted.'
		RETURN
	    ENDIF ELSE inp.infile=file
	  ENDIF
	ENDIF

	CASE inp.b_from OF
	  0: BEGIN	; Entered K value
	    nharm = 1
	    n=[1.0D0]
	    cte = Physical_Constants('ec')/2.0d0/!dpi/ $
		Physical_constants('me')/Physical_constants('c') ; 93.372904
	    bh=[inp.kValue/(cte*inp.per)]
	    ystep = inp.per/(inp.nTrajPoints-1)
	    END
	  1: BEGIN	;	read file with yy[m] B[T]
	    IF CheckFIle(inp.infile) NE 1 THEN $
		Message,'File nor found: '+inp.infile
	    a = rAscii(inp.infile,/Double)
	    yy = Reform(a[0,*])
	    bz = Reform(a[1,*])
	    nmax = N_Elements(bz) + 1
	    nTrajPoints = N_Elements(bz)
	    inp.nTrajPoints= nTrajPoints
	    ystep = yy[1]-yy[0]
	    per  = yy[inp.nTrajPoints-1]-yy[0]
	    inp.per = per
	    print,inp.nTrajPoints ,' points read from file'
	    print,' Period of the ID is : ',inp.per
	    END
	  2: BEGIN	;	read file with period [m], number of periods, harmonics [T]
	    IF CheckFIle(inp.infile) NE 1 THEN $
		Message,'File nor found: '+inp.infile
	    a = rAscii(inp.infile,/Double)
	    n = Reform(a[0,*])
	    bh = Reform(a[1,*])
	    nharm = N_Elements(n)
  
	    print,nharm,' harmonics read from file'
	    print,' Period of the ID is : ',inp.per ,' [m]'
	    print,' Number of periods : ',inp.nper
	    ystep = inp.per/(inp.nTrajPoints-1)
	    END
	ENDCASE


;
;	calculates initial electron speed from its energy
;
	gamma = 1.0D3/physical_constants('mee')*inp.ener_gev
	beta02 = 1.0D0 - gamma^(-2)
	beta0 = sqrt(beta02)

	print,' gamma =',gamma
	print,' beta = v/c =',beta0,Format='(a,G20.10)'
	print,' beta^2 =',beta02,Format='(a,G20.10)'

	start_len=inp.per*(inp.nper-1)/2.d0		

;
;	calculates the integral of the magnetic field bz (proportional
;	to speeds)
;	
	IF (inp.b_from EQ 1) THEN BEGIN
	 ;yint = Total(bz)*ystep
	 ;yint = Int_Tabulated(yy,bz)
	 betaX = DblArr(inp.nTrajPoints)
	 FOR i=1L,inp.nTrajPoints-1 DO BEGIN
	   betaX[i] = Int_Tabulated(yy[0:i],bz[0:i])
	 ENDFOR
	 yInt = betaX(inp.nTrajPoints-1)
	ENDIF ELSE BEGIN
	 phase0=DblArr(inp.nTrajPoints)-!dpi
	 yy = MakeArray1(inp.nTrajPoints,-inp.per/2.d0,inp.per/2.d0)
	 bz = DblArr(inp.nTrajPoints)
	 betaX = DblArr(inp.nTrajPoints)
	 phase=2.0D0*!dpi*(yy/inp.per)
	 FOR n=0,nHarm-1 DO BEGIN
	    bz=bz+bh[n]*cos(phase*(n+1))
	    tmp = (sin(phase*(n+1))-sin(phase0*(n+1)))*(bh[n]/(n+1))
	    betaX=betaX+tmp
	 ENDFOR
	 betax=betax*(inp.per/2.0D0/!dpi)
	ENDELSE

	print,' Integral of B = ',betax(inp.nTrajPoints-1), '  [Tm]',Format='(a,g20.10,a)'

;
;	rescale B to speeds v/c = 0.3/E[GeV] * integral (Bz [T] ds[m])
;
	 betax= -(Physical_Constants('c')*1d-9/inp.ener_gev)*betax
	 betay= sqrt(beta0^2 - betax*betax)
	 emc = Physical_constants('ec')/gamma/Physical_constants('me')/Physical_constants('c')
	 curv= -emc*bz/beta0

;
;	calculates positions as the integral of speeds
;
	yx = DblArr(inp.nTrajPoints)

	IF (inp.b_from EQ 1) THEN BEGIN
	 FOR i=1L,inp.nTrajPoints-1 DO yx[i] = Int_Tabulated(yy[0:i],betaX[0:i])
	ENDIF ELSE BEGIN
		FOR n=0,nharm-1 DO BEGIN
			phase=yy*(2.0D0*!dpi/inp.per)
			yx=yx-(cos(phase*(n+1))-cos(phase0*(n+1)))*(bh[n]/(n+1)^2)
		ENDFOR
		yx=yx*(-3.d-1/inp.ener_gev)*(inp.per/2.0D0/!dpi)^2
	ENDELSE

;
;	creates parameters and file
;

pars = ['                                                              ', $
'##################### Data from xwiggler  ########################', $
'                                                                  ',$
'Period of the ID is [m] :      '+StrCompress(inp.per,/Rem), $
'Number of periods :            '+StrCompress(inp.nper,/Rem), $
'Number of points per periods : '+StrCompress(inp.nTrajPoints,/Rem), $
'Beta = v/c :                   '+String(beta0,Format='(g30.15)'), $
'B = Cte*curv; Cte:             '+StrCompress(-1.0D0*beta0/emc,/Rem), $
'Electron energy [GeV] :        '+StrCompress(inp.ener_gev,/Rem) ] 

CASE inp.b_from OF
  0: BEGIN
     pars = [pars, $
     'K value :                      '+StrCompress(inp.kValue,/Rem), $
     'Maximum of B [T] :             '+StrCompress(bh[0],/Rem) ] 
  END
  1: BEGIN
     pars = [pars, $
     'B[T] profile from file :       '+StrCompress(inp.infile,/Rem), $
     'Integral of B [T.m] : '+StrCompress(betax(inp.nTrajPoints-1),/Rem) ]
  END
  2: BEGIN
     pars = [pars, $
     'B harmonics from file :        '+StrCompress(inp.infile,/Rem), $
     'Number of harmonics :          '+StrCompress(nharm,/Rem) ]
  END
ENDCASE



IF Keyword_Set(ParFile) THEN BEGIN

	openW,unit,parFile,/Get_Lun ;	outfile = rstring ('Output file with parameters ? ')
	  printf,unit,pars
	Free_Lun,unit
ENDIF

;
;	creates trajectory and file 
;

	nPointsTot=inp.nTrajPoints+(inp.nper-1)*(inp.nTrajPoints-1)
	traj = DblArr(8,nPointsTot)
        ii=-1L
	FOR j=0,inp.nper-1 DO BEGIN
	  nn=0
	  IF (j GT 0) THEN nn=1 	; to avoid overlap
	  FOR i=nn,inp.nTrajPoints-1 DO BEGIN
	    ii = ii + 1
	    traj[0,ii] = YX[i]
	    traj[1,ii] = yy[i]+j*inp.per-start_len
            traj[2,ii] = 0.0
	    traj[3,ii] = betax[i]
            traj[4,ii] = betay[i]
            traj[5,ii] = 0.0d0
            traj[6,ii] = curv[i]
            traj[7,ii] = bz[i]
	  ENDFOR
	ENDFOR


IF Keyword_Set(trajFile) THEN BEGIN
	openW,unit,trajFile ;outfile = rstring ('Output file with the trajectory ? ')
	printf,unit,"#F xwiggler.trj"
	printf,unit,"#D "+SysTime()
	printf,unit,"#C Xop/XWiggler results"
	printf,unit,"#S 1 Electron trajectory and velocity"
	printf,unit,"#N 8"
	printf,unit,"#L x[m]  y[m]  z[m]  BetaX  BetaY  BetaZ  Curvature  B[T]"
	FOR j=0,nPointsTot-1 DO BEGIN
	    printf,unit,  traj[*,j] ,Format='(8(1X,G19.12))'
	ENDFOR
	Free_Lun,unit
ENDIF

;
; Display results
;
IF Keyword_Set(xplot) THEN BEGIN
  XPLOT,SPEC="xwiggler.trj",XCOL=2,YCOL=1,title="#s",$
    xtitle='-1',ytitle='-1',WTITLE='xwiggler trajectories: xwiggler.out',no_Block=1
ENDIF

END ; Wiggler_Trajectory
