PRO wiggler_spectrum,inp1, $
  inFile=inFIle, enerMin=enerMin, enerMax=enerMax, $
  nPoints=nPoints,logPlot=logPlot, current=current, $
  outFile=outFile, outVar=out, $
  elliptical=elliptical


;+
; NAME:
;       WIGGLER_SPECTRUM
;
; PURPOSE:
;       Calculates the spectrum of a wiggler using a trajectory file
;	as input. 
;
; CATEGORY:
;       Synchrotron radiation
;
; CALLING SEQUENCE:
;       Wiggler_Spectrum,inp
;	Wiggler_Spectrum [,Keywords]
;
; INPUTS:
;       inp:  A structure of the type "xwiggler" contaning (for example):
;	** Structure XWIGGLER, 12 tags, length=156, data length=156:
;	   [ FIELD           STRING    Array[4] ] 
;	   [ NPERIODS        LONG                 1 ] 
;	   [ ULAMBDA         DOUBLE          0.12500000 ] 
;	   [ K               DOUBLE           14.000000 ] 
;	   [ ENERGY          DOUBLE           6.0400000 ] 
;	   PHOT_ENERGY_MIN DOUBLE           100.00000
;	   PHOT_ENERGY_MAX DOUBLE           100100.00
;	   NPOINTS         LONG               100
;	   LOGPLOT         STRING    Array[3]
;	   [ NTRAJPOINTS     LONG               500 ]
;	   CURRENT         DOUBLE           200.00000
;	   [ FILE            STRING    '?'  ]
;
;		(tags between brackets are not used). 
;	Alternatively, if not input structure is used, keywords should 
;	be used to define input parameters. If an input structure is
;	used, then input keywords are ignored. 
;
; INPUT KEYWORD PARAMETERS (if no input structure is chosen):
;
;       infile:      The file with the electron trajectory
;		     (created by wiggler_trajectory)
;       enerMin:    Minimum photon energy [eV]
;       enerMax:    Maximum photon energy [eV]
;       nPoints:     Number of energy points
;       logPlot:     Flag: 0=linear scaling of energy points, 1=Logarithmic
;	current:     The electron beam current in mA
;
;       outFile:    The name of the file with results
;       outVar:	    A named variable to receive the results:
;		     [energy, flux, power density]
;
; OUTPUTS:
;       A file with the resulting spectrum
;
; PROCEDURE:
;	Based on SHADOW's wiggler_spectrum. Uses Wiggler_NPhoton.pro
;
; MODIFICATION HISTORY:
;       Written by:     M. Sanchez del Rio, srio@esrf.fr, 2002-07-15
;	2002-07-18 srio@esrf.fr adds doc. Use "current" value 
;	2006-06-18 srio@esrf.fr uses hc from Physical_Constants()
;
;-

On_Error,2

IF N_Params() EQ 0 THEN BEGIN
	inp = {inFile:'',i_wig:1,enerMin:0.0D0,enerMax:0.0D0,$
		nPoints:0L,outFile:'',r_intens:100.0D0,logPlot:0}

	IF N_Elements(inFile) NE 0 THEN inp.inFile=inFile
	IF N_Elements(enerMin) NE 0 THEN inp.enerMin=enerMin
	IF N_Elements(enerMax) NE 0 THEN inp.enerMax=enerMax
	IF N_Elements(nPoints) NE 0 THEN inp.nPoints=nPoints
	IF N_Elements(logPlot) NE 0 THEN inp.logPlot=logPlot
	IF N_Elements(current) NE 0 THEN inp.r_intens=current
	IF N_Elements(elliptical) NE 0 THEN inp.i_wig=2

ENDIF ELSE BEGIN
; i_units not used
	inp = {inFile:'xwiggler.trj',i_wig:1, $
	  enerMin:inp1.phot_energy_min, enerMax:inp1.phot_energy_max, $
		nPoints:inp1.nPoints,r_intens:inp1.current,$
		logPlot:Fix(inp1.logPlot[0])}

ENDELSE
;
;	(I_WIG.EQ.1) implies normal wiggler.
;	(I_WIG.EQ.2) implies elliptical wiggler.  (NOT IMPLEMENTED!!)
;

	IF CheckFile(inp.infile) NE 1 THEN Message,'File not found: '+inp.infile

	a = rascii(inp.infile,/double)
	x = Reform(a[0,*])
	y = Reform(a[1,*])
	z = Reform(a[2,*])
	betax = Reform(a[3,*])
	betay = Reform(a[4,*])
	betaz = Reform(a[5,*])
	curv =  Reform(a[6,*])
	np = N_Elements(x)

	Print, 'Read '+StrCompress(NP,/Rem)+' points from input file: '+inp.infile
	STEP	= SQRT((Y[2]-Y[1])^2 + (X[2]-X[1])^2 + (Z[2]-Z[1])^2)
;
; Compute gamma and the beam energy
;
	GAMMA	= 1/SQRT(1-(BETAY[1]^2)-(BETAX[1]^2)-(BETAZ[1]^2))
	BENER	= GAMMA*(9.109D-31)*(2.998d8^2)/(1.602e-19)*1.0d-9
 	Print, 'Beam energy (GeV) = ',BENER

;
; Figure out the limit of photon energy.
;
	CURV_MAX	= 0.0D0
	CURV_MIN	= 1.0D20

	curv_max = Max(ABS(curv))
	curv_min = Min(ABS(curv))

	Print,'Radius of curvature (max.) = ',1.0D0/CURV_MIN,' m'
	Print,'                    (min.) = ',1.0D0/CURV_MAX,' m'

;	TOANGS  =  1.239852D+4
	TOANGS  =  physical_constants('hc')
	PHOT_MIN	= TOANGS*3.0D0*GAMMA^3/4.0D0/!DPI/1.0D10*CURV_MIN
	PHOT_MAX	= TOANGS*3.0D0*GAMMA^3/4.0D0/!DPI/1.0D10*CURV_MAX
	Print, 'Critical Energy (max.) = ',PHOT_MAX,' eV'
	Print, '                (min.) = ',PHOT_MIN,' eV'



	out = DblArr(3,inp.nPoints)
;
;
; starts the loop in energy
;
;	do 555 energy=inp.enerMin,inp.enerMax,esteg

	mul_fac=Abs(curv)*Sqrt(1+(betax/betay)^2+ $
                   (betaz/betay)^2)*1.0D3

	energy_array = MakeArray1(inp.nPoints,inp.enerMin,inp.enerMax,$
		Log=inp.logPlot)

	;estep = (inp1.phot_energy_max-inp1.phot_energy_min)/(inp1.nPoints-1)
	;FOR energy=inp.enerMin,inp.enerMax,estep DO BEGIN

	FOR i=0L,N_Elements(energy_array)-1 DO BEGIN
	energy = energy_array[i]

	;
	; WIGGLER_NPHOTON computes the no. of photons per mrad (ANG_NUM) at each point.
	; It is then used to generate the no. of photons per axial length (PHOT_NUM)
	; along the trajectory S.
	;
	phot_num = DblArr(np)
	rad= Abs(1.0D0/curv)
	ang_num = Wiggler_NPhoton(rad,bener,energy)
;	phot_num=ang_num*Abs(curv)*Sqrt(1+(betax/betay)^2+ $
;                   (betaz/betay)^2)*1.0D3
	phot_num=ang_num*mul_fac
	;
	; Computes CDF of the no. of photon along the trajectory S.
	; In the elliptical case, the entire traversed path length (DS) is computed.
	; In the normal case, only the component (Y) in the direction of propagation
	; is computed.
	;

	;phot_cdf = DblArr(np)

	IF (inp.i_wig EQ 2) THEN BEGIN
 	  ;ds = DblArr(np)
 	  ;dx=ds & dy=ds & dz=ds
 	  ;phot_cdf = DblArr(np)
 	  ;FOR i=1,np-1 DO BEGIN
 	  ;  ds[1-1] = 0.0D0
          ;  DX[I] = X[I] - X[I-1]
          ;  DY[I] = Y[I] - Y[I-1]
          ;  DZ[I] = Z[I] - Z[I-1]
          ;  DS[I] = SQRT(DX[I]^2 + DY[I]^2 + DZ[I]^2) + DS(I-1)
          ;  PHOT_CDF[I]	= PHOT_CDF[I-1] + $
          ;  (PHOT_NUM[I-1] + PHOT_NUM[I])*0.5D0*(DS[I] - DS[I-1])
 	  ;ENDFOR
 	  ;TOT_NUM	= PHOT_CDF(NP-1)
	  dx = x-Shift(x,1)
	  dy = y-Shift(y,1)
	  dz = z-Shift(z,1)
	  ds1 = Sqrt(dx*dx+dy*dy+dz*dz) & ds1[0]=0.0D0
	  ds = DblArr(np)
	  FOR j=1L,np-1 DO ds[j]=Total(ds1[0:j])
          phot_cdf=(Shift(phot_num,1)+phot_num)*0.5D0*(ds-Shift(ds,1))
          phot_cdf[0]=0.0
	  tot_num = Total(phot_cdf)
	ENDIF ELSE BEGIN
	  ;FOR i=1,np-1 DO BEGIN
	  ;   PHOT_CDF[I]	= PHOT_CDF[I-1] + $
	  ;        (PHOT_NUM[I-1] + PHOT_NUM[I])*0.5D0*(Y[I] - Y[I-1])
	  ;ENDFOR
	  ;TOT_NUM	= PHOT_CDF(NP-1)
          phot_cdf=(Shift(phot_num,1)+phot_num)*0.5D0*(y-Shift(y,1))
          phot_cdf[0]=0.0
	  tot_num = Total(phot_cdf)
	ENDELSE

	 ; Print,energy,TOT_NUM*inp.R_INTENS
	 ;Printf,unit,energy,TOT_NUM*inp.R_INTENS

	 out[0,i]=energy
	 out[1,i]=TOT_NUM*inp.R_INTENS
	 out[2,i]=TOT_NUM*inp.R_INTENS*(Physical_Constants('ec')*1d3)
	ENDFOR

IF KeyWord_Set(outFile) THEN BEGIN
  openw,unit,outfile,/get_lun
    printf,unit,"#F "+StrCompress(outFile,/Rem)
    printf,unit,"#D "+SysTime()
    printf,unit,"#C Xop/XWiggler results"
    printf,unit,"#S 1 "
    printf,unit,"#N 3"
    printf,unit,"#L Energy[eV]  Flux[Phot/sec/0.1%bw]  Power[Watts/eV]"
    Printf,unit,out
  Free_Lun,unit
ENDIF

END
