
FUNCTION xy2rphi,img,center, $
   rrange=rrange,nr=nr, phirange=phirange,nphi=nphi, $
   verbose=verbose, axis=axis

;+
; NAME:
;
;	XY2RPHI
;
; PURPOSE:
;
;	this function calculates returns the polar image (i.e.,
;       intensity versus (radii,phi) ) from a cartesian 
;       (i.e., intensity versus (x,y) pixels) image.
;
; CATEGORY:
;
;       Imaging
;
; CALLING SEQUENCE:
;
;	out=xy2rphi(image,center)
;
; INPUTS:
;
;	image:  an IDL variable with the image to process
;	center:  [x0,y0] the pixel values with the center 
;
;       If image and center are undefined, the program creates a 
;       test Gaussian image and siaplays it in xplot2d.
;
; KEYWORDS: 
;
;	RRANGE: [rmin,rmax] The range for radius (in pixels)
;               If undefined, or rmin=rmax, the program computes defaul values.
;       NR: The number of points in R (if undefined, or set to zero, the 
;                program computes defaul values)
;	PHIRANGE: [phiMin,phiMax] The range for phi (in degrees)
;               If undefined, or phimin=phiMax the program  sets the 
;               defaults [0,359]
;       NPHI: The number of points in PHI (if set to zero, the program
;
;	VERBOSE: set this flag to print some working information (default=1)
;
; OUTPUT KEYWORDS:
;
;	AXIS: set this to a names variable to retrieve 
;             [rmin,rFactor,phiMin,phiFactor]
;
;
; ALGORITHM
;
;     based in IDL's INTERPOLATE
;     
;
;     Example:
;     xplot2d,xy2rphi(),scale=1
;
; MODIFICATION HISTORY:
;
;	by M. Sanchez del Rio  srio@esrf.eu 25 April 2008
;
;-	
catch, error_status
 if error_status ne 0 then begin
 message,/info,'error caught: '+!err_string
 if sdep(/w) then itmp = Dialog_Message(/Error,$
   'XY2RPHI: error caught: '+!err_string)
  catch, /cancel
  on_error,2
  RETURN,0
endif

out = 0

IF N_Elements(img) EQ 0 THEN itest = 1 ELSE iTest = 0

IF iTest THEN BEGIN
  ; create a Gaussian
  img = FLTARR(500,500,/NOZERO);*0+100
  Gwidth=.4
  Gcenter=50.0
  center = [200D0,200]
  ;M = sin(rM*!pi*nnn/100)+1
  s=Size(img)
  xM = (findgen(s[1])+0.5) # Replicate(1.0,s[2])
  yM = Replicate(1.0,s[1]) # (findgen(s[2])+0.5)
  rM = sqrt( (xM+0.0-center[0])^2 + (yM+0.0-center[1])^2 ) 
  img = exp(- (rM-Gcenter)^2/2/Gwidth^2)*10
  print,'img limits: ',Max(img),min(img)
  xplot2d,img,scale=1,wtitle='Test Gaussian centered at: '+ $
     vect2string(center)+' Sigma: '+StrCompress(Gwidth,/Rem)
ENDIF


s = Size(img)
IF N_Elements(verbose) EQ 0 THEN verbose = 1
IF N_Elements(center) EQ 0 THEN center = s[1:2]/2.D0

;
; compute (x,y) for input image
;
xM = (findgen(s[1])+0.5) # Replicate(1.0,s[2])
yM = Replicate(1.0,s[1]) # (findgen(s[2])+0.5)
rM = sqrt( (xM+0.0-center[0])^2 + (yM+0.0-center[1])^2 ) 

IF verbose THEN print,'XY2RT: Extrema of r: ',min(rM),max(rM)


;
; calculate r arrays
;

IF N_Elements(rrange) EQ 0 THEN BEGIN
  rmin = Floor(Min(rM))
  rmax = Ceil(Max(rM))
ENDIF ELSE BEGIN
  rmin = rrange[0]
  rmax = rrange[1]
  IF rmin EQ rmax THEN BEGIN
    rmin = Floor(Min(rM))
    rmax = Ceil(Max(rM))
  ENDIF
ENDELSE

rmin = Double(rmin)
rmax = Double(rmax)

IF Not(Keyword_Set(nr)) THEN nr = (Float(rMax)-Float(rMin))+1.0D0
nr = Double(nr)

r = MakeArray1(nr,rMin,rMax) ; +0.5

rShift = (r[1]-r[0])/2
r = r + rShift

IF verbose THEN print,'XT2RT: Rmin: ',rmin
IF verbose THEN print,'XT2RT: Rmax: ',rmax
IF verbose THEN print,'XT2RT: nR: ',nR
IF verbose THEN print,'XT2RT: rShift: ',rShift

;
; calculate phi arrays
;

IF N_Elements(phirange) EQ 0 THEN BEGIN
  phiMin = 0.0D0
  phiMax = 360.0D0
ENDIF ELSE BEGIN
  phiMin = phirange[0]
  phiMax = phirange[1]
  IF phiMin EQ phiMax THEN BEGIN
    phiMin = 0.0D0
    phiMax = 360.0D0
  ENDIF
ENDELSE

phiMin = Double(phiMin) 
phiMax = Double(phiMax) 

IF N_Elements(nphi) EQ 0 THEN nphi = (phiMax-phiMin)+1D0

phi = MakeArray1(nphi,phiMin,phiMax)  ; + 0.5/phiFactor

phiShift = (phi[1]-phi[0])/2
phi = phi+phiShift

IF verbose THEN print,'XT2RT: phiMin: ',phiMin
IF verbose THEN print,'XT2RT: phiMax: ',phiMax
IF verbose THEN print,'XT2RT: nphi: ',nphi
IF verbose THEN print,'XT2RT: phiShift: ',phiShift


;
; (r,phi) arrays
;
rmat = (r+0.0)# Replicate(1.0,n_elements(phi))
phiMat = Replicate(1.0,N_Elements(r)) # (phi+0.0)

;
; calculate (x,y) for the output array
;
x=rmat*cos(phiMat*!pi/180)+center[0]
y=rmat*sin(phiMat*!pi/180)+center[1]

;
; make the main calculation (interpolate)
;
out = INTERPOLATE( img,X-0.5,Y-0.5,CUBIC=-0.5,MISSING=-3)

rFactor = nr/(rmax-rmin)
phiFactor = nphi/(phiMax-phiMin)
axis = [rmin,rFactor,phiMin,phiFactor]

;
; if testing...
;
IF iTest THEN BEGIN
  ;xplot2d,out,scale=1.0,axis=[rmin,rFactor,phiMin,phiFactor]
  IF Verbose THEN BEGIN
    print,'XY2RT: rFactor: ',rFactor
    print,'XY2RT: phiFactor: ',phiFactor
  ENDIF
  ;
  ; exact value
  ;
  
  rex = findgen(1000)/999*100
  
  gau = exp( -(rex-Gcenter)^2/2/Gwidth^2)*10
  ;;window,/free
  ;;plot,rex,gau,LINESTYLE=0,XRANGE=[40,60] ;,YRANGE=[0,2] ; ,

  ;
  ; from polar image
  ;
  ;;oplot,r,total(out,2)/nphi,PSYM=1
  ;;oplot,r,out[*,0],PSYM=2,color=2
  ;;oplot,r,out[*,nphi/2],PSYM=2,color=3
  ;;oplot,r,out[*,nphi-1],PSYM=2,color=4
  ;

  ; from azimuth_integral
  azimuth_integral,img,ar,aint,center=center

  ;;oplot,ar,aint,PSYM=4
  ;;legend,['Exact','From Image','Extrema','Integral'],linestyle=[0,0,0,0],psym=[0,1,2,4]

  buffer = ['plot,data.rex,data.gau,LINESTYLE=0,XRANGE=[40,60]',$
            'oplot,data.r,total(data.out,2)/data.nphi,PSYM=1',$ ; from polar image
            'oplot,data.r,data.out[*,0],PSYM=2,color=2', $
            'oplot,data.r,data.out[*,data.nphi/2],PSYM=2,color=3', $
            'oplot,data.r,data.out[*,data.nphi-1],PSYM=2,color=4', $
            'oplot,data.ar,data.aint,PSYM=4',$ ; from azimuth_integral
            "legend1,['Exact','From Image','Extrema','Integral'],linestyle=[0,0,0,0],psym=[0,1,2,4]" ]

  xwindow,Edit=2,buffer=buffer, wtitle='Integration of image with Gaussian', $
    data={rex:rex,gau:gau,r:r,ar:ar,aint:aint,out:out,nphi:nphi}
  ;
ENDIF

RETURN,out
END
