Function xrf_iupac2siegbahn,line,direction

;______________________________________________________________________________
;+
; NAME:
;       XRF_IUPAC2SIEGBAHN()
;
; PURPOSE:
;       This function translates the X-ray fluorescence line notation 
;	from IUPAC notation to SIEGBAHN and viceversa.
;
; CATEGORY:
;       DABAX. X-ray Fluorescence.
;
; CALLING SEQUENCE:
;       Result = XRF_IUPAC2SIEGBAHN(line,direction)
;
; INPUTS:
;       line: The line descriptor (or annary of descriptors) in one notation.
;       direction:  1  transtales from IUPAC to SIEGBAHN
;	           -1  translates from SIEGBAHN to IUPAC     
;		   default: The program guess the direction, using the fact
;			that the IUPAC notation has a dash character ("-")
;
; OUTPUT: 
;       The line descriptor translated. 
;
; EXAMPLE:
;	print,xrf_iupac2siegbahn('KL3')
;	Ka1
;	print,xrf_iupac2siegbahn(xrf_iupac2siegbahn('Ka1'))
;	Ka1
;
; METHOD:
;	See definition of lines in 
;	http://www.iupac.org/reports/V/spectro/partVIII.pdf
;	Some double lines are defined:
;	        'KL23': Return,'Ka' ; ['Ka2','Ka1']
;	        'KM23': Return,'Kb' ; ['Kb1','Kb3']
;	        'L1O23': Return, 'Lg4' ; ['Lg4p','Lg4']
;	        'L3N45': Return, 'Lb215' ; ['Lb15','Lb2']
;
;	        'Ka12': Return,'KL23'
;	        'Ka': Return,'KL23'
;	        'Kb': Return,'KM23'
;	        'Kb13': Return,'KM23'
;	        'Lg4': Return,'L1O23'
;	        'Lb215': Return,'L3N45'
;		Note: When entering double lines, it is not
;			accepted an array as input. ??
;
; AUTHOR:
;	M. Sanchez del Rio, srio@esrf.fr, 2003/09/09
;	2003-10-16 srio@esrf.fr removed dash in iupac notation
;		Manages an array as input.
;
;______________________________________________________________________________
;-

;IF N_Elements(direction) EQ 0 THEN  $
;  IF (StrPos(line,'-'))[0] EQ -1 THEN direction=-1 ELSE direction=1

IF N_Elements(direction) EQ 0 THEN  BEGIN
  direction=1
  IF Strmatch(line[0],'[KL][Iabgl]*') EQ 1 THEN direction=-1 
ENDIF

;  manages an array as input
IF N_Elements(line) GT 1 THEN BEGIN
  out = StrArr(N_Elements(line))
  FOR i=0,N_Elements(line)-1 DO BEGIN
    out[i]=xrf_iupac2siegbahn(line[i])
  ENDFOR
  RETURN,out
ENDIF

;
; K lines
;
siegbahn = [$
'Ka1',  'Ka2',  'Kb1',  'KIb2', 'KIIb2', 'Kb3',  'KIb4', 'KIIb4', 'Kb4x', 'KIb5','KIIb5']

iupac = [$
'KL3', 'KL2', 'KM3', 'KN3', 'KN2',  'KM2', 'KN5', 'KN4',  'KN4', 'KM5','KM4']


;
; L lines
;
siegbahn = [siegbahn,$
'La1', 'La2', 'Lb1', 'Lb2', 'Lb3', 'Lb4', 'Lb5', 'Lb6', 'Lb7', $
'Lb7p', 'Lb9', 'Lb10', 'Lb15', 'Lb17']

iupac = [iupac,$
'L3M5', 'L3M4', 'L2M4', 'L3N5', 'L1M3', 'L1M2', 'L3O4,5', 'L3N1', 'L3O1',$
'L3N6,7', 'L1M5', 'L1M4', 'L3N4', 'L2M3']


;
; more L lines
;
siegbahn = [siegbahn,$
'Lg1', 'Lg2', 'Lg3', 'Lg4', 'Lg4p', 'Lg5', 'Lg6', 'Lg8', 'Lg8p', 'Le', $
'Ll', 'Ls', 'Lt', 'Lu', 'Lv']

iupac = [iupac,$
'L2N4', 'L1N2', 'L1N3', 'L1O3', 'L1O2', 'L2N1', 'L2O4', 'L2O1', 'L2N6(7)',$
'L2M1', 'L3M1', 'L3M3', 'L3M2', 'L3N6,7', 'L2N6(7)']

;
; M lines
;
siegbahn = [siegbahn,$
'Ma1', 'Ma2', 'Mb', 'Mg', 'Mx']

iupac = [iupac,$
'M5N7', 'M5N6', 'M4N6', 'M3N5', 'M4,5N2,3']


;
; process
;
IF N_Elements(direction) EQ 0 THEN direction=1

IF direction EQ 1 THEN BEGIN
  igood = Where(iupac EQ line) 
  IF igood[0] EQ -1 THEN BEGIN
     Message,/Info,'Undefined line: '+line+'. Looking for non-standad names'
     CASE line OF
	'KL23': Return,'Ka' ; ['Ka2','Ka1']
	'KM23': Return,'Kb' ; ['Kb1','Kb3']
	'L1O23': Return, 'Lg4' ; ['Lg4p','Lg4']
	'L3N45': Return, 'Lb215' ; ['Lb15','Lb2']
	else: 
     ENDCASE
     Message,/Info,'Undefined line: '+line
     RETURN,'-1'
  ENDIF
  Return,siegbahn(igood)
ENDIF ELSE BEGIN
  igood = Where(siegbahn EQ line) 
  IF igood[0] EQ -1 THEN BEGIN
     Message,/Info,'Undefined line: '+line+'. Looking for non-standad names'
     CASE line OF
	'Ka12': Return,'KL23'
	'Ka': Return,'KL23'
	'Kb': Return,'KM23'
	'Kb13': Return,'KM23'
	'Lg4': Return,'L1O23'
	'Lb215': Return,'L3N45'
	else: 
     ENDCASE
     Message,/Info,'Undefined line: '+line
     RETURN,'-1'
  ENDIF
  Return,iupac(igood)
ENDELSE


END
