Function f1f2_calc,input,descriptor,energy,F=F,density=density,$
   theta=theta,rough=rough,group=group, verbose=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       F1F2_CALC
;
; PURPOSE:
;	calculate the elastic Photon-Atom anonalous f1 and f2  coefficients
;	as a function of energy. It also gives the refractive index 
;	components delta and beta (n=1-delta - i beta), the absorption
;	photoelectric coefficient and the reflectivities (s,p and unpolarized).
;	It users data from DABAX data-base.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	f1 = f1f2_calc(input,descriptor[,energy])
; INPUTS:
;	input: a dabax input file as accepted by dabax_access().
;	       The program also accepts the empty string '' and 
;		takes the first cross section file in the list.
;	descriptor: a string  with a description of the material.
;		The string is searched in the scan titles #S in order
;		to find the appropiated data.
; OPTIONAL INPUTS:
;	energy: if undefined, it uses the standard energy grid,
;		which is the grid in the data file for the tabulated
;		cases, and  dindgen(100)/99.*3.0 for the parametrized
;		data. When energy is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data)
;	
; KEYWORDS:
;	F: selects the output:
;		F=0 (default) returns a 2-col array with f1 and f2
;		F=1  returns f1
;		F=2  returns f2
;  		F=3  returns delta  [n = 1 -delta -i beta]
;  		F=4  returns betaf  [n = 1 -delta -i beta]
;  		F=5  returns Photoelectric linear absorption coefficient
;  		F=6  returns Photoelectric mass absorption coefficient
;  		F=7  returns Photoelectric Cross Section
;  		F=8  returns s-polarized reflectivity
;  		F=9  returns p-polarized reflectivity
;  		F=10  returns unpolarized reflectivity
;  		F=11  returns delta/betaf
;	THETA= the incident [grazing] angle in mrad (for reflectivity
;		calculations). It is set to 3.0 (default) when F>=8 and
;		THETA is undefined. It can be an array. In such a case, the
;		returned value os a matrix: out(theta,energy)
;	ROUGH= the rms roughness in Angstroms (for reflectivity
;		calculations. A Debye-Waller model is used.)
;	DENSITY= the density value to be used for the calculations. If
;		not defined, take the value given by atomic_constants().
;	GROUP = The widget ID of the widget that calls the routine.
;		This is passed to widget_message in its dialog_parent
;		keyword to center the window message.
;	VERBOSE = If set, prints some information.
; OUTPUT:
;	out: an array with the values of the selected return parameter.
;
; PROCEDURE:
;	Takes f1 and f2 from the DABAX files. It also takes the
;	Atomic constants from DABAX, and performs the classical 
;	operations to calculate the other parameters.
;
; EXAMPLES:
;	EXAMPLE1: f" vs. energy
;	delvar,energy ; cleans energy variable
;	f = f1f2_calc('f1f2_Henke.dat','Si',energy,f=2)
;	plot,energy,f,/xlog
;
;	EXAMPLE2: unpolarized refl vs. theta and energy
;	; create 20 points in the energy range [1000.,21000.] (eV)
;	energy = (findgen(20)+1)/20.0*10000.0+1000.0
;	; create 20 points in the theta range [1.0,6.0] (mrad)
;	theta = (findgen(20)+1)/20.0*5.0+1.0
;	out = f1f2_calc('f1f2_BrennanCowan.dat','Si',energy,f=10,theta=theta)
;	;plot
;	surface,out,theta,energy,az=285
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;	96-09-27 MSR Many changes. Adds refractive index and
;		reflectivity calculations. Uses #UF1ADD to correct f1 with Z.
;	96-11-19 MSR Adds theta range option.
;	97-01-19 MSR Adpts for windows95.
;	97-02-06 MSR corrects a bug in the calculation of mu in cm^2/cm
;		units pointed out by M.J. Capitan. 
;	97-10-16 MSR adds group keyword.
;	2003-07-07 srio@esrf.fr adds possibility to use empty string as input.
;-
on_error,2



IF N_Elements(verbose) EQ 0 THEN verbose=1

IF input EQ '' THEN BEGIN
list = dabax_ls()
iden_ = 'f1f2_'
dataf1f2 = list[where(strpos(list,iden_) NE -1)]
; take only names with a single "_"
IF N_Elements(dataf1f2) GT 1 THEN BEGIN
  dataf1f2 = dataf1f2(where( strpos(dataf1f2,'_',ipos) EQ -1))
ENDIF
input=dataf1f2[0]
IF verbose THEN Message,/Info,'Using DABAX file: '+input
ENDIF


h = dabax_access(input)

if not(keyword_set(h)) then return,0
dataset = spec_headers(h,-1,'#F',/index,/all)
dataset = strcompress(strsubstitute(dataset,'#F',''),/rem)

igood = dabax_select(h,descriptor,/casematch,group=group)
; added by srio (97/10/16) to avoid stopping the program when 
; an element is not found (i.e. H in muscovite when using Brennan dataset)
if igood EQ 0 then return,0

message,/info,'Using scan: '+strcompress(spec_name(h,igood,/index))
z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))

data = spec_data(h,igood,/index)

energy1 = data(0,*)
; change energy to eV (if necessary)
if input EQ 'f1f2_asf_Kissel.dat' or input EQ 'f1f2_Chantler.dat' or $
   input EQ 'F1F2_AKI.DAT' or input EQ 'F1F2_CHA.DAT' then begin
  message,/info,'Changing Energy from keV to eV for DABAX file '+input
  energy1 = energy1*1000.0
endif

; readjust columns (if necessary)
if ((input EQ 'f1f2_asf_Kissel.dat') OR (input EQ 'F1F2_AKI.DAT')) then begin
  f1 = data(4,*)
  ;;;;f1 = data(2,*)+z ; this is g'
  ;;;;f2 = -data(1,*)
  f2 = abs(data(1,*))
endif else begin
  f1 = data(1,*)
  f2 = data(2,*)
endelse


if n_elements(energy) EQ 0 then energy =  reform(energy1) else begin
  f1 = interpol(f1,energy1,energy)
  f2 = interpol(f2,energy1,energy)
endelse

; adds Z if not included in the data
zadd = strsubstitute(spec_headers(h,igood,'UF1ADD',/index),'#UF1ADD','')
zadd = float(zadd)
if zadd ne 0.0 then f1 = f1+zadd

; returns result( if user requested f)

if not(keyword_set(f)) then f=0
case f of
0:begin
  fout = fltarr(2,n_elements(f1))
  fout(0,*) = f1
  fout(1,*) = f2
  return,fout
  end
1:return,f1
2:return,f2
else:
endcase

;
; calculation of refraction index 
;


if n_elements(density) EQ 0 then $
	density = atomic_constants(z,return='density')
atwt = atomic_constants(z,return='atomicMass')
avogadro = physical_constants('avogadro')
tocm = physical_constants('hc')*1e-8
re = physical_constants('re')

molecules_per_cc = density*avogadro/atwt
wavelength = tocm/energy   ; in cm
k = molecules_per_cc*re*wavelength*wavelength/2.0/!pi

delta = k*f1
betaf = k*f2
mu = 4.0*!pi*betaf/wavelength

case f of
  3:return,delta
  4:return,betaf
  5:return,mu
  6:return,mu/density
  7:return,mu/molecules_per_cc*1.0e+24
  11:return,delta/betaf
  else:
endcase

;
; calculation of reflectivity (piece of code adapted from shadow/abrefc)
;

if n_elements(theta) eq 0 then theta1 = 3.0e-3 else theta1 = theta*1e-3
if n_elements(rough) eq 0 then rough1 = 0.0 else rough1=rough
rough1 = rough1*1.0d-8 ; in cm
; epsi = 1 - alpha - i gamma
alpha = 2.0D0*k*f1
gamma = 2.0D0*k*f2
	for i=0,n_elements(theta1)-1 do begin

     	RHO	=   (SIN(theta1(i)))^2 - ALPHA
     	RHO	=   RHO + SQRT ((SIN(theta1(i))^2 - ALPHA)^2 + GAMMA^2)
     	RHO	=   SQRT(RHO/2)
;** Computes now the reflectivity for s-pol
     	RS1	=   4*(RHO^2)*(SIN(theta1(i))-RHO)^2 + GAMMA^2
     	RS2	=   4*(RHO^2)*(SIN(theta1(i))+RHO)^2 + GAMMA^2
     	RS	=   RS1/RS2
;** Computes now the polarization ratio
     	RATIO1	=   4*RHO^2*(RHO*SIN(theta1(i))-COS(theta1(i))^2)^2 + $
		    GAMMA^2*SIN(theta1(i))^2
     	RATIO2	=   4*RHO^2*(RHO*SIN(theta1(i))+COS(theta1(i))^2)^2 + $
		    GAMMA^2*SIN(theta1(i))^2
     	RATIO	=   RATIO1/RATIO2
;** The reflectivity for p light will be
     	RP	=   RS*RATIO
;** For an unpolarized beam we have
     	RUNP	=   0.5*(RS+RP)
        DebyeWaller = exp( -(4.0*!pi*sin(theta1(i))*rough1/wavelength)^2)
        case f of
          8:out = rs*DebyeWaller
          9:out = rp*DebyeWaller
          10:out = runp*DebyeWaller
          else:
        endcase
	if n_elements(theta1) eq 1 then out1 = out else begin
          if i eq 0 then begin
            out1 = fltarr(n_elements(theta1),n_elements(out))
            out1(0,*) = reform(out)
          endif else out1(i,*) = reform(out)
        endelse
        endfor

return,out1

end
