PRO dabax_crystals,crystalIndex,list=list,str=str,cell=cell, out=out,$
   ask=ask,onlyList=onlyList
;
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       DABAX_CRYSTALS
;
; PURPOSE:
;       retrieve data from dabax file Crystals.dat
;
; CATEGORY:
;       DABAX (DAta Base for Atomic X-rays properties) applications.
;
; CALLING SEQUENCE:
;       DABAX_CRYSTALS [,crystalIndex]
; OPTIONAL INPUTS:
;       crystalIndex:  The index (starting from 0) of the crystal 
;		to be retrieved from the sequence of crystals in the
;		file Crystals.dat (default=0)
;
; INPUT KEYWORDS:
;       GROUP = The widget ID of the widget that calls DABAX_CRYSTAL
;	ASK = Set this keyword to create a menu from where the 
;		crystal is selected.
;	ONLYLIST = Set this keyword if you only need the list of
;		crystals in the file Crystals.dat
;
; OUTPUT KEYWORDS:
;       LIST: The list with the crystals in the dabax file Crystals.dat
;       CELL: A double array of 6 elements containing the cell parameters 
;		a,b,c,alpha,beta,gamma (in A or degrees)
;       STR: A double array of dimension (5,n) containing the atoms in
;		the unit cell. For each atom (from a total of n) we get 
;		 Atomic number, Occupation, X, Y, Z (in fractionary 
;		coordinates)
;	OUT: A structure containing all the information on the chosen crystal. 
;		
; PROCEDURE:
;
; EXAMPLE 1:
;       IDL> dabax_crystals,list=list,/onlyList
; EXAMPLE 2:
;       IDL> dabax_crystals,/ask,out = out
;	IDL> help,/str,out
;
; MODIFICATION HISTORY:
;       07-05-22 Written by M. Sanchez del Rio (srio@esrf.fr)
;
;-

on_error,2
;
; get list of crystals
;
xtalfile = 'Crystals.dat'
h = dabax_access(xtalfile,group=group)
n = spec_access(h)
list = strarr(n)
for i=0,n-1 do list[i]=spec_name(h,i+1,/index)

;print,'List of Crystals: ',list 
IF N_Elements(crystalIndex) EQ 0 THEN crystalIndex = 0


IF Keyword_Set(ask) THEN BEGIN
   tmp = {list:[StrCompress(crystalIndex),list]}
   xscrmenu,tmp,/interpret,/notype,Titles='Crystal',$
	wtitle='Select a crystal',action=action, $
	Group=group, Dialog_Parent=group
   IF ACTION EQ 'DO' THEN crystalIndex=Fix((tmp.list)[0])
ENDIF
IF Keyword_Set(onlyList) THEN RETURN


name = list[crystalIndex]

;
; retrieve str
;
str = double(spec_data(h,crystalIndex+1,/index))
;
; retrieve cell 
;
scell = spec_headers(h,crystalIndex+1,['UCELL'],/index,/Label_Remove)
cell = strsplit( scell,' ',/Extract)
cell = Double(cell)

;
; retrieve all data in headers
;
headers = spec_headers(h,crystalIndex+1,['U'],/index)
pos = strpos(headers,' ')
nn = N_Elements(headers)
titles = strArr(nn)
txt = strArr(nn)
for i=0L,N_Elements(pos)-1 DO titles[i] = strmid(headers[i],2,pos[i]-2)
for i=0L,N_Elements(pos)-1 DO txt[i] = strmid(headers[i],pos[i]+1)
igood = Uniq(titles,sort(titles))

tags = ''
FOR j=0L,N_Elements(igood)-1 DO BEGIN
   ii = igood[j]
   tags = [tags,'<starttext>'+titles[ii]]
   ;print,'<>',titles[ii]
   ;print,txt[where(titles EQ titles[ii])]
   tags = [tags,txt[where(titles EQ titles[ii])]]
   tags = [tags,'<endtext>'+titles[ii]]
ENDFOR
out = tags2struct(tags)
out = Create_Struct('crystalname',name, 'crystalIndex',crystalIndex, $
      'crystallist',list, out,'fcell',cell,'str',str)

END

