C
C This program calculates the diffraction pattern of a crystal.
C It replaces the old second part of the Bragg program.
C Now is an independent module to keep independency of the first part of BRAGG
C and is sharing the CRYSTAL routine with SHADOW. This allows no 
C duplication of code between BRAGG and CRYSTAL, as it was before.
C I avoid to use the term 'rocking curve' as in the old BRAGG, because
C strictly speaking a rocking curve is the result of the combined action
C of two crystal, the crystal to be analyzed, and the analyzer, which is
C scanned to register the transmitted intensity. As this program calculates
C ONE single crystal profile, is more exact to call it "diffraction
C profile" or "diffraction pattern"
C
C MSR/94/03/25
C MSR/97/10/07 Adapts for xop 1.9 (embedded version)
C MSR/00/06/13 Adapts for xop 1.0
C	renamed from diff_pattern to diff_pat
C MSR/08/09/05 Updated TOCM,TOANGS
C Changed by Eric.Lebigot@normalesup.org
C MSR/09/04/14 Updated TOCM,TOANGS everywhere. 
C
	PROGRAM		DIFF_PAT

c        INCLUDE
c     $'/civa/users/b/shadow/shadow2.0/src/trace/./../include/common.blk'

	implicit	none

        real*8  pi,twopi,todeg,torad,tocm,toangs
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /
C        DATA    E2_MC2  /  2.817939 D-13                     / ! e^2/mc^2
                                                               !classical e-
                                                               !radius in cm
C        DATA    AVOG    /  6.022098 D+23                     /


	character*80	rstring
	integer*4	irint,scan_mode,scanunit,scanpoints,i
	real*8		rnumber,phot,q_phot
	real*8		r_s, r_p,phase_s, phase_p, depht_mfp_s
	real*8		delta_ref, theta_b, depht_mfp_p
	real*8		scanmin,scanmax,scanstep,theta,graze,scanval
	real*8		scanval2,ssr,spr
	real*8		vvin(3), vvout(3), bh(3), vnor(3)	!vectors

	character*80	file_refl                           !as in common.blk
	integer*4	f_mosaic,f_bragg_a,f_refrac
C	integer*4	iblank
	real*8		spread_mos,thickness,a_bragg,d_spacing
	real*8		sin_ref,theta_b_ref,scanval_ref,graze_ref


	common	/dpattern/	
     $ file_refl, ssr,spr,
     $ spread_mos,thickness,a_bragg,d_spacing,
     $ f_refrac,f_mosaic,f_bragg_a
C	common	/dpattern/	
C     $ file_refl, ssr,spr,
C     $ spread_mos,thickness,a_bragg,d_spacing
C     $ f_refrac,f_mosaic,f_bragg_a
CCC	common /NAMES/ file_refl
CCC	common /FLAGS/ f_mosaic, f_bragg_a, f_refrac
CCC	common /XTAL/  spread_mos, thickness
CCC	common /GRATING/  a_bragg, d_spacing

C
C Version number also in another place!!!!!!
C
	write(6,*) ' '
	write(6,*) ' '
	write(6,*) ' '
	write(6,*) '*****************************************************'
	write(6,*) '*           DIFF_PAT    v1.4  (14 Apr 2009)         *'
	write(6,*) '*Calculation of a single crystal diffraction profile*'
	write(6,*) '*                                                   *'
	write(6,*) '*****************************************************'
	write(6,*) ' '

	file_refl = 
     $rstring(' Name of file with crystal data (from BRAGG): ')
C
C Inquires about perfect or mosaic crystal
C
        write (6,*) 'What kind of crystal you want to use ?: '
        write (6,*) '        [0] Perfect crystal '
        write (6,*) '        [1] Mosaic crystal '
        f_mosaic = irint(' <?> ')
C
C Inquires about Reflection (Bragg) or Transmission (Laue) geometry.
C If the crystal is perfect we also allow to calculate the transmitted
C (not diffracted beams.
C
        write (6,*) 'What do you want to calculate ?: '
        if (f_mosaic.eq.0) then
          write (6,*) '[0] Diffracted beam in Reflection (Bragg) '//
     $                'geometry'
          write (6,*) '[1] Diffracted beam in Transmission (Laue) '//
     $                'geometry'
          write (6,*) '[2] Transmitted beam in Bragg case'
          write (6,*) '[3] Transmitted beam in Laue case'
        else if (f_mosaic.eq.1) then
          write (6,*) '[0] Diffracted beam in Reflection (Bragg) '//
     $                'geometry  '
          write (6,*) '[1] Diffracted beam in Transmission (Laue) '//
     $                'geometry '
        endif
        f_refrac = irint(' <?> ')

C
C Inquires about mosaic crystal values.
C
        if (f_mosaic.eq.1) then
         spread_mos  = RNUMBER('mosaic angle spread (FWHM) [deg] ? ')
         spread_mos = torad*spread_mos/2.35
        else
         spread_mos   = 0.0d0
        end if
C
C Inquires crystal Thickness. In the Perfect crystal case, we allow
C to use the thick crystal approximation. In such case, the user must
C input any negative value.
C
        WRITE(6,*) 'Input the thickness of the crystal [cm] '
        if (f_mosaic.eq.0) then
           write(6,*) ' [Type a negative value for using the thick '//
     $                'crystal approximation] '
        endif
        thickness = RNUMBER(' <?> ')
C
C Inquires about asymmetrical diffraction
C
        f_bragg_a = 0
        IF (F_MOSAIC.NE.1) THEN
          WRITE(6,*) 'Asymmetric cut angle (deg) between face and '//
     $               'bragg planes (CW)= '
          READ(*,*) A_BRAGG
	 a_bragg = a_bragg*torad
         if (a_bragg.eq.0.0) then
           f_bragg_a = 0
         else
           f_bragg_a = 1
         endif
        ELSE                                  ! mosaic case
	 if (f_refrac.EQ.1) a_bragg = 90.0
	 if (f_refrac.EQ.0) a_bragg = 0.0
	ENDIF
C
C Inquires about the scanning variable
C
	write (6,*) 'Please select scanning variable: '
	write (6,*) ' [1] Incident/Reflected angle [absolute] '
	write (6,*) ' [2] Incident/Reflected angle minus '//
     $              'theta Bragg corrected'
	write (6,*) ' [3] Incident/Reflected angle minus '//
     $              'theta Bragg'
	write (6,*) ' [4] Photon energy '
	write (6,*) ' [5] y variable [Zachariasen] '
C
C Call CRYSTAL to read the file with the crystal data
C
c	write (*,*) '<><> diff: calling crystal. -1, file: '
c     $			//file_refl
	call crystal  (q_phot, vvin, vvout, bh, vnor, 
     $r_s, r_p,phase_s, phase_p, depht_mfp_s, depht_mfp_p,
     $delta_ref, theta_b, -1)
c	write (*,*) '<><> diff: back crystal.'
c	write (*,*) '<><> diff: d_spacing:',d_spacing
C
C Define scan mode (Theta or Energy)
C
	scan_mode = irint('<?> ')
	if (scan_mode.lt.4) then
	  PHOT    = RNUMBER ('... at what energy (eV) ? ')
	  Q_PHOT  = PHOT*TWOPI/TOCM       ! 2 pi / lambda
	else if (scan_mode.eq.4) then
	  THETA = RNUMBER ('... at what grazing angle [deg]: ')
	  IF (THETA.LE.0) THEN
	    PHOT = -1.0D0*THETA
	    Q_PHOT  = PHOT*TWOPI/TOCM       ! 2 pi / lambda
	    THETA = 0
	  ELSE
	    THETA = THETA*torad
	    PHOT = 2.0D0*D_SPACING*SIN(THETA) ! lambda (cm)
	    PHOT = TOCM/PHOT                  ! energy [eV]
	    Q_PHOT  = PHOT*TWOPI/TOCM       ! 2 pi / lambda
	  ENDIF
	else if (scan_mode.eq.5) then
	  PHOT    = RNUMBER ('... at what energy (eV) ? ')
	  Q_PHOT  = PHOT*TWOPI/TOCM       ! 2 pi / lambda
	endif
C
C Call CRYSTAL to calculate useful parameters
C
CCC	if (scan_mode.NE.4) then

C	write (*,*) '<><> diff: calling crystal. 0'
C	kwhat = 0
        WRITE(*,*) ' '
        WRITE(*,*) 'So far, we are working with:'

CL#ifdef vms
CL        OPEN    (23,FILE='DIFF_PAT.PAR',STATUS='NEW')
CL#elif unix
        OPEN    (23,FILE='diff_pat.par',STATUS='UNKNOWN')
        REWIND (23)
CL#endif
	call crystal  (q_phot, vvin, vvout, bh, vnor, 
     $r_s, r_p,phase_s, phase_p, depht_mfp_s, depht_mfp_p,
     $delta_ref, theta_b, 0)
	call crystal  (q_phot, vvin, vvout, bh, vnor, 
     $r_s, r_p,phase_s, phase_p, depht_mfp_s, depht_mfp_p,
     $delta_ref, theta_b, 1)
c	write (*,*) '<><> diff: back crystal.'
c	write (*,*) '<><> diff: d_spacing:',d_spacing
	graze = asin(pi/q_phot/d_spacing) + a_bragg
	graze_ref = asin(pi/q_phot/d_spacing) - a_bragg
	write (*,*) 'grazing [incident] angle = ',graze*TODEG,' degrees'
	write (*,*) 
     $'grazing [reflected] angle = ',graze_ref*TODEG,' degrees'
c	write (*,*) 'DIFF_PAT: theta_b = ',theta_b*todeg

CCC	endif
C
C Units for the scanning variable
C
	if (scan_mode.lt.4) then 
	  write(6,*) 'Select units for the scanning variable (angle):'
	  write(6,*) '[0] radians'
	  write(6,*) '[1] microradians'
	  write(6,*) '[2] degrees'
	  write(6,*) '[3] arc sec'
	  scanunit = irint(' <?>')
c	  SCANUNIT = 1
c	else
	endif

c	write (*,*) '<><> scanunit : ',scanunit

	write(6,*) 'Input scanning variable limits in the chosen units.'
	scanmin = rnumber('Minimum ? ')
	scanmax = rnumber('Maximum ? ')
	scanpoints = irint('Number of points ? ')
	scanstep = (scanmax - scanmin) / (scanpoints - 1)

c	write (*,*) 'DIFF_PAT: scanmin = ',scanmin
c	write (*,*) 'DIFF_PAT: scanmax = ',scanmax
c	write (*,*) 'DIFF_PAT: scanpoints = ',scanpoints
c	write (*,*) 'DIFF_PAT: scanstep = ',scanstep
c
c initialize the vectors 
c
	vvin(1) = 0.0D0
	 vvin(2) = cos(theta_b)
	  vvin(3) = -sin(theta_b)

	vnor(1) = 0.0D0
	 vnor(2) = 0.0D0
	  vnor(3) = 1.0D0

	bh(1) = 0.0D0*TWOPI/d_spacing
	 bh(2) = sin(a_bragg)*TWOPI/d_spacing
	  bh(3) = cos(a_bragg)*TWOPI/d_spacing

	call scat_direction (vvin,bh,vnor,q_phot,vvout)
	call dot (vvout,vnor, sin_ref)
	theta_b_ref = asin(sin_ref)
c	  write(*,*) '<><>vvin: ',vvin
c	  write(*,*) '<><>vvout: ',vvout
c	  write(*,*) '<><>bh: ',bh
c	  write(*,*) '<><> theta_b_ref: ',theta_b_ref
c	  write(*,*) '<><> theta_b_ref [deg]: ',theta_b_ref*todeg
c
c 
c
c        IF (F_BRAGG_A.NE.1) THEN
CL#ifdef vms
CL         OPEN   (20,FILE='DIFF_PAT.DAT',STATUS='NEW')
CL#elif unix
         OPEN   (20,FILE='diff_pat.dat',STATUS='UNKNOWN')
         REWIND (20)
        write(20,'(a)') '#F diff_pat.dat'
        write(20,'(a)') '#S 1 diff_pat run'
        write(20,'(a)') '#C results of diff_pat run'
        write(20,'(a)') '#N 4'

	if (scan_mode.EQ.4) then 
          write(20,'(a)') '#L  E[eV]  Lambda[A]'//
     $'  p-polarized-profile  s-polarized-profile '
	else
	  if (scan_mode.eq.1) then
	    if (scanunit.eq.1) then
               write(20,'(a)')
     $		'#L  Theta{in} [microrad]  Theta{out} [microrad]'//
     $		'  p-polarized  s-polarized'
	    else if (scanunit.eq.2) then
               write(20,'(a)')
     $		'#L  Theta{in} [deg]  Theta{out} [deg]'//
     $		'  p-polarized  s-polarized'
	    else if (scanunit.eq.3) then
               write(20,'(a)')
     $		'#L  Theta{in} [arcsec]  Theta{out} [arcsec]'//
     $		'  p-polarized  s-polarized'
	    endif
	  else if (scan_mode.eq.2) then
	    if (scanunit.eq.1) then
               write(20,'(a)')
     $		'#L  Th-ThBc{in} [microrad]  Th-ThBc{out}'//
     $		' [microrad]  p-polarized  s-polarized'
	    else if (scanunit.eq.2) then
               write(20,'(a)')
     $		'#L  Th-ThBc{in} [deg]  Th-ThBc{out}'//
     $		' [deg]  p-polarized  s-polarized'
	    else if (scanunit.eq.3) then
               write(20,'(a)')
     $		'#L  Th-ThBc{in} [arcsec]  Th-ThBc{out}'//
     $		' [arcsec]  p-polarized  s-polarized'
	    endif
	  else if (scan_mode.eq.3) then
	    if (scanunit.eq.1) then
               write(20,'(a)')
     $		'#L  Th-ThB{in} [microrad]  Th-ThB{out}'//
     $		' [microrad]  p-polarized  s-polarized'
	    else if (scanunit.eq.2) then
               write(20,'(a)')
     $		'#L  Th-ThB{in} [deg]  Th-ThB{out}'//
     $		' [deg]  p-polarized  s-polarized'
	    else if (scanunit.eq.3) then
               write(20,'(a)')
     $		'#L  Th-ThB{in} [arcsec]  Th-ThB{out}'//
     $		' [arcsec]  p-polarized  s-polarized'
	    endif
	  else if (scan_mode.eq.5) then
               write(20,'(a)')
     $		'#L  y  y  p-polarized  s-polarized'
	  endif
	endif
CL#endif
c	ELSE 
c#ifdef vms
c         OPEN   (20,FILE='ROCK_CURVE_INC.S',STATUS='NEW')
c         OPEN   (21,FILE='ROCK_CURVE_INC.P',STATUS='NEW')
c         OPEN   (27,FILE='ROCK_CURVE_REF.S',STATUS='NEW')
c         OPEN   (28,FILE='ROCK_CURVE_REF.P',STATUS='NEW')
c#elif unix
c         OPEN   (20,FILE='rock_curve_inc.s',STATUS='UNKNOWN')
c         REWIND (20)
c         OPEN   (21,FILE='rock_curve_inc.p',STATUS='UNKNOWN')
c         REWIND (21)
c         OPEN   (27,FILE='rock_curve_ref.s',STATUS='UNKNOWN')
c         REWIND (27)
c         OPEN   (28,FILE='rock_curve_ref.p',STATUS='UNKNOWN')
c         REWIND (28)
c#endif
c	END IF

c
c main loop over the scanning points ===== starts ====
c
	do i=1,scanpoints
c
	scanval = scanmin + scanstep*(i-1)
c	write(*,*) '<>><>scanval ',scanval
c
c convert user unit to shadow units (rads, eV)
c
        if (scanunit.eq.1) then
          scanval2 = scanval*1D-6
        else if (scanunit.eq.2) then
          scanval2 = scanval*torad
        else if (scanunit.eq.3) then
          scanval2 = scanval*torad/3600D0
        else
	  scanval2 = scanval
	end if


c
c convert user mode to theta[abs]
c
        if (scan_mode.eq.4) then                  ! energy scan
          PHOT    = scanval2
          Q_PHOT  = PHOT*TWOPI/TOCM       	  ! 2 pi / lambda
	  if (theta.eq.0) theta = theta_b 
c		write(*,*) '<><><> phoT ',phot
c		write(*,*) '<><><> theta_b: ',theta_b 
c		write(*,*) '<><><> theta: ',theta
        else					  ! angle scan
	  if (scan_mode.EQ.1) then 
	    theta = scanval2
	  else if (scan_mode.EQ.2) then
	    theta = theta_b + scanval2
	  else if (scan_mode.EQ.3) then
	    theta = graze + scanval2
c	  else if (scan_mode.EQ.4) then
c	    theta = theta_b + (scanval2)
c	    if (theta.eq.0) theta = theta_b + (scanval2) 
	  else if (scan_mode.EQ.5) then
	    theta = theta_b + scanval2*ssr
c		write(*,*) '<><><>scanval2 ',scanval2
c		write(*,*) '<><><> theta_b: ',theta_b 
c		write(*,*) '<><><> theta: ',theta
c		write(*,*) '<><><> theta/ssr: ',theta
	  endif
	endif

c
c calculate output direction 
c
C		write(*,*) '<><><> theta_b: ',theta_b/torad 
C		write(*,*) '<><><> theta: ',theta/torad
C		write(*,*) '<><><> graze: ',graze/torad
	vvin(2) = cos(theta)
	  vvin(3) = -1.0D0*sin(theta)
	call scat_direction(vvin,bh,vnor,q_phot,vvout)

	call dot (vvout,vnor,sin_ref)

c
c calculate reflectivity
c
	call crystal  (q_phot, vvin, vvout, bh, vnor, 
     $r_s, r_p,phase_s, phase_p, depht_mfp_s, depht_mfp_p,
     $delta_ref, theta_b, 2)

c        write (*,*) '<*> y,theta: ',scanval,theta
c        write (*,*) '<*> r_s,r_p: ',r_s,r_p
cc        write (*,*) '<*> phase_s,phase_p: ',phase_s,phase_p
c	write (*,*) ' '
c	write(20,*)  scanval,r_s**2,r_p**2

c
c if asymmetric is selected, calculate and write down
c the output angle
c
	if (f_bragg_a.EQ.1) then
          call dot (vvout,vnor,sin_ref)

csrio	write(*,*) '<><> theta,theta_ref: ',theta,asin(sin_ref)

          if (scan_mode.EQ.1) then
            scanval_ref = asin(sin_ref)
          else if (scan_mode.EQ.2) then
            scanval_ref =  - theta_b_ref + asin(sin_ref)
          else if (scan_mode.EQ.3) then
            scanval_ref =  - graze_ref + asin(sin_ref)
          else if (scan_mode.EQ.4) then
c           scanval_ref = scanval
          else if (scan_mode.EQ.5) then
           scanval_ref = scanval
          endif

          if (scanunit.eq.1) then
            scanval_ref = scanval_ref/1D-6
          else if (scanunit.eq.2) then
            scanval_ref = scanval_ref/torad
          else if (scanunit.eq.3) then
            scanval_ref = scanval_ref/torad*3600D0
          else
            scanval_ref = scanval_ref
          end if
	else
          scanval_ref = scanval
	endif  
	if (scan_mode.EQ.4) scanval_ref = TOANGS/scanval
	WRITE (20,'(4(g15.8,1x))') scanval,scanval_ref,r_p**2,r_s**2
c
c main loop over the scanning points ===== ends ====
c
	enddo

c
c close files and exit
c
	close(20)
c	close(21)
c	if (f_bragg_a.EQ.1) then 
c	  close(27)
c	  close(28)
c	endif
	write (*,*) '>> '
	write (*,*) '>> Files diff_pat.par (parameters) and '
	if (scan_mode.EQ.4) then 
	  write (*,*) 
     $  '>> diff_pat.dat (E [eV],Lambda [A],p-pol,s-pol)'
	else
	  write (*,*) 
     $  '>> diff_pat.dat (Scan in,Scan out,p-pol,s-pol)'
	endif
	write (*,*) '>> written  to disk.'
	write (*,*) '>> '

	end
	
c========================================================================
c
c	subroutine scat_direction
c	calculates the output scattering direction applying the
c	equations k_out_par = k_in_par + BH_par and |k_in| = |k_out|
c
c	kk% = ( 2 pi / lambda )  * vv%
c	vv% direction versor
c	q_phot = 2 pi / lambda [cm**-1]
c	bh = ( 2 pi / d_spacing )  normal_to_bragg_planes
c	vnor = surface_normal
c
	subroutine scat_direction (vvin,bh,vnor,q_phot,vvout)

        implicit        none

        real*8          vvin(3),vvout(3),bh(3),vnor(3),q_phot
	real*8		kkin(3),kkin_par(3),kkin_perp(3)
	real*8		kkout(3),kkout_par(3),kkout_perp(3)
	real*8		bh_par(3),bh_perp(3)
	real*8		mod2_kkout_par,mod_kkout_perp
	external	sum

	call norm (vnor,vnor)

c	write(*,*) 'SCAT: vvin = ',vvin
c	write(*,*) 'SCAT: vnor = ',vnor
c kkin par and perp components
	call norm (vvin,vvin)
	call scalar (vvin,q_phot,kkin)

	call proj (kkin,vnor,kkin_perp)
	call vector (kkin_perp,kkin,kkin_par)

	call proj (bh,vnor,bh_perp)
	call vector (kkin_perp,kkin,kkin_par)

c	write(*,*) 'SCAT: kkin = ',kkin
c	write(*,*) 'SCAT: kkin_par = ',kkin_par
c	write(*,*) 'SCAT: kkin_perp = ',kkin_perp
c bh par and perp components
	call proj (bh,vnor,bh_perp)
	call vector (bh_perp,bh,bh_par)
c	write(*,*) 'SCAT: bh = ',bh
c	write(*,*) 'SCAT: bh_perp = ',bh_perp
c	write(*,*) 'SCAT: bh_par = ',bh_par

c kkout
	call sum (kkin_par,bh_par,kkout_par)
	call dot(kkout_par,kkout_par,mod2_kkout_par )
	mod_kkout_perp = dsqrt(q_phot**2 - mod2_kkout_par)

c	write(*,*) 'SCAT: q_phot = ',q_phot
c	write(*,*) 'SCAT: mod2_kkout_par = ',mod2_kkout_par
c	write(*,*) 'SCAT: mod_kkout_perp = ',mod_kkout_perp

	call scalar (vnor,mod_kkout_perp,kkout_perp)
	call sum (kkout_par,kkout_perp,kkout)
	call norm (kkout,vvout)
c	write(*,*) 'SCAT: kkout_par = ',kkout_par
c	write(*,*) 'SCAT: kkout_perp = ',kkout_perp
c	write(*,*) 'SCAT: kkout = ',bh_par
c	write(*,*) 'SCAT: **************************'
c	write(*,*) ' '
c
	return
	end
C+++
C	SUBROUTINE	CRYSTAL_FH
C
C	PURPOSE		Computes the structute factor of a crystal
C			from data in bragg's output file.
C
C	ALGORITHM	
C
C	MODIFIED	Created by M. Sanchez del Rio  (Feb. 1996)
C
C			Modified by MSR 96/12/19 to include the number
C			of coefficients in the f0 parametrization. This
C			number is 9 for the CromerMann data and 11 for
C			the Waasmaier Kirfel data. Note that the crystal
C			file has changed!
C
C---
     	SUBROUTINE	CRYSTAL_FH (
     $ KWHAT, FILE, PHOT, THETA			   ! inputs
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC   ! outputs when kwhat>0
     $,RN1,D_SPACING1)				   ! outputs when kwhat<0

C 
C INPUT PARAMETERS:
C   KWHAT: Flag
C     KWHAT<0 reads data file
C     KWHAT>=0 performs calculations
C	in particular, if KWHAT GT 3 it also writes results in the 
C	fortran unit kwhat
C   FILE [string]: File name with crystal data [useless if kwhat GE 0]  
C   PHOT [real]: Photon energy in eV [Useless if kwhat<0]
C   THETA [real]: Scattering grazing angle in rads [Useless if kwhat<0]
C
C OUTPUT PARAMETERS:
C   When kwhat>=0
C   FH           |
C   FH_BAR       |
C   F_0          |    Complex with the returned structure factors 
C   PSI_H        |    F, Psi and the refraction index.
C   PSI_HBAR     |
C   PSI_0        |
C   REFRAC       |
C
C INPUT AND OUTPUT PARAMETERS:
C   RN1: the constant (e^2/mc^2)/V or the ration between the classical e- 
C        radius and the volume of the unit cell [cm^(-2)]
C   D_SPACING1: The crystal d-spacingin Angstroms
C   Note that:
C       IF KWHAT LT 0  then RN1 and D_SPACING1 are returned parameters 
C       (from file)
C       IF KWHAT GE 0  then D_SPACING1 is useless and RN1 is an input 
C       parameter:
C	if negative, forget it and use the saved ones (from file)
C	if positive, consider it for the calculations.
C	(this feature will allow the study of crystal with a gradient in d)
C
C
	implicit none
	integer		NMAXENER,NMAXATOMS
        parameter       (NMAXENER=1000)
        parameter       (NMAXATOMS=100)
                                       

        real*8  pi,twopi,tocm,toangs,todeg
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /



	real*8		PHOT,THETA 
	integer*4	KWHAT

        character*80    file,filename
	character*80	text                                !file
	real*8		rn,d_spacing,rn1,d_spacing1
	real*8		temper(NMAXATOMS),G_0(NMAXATOMS)
	real*8		fract(NMAXATOMS)
        integer*4       nbatom,atnum(NMAXATOMS),npoint,if0center
        complex*16      psi_h,psi_hbar,psi_0,psi_conj
        complex*16      G(NMAXATOMS),G_BAR(NMAXATOMS)
        real*8          f0coeff(NMAXATOMS,11)
	integer*4	nf0coeff
        real*8          energy(NMAXENER)
        real*8		fpp(NMAXATOMS,NMAXENER),fp(NMAXATOMS,NMAXENER)

	integer*4	i,j,nener,ierr
        integer*4	i_debug,itype
	real*8		F0(NMAXATOMS),F1(NMAXATOMS),F2(NMAXATOMS)
	real*8		r_lam0,ratio,version
	complex*16	ci,F(NMAXATOMS),F_0,FH,FH_BAR,STRUCT,REFRAC
	real*8		delta_ref,absorp

C
C SAVE the variables that need to be saved across subsequent invocations
C of this subroutine. Note: D_SPACING is not included in the SAVE block
C because it's included in the COMMON.BLK file.
C
c NOT LONGET TRUE:warning: d_spacing to be removed!!!!!!!
	SAVE		RN, d_spacing,
     $			ATNUM,TEMPER,NBATOM,
     $			G,G_BAR,G_0,
     $			NF0COEFF,F0COEFF,
     $			NPOINT, ENERGY, FP, FPP,
     $			FRACT,filename

C
	i_debug = 0
	CI	= (0.0D0,1.0D0)

	if (i_debug.EQ.1) write (*,*)'<><>CRYSTAL_FH: '//
     $'******** crystal_fh called with flag ********',kwhat

C***************************************************
C If flag is < 0, reads in the reflectivity data
C***************************************************
     	IF (KWHAT.LT.0) THEN
	if (i_debug.EQ.1) write(*,*) '<><> CRYSTAL_FH: file is: ',file
CL#if vms
CL	  OPEN	(25,FILE=file,STATUS='OLD',READONLY,
CL     $FORM='FORMATTED',ERR=77)
CL#elif unix
	  OPEN	(25,FILE=file,STATUS='OLD', FORM='FORMATTED',ERR=77)
CL#endif
        read(25,'(A)',err=79)  text
        read(25,*,err=79)  version, itype
	if (ABS(version-2.3).GT.0.01) then
          write (*,*) 'SHADOW-E-Error: '
          write (*,*) 'Module     : CRYSTAL_FH'
          write (*,*) 'Message    : Data files does not corrspond to '//
     $ ' current version 2.3. Old:',version
	endif
	if (itype.NE.1) then 
	  write (*,*) 'SHADOW-E-Error: '
          write (*,*) 'Module     : CRYSTAL_FH'
          write (*,*) 'Message    : Data file type not yet implemented.'
	  stop
	end if
        read(25,'(A)',err=79)  text
        read(25,*,err=79)  RN,d_spacing  !,TEMPER
        read(25,'(A)',err=79)  text
        read(25,*,err=79)  nbatom
	if (nbatom.GT.NMAXATOMS) then
          write (*,*) 'SHADOW-E-Error: '
          write (*,*) 'Module     : CRYSTAL_FH'
          write (*,*) 'Message    : '//
     $' Maximum number of atoms allowad: ',NMAXATOMS
	end if

        read(25,'(A)',err=79) text
        read(25,*,err=79)  (ATNUM(I),i=1,nbatom)

        read(25,'(A)',err=79) text
        read(25,*,err=79)  (FRACT(I),i=1,nbatom)

        read(25,'(A)',err=79) text
        read(25,*,err=79)  (TEMPER(I),i=1,nbatom)

	read(25,'(A)',err=79) text
	read(25,*,err=79) (G_0(i),i=1,nbatom)

        read(25,'(A)',err=79)  text
	do i=1,nbatom
          read(25,*,err=79)  G(I)
          read(25,*,err=79)  G_BAR(I)
	end do

        read(25,'(A)',err=79) text
	do i=1,nbatom
          read(25,*,err=79)  nf0coeff,(f0coeff(i,j),j=1,nf0coeff)
	end do
        read(25,'(A)',err=79)  text
        read(25,*,err=79)  NPOINT
	read(25,'(A)',err=79)  text
        DO 199 I = 1, NPOINT
          read(25,*,err=79)  energy(i)
	  do j=1,nbatom
            read (25,*,err=79) FP(j,i),FPP(j,i)
	  end do
 199      CONTINUE

C rn and d_spacing are saved, rn1 and d_spacing1 are returned

	  rn1 = rn
	  d_spacing1 = d_spacing
	  filename = file

C
C Check for file reading errors
C
	go to 88
77	write(*,*) ' CRYSTAL_FH: Error opening file: '//file
	stop
79	write(*,*) ' CRYSTAL_FH: Error reading file: '//file
	stop
88	continue
	  CLOSE	(25)
     	  RETURN
     	ELSE
C***************************************************
C If flag is >= 0, general calculation
C***************************************************
C
C If rn1 (NOW INPUTS) is positive, then use it for
C the calculations; otherwise use the saved value.
C
	IF (RN1.lt.0)        rn1 = rn
C
C Computes structure factor at given wavelength and angle.
C
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_FH: working at energy: ',phot
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_FH: working at angle [rads] : ',theta
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_FH: working at angle [deg] : ',theta*todeg

	IF (PHOT.LT.ENERGY(1).OR.PHOT.GT.ENERGY(npoint)) THEN
	  write (*,*) 'SHADOW-E-Error: '
	  write (*,*) 'Module     : CRYSTAL_FH'
	  write (*,*) 'Message    : '//
     $'Incoming photon energy is out of range: ',PHOT
	  ierr = -1
	  write (*,*) 'Error flag : ',IERR
	  RETURN
	END IF	  
C
C Build the fo scattering form factor from its coefficients
C
	ratio 	= sin(theta)/(toangs/phot)
	if (ratio.GT.2) then 
	  write (*,*) 'CRYSTAL_FH: ratio sin(theta)/lambda > 2'
	  write (*,*) '    ratio : ',ratio
	  write (*,*) '    Paramatrizatiog for Fo may fail.'
	end if
	if (i_debug.EQ.1) write (*,*) 'CRYSTAL_FH: ratio is : ',ratio
	if0center = (1+nf0coeff)/2
	do j=1,nbatom
	  f0 (j) = f0coeff(j,if0center)
          do 666 i=1,if0center-1
            f0(j) = f0(j) + f0coeff(j,i) * 
     $		dexp(-1.0d0*f0coeff(j,i+if0center)*ratio**2)
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH:  f0 i,j = '
     $    ,f0(j),i,j
666       continue
	  if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: F0(j) = '
     $     ,F0(j)
	end do
C
C Interpolate for the atomic scattering factor.
C
	DO 299 I = 1, npoint
299	  IF (ENERGY(I).GT.PHOT)	GO TO 101
C
	I	= npoint
101	NENER	= I - 1	
	do j=1,nbatom
	  F1(j)	= FP(j,NENER) + (FP(j,NENER+1) - FP(j,NENER)) * 
     $	  (PHOT - ENERGY(NENER)) / (ENERGY(NENER+1) - ENERGY(NENER))
	  F2(j)	= FPP(j,NENER) + (FPP(j,NENER+1) - FPP(j,NENER)) * 
     $	  (PHOT - ENERGY(NENER)) / (ENERGY(NENER+1) - ENERGY(NENER))
	  if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: F1 = ',F1(j)
	  if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: F2 = ',F2(j)
	end do
	R_LAM0 	= TOCM/PHOT
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH:  R_LAM0  ='
     $     , R_LAM0

	do j=1,nbatom
	  F(j)	= F0(j) + F1(j) + CI*F2(j)
	  if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: F = ',F(j)
	end do
C	
C FH and FH_BAR are the structure factors for (h,k,l) and (-h,-k,-l).
C
	F_0 = (0.0D0, 0.0D0)
	FH = (0.0D0, 0.0D0)
	FH_BAR = (0.0D0, 0.0D0)
C	write (24,*) nbatom
	do i=1,nbatom
C	  write (24,*) TEMPER(i),FRACT(i)
C	  write (24,*) dreal(G(i)), dimag(G(i))
C	  write (24,*) dreal(F(i)), dimag(F(i))
	  FH 	= FH + (G(i) * F(i) * TEMPER(i) * FRACT(i)) 
C From Brennan's private communication:
C Changing the temperature doesn't change F0, but it does change
C Chi0, due to changing the size of the unit cell, which changes
C gamma.
C MSR 96/02/14

	  F_0 	= F_0 + G_0(i) * 
     $	     (atnum(i) + F1(i) + CI * F2(i)) * FRACT(i) 
C	  F_0 	= F_0 + G_0(i) * 
C     $		(atnum(i) + F1(i) + CI * F2(i)) * TEMPER(i) * FRACT(i)
	  FH_BAR	= FH_BAR + 
     $          (G_BAR(i) * F(i) * TEMPER(i) * FRACT(i))
	end do
C	FH =  FH * TEMPER
C	FH_BAR = FH_BAR * TEMPER
	STRUCT 	= sqrt(FH * FH_BAR) 

	if (i_debug.EQ.1) then 
	  write (*,*) '<><>CRYSTAL_FH: FH = ',FH
	  write (*,*) '<><>CRYSTAL_FH: FH_BAR = ',FH_BAR
	  write (*,*) '<><>CRYSTAL_FH: f_0 = ',f_0
	  write (*,*) '<><>CRYSTAL_FH: STRUCT = ',STRUCT
	endif
C
C   PSI_CONJ = F*( note: PSI_HBAR is PSI at -H position and is
C   proportional to fh_bar but PSI_CONJ is complex conjugate os PSI_H) 
C
	psi_h = rn1*r_lam0**2/pi*fh
	psi_hbar = rn1*r_lam0**2/pi*fh_bar
	psi_0 = rn1*r_lam0**2/pi*f_0
	psi_conj = rn1*r_lam0**2/pi*conjg(fh)
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: PSI_H = ',
     $PSI_H
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: PSI_HBAR = ',
     $PSI_HBAR
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: PSI_0 = ',
     $PSI_0
C
C computes refractive index.
C ([3.171] of Zachariasen's book)
C
     	REFRAC = (1.0D0,0.0D0) - R_LAM0**2*RN1*F_0/TWOPI
#if lahey
	DELTA_REF  = 1.0D0 - REAL(REFRAC,8)
	ABSORP	= 2.0d0 * TWOPI *(-AIMAG(REFRAC)) / R_LAM0
#else
	DELTA_REF  = 1.0D0 - DREAL(REFRAC)
	ABSORP	= 2.0d0 * TWOPI *(-DIMAG(REFRAC)) / R_LAM0
#endif
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: REFRAC = ',
     $REFRAC
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: DELTA_REF = ',
     $DELTA_REF
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_FH: ABSORP = ',
     $ABSORP
C
C if kwhat > 3, write text to unit kwhat
C
	if (kwhat.gt.3) then 
        write(kwhat,*) ' '
        write(kwhat,*) 
     $'******************************************************'
        write(kwhat,*) 
     $'           **  DIFF_PAT  v1.4  (14 Apr 2009) **       '
        write(kwhat,*) 'Crystal data from file ',filename
        WRITE(kwhat,*) '       at energy    = ',PHOT,' eV'
        WRITE(kwhat,*) '                    = ',R_LAM0*1E8,' Angstroms'
        WRITE(kwhat,*) '       and at angle = ',THETA*TODEG,' degrees'
        WRITE(kwhat,*) '                    = ',THETA,' rads'
        write(kwhat,*) 
     $'******************************************************'
        WRITE(kwhat,*) ' '
        do j=1,nbatom
          WRITE(kwhat,*) 'For atom ',j,':'
          WRITE(kwhat,*) '       fo + fp+ i fpp = '
          WRITE(kwhat,*) '        ',f0(j),'+',f1(j),'+ i',f2(j),'= '
          WRITE(kwhat,*) '        ',f0(j)+f1(j)+ci*f2(j)
          WRITE(kwhat,*) '       Z = ',atnum(j)
        end do
        WRITE(kwhat,*) 'Structure factor F(0,0,0) = ',F_0
        WRITE(kwhat,*) 'Structure factor FH = ',FH
        WRITE(kwhat,*) 'Structure factor FH_BAR = ',FH_BAR
        WRITE(kwhat,*) 'Structure factor F(h,k,l) = ',STRUCT
        WRITE(kwhat,*) 'Psi_0  = ',psi_0
        WRITE(kwhat,*) 'Psi_H  = ',psi_h
        WRITE(kwhat,*) 'Psi_HBar  = ',psi_hbar
C        WRITE(kwhat,*) 'Temperature factor = ',TEMPER
        WRITE(kwhat,*) 'Refraction index = 1 - delta - i*beta :'
        WRITE(kwhat,*) '           delta = ',DELTA_REF
#if lahey
        WRITE(kwhat,*) '            beta = ',-1.0D0*AIMAG(REFRAC)
#else
        WRITE(kwhat,*) '            beta = ',-1.0D0*DIMAG(REFRAC)
#endif
        WRITE(kwhat,*) 'Absorption coeff = ',ABSORP,' cm^-1'
        WRITE(kwhat,*) ' '
        WRITE(kwhat,*) 'e^2/(mc^2)/V = ',rn,' cm^-2'
        WRITE(kwhat,*) 'd-spacing = ',d_spacing*1.0E8,' Angstroms'
C        WRITE(kwhat,*) 'Photon Energy        = ',PHOT,' eV'
C        WRITE(kwhat,*) 'Lambda               = ',R_LAM0*1E8,' Angstroms'
        WRITE(kwhat,*) 'SIN(theta)/Lambda = ',Ratio
C        WRITE(kwhat,*) 'theta = ',theta,' rads'
C        WRITE(kwhat,*) '      = ',theta*TODEG,' degrees'
        WRITE(kwhat,*) ' '
	
	endif
     	ENDIF
     	END
c
c#if unix
c#include  "/civa/users/b/shadow/shadow2.0/src/trace/./../include/header.txt"
c#elif vms
c     	INCLUDE		'SHADOW$INC:HEADER.TXT/LIST'
c#endif
c
C
C
C+++
C	SUBROUTINE	CRYSTAL
C
C	PURPOSE		Computes the reflectivity of a symmetric Bragg crystal 
C			according to the dynamic theory of x-ray diffraction.
C
C	ALGORITHM	Reference B.E.Warren, X-Ray Diffraction, 
C			Addison-Wesley 1969.  See also M.J.Bedzyk, G.Materlik 
C			and M.V.Kovalchuk, Phy. Rev. B30, 2453(1984).
C			For mosaic crystal reflectivity see Zachariasen, 
C			Theory of x-ray diffraction in crystals, Dover (1966)
C			formula 4.24, and Bacon and Lowde, Acta Crystall.1
C			pag 303 (1948) formula 17. 
C
C	MODIFIED	July 1989, M. Sanchez del Rio for asymmetry part,
C			July 1990, mosaic part.
C			August 1992, laue part.
C
C---
     	SUBROUTINE	CRYSTAL	 (Q_PHOT, VIN, VOUT, BH, SURFNOR,
     $R_S, R_P,PHASE_S, PHASE_P, L_EXT_S, L_EXT_P, 
     $DELTA_REF, THETA_B, KWHAT)
C
C kwhat: flag:
C	<0 read crystal file (call crystal_fh)
C	0 info mode (call crystal_info)
C	1 set mode (call crystal_set)
C	2 reflectivity mode (call crystal_{perfect,mosaic})

c#if unix
CCC     	INCLUDE	'./../../include/common.blk'
c#elif vms
c     	INCLUDE		'SHADOW$INC:COMMON.BLK/LIST'
c#endif
c
CCC
	implicit none
                                       

CCC
        real*8  pi,twopi,todeg,torad,tocm,toangs
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /
C        DATA    E2_MC2  /  2.817939 D-13                     / ! e^2/mc^2
                                                               !classical e-
                                                               !radius in cm
C        DATA    AVOG    /  6.022098 D+23                     /


        integer*4	i_debug
	real*8		Q_PHOT                              !Arguments
	real*8		VIN(3), VOUT(3), BH(3), SURFNOR(3)
	real*8		R_S, R_P,PHASE_S, PHASE_P
	real*8		L_EXT_S, L_EXT_P
	real*8		DELTA_REF, THETA_B
	integer*4	KWHAT

        character*80    file_refl                           !as in common.blk
        integer*4       f_mosaic,f_bragg_a,f_refrac
        real*8          spread_mos,thickness,a_bragg
        real*8          d_spacing,ass_fac


	real*8		rn,rn1,d_spacing1

	real*8		phot,r_lam0
	complex*16	ci,F_0,FH,FH_BAR,REFRAC

	real*8		graze
	real*8		absorp    !,theta_inc_o
	real*8		ssr,spr, q_mos

	complex*16	psi_h,psi_hbar,psi_0

	real*8		theta_b_h,theta_b_sym


	common	/dpattern/	
     $ file_refl, ssr,spr,
     $ spread_mos,thickness,a_bragg,d_spacing,
     $ f_refrac,f_mosaic,f_bragg_a
C        common  /dpattern/   
C     $file_refl,f_mosaic,f_bragg_a,
C     $spread_mos,thickness,a_bragg,f_refrac,d_spacing, 
C     $ssr,spr
	save	rn



C
	i_debug = 0
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL: ******** crystal called with flag ********',kwhat
	CI	= (0.0D0,1.0D0)
C**************************************************************
C If flag is < 0, reads in the reflectivity data, and store the
C returned RN1 in the variable RN (saved) and D_SPACING1 in the
C variable D_SPACING (common block) for further calls.
C**************************************************************
     	IF (KWHAT.LT.0) THEN
	if (i_debug.EQ.1) write (*,*) 
     $'<><> CRYSTAL: file is: ',FILE_REFL

	call crystal_fh (KWHAT, FILE_REFL, PHOT, THETA_B
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC
     $,RN1,D_SPACING1)
	RN = RN1
	D_SPACING = D_SPACING1
	RETURN
	ENDIF

C***************************************************
C If flag is >= 0, general calculation
C***************************************************
C
C
	PHOT	= Q_PHOT/TWOPI*TOCM
	R_LAM0 	= TWOPI/Q_PHOT
	GRAZE = ASIN(R_LAM0/d_spacing/2.0D0)

	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL: working at energy: ',phot
C	IF (PHOT.LT.ENERGY(1).OR.PHOT.GT.ENERGY(npoint)) THEN
C 	  CALL	MSSG ('CRYSTAL ',
C     $			'Incoming photon energy is out of range.',IERR)
C	  R_S	= 0.0D0
C	  R_P	= 0.0D0
C	  PHASE_S = 0.0D0
C	  PHASE_P = 0.0D0
C	  THETA_B = 0.0D0
C	  RETURN
C	END IF	  
C
C call crystal_fh to get the structure factor
C
	if (i_debug.EQ.1) 
     $	  write (*,*) '<><>CRYSTAL: calling crystal_fh: ',kwhat
	RN1 = -1
	D_SPACING1 = -1.
	call crystal_fh (KWHAT, FILE_REFL, PHOT, GRAZE
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC
     $,RN1,D_SPACING1)
	if (i_debug.EQ.1) then
	  write(*,*) '<><>CRYSTAL: back crystal_fh, d_spacing: '
     $      ,d_spacing
	  write(*,*) '<><>CRYSTAL: back crystal_fh, rn: ',rn
	end if

C
C call crystal_set to get THETA_B and L_EXT
C returns also the ssr and spr needed in diff_pat
C
	if (kwhat.eq.0) then
          CALL      CRYSTAL_SET      (Q_PHOT,
     $F_MOSAIC,D_SPACING,F_REFRAC,A_BRAGG,RN,THICKNESS,SPREAD_MOS,
     $THETA_B,THETA_B_H,THETA_B_SYM,L_EXT_S,L_EXT_P,SSR,SPR,ASS_FAC)
	end if
C
C call crystal_info to print and display parameters.
C
	if (kwhat.eq.0) then
          CALL  CRYSTAL_INFO  (Q_PHOT,
     $    F_MOSAIC,D_SPACING,F_REFRAC,A_BRAGG,RN,THICKNESS,SPREAD_MOS)
	endif
C
C Main calculation
C
	if (kwhat.eq.2) then
        IF (F_MOSAIC.EQ.1) THEN
	  Q_MOS = pi**2*abs(psi_h*psi_hbar)/r_lam0/sin(2*graze)
c	write(*,*) '  '
c	write(*,*) '>>>>>>>>>>>>>> CALLING CRYSTAL_MOSAIC'
c	write(*,*) '>>>>>>>>>>>>>> Q_PHOT: ',Q_PHOT
c	write(*,*) '>>>>>>>>>>>>>> VIN: ',VIN
c	write(*,*) '>>>>>>>>>>>>>> BH: ',BH
c	write(*,*) '>>>>>>>>>>>>>> SURFNOR: ',SURFNOR
c	write(*,*) '>>>>>>>>>>>>>> D_SPACING: ',D_SPACING
c	write(*,*) '>>>>>>>>>>>>>> THICKNESS: ',THICKNESS
c	write(*,*) '>>>>>>>>>>>>>> F_REFRAC: ',F_REFRAC
c	write(*,*) '>>>>>>>>>>>>>> SPREAD_MOS: ',SPREAD_MOS
c	write(*,*) '>>>>>>>>>>>>>> Q_MOS: ',Q_MOS
c	write(*,*) '>>>>>>>>>>>>>> REFRAC: ',REFRAC
          CALL      CRYSTAL_MOSAIC   (
     $		Q_PHOT, VIN,
     $		BH, SURFNOR, D_SPACING, THICKNESS, F_REFRAC,SPREAD_MOS,
     $		Q_MOS,REFRAC,
     $		R_S, R_P,PHASE_S, PHASE_P, L_EXT_S, L_EXT_P)
c	write(*,*) '<<<<<<<<<<<<<< R_S: ',R_S
c	write(*,*) '<<<<<<<<<<<<<< R_P: ',R_P
c	write(*,*) '<<<<<<<<<<<<<< PHASE_S: ',PHASE_S
c	write(*,*) '<<<<<<<<<<<<<< PHASE_P: ',PHASE_P
c	write(*,*) '<<<<<<<<<<<<<< L_EXT_S: ',L_EXT_S
c	write(*,*) '<<<<<<<<<<<<<< L_EXT_P: ',L_EXT_P
        ELSE
     	  call	CRYSTAL_PERFECT	 (Q_PHOT, VIN, VOUT, 
     $		BH, SURFNOR,D_SPACING, THICKNESS, F_REFRAC,
     $		PSI_0,PSI_H,PSI_HBAR,
     $		R_S, R_P,PHASE_S, PHASE_P)
	END IF
	end if

	RETURN
     	END

C+++
C	SUBROUTINE	CRYSTAL_INFO
C
C	PURPOSE		C
C
C	ALGORITHM	Reference B.E.Warren, X-Ray Diffraction, 
C
C	MODIFIED	M. Sanchez del Rio
C
C---

     	SUBROUTINE CRYSTAL_INFO	 (Q_PHOT, 
     $F_MOSAIC,D_SPACING,F_REFRAC,A_BRAGG,RN,THICKNESS,SPREAD_MOS)

	implicit none
                                       

CCC
        real*8  pi,twopi,todeg,torad,tocm,toangs
        real*8  theta_o,theta_h
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /
C        DATA    E2_MC2  /  2.817939 D-13                     / ! e^2/mc^2
                                                               !classical e-
                                                               !radius in cm
C        DATA    AVOG    /  6.022098 D+23                     /


        integer*4	i_debug
	real*8		Q_PHOT                              !Arguments
	real*8		L_EXT_S, L_EXT_P
	real*8		THETA_B

        character*80    file_refl                           !as in common.blk
        integer*4       f_mosaic,f_bragg_a,f_refrac
        real*8          spread_mos,thickness,a_bragg
        real*8          d_spacing


	real*8		rn,rn1,d_spacing1

	integer*4	i
	real*8		phot,r_lam0
	complex*16	F_0,FH,FH_BAR,REFRAC

	real*8		graze,sin_gra,ass_fac
	real*8		absorp    !,theta_inc_o
	real*8		ssr,spr, q_mos
	real*8		tmax_mos
	real*8		texts,textp

	complex*16	psi_h,psi_hbar,psi_0

	real*8		theta_b_sym,theta_b_h
	real*8		sigma_gamma,biga_mosaic,omega,kpolsqrt

C
C Computes reflectivities at given wavelength and angle.
C
	i_debug = 0
	PHOT	= Q_PHOT/TWOPI*TOCM
	R_LAM0 	= TWOPI/Q_PHOT
	SIN_GRA = R_LAM0/d_spacing/2.0D0
	GRAZE = ASIN(SIN_GRA)

	RN1 = -1.
	D_SPACING1 = -1.
        call crystal_fh (1, FILE_REFL, PHOT, GRAZE
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC
     $,RN1,D_SPACING1)
#if lahey
	ABSORP  = 2.0d0 * TWOPI *(-AIMAG(REFRAC)) / R_LAM0
#else
	ABSORP  = 2.0d0 * TWOPI *(-IMAG(REFRAC)) / R_LAM0
#endif

     	CALL CRYSTAL_SET	 (Q_PHOT, 
     $F_MOSAIC,D_SPACING,F_REFRAC,A_BRAGG,RN,THICKNESS,SPREAD_MOS,
     $THETA_B,THETA_B_H,THETA_B_SYM,L_EXT_S,L_EXT_P,SSR,SPR,ASS_FAC)

	if (i_debug.eq.1) then
	  write (*,*) '<><>CRYSTAL_INFO ssr: ',ssr
	  write (*,*) '<><>CRYSTAL_INFO spr: ',spr
	  write (*,*) '<><>CRYSTAL_INFO theta_b: ',theta_b
	end if


C***************************************************
c display results on screen (unit=6) and in unit 27 
C***************************************************
C

C
C Calculate mosaic parameters
C
	if (f_mosaic.EQ.1) then
	  if (f_refrac.eq.0) then
	    biga_mosaic = thickness*absorp/sin(torad*graze)    !bragg
	  else if (f_refrac.eq.1) then
	    biga_mosaic = thickness*absorp/cos(torad*graze)    !laue(alpha=90)
	  else 
	    write (*,*) 'CRYSTAL_INFO Error: Option not implemented.'
	    stop
	  endif
	  omega   = (1/sqrt(twopi))*(1/spread_mos)
C
C Q_mos is the mosaic q variable (Zachariasen's [4.33b])
C Sigma_Gamma refers to Zachariasen's formula 4.33a
C
	  q_mos = pi**2*abs(psi_h*psi_hbar)/r_lam0/sin(2*graze)
          sigma_gamma =  omega*q_mos
          kpolsqrt = (cos(2*graze))**2

	  tmax_mos = cos(graze)*dexp(1+2*sigma_gamma/absorp)/2/
     $sigma_gamma/absorp/absorp
C	  texts= r_lam0**2/pi/sin(2*graze)/ssr/2/d_spacing
C	  textp= r_lam0**2/pi/sin(2*graze)/spr/2/d_spacing
	  texts= L_EXT_S*sin(GRAZE)
	  textp= L_EXT_P*sin(GRAZE)
	end if

	DO i=6,23,17       ! write output in units 6(screen) and 23
          WRITE(i,*) ' '
	  RN1 = -1.
	  D_SPACING1 = -1.
          call crystal_fh (i, FILE_REFL, PHOT, GRAZE
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC
     $,RN1,D_SPACING1)


        if (f_refrac.eq.0) then                !bragg diffracted
           THETA_O      = GRAZE + A_BRAGG
           THETA_H      = GRAZE - A_BRAGG
        else if (f_refrac.eq.1) then           !laue diffracted
           theta_o      = abs( - graze + a_bragg )
           theta_h      = abs(   graze + a_bragg )
        else if (f_refrac.eq.2) then           !bragg transmitted
           theta_o      = GRAZE + A_BRAGG
           theta_h      = GRAZE - A_BRAGG
        else if (f_refrac.eq.3) then           !laue transmitted
           theta_o      = abs( - graze + a_bragg )
           theta_h      = abs(   graze + a_bragg )
        endif

          WRITE(i,*) 'Theta (graz)         = ',GRAZE,' rads'
          WRITE(i,*) '                     = ',GRAZE*TODEG,' degrees'
          WRITE(i,*) 'Symmetric Bragg angle (corrected) = ',
     $			THETA_B_SYM,' rad'
          WRITE(i,*) '                                  = ',
     $			THETA_B_SYM*TODEG,' degrees'
          WRITE(i,*) 'Asymmetry factor b=  ',ass_fac
          WRITE(i,*) 'Asymmetry angle alpha= ',a_bragg*todeg,' degrees'
          WRITE(i,*) ' '
C  Following line modified by T. Jach, 10.10.2001
          WRITE(i,*) 
     $'Extinction lengths and depths given here are for amplitude'//
     $'  (for intensities, '
          WRITE(i,*) '   consider half value).'
          WRITE(i,*) 
     $'DEPTH values are measured perpendicular to the crystal '//
     $'surface and LENGTH'
          WRITE(i,*)  '   values are along the incident beam path.'
          WRITE(i,*) 'Extinction length sigma= ',
     $               L_EXT_S*1D4/sin(theta_o),' microns'
C  Following line modified by T. Jach, 10.10.2001
          WRITE(i,*) 'Extinction length pi= ',
     $               L_EXT_P*1D4/sin(theta_o),' microns'
C  Following line modified by T. Jach, 10.10.2001 
          WRITE(i,*) 
     $'Extinction depth sigma= ',L_EXT_S*1D4,' microns'
C  Following line modified by T. Jach, 10.10.2001 
          WRITE(i,*) 
     $'Extinction depth pi= ', L_EXT_P*1D4,' microns'
C  Following line modified by  srio 26.10.2001
          WRITE(i,*) 'Pendellosung period sigma = '//
     $' Extinction depth sigma times pi = ', L_EXT_S*1D4*PI, ' microns'     
C  Following line added  srio 26.10.2001
          WRITE(i,*) 'Pendellosung period pi = '//
     $' Extinction depth pi times pi = ', L_EXT_P*1D4*PI, ' microns'     
		WRITE(i,*) ' '

	  if (f_bragg_a.eq.0) then

           WRITE(i,*) 'width of diffraction profile  s-pol  =  ',
     $              2.0D0*SSR*1.0D+6,' microradians'
           WRITE(i,*) '                                =  ',
     $              2.0D0*SSR*TODEG*3600D0,' arc sec'
           WRITE(i,*) 'width of diffraction profile  p-pol  =  ',
     $              2.0D0*SPR*1.0D+6,' microradians'
           WRITE(i,*) '                                =  ',
     $              2.0D0*SPR*TODEG*3600D0,' arc sec'
	  else
           WRITE(i,*)
     $'The width of the diffraction profile as a function'//
     $' of incident angle is'
           WRITE(i,*)  'width for s-pol  = ',
     $         2.0D0*SSR*1.0D+6,' microradians'
           WRITE(i,*)  '                     =  ',
     $         2.0D0*SSR*TODEG*3600D0,' arc sec'
           WRITE(i,*)  'width for p-pol  = ',
     $         2.0D0*SPR*1.0D+6,' microradians'
           WRITE(i,*)  '                     =  ',
     $         2.0D0*SPR*TODEG*3600D0,' arc sec'
           WRITE(i,*)  'Incident Grazing Angle = ',
     $                (a_bragg+graze)*todeg,' deg'
           WRITE(i,*)  'Incident Corrected Angle = ',
     $                (theta_b)*todeg,' deg'
           WRITE(i,*)   ' '
           WRITE(i,*)   
     $'The width of the diffraction profile as a function'//
     $' of reflected angle is'
           WRITE(i,*)  'width for s-pol  = ',
     $        2.0D0*SSR*1.0D+6*(abs(ass_fac)),' microradians'
           WRITE(i,*)  '                     = ',
     $         2.0D0*SSR*(abs(ass_fac))*TODEG*3600D0,' arc sec'
           WRITE(i,*)  'width for p-pol  = ',
     $         2.0D0*SPR*1.0D+6*(abs(ass_fac)),' microradians'
           WRITE(i,*)  '                     = ',
     $         2.0D0*SPR*(abs(ass_fac))*TODEG*3600D0,' arc sec'
           WRITE(i,*)  'Reflected Grazing Angle = ',
     $                (graze-a_bragg)*todeg,' deg'
           WRITE(i,*)  'Reflected Corrected Angle = ',
     $                (theta_b_h)*todeg,' deg'
	  endif
	END DO
	i = 23 ! only to file
	if (f_mosaic.EQ.1) then
	  write (i,*) '  '
	  write (i,*) '***********  MOSAIC PARAMETERS  ***************'
	  write (i,*) '  '
	  write (i,*) 'spread_mos= ',2.35*spread_mos/TORAD ,' deg fwhm'
	  write (i,*) 'true absorp depth = ',sin(graze)/absorp*1d4,
     $		' microns'
	  write (i,*) 'true absorp length = ',1d4/absorp,' microns'
	  write (i,*) 'peak thickness = ',tmax_mos,' cm'
	  write (i,*) ' '
	  write (i,*) 'For s-polarization we have: '
	  write (i,*) '   Q      = ',q_mos,' cm^-1 '
C	  write (i,*) '   Peak reflectivity = ',r_s
	  write (i,*) '   Primary extinction:'
	  write (i,*) '      mu =',sin(graze)/texts,'cm^-1'
	  write (i,*) '      length =',texts/sin(graze)*1d4,' microns'
	  write (i,*) '      depth =',texts*1d4,' microns'
	  write (i,*) '   Secondary extinction:'
	  write (i,*) '      mu =',sigma_gamma,'cm^-1'
	  write (i,*) '      length =',1d4/sigma_gamma,' microns'
	  write (i,*) '      depth =',sin(graze)/sigma_gamma*1d4,
     $			' microns'

	  write (i,*) ' '
	  write (i,*) 'For p-polarization we have: '
	  write (i,*) '   Q     = ',q_mos*kpolsqrt,' cm^-1 '
C	  write (i,*) '   Peak reflectivity = ',r_p
	  write (i,*) '   Primary extinction:'
	  write (i,*) '      mu =',sin(graze)/textp,'cm^-1'
	  write (i,*) '      length =',textp/sin(graze)*1d4,' microns'
	  write (i,*) '      depth =',textp*1d4,' microns'
	  write (i,*) '   Secondary extinction:'
	  write (i,*) '      mu =',sigma_gamma*kpolsqrt,'cm^-1'
	  write (i,*) '      length =',1d4/sigma_gamma/kpolsqrt,
     $                ' microns'
	  write (i,*) '      depth =',
     $                sin(graze)/sigma_gamma/kpolsqrt*1d4,
     $			' microns'
	  write (i,*) '  '
	endif

	end
C

C+++
C	SUBROUTINE	CRYSTAL_SET
C
C	PURPOSE		C
C
C	ALGORITHM	Reference B.E.Warren, X-Ray Diffraction, 
C
C	MODIFIED	M. Sanchez del Rio
C
C---

     	SUBROUTINE CRYSTAL_SET	 (Q_PHOT, 
     $F_MOSAIC,D_SPACING,F_REFRAC,A_BRAGG,RN,THICKNESS,SPREAD_MOS,
C     $PSI_H,PSI_HBAR,PSI_0,STRUCT,ABSORP,FH,
     $THETA_B,THETA_B_H,THETA_B_SYM,L_EXT_S,L_EXT_P,SSR,SPR,ASS_FAC)

	implicit none
                                       

CCC
        real*8  pi,twopi,todeg,torad,tocm,toangs
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /
C        DATA    E2_MC2  /  2.817939 D-13                     / ! e^2/mc^2
                                                               !classical e-
                                                               !radius in cm
C        DATA    AVOG    /  6.022098 D+23                     /


        integer*4	i_debug
	real*8		Q_PHOT                              !Arguments
	real*8		L_EXT_S, L_EXT_P
	real*8		DELTA_REF, THETA_B

        character*80    file_refl                           !as in common.blk
        integer*4       f_mosaic,f_refrac
        real*8          spread_mos,thickness,a_bragg
        real*8          d_spacing


	real*8		rn,rn1,d_spacing1

	real*8		phot,r_lam0
	complex*16	F_0,FH,FH_BAR,STRUCT,REFRAC

	real*8		graze,sin_gra,theta_o,theta_h,ass_fac
	real*8		absorp    !,theta_inc_o
	complex*16	ssvar,spvar
	real*8		ssr,spr, gammaf, ppol
	real*8		tmp

	complex*16	psi_h,psi_hbar,psi_0

	real*8		theta_b_sym,theta_b_h

C
C Computes reflectivities at given wavelength and angle.
C
	i_debug = 0
	PHOT	= Q_PHOT/TWOPI*TOCM
	R_LAM0 	= TWOPI/Q_PHOT
	SIN_GRA = R_LAM0/d_spacing/2.0D0
	GRAZE = ASIN(SIN_GRA)

	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET working at energy: ',phot
C
C call crystal_fh to get the structure factor
C
	RN1 = -1
	D_SPACING1 = -1.
	call crystal_fh (1, FILE_REFL, PHOT, GRAZE
     $,FH,FH_BAR,F_0,PSI_H,PSI_HBAR,PSI_0,REFRAC
     $,RN1,D_SPACING1)
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET back crystal_fh, d_spacing: ',d_spacing
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET back crystal_fh, rn: ',rn

	STRUCT 	= sqrt(FH * FH_BAR) 
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET FH = ',FH
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET f_0 = ',f_0
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET FH_BAR = ',
     $FH_BAR
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET STRUCT = ',
     $STRUCT
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET PSI_H = ',
     $PSI_H
	if (i_debug.EQ.1) write(*,*) '<><>CRYSTAL_SET PSI_HBAR = ',
     $PSI_HBAR
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_SET PSI_0 = ',
     $PSI_0

#if lahey
	DELTA_REF  = 1.0D0 - REAL(REFRAC,8)
	ABSORP	= 2.0d0 * TWOPI *(-AIMAG(REFRAC)) / R_LAM0
#else
	DELTA_REF  = 1.0D0 - DREAL(REFRAC)
	ABSORP	= 2.0d0 * TWOPI *(-DIMAG(REFRAC)) / R_LAM0
#endif
C
C Calculates the reflection angles and other useful parameters
C


	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET A_BRAGG : ',todeg*a_bragg
C
C Asymmetry factor B
C
        if (f_refrac.eq.0) then                !bragg diffracted
           THETA_O      = GRAZE + A_BRAGG
           THETA_H      = GRAZE - A_BRAGG
        else if (f_refrac.eq.1) then           !laue diffracted
           theta_o      = abs( - graze + a_bragg )
           theta_h      = abs(   graze + a_bragg )
        else if (f_refrac.eq.2) then           !bragg transmitted
           theta_o      = GRAZE + A_BRAGG
           theta_h      = GRAZE - A_BRAGG
        else if (f_refrac.eq.3) then           !laue transmitted
           theta_o      = abs( - graze + a_bragg )
           theta_h      = abs(   graze + a_bragg )
        endif

	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET Uncorrected Bragg angle : ',todeg*theta_o

	ASS_FAC = ABS( sin(theta_o)/sin(theta_h) )
        if ((f_refrac.eq.0).or.(f_refrac.eq.3)) ass_fac = -1.0d0*ass_fac

	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_SET Asymmetry factor is: ',ASS_FAC

C
C THETA_B is the Bragg angle corrected for refraction for sym case,
C following Warren
C
C	THETA_B_SYM = R_LAM0/(1-(DELTA_REF/SIN_GRA**2))/2.0D0/D_SPACING
C	THETA_B_SYM = ASIN(THETA_B_SYM)
C	write (*,*) ' '
C	write (*,*) 'THETA_B_SYM(old) : ',THETA_B_SYM
C Changed to Zachariasen expression ([3.181] when y=0 and b=-1)
#if lahey
	THETA_B_SYM = (real(PSI_0,8)/2/sin_gra)+sin_gra
#else
	THETA_B_SYM = (Dreal(PSI_0)/2/sin_gra)+sin_gra
#endif
	THETA_B_SYM = ASIN(THETA_B_SYM)
C	write (*,*) 'THETA_B_SYM(new) : ',THETA_B_SYM
C
C	delta_theta_s = THETA_B_SYM-GRAZE
C        delta_theta_o = 0.5D0*(1.0D0-1.0D0/ASS_FAC)*delta_theta_s
C        delta_theta_h = 0.5D0*(1.0D0-ASS_FAC)*delta_theta_s
C	if (i_debug.EQ.1) then 
C	  write (*,*) '<><>CRYSTAL_SET THETA_B_SYM: ',THETA_B_SYM
C	  write (*,*) '<><>CRYSTAL_SET DELTA_REF: ',DELTA_REF
C	  write (*,*) '<><>CRYSTAL_SET SIN_GRA: ',SIN_GRA
C	  write (*,*) '<><>CRYSTAL_SET D_SPACING: ',D_SPACING
C	endif
C
c	write(*,*) 'theta_b_sim = ',theta_b*todeg
c	write(*,*) 'delta_theta_s = ',delta_theta_s*todeg
c	write(*,*) 'delta_theta_o = ',delta_theta_o*todeg
c	write(*,*) 'delta_theta_h = ',delta_theta_h*todeg
C	if ((f_refrac.eq.0).or.(f_refrac.eq.2)) then
C	  if (f_bragg_a.eq.1) then
C            THETA_B = GRAZE + A_BRAGG + delta_theta_o
C            THETA_B_H = GRAZE - A_BRAGG + delta_theta_h
C	  else
C	    THETA_B = THETA_B_SYM
C	  endif
C	else if ((f_refrac.eq.1).or.(f_refrac.eq.3)) then
c
c please check the minus sign !!!!!!
c
C	  THETA_B = GRAZE + A_BRAGG -  delta_theta_o
C          THETA_B_H =  GRAZE - A_BRAGG - delta_theta_h
C	endif
C
C	write (*,*) 'THETA_B(old): ',THETA_B
C	write (*,*) 'THETA_B_H(old): ',THETA_B_H
C Changed to Zachariasen expression ([3.181] when y=0)
#if lahey
	THETA_B = (ass_fac-1)/ass_fac*(real(PSI_0,8)/4/sin_gra)+sin_gra
	THETA_B = ASIN(THETA_B) + A_BRAGG
	THETA_B_H = (1-ass_fac)*(real(PSI_0,8)/4/sin_gra)+sin_gra
#else
	THETA_B = (ass_fac-1)/ass_fac*(Dreal(PSI_0)/4/sin_gra)+sin_gra
	THETA_B = ASIN(THETA_B) + A_BRAGG
	THETA_B_H = (1-ass_fac)*(Dreal(PSI_0)/4/sin_gra)+sin_gra
#endif
	THETA_B_H = ASIN(THETA_B_H) - A_BRAGG
C	write (*,*) 'THETA_B(new): ',THETA_B
C	write (*,*) 'THETA_B_H(new): ',THETA_B_H
C	write (*,*) ' '



	if (i_debug.EQ.1) then 
	  write (*,*)
     $'<><>CRYSTAL_SET Corrected Bragg angle : ',todeg*THETA_B
	  write (*,*) '<><>CRYSTAL_SET GRAZE: ',GRAZE
	  write (*,*) '<><>CRYSTAL_SET A_BRAGG: ',A_BRAGG
C	  write (*,*) '<><>CRYSTAL_SET delta_theta_o: ',delta_theta_o
	endif


C
C S%VAR is the variable "s" of Warren.
C S%VAR is the angular width corresponding a y=1 in the Zachariasen notation.
C This value corresponds to the half-width of the Bragg symmetrical 
C reflection in the case of no absorption. 
C This value corresponds to y=1 (y is the Zachariasen variable defined
C in [3.181] under the assumptions [3.180] and [3.116].
C
	gammaf   = RN*(R_LAM0**2)/PI
	PPOL    = ABS(COS(2.0D0*GRAZE))


C	   L_EXT_S = R_LAM0*cos(theta_b)/gammaf/abs(fh)
C Bug fixed 14 Set 2001 srio@esrf.fr
C The extinction length was only for Laue symmetric case.
C Now, the general formula is implemented. 
C Thanks to T. Jach who pointed out the problem, and previous
C remarks by J. Sutter, E. Alp and R. Dejus.
C
C Following line modified by T. Jach 10/10/01
C
C	   L_EXT_S =
C     $     R_LAM0*sin(theta_o)/gammaf/abs(fh)/sqrt(abs(ass_fac))/PI
C
C The formula above is correct for symmetric case (Bragg and Laue). 
C However, to avoid confussion in asymmetric cases, it is better to express 
C it as a function of theta bragg and alpha
C          write (*,*) '<><>THETA_O : ',THETA_O
C          write (*,*) '<><>GRAZE : ',GRAZE
C          write (*,*) '<><>A_BRAGG : ',A_BRAGG
C          write (*,*) '<><>L_EXT_S : ',L_EXT_S
           L_EXT_S =
     $     R_LAM0*sin(graze+a_bragg)/gammaf/abs(fh)/
     1  sqrt(abs(ass_fac))/PI

C          write (*,*) '<><>L_EXT_S : ',L_EXT_S


	L_EXT_P = L_EXT_S/PPOL
C
C width of Bragg reflection
C
C Warren formalism: SSR is the half-width
C        SSVAR   = RN*(R_LAM0**2)*STRUCT/PI/SIN(2.0D0*GRAZE)
	SSVAR   = gammaf*STRUCT/SIN(2.0D0*GRAZE)
        SPVAR   = SSVAR*PPOL
#if lahey
        SSR     = REAL(SSVAR,8)
        SPR     = REAL(SPVAR,8)
#else
        SSR     = DREAL(SSVAR)
        SPR     = DREAL(SPVAR)
#endif


C Zachariasen formalism: SSR is the half-width
C angular value corresponding to y=1
	SSR = sin_gra- (d_spacing/r_lam0/ass_fac/2) *
     $( 2*sqrt(abs(ass_fac)*abs(psi_h*psi_hbar)) +
#if lahey
     $(ass_fac-1)*real(PSI_0,8) )
#else
     $(ass_fac-1)*Dreal(PSI_0) )
#endif
	SSR = ASIN(SSR)


C angular value corresponding to y=-1
	tmp = sin_gra- (d_spacing/r_lam0/ass_fac/2) *
     $( -2*sqrt(abs(ass_fac)*abs(psi_h*psi_hbar)) +
#if lahey
     $(ass_fac-1)*real(PSI_0,8) )
#else
     $(ass_fac-1)*Dreal(PSI_0) )
#endif
	TMP = ASIN(TMP)
C width (2y)
	SSR = (SSR - TMP)/2.0D0
	if (i_debug.eq.1) then
	  write (*,*) ' '
	  write (*,*) '<><>CRYSTAL_SET THETA_B: ',THETA_B
	  write (*,*) 'SSR(new): ',SSR
	end if
C p-pol
C Zachariasen formalism: SSR is the half-width
C angular value corresponding to y=1
        SPR = sin_gra- (d_spacing/r_lam0/ass_fac/2) *
     $( 2*ppol*sqrt(abs(ass_fac)*abs(psi_h*psi_hbar)) +
#if lahey
     $(ass_fac-1)*real(PSI_0,8) )
#else
     $(ass_fac-1)*Dreal(PSI_0) )
#endif
        SPR = ASIN(SPR)
C        write (*,*) 'SPR1(new): ',SPR
C angular value corresponding to y=-1
        tmp = sin_gra- (d_spacing/r_lam0/ass_fac/2) *
     $( -2*ppol*sqrt(abs(ass_fac)*abs(psi_h*psi_hbar)) +
#if lahey
     $(ass_fac-1)*real(PSI_0,8) )
#else
     $(ass_fac-1)*Dreal(PSI_0) )
#endif
        TMP = ASIN(TMP)
C width (2y)
        SPR = (SPR - TMP)/2.0D0
	if (i_debug.eq.1) then
	  write (*,*) ' '
	  write (*,*) '<><>CRYSTAL_SET THETA_B: ',THETA_B
	  write (*,*) '<><>CRYSTAL_SET SSR: ',SSR
	  write (*,*) '<><>CRYSTAL_SET SPR: ',SPR
	end if

	end
C
c
c#if unix
c#include  "./../../include/header.txt"
c#elif vms
c     	INCLUDE		'SHADOW$INC:HEADER.TXT/LIST'
c#endif
c
C
C
C+++
C	SUBROUTINE	CRYSTAL_PERFECT
C
C	PURPOSE		Computes the reflectivity of a perfect crystal 
C			according to the dynamic theory of x-ray diffraction.
C
C	ALGORITHM	Reference 
C			W. H. Zachariasen
C			"Theory of X-Ray diffraction in crystals"
C			Dover, New York, 1967.
C
C       MODIFIED        M. Sanchez del Rio, Feb 1996. This routine was
C			a part of crystal.F, and now is independent.
C
C---
     	SUBROUTINE	CRYSTAL_PERFECT	 (
     $Q_PHOT, VIN, VOUT, 				! inputs (ray)
     $BH, SURFNOR, D_SPACING, THICKNESS, F_REFRAC, 	! inputs (crystal)
     $PSI_0,PSI_H,PSI_HBAR,				! inputs (struct fact)
     $R_S, R_P,PHASE_S, PHASE_P)			! output (reflectivity)
C
C INPUT PARAMETERS:
C   Q_PHOT: Photon wavelength
C   VIN:     Photon incident direction
C   VOUT:    Photon outcoming direction
C   BH:		Bragg planes normal
C   SURFNOR:	Crystal surface normal
C   D_SPACING:	crystal d-spacing
C   THICKNESS:	crystal thickness
C   F_REFRAC:	Mode flag: 0:BraggDiff, 1:LaueDiff, 2:BraggTrans, 3:LaueTrans
C   PSI_0:	|
C   PSI_H:	| Electrical susceptibility (proportional to structure factor)
C   PSI_HBAR:	|
C
C OUTPUT PARAMETERS:
C   R_S:	Sigma reflectivity
C   R_P:	Pi reflectivity
C   PHASE_S:	Sigma phase shift
C   PHASE_P:	Pi phase shift
C
C
	implicit none
        real*8         OEXP		!exponential overflow limit
        parameter      (OEXP=100.0D0)

CCC
        real*8  pi,twopi,todeg,torad,tocm,toangs
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /

        integer*4	i_debug
	real*8		Q_PHOT                              !Arguments
	real*8		VIN(3), VOUT(3), BH(3), SURFNOR(3)
	real*8		R_S, R_P,PHASE_S, PHASE_P

        integer*4       f_refrac
        real*8          thickness    !,a_bragg
        real*8          d_spacing


	real*8		r_lam0,phot
	complex*16	ci

	real*8		graze,sin_gra

	real*8		vtemp(3),sin_brg,alpha_zac,mod2_bh
	real*8		vtemp2(3),tmp
	real*8		sin_q_ang,sin_q_ref,gamma_0,gamma_h
	real*8		cry_b,cry_t,cry_a,cry_alpha
	complex*16	psi_h,psi_hbar,psi_0
	complex*16	cry_q,cry_z,ctemp,br_x1,br_x2
	complex*16	br_delta1,br_delta2,br_c1,br_c2
	complex*16	rcs,rcp
	real*8		c_ppol,pp,qq
c	real*8		oexp            ! exponential overflow limit
	external	sum


C
	i_debug = 0
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_PERFECT: ******** crystal_perfect called ********'
	CI	= (0.0D0,1.0D0)

	PHOT	= Q_PHOT/TWOPI*TOCM
	R_LAM0 	= TWOPI/Q_PHOT
	SIN_GRA = R_LAM0/d_spacing/2.0D0
	GRAZE = ASIN(SIN_GRA)

	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_PERFECT: working at energy: ',phot
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_PERFECT: d_spacing: ',d_spacing

	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_PERFECT: PSI_H = ',
     $		PSI_H
	if (i_debug.EQ.1) write(*,*) '<><>CRYSTAL_PERFECT: PSI_HBAR = ',
     $		PSI_HBAR
	if (i_debug.EQ.1) write (*,*) '<><>CRYSTAL_PERFECT: PSI_0 = ',
     $		PSI_0


C	DELTA_REF  = 1.0D0 - DREAL(REFRAC)
C	ABSORP	= 2.0d0 * TWOPI *(-DIMAG(REFRAC)) / R_LAM0

	call norm (bh,vtemp)
	call dot (vin,vtemp,sin_brg)	
	call dot (vin,surfnor,sin_q_ang)
	call dot (vout,surfnor,sin_q_ref)
	gamma_0 = sin_q_ang        ! sign changed

C
C >>>>>>>>>>>>>>>>>>>> Perfect crystal calculation <<<<<<<<<<<<<<<<<<<
C
C Main calculation (symmetrical case and asym incident case)
C I change to reflectivity formulae of Zachariasen,
C for a definition of ETA taking into account the angle with
C Bragg planes. MSR 6/28/90
C
	ALPHA_ZAC =-((R_LAM0/D_SPACING)**2+2*R_LAM0*
     $SIN_BRG/D_SPACING)
	if (i_debug.EQ.1) 
     $write (*,*) '<><>CRYSTAL_PERFECT: !!!!! sin_brg: ',sin_brg
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_PERFECT: !!!!! alpha_zac_old: ',alpha_zac

	call dot (bh,bh,mod2_bh)
	call dot (vin,bh,alpha_zac)
	alpha_zac = -1D0*(1D0/q_phot)**2 * (mod2_bh -2*q_phot*alpha_zac)
	if (i_debug.EQ.1) write (*,*) 
     $'<><>CRYSTAL_PERFECT: !!!!! alpha_zac: ',alpha_zac

*
* Transmission (Laue) case and general Reflection (Bragg) case
* NB: The general Bragg case is commented (cc). It is not yet being
* used. We still use the Thick Crystal Approximation case.
*
C
C   This part has been written by G.J. Chen and M. Sanchez del Rio. 
C   We use the formula [3.130] of Zachariasen's book.
C
	gamma_h = -1.0*sin_q_ref
c
	if ((f_refrac.eq.0).or.(f_refrac.eq.2)) gamma_h = - gamma_h
C
C changed b factor to its vectorial value (Zachariasen, [3.115])
C
	if (i_debug.EQ.1) then
	  cry_b = gamma_0/gamma_h
	  write (*,*) 'CRYSTAL_PERFECT: b(approx)= ',cry_b
	end if
c numerator
	call dot (surfnor,vin,cry_b)
	cry_b = cry_b*q_phot
c denominator
	call scalar(vin,q_phot,vtemp)
	call sum (vtemp,bh,vtemp2)
	call dot (surfnor,vtemp2,tmp)
c ratio
	cry_b = cry_b / tmp
	if (i_debug.EQ.1) then
	  write (*,*) 'CRYSTAL_PERFECT: b(exact)= ',cry_b
	end if


	cry_t = 0.5*(1./abs(gamma_0) +1./abs(gamma_h))*thickness 
	cry_a = pi/r_lam0*(thickness/gamma_0)
	cry_alpha = -((r_lam0/d_spacing)**2+2*r_lam0*
     $sin_brg/d_spacing)
c

	cry_q = cry_b*psi_h*psi_hbar
	cry_z = (1.-cry_b)*0.5*psi_0 + cry_b*0.5*cry_alpha
	if (i_debug.EQ.1) then
	  write(*,*) '<><>CRYSTAL_PERFECT: cry_b: ',cry_b
	  write(*,*) '<><>CRYSTAL_PERFECT: psi_0: ',psi_0
	  write(*,*) '<><>CRYSTAL_PERFECT: cry_alpha: ',cry_alpha
	  write(*,*) '<><>CRYSTAL_PERFECT: cry_z: ',cry_z
	endif


c
c s-polarization
c
	ctemp = sqrt(cry_q  + cry_z**2)
	br_x1 = (-1.0d0*cry_z+ctemp)/psi_hbar
	br_x2 = (-1.0d0*cry_z-ctemp)/psi_hbar
	br_delta1 = 0.5d0*(psi_0-cry_z+ctemp)
	br_delta2 = 0.5d0*(psi_0-cry_z-ctemp)
	br_c1 = -1.d0*ci*thickness*twopi/(-1.*abs(gamma_0))/
     $		r_lam0*br_delta1
	br_c2 = -1.d0*ci*thickness*twopi/(-1.*abs(gamma_0))/
     $		r_lam0*br_delta2
	if (i_debug.EQ.1) then
	  write(*,*) '<><>CRYSTAL_PERFECT: ctemp: ',ctemp
	endif
c
c a very big exponential produces numerical overflow. If so, the value
c is changed artificially to avoid the overflow. This is equivalent to 
c use the thick crystal approximation
c
c	oexp = 100.d0
#if lahey
	if (real(br_c1,8).gt.oexp.or.real(br_c2,8).gt.oexp) then
	  if (i_debug.EQ.1) write(*,*) 
     $'CRYSTAL: exponential overflow. Corrected.'
	  if (real(br_c1,8).gt.oexp) br_c1 = oexp+ci*aimag(br_c1)
	  if (real(br_c2,8).gt.oexp) br_c2 = oexp+ci*aimag(br_c2)
#else
	if (Dreal(br_c1).gt.oexp.or.Dreal(br_c2).gt.oexp) then
	  if (i_debug.EQ.1) write(*,*) 
     $'CRYSTAL: exponential overflow. Corrected.'
	  if (Dreal(br_c1).gt.oexp) br_c1 = oexp+ci*Dimag(br_c1)
	  if (Dreal(br_c2).gt.oexp) br_c2 = oexp+ci*Dimag(br_c2)
#endif
	endif
c
	br_c1 = exp(br_c1)
	br_c2 = exp(br_c2)
c
	if (f_refrac.eq.0) then
	  rcs = br_x1*br_x2*(br_c1-br_c2)/(br_c2*br_x2-br_c1*br_x1) ! bragg D
	else if (f_refrac.eq.1) then 
	  rcs = br_x1*br_x2*(br_c1-br_c2)/(br_x2-br_x1)             ! laue D
	else if (f_refrac.eq.2) then 
	  rcs = br_c1*br_c2*(br_x1-br_x2)/(br_c2*br_x2-br_c1*br_x1) ! bragg T
	else if (f_refrac.eq.3) then 
	  rcs = (br_x2*br_c1-br_x1-br_c2)/(br_x2-br_x1)             ! laue T
	endif
	if (i_debug.EQ.1) then
	  write(*,*) '<><>CRYSTAL_PERFECT: br_c1: ',br_c1
	  write(*,*) '<><>CRYSTAL_PERFECT: br_c2: ',br_c2
	  write(*,*) '<><>CRYSTAL_PERFECT: br_x1: ',br_x1
	  write(*,*) '<><>CRYSTAL_PERFECT: br_x2: ',br_x2
	endif

c	r_s = (1.0d0/abs(cry_b))*rcs*dconjg(rcs)
c	r_s1 = sqrt((1.0d0/abs(cry_b))*rcs*dconjg(rcs))
c	rcs = rcs/dsqrt(abs(cry_b))
c
c p-polarization
c
	c_ppol = abs(cos(2.0d0*graze))

	ctemp = sqrt(cry_q*c_ppol**2  + cry_z**2)
	br_x1 = (-1.0d0*cry_z+ctemp)/(psi_hbar*c_ppol)
	br_x2 = (-1.0d0*cry_z-ctemp)/(psi_hbar*c_ppol)
	br_delta1 = 0.5d0*(psi_0-cry_z+ctemp)
	br_delta2 = 0.5d0*(psi_0-cry_z-ctemp)
	br_c1 = -1.0d0*ci*thickness*twopi/(-1.0*abs(gamma_0))/
     $		r_lam0*br_delta1
	br_c2 = -1.0d0*ci*thickness*twopi/(-1.0*abs(gamma_0))/
     $		r_lam0*br_delta2
c
c a very big exponential produces numerical overflow. If so the value
c is changed to avoid the overflow. This is equivalent to the thick
c crystal approximation
c
#if lahey
	if (real(br_c1,8).gt.oexp.or.real(br_c2,8).gt.oexp) then
	  if (i_debug.EQ.1) write(*,*) 
     $'CRYSTAL: exponential overflow. Corrected.'
	  if (real(br_c1,8).gt.oexp) br_c1 = oexp+ci*aimag(br_c1)
	  if (real(br_c2,8).gt.oexp) br_c2 = oexp+ci*aimag(br_c2)
#else
	if (Dreal(br_c1).gt.oexp.or.Dreal(br_c2).gt.oexp) then
	  if (i_debug.EQ.1) write(*,*) 
     $'CRYSTAL: exponential overflow. Corrected.'
	  if (Dreal(br_c1).gt.oexp) br_c1 = oexp+ci*Dimag(br_c1)
	  if (Dreal(br_c2).gt.oexp) br_c2 = oexp+ci*Dimag(br_c2)
#endif
	endif
c
	br_c1 = exp(br_c1)
	br_c2 = exp(br_c2)

c	if (f_refrac.eq.1) then
c	  rcp = br_x1*br_x2*(br_c1-br_c2)/(br_x2-br_x1)             ! laue
c	else if (f_refrac.eq.0) then
c	  rcp = br_x1*br_x2*(br_c1-br_c2)/(br_c2*br_x2-br_c1*br_x1) ! bragg
c	endif

	if (f_refrac.eq.0) then
	  rcp = br_x1*br_x2*(br_c1-br_c2)/(br_c2*br_x2-br_c1*br_x1) ! bragg D
	else if (f_refrac.eq.1) then 
	  rcp = br_x1*br_x2*(br_c1-br_c2)/(br_x2-br_x1)             ! laue D
	else if (f_refrac.eq.2) then 
	  rcp = br_c1*br_c2*(br_x1-br_x2)/(br_c2*br_x2-br_c1*br_x1) ! bragg T
	else if (f_refrac.eq.3) then 
	  rcp = (br_x2*br_c1-br_x1-br_c2)/(br_x2-br_x1)             ! laue T
	endif
	
	rcs = (1.0d0/sqrt(abs(cry_b)))*rcs
	rcp = (1.0d0/sqrt(abs(cry_b)))*rcp
	if (i_debug.EQ.1) then
	  write(*,*) '<><>CRYSTAL_PERFECT: rcs: ',rcs
	  write(*,*) '<><>CRYSTAL_PERFECT: rcp: ',rcp
	endif

c please check
c	  IF (GRAZE.GT.45*TORAD) 	RCP = -RCP

	R_S	= ABS(RCS)
#if lahey
	PP	= REAL(RCS,8)
	QQ	= AIMAG(RCS)
#else
	PP	= DREAL(RCS)
	QQ	= DIMAG(RCS)
#endif
	CALL	ATAN_2	(QQ,PP,PHASE_S)
	R_P	= ABS(RCP)
#if lahey
	PP	= REAL(RCP,8)
	QQ	= AIMAG(RCP)
#else
	PP	= DREAL(RCP)
	QQ	= DIMAG(RCP)
#endif
	CALL	ATAN_2	(QQ,PP,PHASE_P)

1122   	RETURN
     	END
c
c#if unix
c#include  "./../../include/header.txt"
c#elif vms
c     	INCLUDE		'SHADOW$INC:HEADER.TXT/LIST'
c#endif
c
C
C
C+++
C	SUBROUTINE	CRYSTAL_MOSAIC
C
C	PURPOSE		Computes the reflectivity of a mosaic crystal 
C			according to the dynamic theory of x-ray diffraction.
C
C	ALGORITHM	Reference 
C			W. H. Zachariasen
C			"Theory of X-Ray diffraction in crystals"
C			Dover, New York, 1967.
C			and
C                       Bacon and Lowde, Acta Crystall. 1 pag 303 (1948) for 17
C
C       MODIFIED        M. Sanchez del Rio, Feb 1996. This routine was
C			a part of crystal.F, and now is independent.
C
C---
     	SUBROUTINE	CRYSTAL_MOSAIC	 (
     $Q_PHOT, VIN, 				! inputs (ray)
     $BH, SURFNOR, D_SPACING, THICKNESS, F_REFRAC,SPREAD_MOS,! inputs (crystal)
     $Q_MOS,REFRAC,				! inputs (struct fact)
     $R_S, R_P,PHASE_S, PHASE_P, L_EXT_S, L_EXT_P)	! output (reflectivity)
C
C INPUT PARAMETERS:
C   Q_PHOT: Photon wavelength = 2 pi / lambda [cm^-1]
C   VIN:     Photon incident direction
C   BH:		Bragg planes normal *  2 pi/d [cm^-1]
C   SURFNOR:	Crystal surface normal
C   D_SPACING   crystal lattice distance [cm]
C   THICKNESS:	crystal thickness
C   F_REFRAC:	Mode flag: 0:BraggDiff, 1:LaueDiff, 2:BraggTrans, 3:LaueTrans
C   SPREAD_MOS: mosaicity
C   Q_MOS: 	Q variable = pi**2*abs(psi_h*psi_hbar)/r_lam0/sin(2*graze)
C   REFRAC:	Refraction index
C
C OUTPUT PARAMETERS:
C   R_S:	Sigma reflectivity
C   R_P:	Pi reflectivity
C   PHASE_S:	Sigma phase shift
C   PHASE_P:	Pi phase shift
C   L_EXT_S:	Scondary extinction length (s-pol)
C   L_EXT_P:	Scondary extinction length (p-pol)
C
C

	implicit none
                                       

        real*8  pi,twopi,todeg,torad,tocm,toangs
        DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C        DATA    PIHALF  /  1.5707 96326 79489 66192 31322 D0 /
        DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
        DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
        DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
C        DATA    TOCM    /  1.239 852    D-4                  /
C        DATA    TOANGS  /  1.239 852    D+4                  /
        DATA    TOCM    /  1.2398 4187572794035    D-4        /
        DATA    TOANGS  /  1.2398 4187572794035    D+4        /


	real*8		Q_PHOT                              !Arguments
	real*8		VIN(3), BH(3), SURFNOR(3)
	real*8		R_S, R_P,PHASE_S, PHASE_P
	real*8		L_EXT_S, L_EXT_P

        integer*4       f_refrac
        real*8          spread_mos,thickness,d_spacing


	real*8		phot,r_lam0

	real*8		graze,sin_gra
	real*8		absorp    
	real*8		q_mos

	real*8		vtemp(3),sin_brg
	real*8		sin_q_ang !,sin_q_ref

	real*8		sigma_gamma,biga_mosaic,ep,omega,kpolsqrt
	real*8		smallas_mosaic,smallap_mosaic,rs_mosaic
	real*8		rp_mosaic
	complex*16	refrac


C
C >>>>>>>>>>>>>>>>>>>> Mosaic crystal calculation <<<<<<<<<<<<<<<<<<<<<<
C
        PHOT    = Q_PHOT/TWOPI*TOCM
        R_LAM0  = TWOPI/Q_PHOT
        SIN_GRA = R_LAM0/d_spacing/2.0D0
        GRAZE = ASIN(SIN_GRA)

#if lahey
	ABSORP  = 2.0d0 * TWOPI *(-AIMAG(REFRAC)) / R_LAM0
#else
	ABSORP  = 2.0d0 * TWOPI *(-DIMAG(REFRAC)) / R_LAM0
#endif

	call norm (bh,vtemp)
	call dot (vin,vtemp,sin_brg)	
	call dot (vin,surfnor,sin_q_ang)
C	call dot (vout,surfnor,sin_q_ref)

        EP      = abs(ASIN(SIN_BRG)) - graze ! scalar def

        BIGA_MOSAIC  = -1.0d0*THICKNESS*ABSORP/SIN_Q_ANG


        OMEGA =(DEXP(-EP**2/2.0D0/(SPREAD_MOS)**2))
     $               /SQRT(TWOPI)/SPREAD_MOS
CCC	Q_MOS = pi**2*cdabs(psi_h*psi_hbar)/r_lam0/sin(2*graze)
	SIGMA_GAMMA =  OMEGA*Q_MOS
	KPOLSQRT = (cos(2*graze))**2

        SMALLAS_MOSAIC = SIGMA_GAMMA/absorp
        SMALLAP_MOSAIC = SMALLAS_MOSAIC*KPOLSQRT

*
* Transmission case
*
	if (f_refrac.eq.1) then
	  rs_mosaic = sinh(smallas_mosaic*biga_mosaic) * 
     $     exp(-biga_mosaic*(1+smallas_mosaic))
	  rp_mosaic = sinh(smallap_mosaic*biga_mosaic) * 
     $     exp(-biga_mosaic*(1+smallap_mosaic))
*
* Reflection case
*
	else  if (f_refrac.eq.0) then
          RS_MOSAIC = 1+SMALLAS_MOSAIC+(SQRT(1+2*SMALLAS_MOSAIC))/
     $		DTANH(bigA_MOSAIC*SQRT(1+2*SMALLAS_MOSAIC))
          RP_MOSAIC = 1+SMALLAP_MOSAIC+(SQRT(1+2*SMALLAP_MOSAIC))/
     $		DTANH(bigA_MOSAIC*SQRT(1+2*SMALLAP_MOSAIC))
          RS_MOSAIC = SMALLAS_MOSAIC / RS_MOSAIC
          RP_MOSAIC = SMALLAP_MOSAIC / RP_MOSAIC
*
* other (not implemented) cases
*
	else
	  write(*,*) 'CRYSTAL_MOSAIC: Error: '//
     $		'Mode (f_refrac) not implemented.'
	  stop
	endif
        R_S     = SQRT(RS_MOSAIC)
        R_P     = SQRT(RP_MOSAIC)
*
* Mean value of depht into the crystal. To be used in MIRROR
*
        L_EXT_S = 1.0D0 /SIGMA_GAMMA
        L_EXT_P = 1.0D0 /SIGMA_GAMMA/KPOLSQRT
*
*No phase change are introduced by now. (The formulae of reflectivity 
*are already intensity, and no complex coefficient are considered).
*This is not important because a mosaic crystal breaks always coherence
*
        PHASE_S = 0.0D0
        PHASE_P = 0.0D0
	RETURN
	END
