PRO periodictable_plot,zMax=zMax,  $
  Fill=pFill, clrfill=pClrFill, $
  Border=pBorder, clrBorder=pClrBorder, $
  clrSymbol=pClrSymbol, background=pBackground, $
  showZ = pShowZ, position=pPosition, $
  noPlot=noPlot,out=out,Font=pFont


; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       PERIODICTABLE_PLOT
;
; PURPOSE:
;	Plots a periodic table in the current display
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	periodictable_plot
;	
; OUTPUT:
;	a graph
;
; KEYWORDS:
;       zMax: Maximum atomic number for the periodic table (default=105)
;       Fill: set to 
;             1 for filling the cells with colors. 
;             0 for no filling
;             -1 for filling with predefined colors (see example)
;       clrfill: When Fill=1, this keyword indicates the color index
;                (scalar) for filling all cells, or the different color 
;                indices (array) for filling the individual cells. 
;                Also, set it to 1 to fill with predefined colors. 
;       Border: set to 1 to draw cell borders
;       clrBorder: When Border=1, this keyword indicates the color index
;       clrSymbol: set to a color index (scalar) or color indices
;                  (array) to color the atomic symbols.
;       background: Set to the color index for the background
;       showZ: set to 1 for displaying atomic numbers (default=1)
;       position: set to [xmin,ymin,xmax,ymax] with the coordinates of
;                 the working area where the periodic table is plotted. 
;                 (like !p.position) default=[.01,.01,.99,.99]
;       noPlot: set this keyword when you do not want to make a plot 
;               (usually for returning the data)
;       Font: set to the font index (default=0)
;       out: set to a named variable to retrieve a structure with all
;            inputs
;
;       out={p:!p,x:!x,y:!y, position:!p.position, background:!p.background, $
;          zTop:zTop, zMax:105, showZ:1, $
;          Fill:0, ClrFill:Replicate(!d.table_Size*0.5,zTop), $
;          Border:1,ClrBorder:!p.color, $
;          clrSymbol:Replicate(!p.color,zTop), $
;          z:z, zx:zx, zy:zy, symbols:symbols, names:names, font:0 }
;
; EXAMPLES:
;	tek_color
;       periodictable_plot,fill=-1
;
; RESTRICTIONS:
;	It does not print correctly (character size calculated for
;       screen resolution)
;
; MODIFICATION HISTORY:
;       2007-10-23 srio@esrf.eu Created
;
;-
;
on_error,2
;
; store all data
;
IF N_Elements(out) EQ 0 THEN BEGIN
  zTop = 119
  ;
  ; define arrays
  ;
  nx = 18D0
  ny = 10D0

  x = MakeArray1(nx+1,1.0D,nx+1)
  y = MakeArray1(ny+1,1.0D,ny+1)

  ;
  ; Z
  ;
  z = LIndGen(zTop) ; z=0 is vacuum
  
  ;
  ; Define coordinates ans symbols for each Z
  ;
  
  zx = [0,$
        1,                              18, $
        1,2,                  Fix(MakeArray1(6,13,18)), $
        1,2,                  Fix(MakeArray1(6,13,18)), $
        Fix(MakeArray1(18,1,18)),  $
        Fix(MakeArray1(18,1,18)),  $
        1,2,Fix(MakeArray1(15,4,18)), Fix(MakeArray1(15,4,18)), $
        1,2,Fix(MakeArray1(15,4,18)), Fix(MakeArray1(15,4,18)) ] 
  
  zy = [0, $
        1,                                 1, $
        2,2,                  Replicate(2,6), $
        3,3,                  Replicate(3,6), $
        Replicate(4,18), $
        Replicate(5,18), $
        6,6,Replicate(9,15), Replicate(6,15), $
        7,7,Replicate(10,15),Replicate(7,15) ]
  
  symbols =  atomic_symbols()
  names = atomic_names()
    
  zy=11-zy ; first row (H) is at the top

  out={p:!p,x:!x,y:!y, position:!p.position, background:!p.background,  $
     zTop:zTop, zMax:105, showZ:1, $
     Fill:0, ClrFill:Replicate(!d.table_Size*0.5,zTop), $
     Border:1,ClrBorder:!p.color, $
     clrSymbol:Replicate(!p.color,zTop), $
     z:z, zx:zx, zy:zy, symbols:symbols, names:names, font:0 }
ENDIF

;
; rearrange inputs
;

IF N_Elements(zMax) NE 0 THEN BEGIN
  zMax = zMax < (out.zTop-1) > 0
  out.zMax = zMax
ENDIF

IF N_Elements(pFill) NE 0 THEN out.fill = pFill

CASE N_Elements(pClrFill) OF
  0:
  1: out.clrFill = out.clrFill*0+pClrFill
  else: out.clrFill = pClrFill
ENDCASE

IF N_Elements(pBorder) NE 0 THEN out.border = pBorder
IF N_Elements(pClrBorder) NE 0 THEN out.ClrBorder = pClrBorder

CASE N_Elements(pClrSymbol) OF
  0:
  1: out.clrSymbol = out.clrSymbol*0+pClrSymbol
  else: out.clrSymbol = pClrSymbol
ENDCASE


IF N_Elements(pShowZ) NE 0 THEN out.ShowZ=pShowZ
IF N_Elements(pBackground) NE 0 THEN out.background=pBackground

IF N_Elements(pPosition) NE 4 THEN BEGIN
  marginX= 0.01D
  marginY= 0.01D
  pPosition=[marginX,marginY, 1.0D0-marginX,1.0D0-marginY]
ENDIF
out.position=pPosition

IF N_Elements(pFont) NE 0 THEN out.font=pFont


IF out.fill EQ -1 THEN BEGIN
  tmp = out.ClrFill*0
  tmp[1]=15  ; H
  tmp[[3,11,19,37,55,87]]=8 ; Alkali
  tmp[[4,12,20,38,56,88]]=7 ; Alkali earth

  ; transition metals
  tmp[21:30]=22
  tmp[39:48]=22
  tmp[72:80]=22
  tmp[104:112]=22
  tmp[[57,89]]=22

  tmp[[13,31,32,49,50,51,81,82,83,84,113,114,115,116]]=5 ; Poor metals
  tmp[[5,6,7,8,9,14,15,16,17,33,34,35,52,53,85]]=17 ; Nonmetals
  tmp[[2,10,18,36,54,86]]=27

  tmp[58:71]=31  ; Lanthanides
  tmp[90:103]=6 ; Actinides

  out.ClrFill=tmp
ENDIF

;
; make plot
;

IF NOT(Keyword_Set(noPlot)) THEN BEGIN
p_old=!p
x_old=!x
y_old=!y
!p.position=out.position
!x.range=[1D,19D0+18D0/100]
!y.range=[1D,11D0+10D0/100]
!x.style=4+1
!y.style=4+1
!p.background=out.background
plot,[0D,0],/noData ; ,_extra=extra

FOR i=1,out.zmax  DO BEGIN
    zi = out.z[i] ; zi=i
    x0 = out.zx[i]
    y0 = out.zy[i]
    IF out.Fill NE 0 THEN $
    polyFill,[0.D0,1,1,0,0]+x0,[0.,0,1,1,0]+y0, $
      color=out.ClrFill[i],_extra=extra
    IF out.Border NE 0 THEN $
    oplot,[0.D0,1,1,0,0]+x0,[0.,0,1,1,0]+y0, $
      color=out.ClrBorder,_extra=extra
    xyouts,x0+0.2,y0+0.2,out.symbols[i],color=out.ClrSymbol[i], $
     CharSize=!d.X_SIZE/18./20,font=out.font,_extra=extra
    IF out.ShowZ NE 0 THEN $
    xyouts,x0+0.1,y0+0.8,StrCompress(zi,/Rem), $
     color=out.ClrSymbol[i],CharSize=(!d.X_SIZE/18./20)*0.6, $
     font=out.font,_extra=extra
ENDFOR

;
; back to original setting 
;
out.x=!x
out.y=!y
out.p=!p
!x=x_old
!y=y_old
!p=p_old

ENDIF


END
