FUNCTION XOP_Execute, __command, track=__track, _extra=__extra, $
  verbose=__verbose, highLightId=__highLightId


;+
; NAME:
;	xop_execute
;
; PURPOSE:
;	This function is similar to IDL's execute(), but: 
;	i) It accepts a bunch of command to be executed sequentiuall
;	ii) It has the option to copy the created modified variables 
;	   to the IDL's $MAIN$ level. This is useful to store data 
;	   in sucessive calls to xop_execute()
;
; CALLING SEQUENCE:
;	itmp = Xop_Execute(inCode) 
;
; INPUTS:
;	inCode:	An array of strings with the code to be executed 
;	
; KEYWORD PARAMETERS:
;	TRACK = 0: The program does not retrieve variables from $MAIN$ 
;		before execution. It does not send the create variables
;		to $MAIN$. 
;		1: the created variables are copied to the $MAIN$ IDL's 
;		level.  Also, the variables at $MAIN$ are retrieved and 
;		made available to the code at running time. 
;
;	VERBOSE = set this keyword to print the processed code.
;	HighLightId = set this keyword to a text_widget Id with the 
;		commands to be executed in order to highlith the
;		commands during execution. Used by Xop_Macro
;	_Extra = Other keywords to be passed to Dialog_Message
;
; OUTPUTS:
;
;
; EXAMPLE1:
;	IDL> delvar,a
;	IDL> itmp = xop_execute('a=10',track=0)
;	IDL> itmp = xop_execute('help,a')
;	A               UNDEFINED = <Undefined>
;
; EXAMPLE2:
;	IDL> delvar,a
;	IDL> itmp = xop_execute('a=10',track=1)
;	IDL> itmp = xop_execute('help,a')
;	A               INT       =       10
;
; EXAMPLE3:
;	IDL> delvar,a
;	IDL> itmp = xop_execute(['a=10','help,a'],track=0)
;	A               INT       =       10
;
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio (srio@esrf.eu) 2007/10/10
;	
;	Modified: 
;
;-

catch, __error_status
if __error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   __itmp = Dialog_Message(_Extra=__extra, $
   /Error,'XOP_EXECUTE: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN,0
endif

IF N_Elements(__track) EQ 0 THEN __track=1
IF N_Elements(__verbose) EQ 0 THEN __verbose=1

;
; retrieve all variables from $MAIN$ level
;
IF __track EQ 1 THEN BEGIN
  __varsMain = scope_varname(Level=1,Count=__nn)
  IF __nn GT 0 THEN BEGIN
    FOR __i=0L,__nn-1 DO BEGIN
      __s = n_elements(Scope_VarFetch(__varsMain[__i], Level=1, /Enter))
      ;Print,'>>> Fetching variable ' + __varsMain[__i] + ' with n_elements ' + string(__s)
      if (__s gt 0) then begin
          __itmp = Scope_VarFetch(__varsMain[__i], Level=1)
          (Scope_VarFetch(__varsMain[__i], Level=0, /Enter)) = __itmp
      ;Print,'>>> Retrieved from $MAIN$: ',__varsMain[__i],' ',__itmp
      endif 
    ENDFOR
  ENDIF
ENDIF

FOR __i=0L,N_Elements(__command)-1 DO BEGIN

  IF Keyword_Set(__highLightId) THEN  BEGIN
    __ixy = Widget_Info(__highLightId,Text_XY_To_OffSet=[0,__i])
    Widget_Control,__highLightId,Set_Text_Select=[__ixy,0]
    Widget_Control,__highLightId,Set_Text_Select=[__ixy,StrLen(__command[__i])]
  ENDIF

  IF __verbose GT 1 THEN message,/info,' executing command: '+__command[__i]
  __tmp = execute(__command[__i])
  IF __verbose GT 0 THEN BEGIN
    IF __tmp NE 1 THEN BEGIN
     IF N_Elements(__command) GT 1 THEN BEGIN
        __itmp = Dialog_Message(/Question,['Error executing: ',__command[__i],$
           'Continue?'],/Cancel,_extra=__extra)
     ENDIF ELSE BEGIN
        __itmp = Dialog_Message(/Error,'Error executing: '+__command[__i],$
           _extra=__extra)
     ENDELSE
     IF __itmp EQ 'Cancel' THEN RETURN,__tmp
     IF __itmp EQ 'No' THEN GoTo,masCosas
    ENDIF
  ENDIF

ENDFOR
IF Keyword_Set(__highLightId) THEN Widget_Control,__highLightId, $
  Set_Text_Select=[0,0]

MasCosas:

IF __track EQ 1 THEN BEGIN
  ;
  ; Send all local variables (except __*) to $MAIN$
  ;
  __vars = Scope_VarName(Level=0,Count=__nn)
  IF __nn GT 0 THEN BEGIN
    __igood = where(strmid(__vars,0,2) NE '__')
    IF __igood[0] NE -1 THEN BEGIN 
      __vars = __vars[__igood]
      FOR __i=0L,N_Elements(__vars)-1 DO BEGIN
        ; this is to avoid errors with variables referenced but undefined
        __s = n_elements(Scope_VarFetch(__vars[__i], Level=0))
        if (__s gt 0) then begin
            __itmp = Scope_VarFetch(__vars[__i], Level=0)
            (Scope_VarFetch(__vars[__i], Level=1, /Enter)) = __itmp
            ;Print,'>>> Sent to $MAIN$: ',__vars[__i]
        endif 
      ENDFOR
    ENDIF
  ENDIF
ENDIF

RETURN,__tmp

END ; xop_execute
