
FUNCTION xy2rphi_axial,img,v0in,vSin, $
   useEllipses=useEllipses,  $
   twothetarange=twothetarange, $ 
   twothetastep=twothetastep, $
   phirange=phirange, $ 
   phistep=phistep, $
   verbose=verbose, axis=axis, $
   xx=xx, yy=yy  ; , twoThetaMat=twoThetaMat, phiMat=phiMat

;+
; NAME:
;
;	XY2RPHI_AXIAL
;
; PURPOSE:
;
;	this function calculates returns the polar image (i.e.,
;       intensity versus (radii,phi) ) from a cartesian 
;       (i.e., intensity versus (x,y) pixels) image.
;
; CATEGORY:
;
;       Imaging
;
; CALLING SEQUENCE:
;
;	out=xy2rphi_axial(image,v0,vS)
;
; INPUTS:
;
;    image:  an IDL variable with the image to process
;    v0:  [x0,y0] the value with the direct beam position [pixels]
;    vS:  [xS,yS,zS] the sample coordinates [in pixel units] 
;
;
; KEYWORDS: 
;
;    useEllipses: set this Flag to zero (for circles, default) and 1 
;         (to ellipses). If useEllipses=0, it sets vS[0:1]=v0[0:1]
;
;    TwoThetaRange: [min,max] The range for twoTheta (in deg)
;         If undefined, or min=max, the program computes defaul values.
; 
;    twoThetaStep: the step in twotheta [deg]. It has priority over NtwoTheta
;	
;    PHIRANGE: [phiMin,phiMax] The range for phi (in degrees)
;               If undefined, or phimin=phiMax the program  sets the 
;               defaults [0,359]
; 
;    PhiStep: the step in phi [deg]. It has priority over Nphi
;
;    VERBOSE: set this flag to print some working information (default=0)
;
;    AXIS: set this to a names variable to retrieve 
;             [twoThetaMin,twoThetaFactor,phiMin,phiFactor]
;
;
; OUTPUT KEYWORDS (named variables):
;
;   xx: The matrix with x coordinates (to be used in interpolation)
;   yy: The matrix with y coordinates (to be used in interpolation)
;
; ALGORITHM
;
;     start creating a grid in (2theta,phi), then use the Rodrigues' 
;     formula to convert to cartesian, and then use IDL's INTERPOLATE
;     with the original image to obtain the intensity values. 
;     
; MODIFICATION HISTORY:
;
;	by M. Sanchez del Rio  srio@esrf.eu 29 April 2008
;
;-	
catch, error_status
 if error_status ne 0 then begin
 message,/info,'error caught: '+!err_string
 if sdep(/w) then itmp = Dialog_Message(/Error,$
   'XY2RPHI_AXIAL: error caught: '+!err_string)
  catch, /cancel
  on_error,2
  RETURN,0
endif

out = 0

s = Size(img)

IF N_Elements(verbose) EQ 0 THEN verbose = 0
IF N_Elements(useEllipses) EQ 0 THEN useEllipses = 0


IF N_Elements(v0in) EQ 0 THEN v0 = s[1:2]/2.D0 ELSE v0=Double(v0in)
IF N_Elements(vSin) EQ 0 THEN vS = [v0,1000D0] ELSE vS=Double(vSin)

IF Not(useEllipses) THEN BEGIN
  vS[0:1]=v0[0:1]
ENDIF


IF N_Elements(twoThetaRange) NE 2 THEN BEGIN
     TwoThetaMin=0D0
     TwoThetaMax=40D0
ENDIF ELSE BEGIN
     TwoThetaMin=Double(twoThetaRange[0])
     TwoThetaMax=Double(twoThetaRange[1])
  IF twoThetaRange[0] EQ twoThetaRange[1] THEN BEGIN
     TwoThetaMin=0D0
     TwoThetaMax=40D0
  ENDIF 
ENDELSE

IF verbose THEN BEGIN
  print,'XY2RPHI_AXIAL: useEllipses : '+String(useEllipses)
  print,'XY2RPHI_AXIAL: v0 : '+vect2string(v0)
  print,'XY2RPHI_AXIAL: vS : '+vect2string(vS)
  print,'XY2RPHI_AXIAL: TwoTheta Extrema [deg]: '+vect2string([TwoThetaMin,TwoThetaMax])
ENDIF

IF N_Elements(twoThetaStep) EQ 0 THEN $
    twoThetaStep=0.1D0 ELSE twoThetaStep=Double(twoThetaStep)

IF N_Elements(phiStep) EQ 0 THEN $
    phiStep=1D0 ELSE phiStep=Double(phiStep)

;
; calculate twoTheta arrays
;

nTwoTheta = (twoThetaMax-twoThetaMin)/twoThetaStep + 1
twoTheta = MakeArray1(nTwoTheta,TwoThetaMin,TwoThetaMax) 
;
; remove last point
;
twoTheta = twoTheta[0:nTwoTheta-2]
ntwoTheta = Double(N_Elements(twoTheta))

; shift data to make calculation in the center of the future pixel
twoThetaShift = (twoTheta[1]-twoTheta[0])/2
twoTheta = twoTheta + twoThetaShift

IF verbose THEN print,'XY2RPHI_AXIAL: 2thetaMin: ',TwoThetaMin,twoTheta[0]
IF verbose THEN print,'XY2RPHI_AXIAL: 2thetaMax: ',TwoThetaMax,twoTheta[nTwoTheta-1]
IF verbose THEN print,'XY2RPHI_AXIAL: nTwoTheta: ',nTwoTheta
IF verbose THEN print,'XY2RPHI_AXIAL: twoThetaShift: ',twoThetaShift

;
; calculate phi arrays
;

IF N_Elements(phirange) NE 2 THEN BEGIN
  phiMin = 0.0D0
  phiMax = 360.0D0
ENDIF ELSE BEGIN
  phiMin = Double(phirange[0])
  phiMax = Double(phirange[1])
  IF phiMin EQ phiMax THEN BEGIN
    phiMin = 0.0D0
    phiMax = 360.0D0
  ENDIF
ENDELSE

nPhi = (phiMax-phiMin)/phiStep + 1
phi = MakeArray1(nphi,phiMin,phiMax)  
;
; remove last point
;
phi = phi[0:nphi-2]
nphi = Double(N_Elements(phi))

; shift data to make calculation in the center of the future pixel
phiShift = (phi[1]-phi[0])/2
phi = phi+phiShift

IF verbose THEN BEGIN
  IF verbose THEN print,'XY2RPHI_AXIAL: phiMin: ',phiMin,phi[0]
  IF verbose THEN print,'XY2RPHI_AXIAL: phiMax: ',phiMax,phi[nPhi-1]
  IF verbose THEN print,'XY2RPHI_AXIAL: nphi: ',nphi
  IF verbose THEN print,'XY2RPHI_AXIAL: phiShift: ',phiShift
ENDIF

xx = DblArr(N_Elements(twoTheta),n_elements(phi),/NoZero)
yy = xx

;
; beam direction: v1 = v0-vS
;

v1 = [v0,0]-vS

;
;
; THIS IS THE FULL LOOP OVER THE 
; (twoTheta,phi) MATRICES USING VECTOR OPERATION. 
; IT IS EXTREMELY SLOW, THE SECTION AFTER IT DOES THE SAME 
; THING, FROM THE PROGRAMMING POINT OF VIEW MUCH MORE 
; COMPLEX, BUT MUCH FASTER...
;
;twoThetaMat = (twoTheta+0.0)# Replicate(1.0,n_elements(phi))
;phiMat = Replicate(1.0,N_Elements(twoTheta)) # (phi+0.0)
;FOR ii=0L,N_Elements(twoThetaMat)-1 DO BEGIN
;  v2 = vecrotvec([0,1,0],v1,twoThetaMat[ii],/Deg)
;  v3 = vecrotvec(v1,v2,PhiMat[ii],/Deg)
;  ; intersection with CCD
;  lambda = -vS[2]/v3[2] 
;  xx[ii] = vS[0]+lambda*v3[0]
;  yy[ii] = vS[1]+lambda*v3[1]
;  ;;print,twoThetaMat[ii],PhiMat[ii],lambda,xx[ii],yy[ii]
;ENDFOR

;print,'Startling loop... v1: '+vect2string(v1)

;
; normalize v1
;
v1=v1/Sqrt(Total(v1^2,1))

FOR ii=0L,N_Elements(twoTheta)-1 DO BEGIN
  angle = (twoTheta[ii])*!dpi/180
  ca = cos(angle)
  sa = sin(angle)
  ; 
  ; a1 (rotation axis) is perpendicular to v1
  ;
  a1 = [0D0,-v1[2],v1[1]]
  a1=a1/Sqrt(Total(a1^2,1))

  v2 = v1*ca  + $
  [ a1[1]*v1[2]-a1[2]*v1[1],-(a1[0]*v1[2]-a1[2]*v1[0]),a1[0]*v1[1]-a1[1]*v1[0] ] $
  * sa + Total(a1*v1,1)*(1D0-ca)*a1

  ;; THE INNER LOOP HAS BEEN VECTORIZED!!!
  ;;FOR jj=0L,N_Elements(Phi)-1 DO BEGIN
    ;;angle = phi[jj]*!dpi/180
    angle = phi*!dpi/180
    ca = cos(angle)
    sa = sin(angle)

    fac1 = Total(v1*v2,1)*(1D0-ca)

    v3x =   v2[0]*ca  + $
      ( v1[1]*v2[2]-v1[2]*v2[1] ) $
    * sa + fac1*v1[0]
    
    v3y =   v2[1]*ca  + $
      ( -(v1[0]*v2[2]-v1[2]*v2[0]) ) $
    * sa + fac1*v1[1]

    v3z =   v2[2]*ca  + $
      ( v1[0]*v2[1]-v1[1]*v2[0] ) $
    * sa + fac1*v1[2]

    ;
    ; intersection with CCD
    ;

    lambda = -vS[2]/v3z
    xx[ii,*] = vS[0]+lambda*v3x
    yy[ii,*] = vS[1]+lambda*v3y

  ;;ENDFOR
ENDFOR

twoThetaFactor = ntwoTheta/(twoThetaMax-twoThetaMin)
phiFactor = nphi/(phiMax-phiMin)
axis = [twoThetamin,twoThetaFactor,phiMin,phiFactor]


;
; interpolate to obtain intensities (new pixel values)
;
t0 = systime(1)
IF verbose THEN print,'Start interpolation...'
;;out = INTERPOLATE( img,XX-0.5,YY-0.5,CUBIC=-0.5,MISSING=-3)
out = INTERPOLATE( img,XX,YY,CUBIC=-0.5,MISSING=-3)
IF verbose THEN print,'End interpolation...',sysTime(1)-t0

RETURN,out
END
