Function physical_constants,descriptor,double = use_double

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       PHYSICAL CONSTANTS
;
; PURPOSE:
;	Returns physical constants.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = physical_constants(id)
; INPUTS:
;	id: the constant  identifier. Valid identifiers (case insensitive) are:
;	C    (Speed of light [m/s])
;	ME   (Electron rest mass [kg]) 
;	MEE  (Electron rest mass [MeV])
;	EC   (Elementary charge [C])
;	H    (Planck's constant [Js])
;	HBAR (Planck's constant/2Pi [Js])
;	EPSZ (Permittivity of vacuum [Fm^-1])
;	AVOGADRO (Avogadro constant [mol^-1])
;
;	C_M_CM   (m -> cm)
;	C_M_A    (m -> Angstrom)
;	C_B_CM2  (barn -> cm^2)
;	C_EV_MEV (eV -> MeV)
;	C_MEV_EV (MeV -> eV)
;	HC       (Angstrom <-> eV) for particles with no mass (photons)
;				L(Angstrom)=HC/E(eV)
;	HCE	 (Angstrom <-> sqrt(eV) for electrons)
;				L(A)=HCE/sqrt(E(eV))
;	HCN	(Angstrom <-> sqrt(eV) for neutrons)
;				 L(A)=HCN/sqrt(E(eV))
;	HCP	( Angstrom <-> sqrt(eV) for protons)
;				 L(A)=HCP/sqrt(E(eV))
;
;	RE       (classical electron radius (cm) 2.8179409e-13 cm)
;	KB       (Boltzmann constant in J K^-1)
;	AM       (Atomic mass (i.e., 1.0/Avogadro in Kg)
;
;	MN   	(Neutron rest mass [kg])
;	MNE  	(Neutron rest mass [MeV])
;	MP   	(Proton rest mass [kg])
;	MPE  	(Proton rest mass [MeV])
;
;
; OUTPUT:
;	out: the value of the selected parameter
;
; KEYWORD VALUES:
;	DOUBLE: Set this keyword to return value in double precision
;		(default=yes)
;
; PROCEDURE:
;     Copied from asf.pro from Roger J. Dejus. Most of them are from
;     Fundamental physical constants; Physics Today Aug. 1990
;     These and many other physical constants can be found at the URL:
;     http://physics.nist.gov/PhysRefData/codata86/codata86.html
;
;     Updated from same web site but with 1998 codata dataset published
;     in:
;     Journal of Physical and Chemical Reference Data, Vol. 28, 
;     No. 6, pp. 1713-1852, 1999 
;     and in: 
;     Reviews of Modern Physics, Vol. 72, No. 2, pp. 351-495, 2000
;     http://physics.nist.gov/cuu/index.html
;
;     2006 updated CODATA values from:
;     http://physics.nist.gov/cuu/Constants/index.html
;
;
; EXAMPLES:
;	print,physical_constants('avogadro')
;
; MODIFICATION HISTORY:
;       96-09-23 Written by M. Sanchez del Rio (srio@esrf.fr)
;	01-11-13 Updated by srio and C. Boisseau (boisseau@esrf.fr)
;	02-10-09 srio@esrf.fr adds Boltzmann constant and atomic mass.
;	03-04-08 Updated by srio and C. Boisseau (boisseau@esrf.fr)
;		 Proton and neutron conversion. 
;	07-12-18 srio@esrf.eu updates with new 2006 CODATA values 
;		http://physics.nist.gov/cuu/Constants/index.html
;-
on_error,2

IF N_Elements(use_double) EQ 0 THEN use_double=1

; Fundamental physical constants; Physics Today Aug. 1990:
C    =2.99792458D8      ;; Speed of light [m/s]
; ME   =9.1093897D-31     ;; Electron rest mass [kg]
; ME   =9.109381887D-31     ;; Electron rest mass [kg]
  ME   =9.1093821545D-31     ;; Electron rest mass [kg]
; MEE  =0.51099906D0      ;; Electron rest mass [MeV]
;  MEE  =0.510998902D0      ;; Electron rest mass [MeV]
  MEE  =0.51099891013D0      ;; Electron rest mass [MeV]
; EC   =1.60217733D-19    ; Elementary charge [C]
; EC   =1.602176462D-19    ; Elementary charge [C]
  EC   =1.60217648740D-19    ; Elementary charge [C]
; H    =6.6260755D-34     ;; Planck's constant [Js]
; H    =6.62606876D-34     ;; Planck's constant [Js]
  H    =6.6260689633D-34     ;; Planck's constant [Js] 2006 value
; HBAR =1.05457266D-34    ;; Planck's constant/2Pi [Js]
; HBAR =1.054571596D-34    ;; Planck's constant/2Pi [Js]
  HBAR =1.05457162853D-34    ;; Planck's constant/2Pi [Js]
EPSZ =8.854187817D-12   ;; Permittivity of vacuum [Fm^-1]
;KB = 1.3806503D-23      ;; Boltzmann constant J K-1 
 KB = 1.380650424D-23      ;; Boltzmann constant J K-1 

; neutrons and protons
;MN   =1.67492716D-27     ;; Neutron rest mass [kg]
 MN   =1.67492721184D-27     ;; Neutron rest mass [kg]
;MNE  =939.565330D0      ;; Neutron rest mass [MeV]
 MNE  =939.56534623D0      ;; Neutron rest mass [MeV]
;MP   =1.67262158D-27     ;; Proton rest mass [kg]
 MP   =1.67262163783D-27     ;; Proton rest mass [kg]
;MPE  =938.271998D0      ;; Proton rest mass [MeV]
 MPE  =938.27201323D0      ;; Proton rest mass [MeV]

; Physicochemical constants; Physics Today Aug. 1990:
; AVOGADRO = 6.0221367D23 ; Avogadro constant [mol^-1]
; AVOGADRO = 6.02214199D23 ; Avogadro constant [mol^-1]
  AVOGADRO = 6.0221417930D23 ; Avogadro constant [mol^-1]

AM = 1D-3/AVOGADRO ;; atomic mass
; conversion factors
C_M_CM   = 1.0D+2       ; m -> cm
C_M_A    = 1.0D+10      ; m -> Angstrom
C_B_CM2  = 1.0D-24      ; barn -> cm^2
C_EV_MEV = 1.0D-6       ; eV -> MeV
C_MEV_EV = 1.0D+6       ; MeV -> eV
HC       = H*C/EC*C_M_A ; conversion Angstrom <-> eV for particles
                        ; without mass => L(Angstrom)=HC/E(eV)
HCE      = H*C_M_A/sqrt(2.d0*ME*EC) ; conversion Angstrom <-> sqrt(eV)
                        ; for electrons. => L(A)=HCE/sqrt(E(eV))
HCN      = H*C_M_A/sqrt(2.d0*MN*EC) ; conversion Angstrom <-> sqrt(eV)
                        ; for neutrons. => L(A)=HCE/sqrt(E(eV))
HCP      = H*C_M_A/sqrt(2.d0*MP*EC) ; conversion Angstrom <-> sqrt(eV)
                        ; for protons. => L(A)=HCE/sqrt(E(eV))


; derived constants
RE  = EC*1.0D-6/(4.0D0*!DPI*EPSZ*MEE)*C_M_CM ; classical electron radius (cm)
                                             ; = 2.81794092e-13 cm

out = 0.0D0
itmp = execute('out = '+descriptor)
if use_double EQ 0  then out = float(out)
return,out
end
