PRO CRL_Plot1,out,source,Group=group, Distance=distance
;
; Makes a 1-dimensional histogram with CRL results
;
;
; Modification history:
; 2007-06-25 srio@esrf.eu uses histo_nd2 instead of histogramw2
;

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL_PLOT1: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;
; trace the source in vacuum (no CRL)
;
IF N_Elements(distance) EQ 0 THEN distance = 0.0
z1=source.z+source.zp*distance

;
; calculate histo limits
;
xmin=min(out.z,max=xmax)

;
; create structures to pass data into XWindow
;
histdata={z:out.z,z1:z1,int:out.int,int1:source.int}
parms = {xmin:xmin, xmax:xmax, ymin:0.0, ymax:0.0, nbins:201, charsize:1.0, $
   over:['1','No','Yes'],Title:'',diskfile:['0','No','Yes']} 
titles=['X min [microns]','X max [microns]','Y min (default=0)','Y max (default=0)', $
  'Number of bins', 'plot character size','Overplot non-focused beam?',$
  'Top title','write results to a file?']

;
; calculate total intensities
;
intTotInp = total(source.int)
intTot = total(out.Int)
intNorm = intTot/n_elements(out.Int)

;
; create XWindow's buffer and call XWindow.
;
buffer = ["; CRL results. It is recommended NOT to change this buffer unless"+$
  " you are an advanced user.  Use 'Edit Parameters' button instead.",$
 "a0=(parms.xmax-parms.xmin)/parms.nbins",$
 "c1 = histogramw(data.z,data.int,min=parms.xmin,max=parms.xmax, binsize=a0)",$
 "c2 = histogramw(data.z1,data.int1,min=parms.xmin,max=parms.xmax, binsize=a0)",$
 "c0=(FIndGen(n_elements(c1(*,0)))*(parms.xmax-parms.xmin)/parms.nbins+parms.xmin)",$
 "Plot, c0, c1, XTitle='x [!7l!Xm]',YTitle='Intensity [a.u.]',CharSize=parms.charsize, "+$
   "Thick=2,YRange=[0.0,parms.ymax],PSym=10,Title=parms.title",$
 "IF StrCompress(parms.over[0],/Rem) EQ '1' THEN OPlot, c0, c2, PSym=10", $
 "b1 = GetFWHM(Make_Set(c0,c1))",$
 "b2 = GetFWHM(Make_Set(c0,c2))",$
 "Legend1,PSym=[3,3,3,3,3],['Focused FWHM: '+StrCompress(b1),'Unfocused FWHM: '"+$
	"+StrCompress(b2),'Source intensity: "+StrCompress(intTotInp,/Rem)+$
	"','Transmitted intensity: "+StrCompress(intTot,/Rem)+$
	"','Normalized intensity: "+StrCompress(intNorm,/Rem)+"']",$
 "IF Fix(parms.diskfile[0]) EQ 1 THEN c3=Make_Set(c0,c2,c1,ToFile='?',File='crl.dat')" ]

xwindow,buffer=buffer,data=histdata,parms=parms,/edit, $
  /Interp,Titles=titles, /NoType, Wtitle='CRL histogram',NCol=2, $
  /ZoomFlag

END ; CRL_Plot1

;
;=================================================================
;
PRO CRL_Plot2,out,Group=group
;
; Makes a 2-dimensional histogram with CRL results
;

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL_PLOT2: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;
; calculate histo limits
;
xmax = 0 & ymax = 0
xmin=min(out.z,max=xmax)
ymin=min(out.zp,max=ymax)
; Print, '>>>> ', xmax,xmin,ymax,ymin,' <<<<'

; 
; define structures to pass into XWindow
;
histdata={z:out.z,zp:out.zp,int:out.int}
parms = {xmin:xmin, xmax:xmax, ymin:ymin, ymax:ymax, nbins:51, charsize:1.0, $
   Title:'',kind:['0','Contour','Surface'], nlevels:10, $
   fill:['0','No','Yes'], az:30} 
titles=['X min [microns]','X max [microns]','Y min [microrad]', $
   'Y max [microrad]', 'Number of bins', $
  'plot character size','Top title','Kind of plot',$
  'Contour lines','Contour filling','Rotation angle [deg]']
Flags=[replicate('1',8),'w(7) EQ 0','w(7) EQ 0','w(7) EQ 1']

;
; create XWindow's buffer and call XWindow.
;

;data=histdata
;parms=parms

;c2 = hist_nd2(Make_Set(Reform(data.z),Reform(data.zp)),wei=data.int,min=[parms.xmin,parms.ymin],max=[parms.xmax,parms.ymax],nbins=parms.nbins,location=loc )
;xx = loc[0:parms.nbins-1] 
;yy = loc[parms.nbins:2*parms.nbins-1]
;contour,c2,xx,yy


buffer = ["; CRL results. It is recommended NOT to change this buffer unless"+$
  " you are an advanced user.  Use 'Edit Parameters' button instead.",$
 "c2 = hist_nd2(Make_Set(Reform(data.z),Reform(data.zp)),wei=data.int,min=[parms.xmin,parms.ymin],max=[parms.xmax,parms.ymax],nbins=parms.nbins,location=loc)",$
 "xx = loc[0:parms.nbins-1]",$
 "yy = loc[parms.nbins:2*parms.nbins-1]",$
 "IF Fix(parms.kind[0]) EQ 0 THEN Contour, c2, xx, yy, XTitle='x [!7l!Xm] ', YTitle='xp [!7l!Xrad]', CharSize=parms.charsize, NLevels=parms.nlevels, Fill=Fix(parms.fill[0])",$
 "IF Fix(parms.kind[0]) EQ 1 THEN Surface, c2, xx, yy, XTitle='x [!7l!Xm] ', YTitle='xp [!7l!Xrad]', CharSize=parms.charsize, /Lego, ZTitle='Intensity [a.u.]', AZ=parms.az" ]

xwindow,buffer=buffer,data=histdata,parms=parms,/edit, $
  /Interp,Titles=titles, /NoType, Wtitle='CRL phase-space',NCol=2, $
  Flags=flags, /ZoomFlag

END ; CRL_Plot2
;
;=================================================================
;

function crl_ray_source, sigmas,nrays=nrays,seed=seed,method=method, $
  Group=group
;
; Calculates a gaussian source for ray tracing
;

catch, error_status
if error_status ne 0 then begin
   message,/info,'error catched: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,Dialog_Parent=group,$
	'CRL_RAY_SOURCE: error catched: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

if n_params() EQ 0 then begin
  sigma = 1.0
  sigmapr = 1.0
endif else begin
  sigma = sigmas[0]
  sigmapr = sigmas[1]
endelse

if not(keyword_set(seed)) then Seed=10822541L else Seed=seed
if not(keyword_set(nrays)) then nrays=10000L
if not(keyword_set(method)) then method=0

case method of
   0:begin
     ; source=(2*RANDOMU(Seed,Nrays,2)-1)
     source=(2*RANDOMU1(Seed,Nrays,2)-1)

     z=source(*,0)*3*sigma*1000 ; X (source)
     zp=source(*,1)*3*sigmapr*1000000 ; X' (source)
     int=exp(-((z/sigma/1000)^2+(zp/sigmapr/1000000)^2)/2)
     end
   1:begin
     ; source=(RANDOMN(Seed,Nrays,2))
     source=(RANDOMN1(Seed,Nrays,2))

     z=source(*,0)*sigma*1000 ; X (source)
     zp=source(*,1)*sigmapr*1000000 ; X' (source)
     int=FltArr(nrays)
     int=int+1
     end
endcase

return,{z:z,zp:zp,int:int}
end ; crl_ray_source
;
;=================================================================
;
;+
; NAME:
;	crl_ray_parabol
;
; PURPOSE:
;       This function calculate the final position (in the phase space)
;       and the attenuation of an incoming ray, for a CRL with parabolic
;       holes of specified dimensions
;
; CATEGORY:
;	optics
;
; CALLING SEQUENCE:
;       Result = parabol(X, Xprime, Intens, Aperture, Delta, Nholes, $
;		Rholed, d, Xmax, Slendist, Lfindist )
;
; INPUTS:
;       X, Xprime: coordinates in phase space of ray at source (um and urad)
;       Intens: relative intensity at coordinate X, Xprime (arb. unit)
;       Aperture: aperture of lens (mm) Rif. ESRF/MACH ID 97/33
;       Delta: coefficient delta of refr. index (n = 1 - delta - i * beta)
;       Nholes, Rholes: Number of holes and radius (mm) of curvature of lens
;       d: minimum distance between holes (mm)
;       Xmax: half of max lateral dimension of hole (mm)
;       Slendist, Lfindist: distances source-lens, lens-final plane (plane
;       of interest) in meters
;
; KEYWORD PARAMETERS:
;	Group: the id of the called widget (to manage error window).
;
; OUTPUTS:
;       This function return the coordinates (X1, X1') in the 
;       phase space and the relative intensity of the ray at the final plane 
;       (plane of interest), in form of a 3-tags structure
;
; PROCEDURE:
;       
;       Calculate the deflection of the ray and the transversal displacement 
;       due to drift-spaces
;       Reference. P.Elleaume - - NIM A412 (1998) pp 483-506 (formula 9)
;
;
; EXAMPLE:
;       out=ray_calc(45,12,.5,.3,5.9e-7,51,.4,.1,1,24,8.2)
;       (out is the position in phase space and the relative intensity of the
;        ray at the exit of the lens (at a distance 8.2 meters from it))
;
; MODIFICATION HISTORY:
; 	Written by:	G. Apostolo (apostolo@esrf.fr), 98-3-4
;	98/12/22 srio@esrf.fr adapts to XOP2.0. Vectorize the
;		calculation loop. Renamed from crl_Parabol.
;
;-


FUNCTION crl_ray_parabol, X, Xprime, Intens, Aperture, Delta, Nholes, Rholes, $
  d, Xmax, Slendist, Lfindist , Group=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL_RAY_PARABOL: error caught (return 0): '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN, 0
ENDIF

IF (N_Params() NE 11) THEN Message, 'Number of parameters wrong'

;out=DblArr(3)
nn=n_elements(x)
out = { z:dblarr(nn), zp:dblarr(nn), int:dblarr(nn) }

Ymax=(Xmax^2)/(2*Rholes)
xa=Xprime*Slendist+X

;  ;=================
;  
;  xa=Xprime*Slendist+X
;  
;  ;IF ((xa GE Rholes*1000) OR (xa LE -Rholes*1000)) THEN $
;  ;   dx=0 ELSE dx=2*Nholes*Delta*tan(asin(xa/(1000*Rholes)))
;  ;dx=2*Nholes*Delta*tan(asin(xa/(1000*Rholes)))
;  dx=tan(asin(xa/(1000*Rholes)))*(2*Nholes*Delta(0))
;  
;  indices = where(((xa GE Rholes*1000) OR (xa LE -Rholes*1000)),count)
;  if count GT 0 then dx[indices]=0
;  
;  out.zp=Xprime-dx*1e6 ; final angle
;  out.z=xa+out.zp*Lfindist  ; final displacement
;  ;IF ((xa GE Rholes*1000) OR (xa LE -Rholes*1000)) THEN $
;  ; out.int=Intens*exp(-(d+2*Rholes)*Rholes/(2*Aperture^2)) ELSE BEGIN
;  ; out.int=Intens*exp(-(Rholes/Aperture)^2*(1-cos(asin(xa*.001/Rholes)))-$
;  ; Rholes*d/(2*Aperture^2))
;  
;  Print, aperture
;  out.int=Intens*exp(-(Rholes/Aperture[0])^2*(1-cos(asin(xa*.001/Rholes)))-$
;    Rholes*d/(2*(Aperture[0])^2))
;  
;  if count GT 0 then out.int[indices]= $
;      Intens[indices]*exp(-(d+2*Rholes)*Rholes/(2*Aperture[0]^2)) 
;  
;  Return, out
;  
;  END
;  ;=================


;IF ((xa GE Xmax*1000) OR (xa LE -Xmax*1000)) THEN dx=0 $
;   ELSE dx=2*Nholes*Delta*xa/(1000*Rholes)

dx=xa * ( 2*Nholes*Delta(0)/(1000*Rholes) )

indices = where(((xa GE xmax*1000) OR (xa LE -xmax*1000)),count)
if count GT 0 then dx[indices]=0

out.zp=Xprime-dx*1e6 ; final angle
out.z=xa+out.zp*Lfindist  ; final displacement
;IF ((xa GE Xmax*1000) OR (xa LE -Xmax*1000)) THEN $
;  out.int=Intens*exp(-(d+2*Ymax)*Rholes/(2*Aperture^2)) ELSE BEGIN
;   out.int=Intens*exp(-((xa*.001)^2)/(2*Aperture^2)-Rholes*d/(2*Aperture^2))
;ENDELSE

; formula 6, Elleaume's paper
out.int=Intens*exp(-((xa*.001)^2)/(2*Aperture[0]^2)-Rholes*d/(2*Aperture[0]^2))

if count GT 0 then out.int[indices]= $
    Intens[indices]*exp(-(d+2*Ymax)*Rholes/(2*Aperture[0]^2)) 
Return, out

END ; CRL_Ray_Parabol

;
;=================================================================
;
;+
; NAME:
;	crl_ray_cyl
;
; PURPOSE:
;       This function calculate the final position (in the phase space)
;       and the attenuation of an incoming ray, for a CRL with cylindrical
;       holes of specified dimensions
;
; CATEGORY:
;	Optics
;
; CALLING SEQUENCE:
;       Result = crl_ray_cyl(X, Xprime, Intens, Aperture, Delta, Nholes, $
;		Rholed, d, Slendist, Lfindist )
;
; INPUTS:
;       X, Xprime: coordinates in phase space of ray at source (um and urad)
;       Intens: relative intensity at coordinate X, Xprime (arb. unit)
;       Aperture: aperture of lens (mm) Rif. ESRF/MACH ID 97/33
;       Delta: coefficient delta of refr. index (n = 1 - delta - i * beta)
;       Nholes, Rholes: Number and radius (mm) of lens holes
;       d: minimum distance between holes (mm)
;       Slendist, Lfindist: distances source-lens, lens-final plane (plane
;       of interest) in meters
;
; KEYWORD PARAMETERS:
;	Group: the id of the called widget (to manage error window).
;
; OUTPUTS:
;       This function return the coordinates (X1, X1') in the 
;       phase space and the relative intensity of the ray at the final plane 
;       (plane of interest), in form of a 3-tags structure ({x,xp,int})
;
; PROCEDURE:
;       
;       Calculate the deflection of the ray and the transversal displacement 
;       due to drift-spaces
;       Reference: P.Elleaume - NIM A412 (1998) pp 483-506 (formula 32)
;
; EXAMPLE:
;       EXAMPLE1: 
;       out=ray_cyl(45,12,.5,.3,5.9e-7,51,.4,.1,24,8.2)
;       (out is the position in phase space and the relative intensity of the
;        ray at the exit of the lens (at a distance 8.2 meters from it))
;
; MODIFICATION HISTORY:
; 	Written by:	G. Apostolo (apostolo@esrf.fr), 98-3-4
;	98/12/22 srio@esrf.fr adapts for XOP2.0. Renamed from crl_ray_calc.
;
;
;-

FUNCTION crl_ray_cyl, X, Xprime, Intens, Aperture, Delta, Nholes, Rholes, $
  d, Slendist, Lfindist, Group=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL_RAY_CYL: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

IF (N_Params() NE 10) THEN Message, 'Number of parameters wrong'

;out=DblArr(3)

nn = n_elements(x)
out={z:dblarr(nn),zp:dblarr(nn),int:dblarr(nn)}

xa=Xprime*Slendist+X

;IF ((xa GE Rholes*1000) OR (xa LE -Rholes*1000)) THEN $
;   dx=0 ELSE dx=2*Nholes*Delta*tan(asin(xa/(1000*Rholes)))
;dx=2*Nholes*Delta*tan(asin(xa/(1000*Rholes)))
dx=tan(asin(xa/(1000*Rholes)))*(2*Nholes*Delta(0))

indices = where(((xa GE Rholes*1000) OR (xa LE -Rholes*1000)),count)
if count GT 0 then dx[indices]=0

out.zp=Xprime-dx*1e6 ; final angle
out.z=xa+out.zp*Lfindist  ; final displacement
;IF ((xa GE Rholes*1000) OR (xa LE -Rholes*1000)) THEN $
; out.int=Intens*exp(-(d+2*Rholes)*Rholes/(2*Aperture^2)) ELSE BEGIN
; out.int=Intens*exp(-(Rholes/Aperture)^2*(1-cos(asin(xa*.001/Rholes)))-$
; Rholes*d/(2*Aperture^2))

; formula 32 Elleaume's paper
out.int=Intens*exp(-(Rholes/Aperture[0])^2*(1-cos(asin(xa*.001/Rholes)))-$
  Rholes*d/(2*(Aperture[0])^2))

if count GT 0 then out.int[indices]= $
    Intens[indices]*exp(-(d+2*Rholes)*Rholes/(2*Aperture[0]^2)) 

Return, out

END ; CRL_Ray_Cyl

;
;=================================================================
;
;+
;
; NAME:
;	crl_ray_tracing
;
; PURPOSE:
;       This function ray-traces a source trough a Compound
;	Refractive Lense (either cylindrical or parabolic).
;
; CATEGORY:
;	Optics
;
; CALLING SEQUENCE:
;       Result = crl_ray_tracing(source, Delta, Mu, $
;	         Nholes, Rholed, d, Xmax, Slendist, Lfindist)
;
; INPUTS:
;	source: a structure with the source (as created by CRL_Tay_Source)
;	Delta: The delta component of the refraction index of the CRL material.
;	Mu; attenuation coeficient od the CRL material.
;
;       Nholes: Number of lens holes
;       Rholes: Radius (mm) of lens holes
;       d: minimum distance between holes (mm)
;       Xmax: half of max lateral dimension of hole (mm) (only for parabolic
;             holes, Cil = 0)
;       Slendist: distance source-lens in meters
;       Lfindist: distances lens-final plane (plane of interest) in meters
;
; KEYWORD PARAMETERS:
;       Cil: flag (1 = cylindrical holes (default), 0 = parabolic holes)
;
;
; OUTPUTS:
;	a structure of same type as source with the image {z,zp,int}
;
; PROCEDURE:
;       
;       Calculate the parameters of the lens (Focal distance, Aperture),
;       generate Nrays rays and calculate the deviation of each ray through
;       the lens
;
; EXAMPLE:
;       EXAMPLE1: 
;       space=crl_ray_tracing(.039,2.5,24,1.6,'Be',1.85,51,.4,.1,1,0,0,24,8.2,$
;		100000,Inttot,1,1)
;       (generate the flux versus the lateral displacement for a lens in
;        Beryllium (51 cylindricalholes of radius 0.4 mm) at a distance 
;	 8.2 meters from the lens)
;
; MODIFICATION HISTORY:
; 	Written by:	G. Apostolo (apostolo@esrf.fr), 98-3-4
;	98/05/18 srio@esrf.fr adapts for XOP.
;	98/12/22 srio@esrf.fr adapts for XOP2.0. Renamed from crl_ray_stat1.
;
;
;-

FUNCTION crl_ray_tracing, source, Delta, Mu, $
  Nholes, Rholes, d, Xmax, Slendist, Lfindist, CIL=cil, Group=group

catch, error_status
if error_status ne 0 then begin
   message,/info,'error catched: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	Dialog_Parent=group, $
	'CRL_RAY_TRACING: error catched: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

if n_params() NE 9 then message,'Wrong number of inputs'

if n_elements(cil) EQ 0 then cil=1 

F=Rholes/(Nholes*2.0*Delta*1000)
Aperture=((Delta*F/100/mu)^.5)*1000

; Print,'rholes [mm] : ',rholes
; Print,'delta : ',delta
; Print,'mu : ',mu
; Print,'F: ',F
; Print,'Lens aperture: ',aperture

out=DblArr(3)
ist=DblArr(201,2)
out1=DblArr(201,201)

i=source.z ; source(*,0)*3*sigma*1000 ; X (source)
j=source.zp ; source(*,1)*3*sigmapr*1000000 ; X' (source)
Int=source.int ; exp(-((i/sigma/1000)^2+(j/sigmapr/1000000)^2)/2) or 1 (gauss random generation)

IF (Cil EQ 1) THEN $
     out=crl_ray_cyl(i, j, Int, Aperture, Delta, Nholes, Rholes, d, $
         Slendist, Lfindist) ELSE $
     out=crl_ray_parabol(i, j, Int, Aperture, Delta, Nholes, Rholes, d, Xmax, $
         Slendist, Lfindist)

; for i1=0,5 do print,'<><>',out.z(i1),out.zp(i1),out.int(i1)

return,out

END ; CRL_Ray_Tracing

;
;=================================================================
;
Pro crl_calc_rt,ptr,Group=group
;+
; NAME:
;	crl_calc_rt
;
; PURPOSE:
;       main interface for performing ray tracing with CRLs
;
; CATEGORY:
;	Optics
;
; CALLING SEQUENCE:
;       crl_calc_rt,pointer
;
; INPUTS:
;       pointer: a pointer to a structure containg the CRL parameters
;		(as obtained from Dabax_Defaults('crl') )
;
; KEYWORD PARAMETERS:
;	Group: the id of the called widget (to manage error window).
;
; OUTPUTS:
;       performs the computation and creates the graph windows.
;
; PROCEDURE:
;	1) Use dabax to get the refraction index and attenuation coefficient
;	2) Create the gaussian source using CRL_Ray_Source()
;	3) Trace the source trough the CRL using CRL_Ray_Tracing, which
;	   calls either CRL_Ray_Cyl (for cylindrical lenses) or 
;	   CRL)Ray_Parabol (for parabolic lenses) 
;	4) Calls CRL_Plot1 and/or CRL_Plot2 for creating the graphs
;	   (image profile and image phase-space, respectively).
;
;       Reference: P.Elleaume - NIM A412 (1998) pp 483-506 
;	 (formula 9 for cylindrical lenses and formula 32 for parabolic ones)
;
; EXAMPLE:
;       CRL_Calc_RT,Ptr_New(dabax_defaults('crl'))
;
; MODIFICATION HISTORY:
;
; 	Written by:	M. Sanchez del Rio (srio@esrf.fr), 98-12-22
;
;
;-
;
; calculate pseudo-ray-tracing
;
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL_CALC_RT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

str_par = (*ptr).parameters

;get f1f2 file
f1f2 = str_par.f1f2(fix( str_par.f1f2[0])+1)
;get Mu file
CS = str_par.CS(fix( str_par.CS[0])+1)
;get Energy
energy=str_par.e

;
; get descriptor
;

; mat_flag : 0=element, 1=mixture(form) 2=mixture(table)
mat_flag = fix( str_par.mat_flag[0] )
case mat_flag of
0: begin
    iname = 0 
           descriptor = strcompress(str_par.descriptor,/rem)
           if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
    imix = 0
   end
1: begin
    iname = 0 
           descriptor = strcompress(str_par.descriptor,/rem)
           if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
    density = str_par.density
    imix = 1
   end
2: begin
    iname = 1 & descriptor = str_par.mat_list($
	fix(str_par.mat_list(0))+1)
    density = 0
    imix = 1
   end
endcase

;
; calculate delta and Mu
;
if imix then begin
  delta = f1f2_calc_mix(f1f2,descriptor,energy,f=3,name=iname,density=density,$
  group=group) 
  betaf = f1f2_calc_mix(f1f2,descriptor,energy,f=4,name=iname,density=density,$
  group=group) 
  Mu = cross_calc_mix(CS,descriptor,energy,partial='TotalCrossSection',$
  name=iname,density=density,group=group,unit=3) 
endif else begin
  delta = f1f2_calc(f1f2,descriptor,energy,f=3,group=group)
  betaf = f1f2_calc(f1f2,descriptor,energy,f=4,group=group)
  Mu = cross_calc(CS,descriptor,energy,partial='TotalCrossSection',$
  group=group,unit=3) 
endelse
delta=delta(0)
betaf=betaf(0)
mu=mu(0)

;
; create the rays of the source
;
; print,'*************************************************************'
sigmax = (*ptr).parameters.sourceSize*1e-3
sigmaxp = (*ptr).parameters.sourceDiv*1e-6
method = Fix( (*ptr).parameters.calc_method[0] )
nRays = (*ptr).parameters.nRays
seed = (*ptr).parameters.seed
; print,'Running CRL_RAY_SOURCE with inputs: '
; print,'   sigmax [mm]: ',sigmax
; print,'   sigmaxp [mm]: ',sigmaxp
; print,'   method: ',method
; print,'   nrays: ',nrays
; print,'   seed: ',seed
source = crl_ray_source([sigmax,sigmaxp],method=method, $
  NRays=nRays, Seed=seed, Group=group)
; help,/str,source
; print,'*************************************************************'

;
; doing the ray tracing
; (uses internally crl_ray_tracing wich calls either crl_rat_cyl or
;  crl_ray_parab)
;
; print,'*************************************************************'
; print,'Starting ray tracing with: '
; help,/str,source
; print,'   Delta: ',Delta
; help,mu
; print,'   mu: ',mu
nholes = (*ptr).parameters.n
slendist = (*ptr).parameters.p
lfindist = (*ptr).parameters.q
d = (*ptr).parameters.dholes * 1e-3 ; from microns to mm
rholes = (*ptr).parameters.r
xmax = (*ptr).parameters.parab_aperture
tmp = Fix( (*ptr).parameters.cyl_flag[0] )
IF tmp EQ 0 THEN cil=1 ELSE cil=0 ; cil=1 cylindrical, cil=o parabolic
; print,'   nholes: ',nholes
; print,'   rholes: ',rholes
; print,'   d: ',d
; print,'   xmax: ',xmax
; print,'   slendist: ',slendist
; print,'   lfindist: ',lfindist
; print,'   cil: ',cil
; help,slendist
out=crl_ray_tracing( source, Delta, Mu, $
   nholes , rholes , d , xmax , slendist , lfindist , CIL=cil)
; help,slendist
; help,/str,out
; print,'*************************************************************'

;
; histogramming
;
; print,'*************************************************************'
CASE Fix( (*ptr).parameters.rt_show[0] ) OF
  0: crl_plot1,out,source,Group=group,Distance=slendist+lfindist
  1: crl_plot2,out,Group=group
  2: crl_plot2,source,Group=group
  3: BEGIN
	crl_plot1,out,source,Group=group,Distance=slendist+lfindist
	crl_plot2,out,Group=group
	END
ENDCASE

END ; crl_calc_rt
