Function bragg_inp,inp1,ask=ask

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       BRAGG_INP
;
; PURPOSE:
;	Returns the structure with inputs for bragg_calc.
;	
; CATEGORY:
;       X-Ray optics.
;
; CALLING SEQUENCE:
;	out = bragg_inp([inp])
; OPTIONAL INPUTS:
;	inp = An existing input structure.
;	
; KEYWORDS:
;	ask = three possible values:
;		0 (default). Returns the default structure (only calls
;			dabax_defaults('bragg'))
;		1 Ask for the new parameters by using command lines.
;			(To be used with non-X applications).
;		2 Ask for the new parameters by using widgets.
;
; OUTPUT:
;	out: a structure with the output data. This structure is the
;		input bragg_calc and can be edited with bragg_inp.
;
; PROCEDURE:
;	It calls dabax_defaults('bragg') which uses the DABAX files:
;	  CrystalStructures.dat: to get the crystal tyle list
;	  f1f2_*.dat: to get the f1f2 DABAX file list.
;	  f0*.dat: to get the f0 DABAX file list.
;
; EXAMPLE:
;	inp = bragg_inp(ask=2)
;
; SEE ALSO:
;	bragg_calc, bragg_out, dabax_defaults.
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;	96-12-03 srio@esrf.fr: small changes in input structure.
;-
on_error,2
;
; ask = 0 (default) ; It does not ask, just return defaults
; ask = 1 It asks (command lines)
; ask = 2 It asks (widget)
;
; definition section
;
if not(keyword_set(ask)) then ask=0

if n_params() EQ 0 then inp = dabax_defaults('bragg') else inp=inp1

  filef0 = inp.parameters.filef0(1:n_elements(inp.parameters.filef0)-1)
  ifilef0 = inp.parameters.filef0(0) + 1
  filef1f2 = inp.parameters.filef1f2(1:n_elements(inp.parameters.filef1f2)-1)
  ifilef1f2 = inp.parameters.filef1f2(0) + 1

  ilattice = inp.parameters.ilattice(0) + 1
  xtals = inp.parameters.ilattice(1:n_elements(inp.parameters.ilattice)-1)
  hmiller=inp.parameters.hmiller 
   kmiller=inp.parameters.kmiller 
    lmiller=inp.parameters.lmiller
  I_ABSORP = inp.parameters.i_absorp(0)
  TEMPER = inp.parameters.temper
  EMIN = inp.parameters.emin 
   EMAX = inp.parameters.emax 
    ESTEP = inp.parameters.estep
  OUTFIL = inp.parameters.outfil

;
; input section
;
if ask EQ 1 then begin
  print,'Please selected a file for f0: '
  for i=0,n_elements(filef0)-1 do print,'   ',i+1,': ',filef0(i)
  read,'Give me f0-file index: ',ifilef0

  print,'Please selected a file for f1f2: '
  for i=0,n_elements(filef1f2)-1 do print,'   ',i+1,': ',filef1f2(i)
  read,'Give me f1f2-file index: ',ifilef1f2

  print,'Please selected a crystal: '
  for i=0,n_elements(xtals)-1 do print,'   ',i+1,': ',xtals(i)
  read,'Give me crystal index: ',ilattice

  read,' Miller index h: ',hmiller
  read,' Miller index k: ',kmiller
  read,' Miller index l: ',lmiller

  read,'Do you want to include crystal absorption [1/0] ? ',I_ABSORP
  read,'Temperature (Debye-Waller) factor ? ',TEMPER
  print, '**********************************************'
  read,'minimum photon energy (eV)  ? ',EMIN
  read,'maximum photon energy (eV)  ? ',EMAX
  read,'energy step (eV)            ? ',ESTEP
  read,'Output file name            : ',OUTFIL
endif


inp.parameters.filef0=[strcompress(ifilef0-1,/rem),filef0]
inp.parameters.filef1f2=[strcompress(ifilef1f2-1,/rem),filef1f2]
inp.parameters.ilattice=[strcompress(ilattice-1,/rem),xtals]
inp.parameters.hmiller=hmiller
inp.parameters.kmiller=kmiller
inp.parameters.lmiller=lmiller
inp.parameters.I_ABSORP(0)=strcompress(I_ABSORP,/rem)
inp.parameters.TEMPER=TEMPER
inp.parameters.EMIN=EMIN
inp.parameters.EMAX=EMAX
inp.parameters.ESTEP=ESTEP
inp.parameters.OUTFIL=OUTFIL


if ask EQ 2 then begin
  tmp = inp.parameters
  XScrMenu,tmp,/Interp,/NoType,action=action,Ncol=4,titles=inp.titles, $
	flags=inp.flags,wtitle='Bragg inputs'
  if  action EQ 'DO' then inp.parameters = tmp
endif

return,inp

end


