
;+
; 
; NAME: 
;	F3EQUATION__DEFINE
;
; PURPOSE:
;	Object class and methods to handle a function made by adding several 
;	components (or lines). The components are:
;	  IF N_Elements(equation) NE 0 THEN type1='UserDefined'
;		Polynomials: (StraightLine and polynomial)
;		Peaks (Gaussian, Lorentzian and PseudoVoigt)
;		Steps (StepAtan, StepErrorF)
;		UserDefined
;
; CATEGORY:
;	Object tools
;
; CALLING SEQUENCE:
;	o1 = Obj_New('f3equation',input)
;	o1-><method>
;
; F=Function, P=Procedure
;
; METHODS:
;	INIT(F)	called when the object is created with o1=Obj_New('f3equation',input)
;		OPTIONAL INPUTS:
;			input: A string containing the equation. It uses a syntax 
;				y=f(x;p[!]). The parameters have to be
;				explicitely marked as "p[!]". The application will 
;				automatically numner them.
;		KEYWORDS:
;			PARAM: A vector with the associated parameters (guess values)
;			GAUSS: Set to 1 to define a Gaussian (not input needed)
;			LORENTZIAN: Set to 1 to define a Lorentzian (not input needed)
;			PSEUDOVOIGT: Set to 1 to define a PseudoVoigt (not input needed)
;			STEPATAN: Set to 1 to define a StepAtan (not input needed)
;			STEPERRORF: Set to 1 to define a StepErrorF (not input needed)
;			STRAIGHTLINE: Set to 1 to define a StraightLine 
;				(not input needed)
;			POLYNOMIAL: Set to 1 to define a polynomial (not input needed)
;			DEGREE = degree of the polymomial (is POLYNOMIAL kw is set)
;			
;
;	CLEANUP(P)	called when the object is destroyed with Obj_destroy,o1
;
;	INFO(F)	returns 1
;		KEYWORDS:
;			VERBOSE: if set, displays on the terminal the info
;
;	DELETELINE(P)	Deletes a component
;		INPUTS: lineIndex
;			lineIndex: the index of the line 
;
;	ADDLINE(P)	Deletes a component
;		INPUTS: equation
;			equation: The equation template for the line 
;		KEYWORDS: 
;			PARAM: A vector with the associated parameters (guess values)
;			WID: A widget id asociated to the new line.
;			GAUSS: Set to 1 to define a Gaussian (not input needed)
;			LORENTZIAN: Set to 1 to define a Lorentzian (not input needed)
;			PSEUDOVOIGT: Set to 1 to define a PseudoVoigt (not input needed)
;			STEPATAN: Set to 1 to define a StepAtan (not input needed)
;			STEPERRORF: Set to 1 to define a StepErrorF (not input needed)
;			STRAIGHTLINE: Set to 1 to define a StraightLine 
;				(not input needed)
;			POLYNOMIAL: Set to the polynomial DEGREE to define a 
;				polynomial (not input needed)
;			DEGREE = degree of the polymomial (is POLYNOMIAL kw is set)
;
;	EVALUATE(F) Evaluates the equation using the stored parameters for an
;			array of abscissas X
;		INPUTS:
;			X: the abscissas array
;		KEYWORDS:
;			LINE: The index of the line to evaluate (default: the full
;				equation is evaluated)
;	GETPROPERTY(F)	 returns an array with all parameters
;		INPUTS: None
;		KEYWORDS:
;			TEXT: The parameter vertor is in string format (e.g., [2.0,3.0])
;			STR: returns an array of structures, one for each parameter. 
;				The structure tags are: 
;				  tmp = { value:0.0D0, $
;				        fixed:0, $
;				        limited:[0,0], $
;				        limits:[0.0D0,0.0D0] }
;			NLINES: returns a vector containing in each element the number 
;				of parameters for each line equation. 
;			LISTEQUATIONS: returns a vector containing the names of
;				the equation components available in the application. 
;			NPARS: Returns the total number of parameters (for all lines)
;			NINDICES: Returns an array with the number of parameters 
;				per line (a line for each element)
;			NAMES: Return an array of strings with the names of the
;				components.
;			WID: Returns an array with the widget ids associated to
;				each lines. 
;
;	GETPROPERTYLINE(F)	 returns an array with all parameters
;		OPTIONAL INPUTS: 
;			lineIndex: the line index (default: the last line)
;		KEYWORDS:
;			STR: returns an array of structures, one for each parameter. 
;				The structure tags are: 
;				  tmp = { value:0.0D0, $
;				        fixed:0, $
;				        limited:[0,0], $
;			NINDICES: If set, eeturns number of parameters for this line 
;			NAMES: If set, return a strings with the names of the component.
;			WID: If set, returns the widget ids associated to this lines.
;			PARLINE: If set, returns the parameters (guess) of this line
;			STARTINDEX: If set, returns the starting index for numbering the 
;					line parameters
;			TEMPLATE: If set, returns the equation template (i.e., with 
;				p[!] symbols)
;			EQUATION: Is set returns the equation of the line.
;
;
;
;	SETPROPERTY(P)	 
;		KEYWORDS:
;			PARAM: A vector with the associated parameters (guess values)
;			STR: A structure array with the associated parameters 
;			UPDATEWIDGETS: Set this keyword to update the corresponding 
;				widgets
;	SETPROPERTYLINE(P)	 
;		INPUTS:
;			lineIndex: line index (default: the last line)
;		KEYWORDS:
;			PARAM: A vector with the associated parameters (guess values)
;			STR: A structure array with the associated parameters 
;			WID: Set an array with the widget ids associated to
;				each lines. 
;	SETPROPERTYITEM(P)	 
;		INPUTS:
;			lineIndex: line index 
;			itemIndex: item index 
;		KEYWORDS:
;			STR: returns an array of structures, one for each parameter.
;                               The structure tags are:
;                                 tmp = { value:0.0D0, $
;                                       fixed:0, $
;                                       limited:[0,0], $
;                                       limits:[0.,0] }
;			VALUE: returns the value (first tag)
;			FIXED: returns the "fixed" flag (second tag)
;			LIMITED: returns the "limited" flag (third tag)
;			LIMITS: returns the "limits" flag (four tag)
;
;
; EXAMPLE:
;		See and run the file f3equation_test.pro
;
; MODIFICATION HISTORY:
;	Initial version by M. Sanchez del Rio, December 2001
;	2006-09-25 srio@esrf.fr fixes a bug afeccting the names after 
;			deleting lines. Updated doc.
;
;-

;
;===============================================================================
;
FUNCTION __f3equation_getMPFitStr
   tmp = { value:0.0D0, $
	fixed:0, $
	limited:[0,0], $
	limits:[0.0D0,0.0D0] }
RETURN,tmp

END
;
;===============================================================================
;
; calculate the number of parameters in a generic equation form (counting
; the number of p[!] occurences)
;
FUNCTION __f3equation_getNIndices,equation
n=0
tmp=0
iStart=0

equation=StrCompress(equation,/Rem)
WHILE tmp ne -1 DO BEGIN
  tmp = strpos(equation,'p[!]',iStart)
  IF tmp NE -1 THEN BEGIN
    iStart=tmp+1
    n=n+1
  ENDIF
ENDWHILE
RETURN,n
END

;
;===============================================================================
;

FUNCTION __f3equation_getTemplate,type1,Degree=degree
;
; create the equation template (i.e., using p[!] for parameters)
;
CASE StrUpCase(type1) OF
  'GAUSSIAN': equation= 'p[!]*exp( - ( (x-p[!]) *2.0*sqrt(alog(2.))/p[!])^2 )'
  'LORENTZIAN': equation= 'p[!]/(1.0 + ((x-p[!])/(0.5*p[!]))^2)'
  'PSEUDOVOIGT': equation= 'Voigt1(x,[p[!],p[!],p[!],p[!]])'
  'STEPATAN': equation= 'step_atan(x,[p[!],p[!],p[!]])'
  'STEPERRORF': equation= 'step_errorf(x,[p[!],p[!],p[!]])'
  'STRAIGHTLINE': equation= 'p[!]*x^0 + p[!]*x '
  'POLYNOMIAL': BEGIN
	IF N_Elements(degree) EQ 0 THEN degree=2
	equation= ''
                for i=0,degree-1,+1 do equation = equation+ $
                  ' x^'+strcompress(i,/rem)+ $
                  '*p[!]+'
                equation = Equation +  $
                 'x^'+strcompress(degree,/rem) + $
                 '*p[!]'
	END
  else: BEGIN
	;Print,'**Input equation undefined.'
        equation=''
	END
ENDCASE
RETURN,equation
END

;
;===============================================================================
;
FUNCTION f3equation::evaluate,x,Line=line

y=0

IF N_Elements (line) EQ 0 THEN BEGIN	; Evaluate full equation
  p = self->getProperty()
  equation = self->getEquation()
ENDIF ELSE BEGIN 			; Evaluate only one line
  p = self->getPropertyLine(line,/parLine)
  equation = self->getPropertyLine(line,/equation,startIndex=0)
ENDELSE
  
itmp = Execute('y = '+equation)
  
IF itmp NE 1 THEN BEGIN
  tmp = Dialog_Message(/Error,'Error executing: y = '+equation)
ENDIF

RETURN,y
END

;
;===============================================================================
;
FUNCTION f3equation::getProperty, Text=text, Str=str1, $
   NLines=NLines, ListEquations=listEquations, NPars=npars, $
   NIndices=nIndices,Names=names,WId=wId

; returns an array with all parameters



IF Keyword_Set(listEquations) NE 0 THEN BEGIN
   list =  ['StraightLine', 'Polynomial', $
	'Gaussian', 'Lorentzian', 'PseudoVoigt', $
	'StepErrorF', 'StepAtan', $
	'UserDefined']
   RETURN,list
ENDIF

IF N_Elements(NLines) NE 0 THEN BEGIN
  RETURN,self.nLines
ENDIF


IF Keyword_Set(wId) THEN BEGIN
  tmp = *(self.ptrWId)
  RETURN,tmp
ENDIF

IF Keyword_Set(names) THEN BEGIN
  tmp = *(self.ptrNames)
  RETURN,tmp
ENDIF

IF Keyword_Set(nIndices) THEN BEGIN
  tmp = *(self.ptrNIndices)
  RETURN,tmp
ENDIF

IF Keyword_Set(npars) THEN BEGIN
  tmp = Long( Total( *(self.ptrNIndices) ) )
  RETURN,tmp
ENDIF


; if structures are wanted
IF KeyWord_Set(str1) THEN BEGIN
  pars = self->getPropertyLine(0,/str) 
  IF self.nlines GT 0 THEN BEGIN
    pars = self->getPropertyLine(0,/str) 
    par1 = pars[0]
    pars = Replicate(par1,self->GetProperty(/nPar) )
    counter = 0L
    FOR i=0L,self.nlines-1 DO BEGIN	; loop on lines
      addpars = self->getPropertyLine(i,/str)
      FOR j=0L,N_Elements(addpars)-1 DO BEGIN
        tmp = pars[counter]
        copy_structure,addpars[j],tmp
        pars[counter]=tmp
	counter = counter+1
      ENDFOR
    ENDFOR
  ENDIF
  RETURN,pars
ENDIF


; if params are wanted
pars = self->getPropertyLine(0,/parLine)
IF self.nlines GT 0 THEN BEGIN
  FOR i=1L,self.nlines-1 DO BEGIN
     pars=[pars,self->getPropertyLine(i,/parLine)]
  ENDFOR
ENDIF
IF Keyword_Set(text) THEN pars=vect2String(pars)
RETURN,pars


END

;
;===============================================================================
;
FUNCTION f3equation::getEquation
;
; returns the equation
;
equation = self->getPropertyLine(0,/equation)
IF self.nlines EQ 0 THEN RETURN,equation
FOR i=1L,self.nlines-1 DO BEGIN
  equation=equation+' + '+self->getPropertyLine(i,/equation)
ENDFOR
RETURN,equation
END



;
;===============================================================================
;
PRO f3equation::setProperty, param=param, UpdateWidgets=updateWidgets, str=str


nlines=self->GetProperty(/NLines)
nIndices= self->GetProperty(/NIndices)

IF N_Elements(param) NE 0 THEN BEGIN
  istart = -1
  iend = -1
  FOR i=0L,nlines-1 DO BEGIN
    n_i = nIndices[i]
    istart = iend+1
    iend = istart+(nIndices[i]-1)
    self->SetPropertyLine,i,Param=param[istart:iend]
    IF Keyword_Set(updateWidgets) THEN BEGIN
      wId = self->GetPropertyLine(i,/wId)
      Widget_Control,wId,Get_Value=tmp
      tmp.coefs = param[istart:iend]
      Widget_Control,wId,Set_Value=tmp
    ENDIF
  ENDFOR
ENDIF


IF N_Elements(str) NE 0 THEN BEGIN
  istart = -1
  iend = -1
  FOR i=0L,nlines-1 DO BEGIN
    n_i = nIndices[i]
    istart = iend+1
    iend = istart+(nIndices[i]-1)
    self->SetPropertyLine,i,str=str[istart:iend]
    IF Keyword_Set(updateWidgets) THEN BEGIN
      wId = self->GetPropertyLine(i,/wId)
      Widget_Control,wId,Get_Value=tmp
      val = (str[istart:iend]).value
      tmp.coefs = val
      Widget_Control,wId,Set_Value=tmp
    ENDIF
  ENDFOR
ENDIF



END ; f3equation::setProperty

;
;===============================================================================
;
PRO f3equation::setPropertyLine,lineIndex, param=param,WId=wId,str=str
;
; set properties for the full line
;

nlines = self.nLines
IF nlines EQ 0 THEN RETURN
;
IF N_Elements(LineIndex) NE 0 THEN BEGIN
  IF lineIndex LT 0 OR lineIndex GT self.nLines THEN $
	Messsage,'Bad line index value'
ENDIF ELSE lineIndex = nLines-1

npar = (*(self.ptrNIndices))[LineIndex]

;
; set new parameter values (from an array)
;
IF N_Elements(param) NE 0 THEN BEGIN
  nparam = N_Elements(param)
  IF nparam NE npar THEN BEGIN
       Print,'Wrong number of parameters input.'
       RETURN
  ENDIF 
  arrayOfPointers = *(self.ptrPars)
  arrayOfStructures = *(arrayOfPointers[lineIndex])
  FOR i=0L,nparam-1 DO BEGIN
    tmpstr = arrayOfStructures[i]
    tmpstr.value=param[i]
    arrayOfStructures[i]=tmpstr
  ENDFOR
  *(arrayOfPointers[lineIndex]) = arrayOfStructures
ENDIF


;
; set new str (from an array)
;
IF N_Elements(str) NE 0 THEN BEGIN
  nparam = N_Elements(str)
  IF nparam NE npar THEN BEGIN
       Print,'Wrong number of parameters input.'
       RETURN
  ENDIF 
  arrayOfPointers = *(self.ptrPars)
  arrayOfStructures = *(arrayOfPointers[lineIndex])
  FOR i=0L,nparam-1 DO BEGIN
    tmpstr = arrayOfStructures[i]
    tmpstr=str[i]
    arrayOfStructures[i]=tmpstr
  ENDFOR
  *(arrayOfPointers[lineIndex]) = arrayOfStructures
ENDIF



IF N_Elements(wId) NE 0 THEN BEGIN
  w_Ids = *(self.ptrWId)
  w_Ids[lineIndex]=Long(wId)
  *(self.ptrWId)=w_Ids
ENDIF


END


;
;===============================================================================
;
PRO f3equation::setPropertyItem,lineIndex, itemIndex, $
  value=value, fixed=fixed, limited=limited, limits=limits, $
  str=str
;
; set properties for an individual item of a line
;

nlines = self.nLines
IF nlines EQ 0 THEN RETURN
;
IF N_Elements(lineIndex) EQ 0  OR N_Elements(lineIndex) EQ 0 THEN $
  Message,'Usage: o->setPropertyItem,lineIndex,itemIndex'

nitems = (*(self.ptrNIndices))[LineIndex]


arrayOfPointers = *(self.ptrPars)
arrayOfStructures = *(arrayOfPointers[lineIndex])
tmpstr = arrayOfStructures[itemIndex]
;
; set new parameter value
;
IF N_Elements(value) NE 0 THEN BEGIN
  tmpstr.value=value
ENDIF
;
; set new "str" flag
;
IF N_Elements(str) NE 0 THEN BEGIN
  tmpstr=str
ENDIF
;
; set new "fixed" flag
;
IF N_Elements(fixed) NE 0 THEN BEGIN
  tmpstr.fixed=fixed
ENDIF
;
; set new "limited" flags
;
IF N_Elements(limited) NE 0 THEN BEGIN
  tmpstr.limited=limited
ENDIF
;
; set new "limits" values
;
IF N_Elements(limits) NE 0 THEN BEGIN
  tmpstr.limits=limits
ENDIF


arrayOfStructures[itemIndex]=tmpstr
*(arrayOfPointers[lineIndex]) = arrayOfStructures

END




;
;===============================================================================
;
FUNCTION f3equation::getPropertyLine,lineIndex, $
  WId=wId, ParLine=parLine, StartIndex=startIndex, str=str, $
  Names=names,Template=template,equation=equation, $
  NIndices=nIndices

;;   NLines=NLines, ListEquations=listEquations, NPars=npars, $
;
; startIndex is a keyword to set the starting index fror the line
; equation. If not set, it takes the assigned index.
;

IF N_Elements(lineIndex) EQ 0 THEN lineIndex=self.nlines-1>0


IF Keyword_Set(wId) THEN BEGIN
  tmp = *(self.ptrWId)
  tmp=tmp[lineIndex]
  RETURN,tmp
ENDIF

IF Keyword_Set(names) THEN BEGIN
  tmp = *(self.ptrNames)
  tmp=tmp[lineIndex]
  RETURN,tmp
ENDIF

IF Keyword_Set(template) NE 0 THEN BEGIN
  IF lineIndex LT 0 OR lineIndex GT self.nLines THEN $
	Messsage,'Bad line index value'
  line = (*(self.ptrLines))[LineIndex]
  RETURN,line
ENDIF


IF Keyword_Set(parLine) NE 0 THEN BEGIN
  arrayOfPointers = *(self.ptrPars)
  arrayOfStr = *(arrayOfPointers[lineIndex])
  RETURN,arrayOfStr.value
ENDIF


IF Keyword_Set(str) NE 0 THEN BEGIN
  arrayOfPointers = *(self.ptrPars)
  arrayOfStr = *(arrayOfPointers[lineIndex])
  RETURN,arrayOfStr
ENDIF

IF Keyword_Set(nIndices) THEN BEGIN
  tmp = *(self.ptrNIndices)
  tmp = tmp[lineIndex]
  RETURN,tmp
ENDIF

IF keyword_Set(equation) NE 0 THEN BEGIN
  IF lineIndex LT 0 OR lineIndex GT self.nLines THEN $
	Messsage,'Bad line index value'
  line = (*(self.ptrLines))[lineIndex]
  npar = (*(self.ptrNIndices))[lineIndex]
  IF N_Elements(startIndex) EQ 0 THEN $
	startIndex = (*(self.ptrStartIndices))[LineIndex]
  FOR j=startIndex,startIndex+npar-1 DO BEGIN
     line = strSubstitute(line,'[!]','['+StrCompress(j,/Rem)+']',/NoGlobal)
  ENDFOR
  RETURN,line
ENDIF
;

RETURN,1
END

;
;===============================================================================
;
PRO f3equation::deleteLine, line
;
; cleans a line

nlines =  self.nlines
IF N_Elements(line) EQ 0 then line = nlines-1
IF line GT nlines-1 THEN BEGIN
  Print,'Line does not exist: '+String(line)
  IF SDep(/W) THEN $
     itmp=Dialog_Message(/Info,'Line does not exist: '+String(line))
  RETURN
ENDIF

IF nlines GT 1 THEN BEGIN
  indices = IndGen(nlines)
  newIndices = Where(indices NE line)

  myLines = *(self.ptrLines)
  myLines = myLines(newIndices)
  *(self.ptrLines) = myLines
  
  myNames = *(self.ptrNames)
  myNames = myNames(newIndices)
  *(self.ptrNames) = myNames
  
  nIndices = *(self.ptrNIndices)
  myNIndex= nIndices[line] ; to be used later
  nIndices = nIndices[newIndices]
  *(self.ptrNIndices) = nIndices
  
  startIndices = *(self.ptrStartIndices)
  startIndices[line:nlines-1] = startIndices[line:nlines-1]-myNIndex
  startIndices = startIndices[newIndices]
  *(self.ptrStartIndices) = startIndices
  
  arrayOfPointers = *(self.ptrPars)
  ii = arrayOfPointers[line]
  arrayOfPointers = arrayOfPointers[newIndices]
  IF Ptr_Valid(ii) THEN Ptr_Free,ii
  *(self.ptrPars) = arrayOfPointers
  w_id = *(self.ptrWId)
  IF Widget_Info(w_Id[line],/Valid_Id) THEN Widget_Control,w_Id[line],/Destroy
  w_Id = w_Id[newindices]
  *(self.ptrWId) = w_Id
ENDIF ELSE BEGIN  			; deleting last line
  *(self.ptrLines) = ''
  *(self.ptrNIndices) = 0
  *(self.ptrNames) = ''
  *(self.ptrStartIndices) = 0
  arrayOfPointers = *(self.ptrPars)
  ii = arrayOfPointers[line]
  IF Ptr_Valid(ii) THEN Ptr_Free,ii

  ptrParameters = Ptr_New()
  *(self.ptrPars) = [ptrParameters]


  w_id = *(self.ptrWId)
  IF Widget_Info(w_Id[0],/Valid_Id) THEN Widget_Control,w_Id[0],/Destroy
  *(self.ptrWId) = [0L]
ENDELSE
  
self.nlines = nlines-1

END

;
;===============================================================================
;
PRO f3equation::addLine, equation , $
  Gaussian=gauss, Lorentzian=lorentzian, PseudoVoigt=pseudovoigt, $
  Polynomial=polynomial, StraightLine=straightLine, $
  StepErrorF=steperrorf, StepAtan=stepAtan, $
  Param=param,WId=wId


nLines = self.nLines
type1='UNDEFINED'
IF N_Elements(equation) NE 0 THEN type1='UserDefined'
IF N_Elements(wId) EQ 0 THEN wId=0L
IF Keyword_Set(gauss) THEN type1='Gaussian'
IF Keyword_Set(lorentzian) THEN type1='Lorentzian'
IF Keyword_Set(pseudovoigt) THEN type1='PseudoVoigt'
IF Keyword_Set(stepatan) THEN type1='StepAtan'
IF Keyword_Set(steperrorf) THEN type1='StepErrorF'
IF Keyword_Set(straightLine) THEN type1='StraightLine'
IF Keyword_Set(polynomial) THEN BEGIN
   type1='Polynomial'
   degree=polynomial
ENDIF

IF type1 NE 'UserDefined' THEN equation=__f3equation_getTemplate(type1,Degree=degree)

IF equation EQ '' THEN RETURN

npar = __f3equation_getNIndices(equation)
IF npar EQ 0 THEN BEGIN
   itmp = Dialog_Message(/Error,'Equation syntax error: no parameters found')
   RETURN
ENDIF

IF nLines EQ 0 THEN newequation = [equation]  ELSE $
        newequation = [*(self.ptrLines),equation]
*(self.ptrLines) = newequation

nIndices = *(self.ptrNIndices)


startIndices  = *(self.ptrStartIndices)
IF nLines EQ 0 THEN newStartIndices = [Long(Total(NIndices))] ELSE $
        newStartIndices = [startIndices,Long(Total(NIndices)) ]
*(self.ptrStartIndices) = newStartIndices

IF nLines EQ 0 THEN newNIndices = [npar] ELSE newNIndices = [nIndices,npar]
*(self.ptrNIndices) = newNIndices

self.nLines= nLines+1

IF Keyword_Set(param) THEN BEGIN
  IF npar NE N_Elements(param) THEN BEGIN
    itmp = Dialog_Message(/Error, ['Invalud number of input parameters',$
	'Set them to zero'])
    param = FindGen(npar)
  ENDIF
ENDIF

arrayOfStructures = Replicate(__f3equation_getMPFitStr(),npar)

IF nLines EQ 0 THEN BEGIN
  arrayOfPointers = *(self.ptrPars)
  IF Ptr_Valid(arrayOfPointers[0]) THEN Ptr_Free,arrayOfPointers[0]
  newArrayOfPointers = [Ptr_New(arrayOfStructures)]
  *(self.ptrPars)=newArrayOfPointers

  IF Ptr_Valid(self.ptrWId) THEN Ptr_Free,self.ptrWId
  self.ptrWId=Ptr_New([wId])

  IF Ptr_Valid(self.ptrNames) THEN Ptr_Free,self.ptrNames
  self.ptrNames=Ptr_New([type1])
ENDIF ELSE BEGIN
  arrayOfPointers = *(self.ptrPars)
  newArrayOfPointers = [arrayOfPointers,Ptr_New(arrayOfStructures)]
  *(self.ptrPars)=newArrayOfPointers

  tmp = *(self.ptrWId)
  tmp = [tmp,wId]
  *(self.ptrWId) = tmp

  tmp = *(self.ptrNames)
  tmp = [tmp,type1]
  *(self.ptrNames) = tmp
ENDELSE

nparam = N_Elements(param)
IF nparam NE 0  THEN self->setPropertyLine,param=param
;===============================================================================


;===============================================================================

END

;
;===============================================================================
;
FUNCTION f3equation::Info,Verbose=verbose

verbose=1
nLines = self.nLines

txt=['********************************************************************',$
'Number of lines    : '+StrCompress(nLines), $
'',$
'Lines       : '+StrCompress(*(self.ptrLines),/Rem), $
'StartIndices: '+Vect2String(*(self.ptrStartIndices)),$
'NIndices    : '+Vect2String(*(self.ptrNIndices),/Rem), $
'WId    : '+Vect2String(*(self.ptrWId),/Rem), $
'Names    : '+Vect2String(self->getProperty(/Names)), $
'Total NIndices    : '+StrCompress(self->getProperty(/npar),/Rem), $
'']

IF nLines EQ 0 THEN GoTo,out

FOR i=0L,nLines-1 DO BEGIN
  line = self->getPropertyLine(i,/equation)
  txt=[txt,'','','','Equation for line '+StrCompress(i,/Rem)+' is: '+line]
  values = self->getPropertyLine(i,/parLine)
  txt=[txt,'Pars are     : '+ vect2string(values), $
           'Name is : '+ self->getPropertyLine(i,/Names)]
ENDFOR
help,/str,str,Output=tmp
txt=[txt,'Str info is : ','     '+tmp] 


str = self->getProperty(/str)
FOR i=0,N_Elements(str)-1 DO BEGIN
  txt=[txt, '','p['+StrCompress(i,/Rem)+']',$
       '     Value: '+StrCompress(str[i].value,/Rem), $
       '     Fixed: '+StrCompress(str[i].limited,/Rem), $
       '     Limited: '+Vect2String(str[i].limited), $
       '     Limits: '+Vect2String(str[i].limits)]
ENDFOR

txt=[txt,'********************************************************************']

out:
IF Keyword_Set(verbose) THEN BEGIN
  FOR i=0L,N_Elements(txt)-1 DO print,txt[i]
ENDIF

RETURN,txt
END


;
;===============================================================================
;
PRO f3equation::cleanup
  IF Ptr_Valid(self.ptrLines) THEN Ptr_Free,self.ptrLines
  IF Ptr_Valid(self.ptrStartIndices) THEN Ptr_Free,self.ptrStartIndices
  IF Ptr_Valid(self.ptrNIndices) THEN Ptr_Free,self.ptrNIndices
  
  IF Ptr_Valid(self.ptrPars) THEN BEGIN
     arrayOfPointers = *(self.ptrPars)
     FOR i=0L,N_Elements(arrayOfPointers)-1 DO BEGIN
        ptr_i = arrayOfPointers[i]
        IF Ptr_Valid(ptr_i) THEN Ptr_Free,ptr_i
     ENDFOR
  ENDIF
  IF Ptr_Valid(self.ptrPars) THEN Ptr_Free,self.ptrPars
  IF Ptr_Valid(self.ptrWId) THEN Ptr_Free,self.ptrWId
  IF Ptr_Valid(self.ptrNames) THEN Ptr_Free,self.ptrNames
END


;
;===============================================================================
;
FUNCTION f3equation::init, equation, $
  Gauss=gauss, Lorentzian=lorentzian, PseudoVoigt=pseudovoigt, $
  Polynomial=polynomial, StraightLine=straightLine, $
  StepErrorF=steperrorf, StepAtan=stepatan, $
  Param=param

type1='UNDEFINED'
IF N_Elements(equation) NE 0 THEN type1='UserDefined'
IF Keyword_Set(gauss) THEN type1='Gaussian'
IF Keyword_Set(lorentzian) THEN type1='Lorentzian'
IF Keyword_Set(pseudovoigt) THEN type1='PseudoVoigt'
IF Keyword_Set(stepatan) THEN type1='StepAtan'
IF Keyword_Set(steperrorf) THEN type1='StepErrorF'
IF Keyword_Set(straightLine) THEN type1='StraightLine'
IF Keyword_Set(polynomial) THEN BEGIN
   type1='POLYNOMIAL'
   degree=polynomial
ENDIF

IF type1 NE 'UserDefined' THEN $
	equation=__f3equation_getTemplate(type1,Degree=degree)

IF equation NE '' THEN nLinesAdd=1 ELSE nLinesAdd=0
npar = Long(__f3equation_getNIndices(equation))

IF npar EQ 0 THEN strArray = Replicate(__f3equation_getMPFitStr(),1) ELSE $
	          strArray = Replicate(__f3equation_getMPFitStr(),npar)  

;
; Insert parameter values in the structure
;

     
;
; define pointer to structure array with parameters
;
IF npar EQ 0 THEN BEGIN
  ptrParameters = Ptr_New()
ENDIF ELSE BEGIN
  ptrParameters = Ptr_New(strArray)
ENDELSE

self.nlines = self.nlines + nLinesAdd
self.ptrLines = Ptr_New([equation])
self.ptrStartIndices = Ptr_New([0L])
self.ptrNIndices =Ptr_New([npar])
self.ptrPars = Ptr_New( [ptrParameters] )
self.ptrWId = Ptr_New([0L])
self.ptrNames = Ptr_New([type1])

IF KeyWord_Set(param) THEN self->setPropertyLine,param=param

out:
RETURN,1
END

;
;===============================================================================
;
PRO f3equation__define
  tmp = {f3equation, 	$
		nlines:0L, $			; number of eq lines
		ptrLines:Ptr_New(), $		; ptr to array with lines
		ptrStartIndices:Ptr_New(), $ ;ptr to starting index of each line
		ptrNIndices:Ptr_New(), $  ; ptr to number of pars per line 
		ptrPars:Ptr_New(), $      ; ptr to parameters arrays of str
		ptrWId:Ptr_New(), $       ; ptr to array with widget indices
		ptrNames:Ptr_New() }      ; ptr to array with names
END

