

;+
; NAME:
;	AZIMUTH_INTEGRAL_AXIAL
; PURPOSE:
;	this procedure permorms the azimuthal integration of an image
;       knowing the beam impact point and the sample position
;
; CALLING SEQUENCE:
;	azimuth_integral_axial, M, xh, intensity
; INPUTS:
;	M:  The array with 2D data matrix (may 
;           contain -1 for masked values)
; OUTPUTS:
;       xh: the abscissas array of the integral (in "pixels")
;       intensity: the intensity array of the integral
;
; KEYWORD PARAMETERS:
;       v0: [Xcenter,Ycenter] the direct beam impact point on the detector
;	vS: [xS,yS,zS] the coordinates of the sample
;	INCREMENT: The integration step in degrees (default=0.02)
;	FORCE: set this keyword to 1 for forcing full calculations. 
;	    If force=0, the geometrical part of the computation is 
;           saved in common blocks and further reused. This is valid if
;           the image has the same dimension and the same mask as the 
;           previous one, and V0 and VS are unchanges. 
;           default: force=1
;        NOMASK: set this keyword to 1 for considering also negative 
;           pixel values in the calculation (Default NOMASK=0, meaning
;           that negative pixel values are not used, i.e., masked)
;
;	
; MODIFICATION HISTORY:
;	2007/11/29 srio@esrf.eu 
; 
;       2009-12-04 srio@esrf.eu added NoMask, default Force=1
;-	
pro azimuth_integral_axial, M, xh, intensity, v0=v0, vS=vS, $
  INCREMENT=incr, Force=force, NoMask=NoMask ; , ERROR=ierror

COMMON azimuth_integral_axial1, v01, vS1, incr1, xh1, intensity1,  $
       nMask1, w, wm, rev


Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,/Cancel,$
      ['azimuth_integral_axial: error caught: '+!error_state.msg,$
       'retry forcing full calculation? '])
      IF itmp EQ 'Cancel' THEN BEGIN
        xh=0
        intensity=0
        RETURN
      ENDIF ELSE BEGIN
        force=1
        goto,restart
      ENDELSE
      catch, /cancel
      on_error,2
endif

restart:

t00 = systime(1)

IF N_Elements(incr) EQ 0 THEN incr=0.02D0 ; angular increment in deg
IF N_Elements(force) EQ 0 THEN force=1

;force=1
IF Not(Keyword_Set(Force)) THEN BEGIN
  IF N_Elements(v01) NE 0 AND N_Elements(vS1) AND N_Elements(incr1) NE 0 THEN BEGIN
    IF norm( [v01,vS1,incr1]-[v0,vS,incr] ) LE 1e-8 THEN BEGIN
      tmp = where(M LT 0,nMask)
      IF nMask EQ nMask1 THEN BEGIN
        print,'AZIMUTH_INTEGRAL_AXIAL: Re-using calculation.'
        xh=xh1
        intensity=intensity1*0
        goto,jumpHere
      ENDIF
    ENDIF
  ENDIF
ENDIF


s=Long(size(M))
nx = s[1]
ny = s[2]

IF (s[0] ne 2) or (s[1] lt 2) or (s[2] lt 2) THEN BEGIN
  xh=0 
  intensity=0
  RETURN
ENDIF

IF n_elements(v0)  ne 2 then v0=([s[1],s[2]]-1.)/2. else v0=float(v0)
IF n_elements(vS)  ne 3 then v3=[v0,5*norm(v0)] else vS=float(vS)

incr=abs(incr) ; <(s[1]/2) <(s[2]/2) ; > 0.01

t0=systime(1)
;
; compute matrices with the coordinates of the center of pixel
;
xx = (MakeArray1(nx,0.,nx-1)+0.5)#Replicate(1,ny)
yy = Replicate(1,nx)#(MakeArray1(ny,0.,ny-1)+0.5)

print,'AZIMUTH_INTEGRAL_AXIAL: time for creating xx,yy: ',systime(1)-t0

;
; apply mask (do not considere negative pixels)
; for masked pixels, set both intensity and weight zero
;
t0=systime(1)
IF keyword_Set(noMask) THEN BEGIN
  tmp = [-1]
ENDIF ELSE BEGIN
  tmp = where(m LT 0,nMask1)
ENDELSE

IF tmp[0] EQ -1 THEN BEGIN
  Print,'AZIMUTH_INTEGRAL_AXIAL: Masked values: None'
ENDIF ELSE BEGIN
  Print,'AZIMUTH_INTEGRAL_AXIAL: Masked values: ', N_Elements(tmp)
  igood = LIndGen(s[1]*s[2])
  igood[tmp]=-1
  igood = igood[where(igood GE 0)]
  IF igood[0] EQ -1 THEN Message,'No good points in image.'
  m=m[igood]
  xx=xx[igood]
  yy=yy[igood]
ENDELSE
print,'AZIMUTH_INTEGRAL_AXIAL: time for masking : ',systime(1)-t0



t0=systime(1)

; 
; calculate discances from pixel to v0 (tau2) and to vS (rho2)
;
rho2 = (xx-vS[0])^2 + (yy-vS[1])^2 + vS[2]^2
tau2 = (xx-v0[0])^2 + (yy-v0[1])^2

print,'AZIMUTH_INTEGRAL_AXIAL: time for creating distance matrices : ',systime(1)-t0
t0=systime(1)
;
; apply cosine theorem to get the angle subtended from each pixel-vS
; with the beam
;
vv2 = (v0[0]-vS[0])^2 + (v0[1]-vS[1])^2 + vS[2]^2
;cosTheta = ((tau2 - rho2) - vv2) / ( (-2D0*Sqrt(vv2))*Sqrt(rho2) )
;Theta = acos(Temporary(cosTheta))
theta = acos ( ( (tau2-rho2)-vv2 )/( (-2D0*Sqrt(vv2))*Sqrt(rho2) ) )

print,'AZIMUTH_INTEGRAL_AXIAL: time for computing angular matrices : ',systime(1)-t0
t0=systime(1)

ThetaMin = min(theta,max=ThetaMax)
;rmax=ceil(rmax)
;rmin=floor(rmin)

;print,'Min Theta [rad,deg]: ',ThetaMin,ThetaMin*180/!pi
;print,'Max Theta [rad,deg]: ',ThetaMax,ThetaMax*180/!pi

print,'AZIMUTH_INTEGRAL_AXIAL: time for preparing arrays:',systime(1)-t0
;
; compute histogram of distances
;
t0=systime(1)
h = histogram(theta,binsize=incr*!dpi/180,min=thetaMin,reverse_indices=rev, locations=xh)

; the abscissas of histogram are shifted to the center of the bin
xh=xh+0.5*(incr*!dpi/180) 
nh = N_Elements(xh)

print,'AZIMUTH_INTEGRAL_AXIAL: time for histogramming:',systime(1)-t0
;
; define output arrays
;
intensity=xh*0.0
;ierror=xh*0.0


;===============================================================================
; 
; pixels contributing to main ring
;

;
; theta_rounded is a matrix of the same dimension as theta, where each value of theta has
; been substituted by the closest value in the histogram (i.e., rounded)
;
theta_rounded = theta*0
t0 = systime(1)
FOR i=0L,nh-1 DO BEGIN
  ;Set all elements of rr_rounded that are in the ith bin of rr to xh[i]
  IF REV[i] NE REV[i+1] THEN theta_rounded[Rev[Rev[I] : Rev[i+1]-1]] = xh[i]
ENDFOR
print,'AZIMUTH_INTEGRAL_AXIAL: time for first loop:',systime(1)-t0


; 
; Build the pixel weight for the integration
;
w=(1.-(abs(theta_rounded-theta)/(incr*!dpi/180) )) >0

;
; store data in COMMON block
;
v01=v0
vs1=vS
incr1=incr
intensity1=intensity
xh1=xh


jumpHere:

; Multiply the matrix m by weight
wm = w*m


; fill the output arrays
t0 = systime(1)
FOR i=0L,N_Elements(xh)-1 DO BEGIN
  IF REV[i] NE REV[i+1] THEN BEGIN
    num1 = total(wm[Rev[Rev[I] : Rev[i+1]-1]])
    den1 = total(w[Rev[Rev[I] : Rev[i+1]-1]])
    IF den1 NE 0 THEN intensity[i]=intensity[i]+num1/den1 ;ELSE intensity[i]=0
    ;ierror[i]=ierror[i]+(sqrt(num1)/den1) ; CHECK! NOT SURE
  ENDIF
ENDFOR
print,'AZIMUTH_INTEGRAL_AXIAL: time for second loop:',systime(1)-t0


;xplot,/no,xh*180/!pi,intensity


print,'AZIMUTH_INTEGRAL_AXIAL: Total time for azimuth_integral_axial: ',systime(1)-t00

END
