FUNCTION intersec_trial, patch, plane, ray, tolerance=lim, index=index,  $
  maxiter=n,verbose=verbose,useGuess=useGuess

;+
; FUNCTION intersec_trial, patch, plane,lim=lim, index=index
;
; Finds the intersection parameters (u,v) for a number of rays
; returns an array of structures with the points, derivatives and flag
;
;
; Inputs: 
;        patch: the beziers patches of the surface (array of structures)
;        plane: the rays. In fact, each ray is represented by the
;               intersection of two planes. 
;	 ray: matrix 7xn with the rays (used with useGuess kw)
;        tolerance: tolerance. When lim is reached, the iterative search is
;		stopped. (default: 1D-3)
;        maxiter: maximum number of iterations (default=30)
; 
; Output: 
;	 index: index of the patch where the intersection is found.
;
;
; Keywords: 
;        verbose: 1 for printing info
;        useGuess: if set, precomputes guess for better calculation
;
;-


time0=systime(1)

IF N_Elements(lim) EQ 0 THEN lim = 1D-3
IF N_Elements(n) EQ 0 THEN n=30
IF N_Elements(verbose) EQ 0 THEN verbose=0

;=== REGISTER VARIABLES ===;
;=== counter variables ===;

np = n_elements(plane)   ;number of plane
nbzr = n_elements(patch) ;number of patches
index=lonarr(np)	 ;which ray with which patch

;int={pnt:dblarr(3),gradu:dblarr(3),gradv:dblarr(3), flag:-1}
int={xyz:dblarr(3),normal:dblarr(3), flag:-1}
int=replicate(int,np)

;parametric=dblarr(3, np) ; initialize returned (u,v,flag)
;parametric[2,*]=-1



FOR jj=0L, np-1 DO BEGIN

  ;=== it checks if an interesection occurs ===;
  
  ;binary array npatch-by-nray (0 eq no intersection 1 eq intersection)

  ; dchek is an array with flags: 1=intersection, 0=No
  ; dist is the new created NURBS with the distance plane
  ; *******************************************************************
  dcheck=int_check(patch, plane[jj],dist=dist) 

  e=where(dcheck EQ 1) ;which ray hit the JJth patch
  index[jj]=e

  dim=4
  IF KEYWORD_SET(verbose) THEN print, '********************** ray: ', jj,' patch: ', e

  ;condition of at least one intersection with the JJth patch
  IF e[0] NE -1 THEN BEGIN 
    ; computes guess
    IF Keyword_Set(useGuess) THEN BEGIN
      uvguess=int_guess(patch[e], ray[*,jj],verbose=verbose)
      uu = uvguess.u 
      vv = uvguess.v
    ENDIF ELSE BEGIN
      ;starting parametric points;
      ; uu and vv are the limits of the knots for the given patch
      uu = patch[e].uknots(uniq(patch[e].uknots))
      vv = patch[e].vknots(uniq(patch[e].vknots))
      IF Keyword_Set(verbose) THEN print,'INTERSEC_TRIAL u v:',uu,vv
    ENDELSE

    temp0=dist

    ff=0 ;iteration counter

    ;IF Keyword_Set(verbose) THEN print, 'INTERSEC_TRIAL ray: ', jj,' patch: ', e
    u_lim=uu[1]-uu[0]
    v_lim=vv[1]-vv[0]
    ; order of the patch
    pdeg=temp0.p+1
    qdeg=temp0.q+1


    c=temp0.coefs
    trim=where(abs(c) lt 1e-5)
    distx=temp0.xgrid
    disty=temp0.ygrid
    grid=dblarr(pdeg,3,qdeg)

    WHILE ((u_lim GE lim) or (v_lim GE lim)) AND (ff LE n) DO BEGIN

      IF trim[0] NE -1 THEN c[trim]=0.0
      ; calculates convergence planes 
      ; xydirection=0, users distance to x,y axes
      ; xydirection=1 uses formulas in 4.3.3 of Efremov Master thesis
      ; 
      b=moddist_trial(distx,disty,ff,xydirections=1)
      temp1=temp0

      IF u_lim GT lim THEN BEGIN
        uu=u_par_trial(b,uu,verbose=verbose)
        ;IF Keyword_Set(verbose) THEN print,'INTERSEC_TRIAL: ff,uu:',ff,uu
      ENDIF

      u_lim=uu[1]-uu[0]

      IF v_lim GT lim THEN BEGIN
        vv=v_par_trial(b,vv,verbose=verbose)
        ;IF Keyword_Set(verbose) THEN print,'INTERSEC_TRIAL: ff,vv:',ff,vv
      ENDIF
      v_lim=vv[1]-vv[0]



      utemp=nrblinear(uu[0],uu[1],pdeg)
      vtemp=nrblinear(vv[0],vv[1],qdeg)


      ;
      ; De Casteljau algorithm (see Efremov p 12 or NURBS book)
      ; A Davide non piace. 
      ; 
      ;FOR mm=0,qdeg-1 DO BEGIN
      ;  FOR nn=0,pdeg-1 DO grid[nn,0:2,mm]=decasteljauS(temp0.coefs,utemp[nn],vtemp[mm])
      ;ENDFOR
      ; otherwise:
      grid=nrbeval(temp0,utemp,vtemp,/space)

      c=grid
      trim=where(abs(c) lt 1e-5)
      distx=__nrbgrid2mat(grid,v=0)
      disty=__nrbgrid2mat(grid,v=1)

      ff++

    ENDWHILE
    IF Keyword_Set(verbose) THEN print,'INTERSEC_TRIAL last u v:',uu,vv
    flg = 1
    ;if ff ge n then begin
    ;  flg=-1
    ;  IF Keyword_Set(verbose) THEN print, 'INTERSEC_TRIAL ray: ', jj,' patch: ', e
    ;  IF Keyword_Set(verbose) THEN print, 'INTERSEC_TRIAL more than', n, ' iterations'
    ;endif
    ;IF (u_lim GE lim) THEN flg=2
    ;IF (v_lim GE lim) THEN flg=3
    ;IF (ff LE n) DO BEGIN
    ; parametric[*,jj]=[total(uu)/2,total(vv)/2,1]
    parametric=[total(uu)/2,total(vv)/2]
    ; 
    ; computes now the point and its derivatives
    ;
    p=nrbdeval(patch[e],parametric[0],parametric[1],jacob=der,/dnrb)
    IF Keyword_Set(verbose) THEN print,'INTERSEC_TRIAL: intersection point: ',p
    int[jj].xyz = p
    int[jj].normal = vecproduct(Reform(der.jacu),Reform(der.jacv))
    int[jj].flag = flg
  ENDIF
ENDFOR
IF Keyword_Set(verbose) THEN print, 'INTERSEC_TRIAL final u v:', uu,vv

print,'intersec.pro time: ',systime(1)-time0

;ind=where(parametric[2,*] eq 0)
;IF ind[0] NE -1 THEN BEGIN
;parametric[2,ind] = -1
;ENDIF

;RETURN, parametric
RETURN, int


END
