pro ray_prop,shadow_in,col1,col2,RANGE=range,ITERS=iters,          $
XRANGE=xrange,YRANGE=yrange,TITLE=title,XTITLE=xtitle,YTITLE=ytitle,  $
NOLOST=nolost,PRINT=print,CART=cart,WAIT=wait,GROUP=group,xplot=ixplot, $
evol=evol,noplot=noplot,mpeg_create=mpeg_create,$
nbins=nbins
;+
; NAME:
;	RAY_PROP 
; PURPOSE:
;	plot the evolution of an image (cols 1,3) in vacuum along the beamline
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	ray_prop,shadow_in,keyword_parameters
; INPUTS:
;	shadow_in   IDL structure with SHADOW data
; OPTIONAL INPUTS:
;	col1 [default=1] The column to be placed in the Horizontal axis.
;	col2 [default=2] The column to be placed in the Verical axis.
; KEYWORD PARAMETERS:
;	RANGE  = [min,max]  limits of Y axis (col 2) to where retrace the image 
;	ITERS  = number of iterations (default=20)
;	XRANGE = [min,max], limits of the x scale. 
;	YRANGE = [min,max], heigth limits of the y scale. 
;	NOLOST = consider losses
;		0 all rays (default option)
;		1 exclude losses 
;		2 only losses 
;       TITLE = top title
;       XTITLE = title in X column
;       YTITLE = title in Y column
;	PRINT  = if set to 1, set to screen all the images. This
;		option is suitable for generating the PS file.
;	CART = if set, force cartesian axes (same H and V aspect ratio)
;	WAIT = time between frames (default=0)
;	XPLOT = set this keyword to launch an xplot window with the 
;		statistical parameters (mean sizes and st dev) versus
;		beam position.
;       GROUP The parent id for the caller widget (used to positioning
;               error and information windows).
;	EVOL Set this keyword to a named variable to receive a multicolum 
;		array with: 
;		Col1: dist [cm]
;		Col2: Mean X
;		Col3: Mean Y
;		Col4: St Dev X
;		Col5: St Dev Y
;		Col6: weighted Mean X
;		Col7: weighted Mean Y
;		Col8: weighted StDev X
;		Col9: weighted StDev Y
;	NOPLOT Set this keyw avoid showing the animation (useful when 
;		one is only interested in statistical values). 
;	MPEG_CREATE Set this keyw to create a mpeg movie file ray_prop.mpg
;       NBINS: set this keyword to the number of bins to be used in the
;              calculations of the histogram FWHM. (Default: NBINS=0,
;              i.e., no histogram calculation)
;              Note that the results of these calculations are not
;              plotted, but stored and displayed using the XPLOT=1 option
;	
; OUTPUTS:
;	the interactive view of the beam evolution
; OPTIONAL OUTPUT PARAMETERS:
; COMMON BLOCKS:
; SIDE EFFECTS:
; 	None.
; RESTRICTIONS:
;	None.
; PROCEDURE:
;	Similar to SHADOW's RETRACE
; MODOFICATION HISTORY:
;	Original from Steve Turner
;       92/10/13 MSR lots of modifications
;	98/04/30 srio@esrf.fr Almost rewritten. Adapted for SHADOWVUI.
;	07/10/03 srio@esrf.eu added XPLOT keyword
;	2010/01/29 srio@esrf.eu added histogram FWHM calculations (NBINS)
;
;-
;
catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message(/Error,Dialog_Parent=group,$
	'RAY_PROP: Error caught: '+!err_string)
   goto,out
endif

p_position_old= !p.position
p_charsize_old = !p.charsize

IF N_Elements(nbins) EQ 0 THEN nbins=0

; set graphic environment and font's size independently from device
;
!p.position = [0.04,0.04,0.7,0.9]
bfy = float(!d.y_ch_size)/!d.y_size
bfx = float(!d.x_ch_size)/!d.x_size
xsize = 0.0065/bfx
ysize = 0.018/bfy
!p.charsize = min([xsize,ysize]) 
if not(keyword_set(title)) then  title = ' '
if not(keyword_set(wait)) then  wait = 0
if n_params() eq 1 then begin
  col1=1
  col2=3
endif
if not(keyword_set(xtitle)) then  xtitle = column_name(col1)
if not(keyword_set(ytitle)) then  ytitle = column_name(col2)

IF keyword_set(mpeg_create) THEN mpegID = MPEG_OPEN([!d.x_size,!d.y_size])
mpegindex=0L
;
; load shadow-idl structure and define arrays and constants
;
shadow_out = readsh(shadow_in,NoLost=nolost)
;
;
arr1 = getshcol(shadow_out,col1)
arr2 = getshcol(shadow_out,col2)
i0 = getshcol(shadow_out,23)

if not(keyword_set(xrange)) then xrange = [min(arr1,max=tmp),tmp]
if not(keyword_set(yrange)) then yrange = [min(arr2,max=tmp),tmp]
;
if not(keyword_set(iters)) then iters = 20L
if not(keyword_set(range)) then range = [-10.0,10.0] 
stepsize = (range(1)-range(0))/float(iters)
if not(keyword_set(noplot)) then begin
  if keyword_set(cart) then begin
     cart_axes,xrange,yrange,position=!p.position,$
      title=title,xtitle=xtitle,ytitle=ytitle
  endif else begin
     plot,[0,0],xrange=xrange,yrange=yrange,/NoData,$
      title=title,xtitle=xtitle,ytitle=ytitle
  endelse
endif

;ppoints = getshcol(shadow_out,[col1,col2])
;npoints = iters+1
npoints = iters
stdevx = DblArr(npoints)
stdevy = DblArr(npoints)
Wstdevx = DblArr(npoints)
Wstdevy = DblArr(npoints)
meanX   = DblArr(npoints)
meanY   = DblArr(npoints)
WmeanX   = DblArr(npoints)
WmeanY   = DblArr(npoints)
fwhmX   = DblArr(npoints)
fwhmY   = DblArr(npoints)
WfwhmX   = DblArr(npoints)
WfwhmY   = DblArr(npoints)


yy = MakeArray1(npoints,range[0],range[1])

icounter=0L
tmpxx=0
tmpyy=0
totI0=Total(i0)
Print,'Total Intensity: ',totI0
Print,'Number of rays: ',N_Elements(i0)
IF nbins GT 0 THEN BEGIN
  hall = dblarr(5,double(npoints)*(nbins) ) ; added 2 bins just in case of resize
  hi = 0L
ENDIF
for i=0L,npoints-1 do begin
    tmp = retrace(shadow_out,dist=yy[i])
    tmpx = getshcol(tmp,col1)
    tmpy = getshcol(tmp,col2)

    mX = moment(tmpx)
    mY = moment(tmpy)
    meanX[i] = mX[0]
    meanY[i] = mY[0]
    stDevX[i] = Sqrt(mX[1])
    stDevY[i] = Sqrt(mY[1])
    WmeanX[i] = Total(i0*tmpx)/totI0
    WmeanY[i] = Total(i0*tmpy)/totI0
    wstDevX[i] = Sqrt( Total(i0*(tmpx-WmeanX[i])^2)/totI0 )
    wstDevY[i] = Sqrt( Total(i0*(tmpy-WmeanY[i])^2)/totI0 )

    ; histograms
    IF nbins GT 0 THEN BEGIN
      hx = histosh(tmpx,tmpx*0+1,nbins=nbins,xrange=xrange,/noshift)
      Wfwhmx[i]  = getfwhm(hx,/zeroBase)
      hy = histosh(tmpy,tmpx*0+1,nbins=nbins,xrange=yrange,/noshift)
      Wfwhmy[i] = getfwhm(hy,/zeroBase)

      hx = histosh(tmpx,i0,nbins=nbins,xrange=xrange,/noshift)
      fwhmx[i]  = getfwhm(hx,/zeroBase)
      hy = histosh(tmpy,i0,nbins=nbins,xrange=yrange,/noshift)
      fwhmy[i] = getfwhm(hy,/zeroBase)

      ; be careful that histosh can change the number of bins...
      nhx = N_Elements(hx[0,*])
      nhy = N_Elements(hy[0,*])
      hx=hx[*,0:(nbins<nhx)-1]
      hy=hy[*,0:(nbins<nhy)-1]
      nhx = N_Elements(hx[0,*])
      nhy = N_Elements(hy[0,*])
      FOR jj=0L,nbins-1 DO BEGIN
        hall[0,hi] = yy[i]
        hall[1,hi] = hx[0,jj<(nhx-1)]
        hall[2,hi] = hx[1,jj<(nhx-1)]
        hall[3,hi] = hy[0,jj<(nhy-1)]
        hall[4,hi] = hy[1,jj<(nhy-1)]
        hi=hi+1
      ENDFOR
    ENDIF

if not(keyword_set(noplot)) then begin
    if not(keyword_set(print)) then begin
      oplot,tmpx,tmpy,psym = 3

      IF keyword_set(mpeg_create) THEN BEGIN
        tmp = rotate(tvrd(),7)
        mpegindex=mpegindex+1
        MPEG_PUT, mpegID, IMAGE=tmp, FRAME=mpegindex
      ENDIF
      if icounter NE 0 then begin
        oplot,tmpxx,tmpyy, psym = 3,color = 0
        IF keyword_set(mpeg_create) THEN BEGIN
          tmp = rotate(tvrd(),7)
          mpegindex=mpegindex+1
          MPEG_PUT, mpegID, IMAGE=tmp, FRAME=mpegindex
        ENDIF
      endif
      wait,wait
    endif
endif
    tmpxx = tmpx  &  tmpyy=tmpy
    icounter = icounter+1
endfor

if not(keyword_set(noplot)) then begin
  if not(keyword_set(print)) then oplot,tmpxx,tmpyy, psym = 3,color = 0
  oplot,arr1,arr2,psym=3
endif

;
; plot stdev's
;
!p.position = [0.74,0.04,0.9,0.43]
off=0.1*abs(max(stdevx)-min(stdevx))
if off lt 1e-10 then off=0.5*max(stdevx)
plot,yy,stdevx,psym=1,/noerase,yrange=[min(stdevx)-off,max(stdevx+off)]
oplot,yy,stdevx
!p.position = [0.74,0.47,0.9,0.9]
off=0.1*abs(max(stdevy)-min(stdevy))
if off lt 1e-10 then off=0.5*max(stdevy)
plot,yy,stdevy,psym=1,/noerase,yrange=[min(stdevy)-off,max(stdevy)+off]
oplot,yy,stdevy
;
;
cd,current=pwd
if sdep() eq 'UNIX' then user=getenv('USER')+'@'+getenv('HOST') else user='' 
xyouts,.04,.96,/norm,pwd+sdep(/ds)+shadow_out.name+'  '+$
  '   '+user+'   '+systime()
xyouts,0.76,0.85,/norm,'StDev Vert: '
xyouts,0.76,0.83,/norm,column_name(col2)
xyouts,0.76,0.38,/norm,'StDev Horiz:'
xyouts,0.76,0.36,/norm,column_name(col1)
;
IF keyword_set(mpeg_create) THEN BEGIN
  tmp = rotate(tvrd(),7)
  MPEG_PUT, mpegID, IMAGE=tmp, FRAME=0
  MPEG_SAVE, mpegID, FILENAME='ray_prop.mpg'
  print,'RAY_PROP: File written to disk ray_prop.mpg'
ENDIF

IF N_Elements(iXplot) EQ 0 THEN iXplot=0

CASE nbins of
  0: BEGIN ; no histogram
     Evol = Make_Set(yy,meanX,meanY,stDevX,stdevY,wMeanX,wMeanY,wStDevX,wStDevY)
     colTitles=['dist [cm]','Mean H','Mean V','St Dev H','St Dev V',$
              'weighted Mean H','weighted Mean V', $
              'weighted StDev H','weighted StDev V']
     END
  else: BEGIN
     Evol = Make_Set(yy,meanX,meanY,stDevX,stdevY, $
                     wMeanX,wMeanY,wStDevX,wStDevY, $
                     fwhmX,fwhmY,wfwhmX,wfwhmY)
     colTitles=['dist [cm]','Mean H','Mean V','St Dev H','St Dev V',$
              'weighted Mean H','weighted Mean V', $
              'weighted StDev H','weighted StDev V',$
              'histo FWHM H','histo FWHM V',$
              'weighted histo FWHM H','weighted histo FWHM V']
     print,' '
     print,'RAY_PROP: histogram fwhm:'
     print,'      dist[cm]       FWHM_H      FWHM_H(weighted)     FWHM_V     FWHM_V(weighted) '
     for ii=0,iters-1 do print,yy[ii],fwhmX[ii],wfwhmX[ii],fwhmY[ii],wfwhmY[ii]
     print,' '
     END
ENDCASE

IF nbins GT 0 THEN BEGIN
  xplot,Parent=p,hall,colTitles=['Dist [cm]','H','Histo H','V','Histo V'],$
       xcol=4,ycol=5,YTitle='Distance [cm]',WTitle='Histograms'
  xplot_mesh,p,flag=1,kind=0,col=-1,interactive=0
  xplot_refresh,p
ENDIF

out:
!p.position = p_position_old
!p.charsize = p_charsize_old
IF iXplot EQ 1 THEN BEGIN
  Xplot,Evol,/No_Block,WTitle='Ray_Prop statistics',colTitles=colTitles
ENDIF

end
